﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/init.h>
#include <nn/os.h>

#include <nn/pcv/detail/pcv_Log.h>
#include <nn/pcv/driver/pcv.h>
#include <nn/pcv/server/pcv_ArbitrationServer.h>
#include <nn/pcv/pcv.h>
#include <nn/pcv/server/pcv_PcvHipcServer.h>

#include "pcv_PcvServer.h"

namespace nn { namespace pcv {

namespace {

const size_t PcvStackSize = 0x4000;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_PcvServerStack[PcvStackSize];
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_PcvImmediateServerStack[PcvStackSize];
nn::os::ThreadType g_PcvServerThread;
nn::os::ThreadType g_PcvImmediateServerThread;

void PcvHipcServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // PCV 用サーバーのループ処理
    NN_DETAIL_PCV_INFO("Start PCV Server.\n");
    nn::pcv::server::LoopPcvServer();
}

void PcvImmediateHipcServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // PCV 用サーバーのループ処理
    NN_DETAIL_PCV_INFO("Start PCV Immediate Server.\n");
    nn::pcv::server::LoopPcvImmediateServer();
}

}

void InitializePcvServer() NN_NOEXCEPT
{
    NN_DETAIL_PCV_INFO("Start PCV Arbitration Server.\n");

    // PCV の Clock And Reset の調停サービスの起動。
    nn::pcv::server::InitializeForArbitration();

    NN_DETAIL_PCV_INFO("Wait for arbitration.\n");

    // Clock And Reset を使用する全モジュールとの調停待ち。
    nn::pcv::server::WaitForArbitration();

    NN_DETAIL_PCV_INFO("Stop PCV Arbitration Server.\n");

    // PCV の Clock And Reset の調停サービスの終了。
    nn::pcv::server::FinalizeForArbitration();

    // Clock And Reset の調停はドライバライブラリの初期化前に完了しなければならない。

    // PCV ドライバライブラリの初期化。
    nn::pcv::driver::Initialize();

    // SD2 の電圧を 1.32 V に修正
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::SetVoltageValue(PowerDomain_Max77620_Sd2, 1325000));

    // (SIGLO-67830) 明示的な SD2 の有効化。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::SetVoltageEnabled(PowerDomain_Max77620_Sd2, true));

    // (SIGLO-79931) Hoag 向けの MCU 電源の初期化
    auto result = nn::pcv::driver::PowerOn(PowerControlTarget_SioMcuA, 2800000);
    if ( result.IsSuccess() )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(11000)); // SIGLO-79931: 11 ms
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::PowerOn(PowerControlTarget_SioMcu, 3100000));
    }
    else if ( ResultInvalidPowerControlTarget::Includes(result) )
    {
        // Iowa, Icosa, Copper ではサポートされていないので正常系とします
    }
    else
    {
        // ここを通る時点で UNLESS_RESULT_SUCCESS は確定だが、整形済みのエラーコード文字列が表示されるのがうれしいので
        // NN_ABORT_UNLESS_RESULT_SUCCESS を使用します
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // PCV 用サーバの初期化
    nn::pcv::server::InitializePcvServer();
}

void StartPcvServer() NN_NOEXCEPT
{
    auto result = nn::os::CreateThread(&g_PcvServerThread, &PcvHipcServerFunction, nullptr,
                                       g_PcvServerStack, sizeof(g_PcvServerStack),
                                       NN_SYSTEM_THREAD_PRIORITY(pcv, IpcServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_PcvServerThread, NN_SYSTEM_THREAD_NAME(pcv, IpcServer));

    nn::os::StartThread(&g_PcvServerThread);

    result = nn::os::CreateThread(&g_PcvImmediateServerThread, &PcvImmediateHipcServerFunction, nullptr,
                                   g_PcvImmediateServerStack, sizeof(g_PcvImmediateServerStack),
                                   NN_SYSTEM_THREAD_PRIORITY(pcv, ImmediateIpcServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_PcvImmediateServerThread, NN_SYSTEM_THREAD_NAME(pcv, ImmediateIpcServer));

    nn::os::StartThread(&g_PcvImmediateServerThread);
}

void RequestStopPcvServer() NN_NOEXCEPT
{
    // PCV サーバ終了をリクエスト
    nn::pcv::server::RequestStopPcvServer();
}

void WaitAndFinalizePcvServer() NN_NOEXCEPT
{
    // PCV 用サーバ処理スレッド終了待機と破棄
    nn::os::WaitThread(&g_PcvServerThread);
    nn::os::DestroyThread(&g_PcvServerThread);

    nn::os::WaitThread(&g_PcvImmediateServerThread);
    nn::os::DestroyThread(&g_PcvImmediateServerThread);

    // PCV サーバ自体の Finalize
    nn::pcv::driver::Finalize();

    // PCV サーバの終了処理
    nn::pcv::server::FinalizePcvServer();
}

}} // namespace nn::pcv
