﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/timesrv/timesrv_Api.h>

#include "pcv_BpcServer.h"
#include "pcv_PcvServer.h"

#define ENABLE_TIME

#ifdef ENABLE_TIME
#define ENABLE_FS_USE // time が fs を使う
#endif

#ifdef ENABLE_FS_USE
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/lmem/lmem_ExpHeap.h>
#endif

namespace nn { namespace pcv {

namespace {

#ifdef ENABLE_TIME
nn::os::ThreadType g_TimeThread;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_TimeServerStack[1024 * 4];
bool g_IsTimeThreadStarted = false;
#endif

#ifdef ENABLE_FS_USE
struct FsHeap
{
    static const size_t HeapSize = 20 * 1024;
    uint8_t heapBuffer[HeapSize];
    nn::lmem::HeapHandle heapHandle;

    void CreateHeap() NN_NOEXCEPT
    {
        this->heapHandle = nn::lmem::CreateExpHeap(
            this->heapBuffer, HeapSize, nn::lmem::CreationOption_NoOption);
        NN_ASSERT(this->heapHandle != nullptr, "[pcv] Failed to create fs heap.");
    }

    void DestroyHeap() NN_NOEXCEPT
    {
        nn::lmem::DestroyExpHeap(this->heapHandle);
    }

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        return nn::lmem::AllocateFromExpHeap(this->heapHandle, size);
    }

    void Deallocate(void* ptr, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        nn::lmem::FreeToExpHeap(this->heapHandle, ptr);
    }
};

FsHeap g_FsHeap;

#endif

void InitializeHipcServer() NN_NOEXCEPT
{
    nn::pcv::InitializeBpcServer();

    nn::pcv::StartBpcServer();

    nn::pcv::InitializePcvServer();

    nn::pcv::StartPcvServer();

#ifdef ENABLE_TIME
#if defined(NN_PCV_BUILD_TYPE_SAFE_MODE)
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
#endif

    nn::timesrv::Initialize();

    // Timeサービス用処理スレッドの作成/開始
    {
        // TODO:スレッド優先度の確認
        auto result = nn::os::CreateThread(&g_TimeThread,
                                           [] (void*) { nn::timesrv::LoopAutoTimeServer(); }, nullptr,
                                           g_TimeServerStack, sizeof(g_TimeServerStack), NN_SYSTEM_THREAD_PRIORITY(time, IpcServer));
        NN_ASSERT(result.IsSuccess()); // release 環境において time 要因で pcv プロセスが止まると被害が大きいので ABORT でなく ASSERT で.

        if(result.IsSuccess())
        {
            NN_SDK_LOG("[pcv] Start time server\n");
            nn::os::SetThreadNamePointer(&g_TimeThread, NN_SYSTEM_THREAD_NAME(time, IpcServer));
            nn::os::StartThread(&g_TimeThread);
            g_IsTimeThreadStarted = true;
        }
    }
#endif
}

#ifdef ENABLE_FS_USE
void* AllocateForFs(size_t size) NN_NOEXCEPT
{
    return g_FsHeap.Allocate(size);
}

void DeallocateForFs(void* ptr, size_t size) NN_NOEXCEPT
{
    g_FsHeap.Deallocate(ptr, size);
}
#endif

void FinalizeHipcServer() NN_NOEXCEPT
{

#ifdef ENABLE_TIME
    if(g_IsTimeThreadStarted)
    {
        // Time サーバー停止要求
        nn::timesrv::RequestStopTimeServer();
        nn::os::WaitThread(&g_TimeThread);
        nn::os::DestroyThread(&g_TimeThread);
    }
    nn::timesrv::Finalize();
#endif

    nn::pcv::RequestStopPcvServer();
    nn::pcv::WaitAndFinalizePcvServer();

    nn::pcv::RequestStopBpcServer();
    nn::pcv::WaitAndFinalizeBpcServer();
}

} // namespace

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
#if defined(NN_PCV_BUILD_TYPE_SAFE_MODE)
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
#endif

    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(pcv, Main));

#ifdef ENABLE_FS_USE
    nn::fs::InitializeWithMultiSessionForSystem();

    g_FsHeap.CreateHeap();
    nn::fs::SetAllocator(AllocateForFs, DeallocateForFs);
#endif

    InitializeHipcServer();

    // 終了イベント処理を実装するまでの暫定
    while ( NN_STATIC_CONDITION(true) )
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(100000));
    }

    // TORIAEZU:終了はしない
    NN_UNUSED(FinalizeHipcServer);

#ifdef ENABLE_FS_USE
    g_FsHeap.DestroyHeap();
#endif
}

}} // namespace nn::pcv
