﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_OlscServer.h"


#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/srv/olsc_Executor.h>
#include <nn/olsc/srv/olsc_ServiceNames.h>
#include <nn/olsc/srv/olsc_ServiceResource.h>
#include <nn/olsc/srv/util/olsc_Os.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcSimpleAllInOneServerManager.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace olsc { namespace srv {
namespace
{

// TODO: この辺は要調整
struct OlscServerOption
{
    static const size_t PointerTransferBufferSize = 0;
    static const bool CanDeferInvokeRequest = false;
    static const int SubDomainCountMax = 4;
    static const int ObjectInSubDomainCountMax = 64;
};

const int MaxSessionCount = 4;
const int MaxPortCount = 1;

class OlscServerManager : public nn::sf::HipcSimpleAllInOneServerManager<MaxSessionCount, MaxPortCount, OlscServerOption>
{};
const int OlscServerThreadStackSize = 16 * 1024;

NN_OS_ALIGNAS_THREAD_STACK char g_OlscServerThreadStack[OlscServerThreadStackSize];
nn::os::ThreadType g_OlscServerThread;

std::aligned_storage<sizeof(OlscServerManager), NN_ALIGNOF(OlscServerManager)>::type g_OlscServerStorage;
OlscServerManager* g_OlscServerManager = nullptr;
std::aligned_storage<sizeof(ServiceResource), NN_ALIGNOF(ServiceResource)>::type g_ServiceResourceStorage;
ServiceResource* g_ServiceResource = nullptr;


// ----------------------------------------------------------------------------
// Executorスレッド関連
// ----------------------------------------------------------------------------
const int OlscExecThreadStackSize = 16 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_OlscExecThreadStack[OlscExecThreadStackSize];
os::ThreadType g_OlscExecThread;
util::ManualClearEvent g_OlscExecThreadTerminator;

struct ExecutionContext
{
    olsc::srv::Executor* pExecutor;
    olsc::srv::ExecutionResource resource;
    olsc::srv::util::ManualClearEvent* pTerminator;
} g_ExecutionContext = {};

void ExecutionThreadFunction(void* p) NN_NOEXCEPT
{
    auto& context = *reinterpret_cast<ExecutionContext*>(p);
    auto& executor = *context.pExecutor;
    auto& resource = context.resource;
    auto& terminator = *context.pTerminator;

    resource.curlHandle = curl_easy_init();
    NN_ABORT_UNLESS_NOT_NULL(resource.curlHandle);
    NN_UTIL_SCOPE_EXIT{
        curl_easy_cleanup(resource.curlHandle);
        resource.curlHandle = nullptr;
    };
    util::ManualClearTimerEvent curlTimer;

    bool isAlive = true;
    while (isAlive)
    {
        switch (os::WaitAny(
            terminator.GetBase(),
            curlTimer.GetBase(),
            executor))
        {
        case 0: // terminator
            isAlive = false;
            break;

        case 1: // curlTimer
            curlTimer.Clear();
            curl_easy_cleanup(resource.curlHandle);
            resource.curlHandle = curl_easy_init();
            NN_ABORT_UNLESS_NOT_NULL(resource.curlHandle);
            break;

        case 2: // executor
            executor.ClearSignal();
            executor.Execute(resource);
            // curlHandle のキャッシュを、連続するリクエストの 10 秒後に破棄する。
            curlTimer.StartOneShot(TimeSpan::FromSeconds(10));
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

Result InitializeExecutorThread(olsc::srv::ServiceResource& resource) NN_NOEXCEPT
{
    g_ExecutionContext.pExecutor = &resource.GetExecutor();
    g_ExecutionContext.pTerminator = &g_OlscExecThreadTerminator;

    NN_RESULT_DO(os::CreateThread(
        &g_OlscExecThread,
        ExecutionThreadFunction, &g_ExecutionContext,
        g_OlscExecThreadStack, sizeof(g_OlscExecThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(olsc, DeferredExecution)));
    os::SetThreadNamePointer(
        &g_OlscExecThread,
        NN_SYSTEM_THREAD_NAME(olsc, DeferredExecution));

    os::StartThread(&g_OlscExecThread);

    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------------------------------------------------------
// サービスサーバ
// ---------------------------------------------------------------------------------------------------
nn::Result InitializeServerThread() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::os::CreateThread(&g_OlscServerThread, [](void* p) -> void {
        auto server = reinterpret_cast<OlscServerManager*>(p);
        server->LoopAuto();
    }, g_OlscServerManager, g_OlscServerThreadStack, OlscServerThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(olsc, IpcServer)));
    os::SetThreadName(&g_OlscServerThread, NN_SYSTEM_THREAD_NAME(olsc, IpcServer));

    os::StartThread(&g_OlscServerThread);

    NN_RESULT_SUCCESS;
}

void FinalizeServerThread() NN_NOEXCEPT
{
    g_OlscServerManager->RequestStop();
    os::DestroyThread(&g_OlscServerThread);
}

Result InitializeOlscServerManager() NN_NOEXCEPT
{
    g_ServiceResource = new (&g_ServiceResourceStorage) ServiceResource();
    g_ServiceResource->Initialize();

    g_OlscServerManager = new (&g_OlscServerStorage) OlscServerManager();

    NN_RESULT_DO(g_OlscServerManager->RegisterObjectForPort(
        g_ServiceResource->GetServiceForSystemService(),
        MaxSessionCount,
        ServiceNameForSystemService));

    g_OlscServerManager->Start();
    NN_RESULT_SUCCESS;
}

void FinalizeOlscServerManager() NN_NOEXCEPT
{
    g_OlscServerManager->~OlscServerManager();
    g_ServiceResource->~ServiceResource();
}

} // ~namespace nn::<anonymous>

nn::Result StartOlscServer() NN_NOEXCEPT
{
    NN_RESULT_DO(InitializeOlscServerManager());
    NN_RESULT_DO(InitializeExecutorThread(*g_ServiceResource));
    NN_RESULT_DO(InitializeServerThread());
    NN_RESULT_SUCCESS;
}

void StopOlscServer() NN_NOEXCEPT
{
    FinalizeServerThread();
    FinalizeOlscServerManager();
}

void InitializeOlscLibraryForDfc() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_ServiceResource);
    InitializeWithServiceObject(g_ServiceResource->GetServiceForSystemService());
}

}}} // ~namespace nn::olsc::srv
