﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/npns/detail/npns_Utility.h>
#include <nn/util/util_FormatString.h>
#include <nn/nifm/nifm_ApiProxy.h>
#include <nn/bgsu/bgsu_HeapTemplate.h>

#include <mutex>
#include <cstdlib>
#include <cstring>
#include <algorithm>

extern "C" {
#include <src/common.h>
}

#include "npns_Xmpp.h"
#include "npns_XmppStanza.h"
#include "npns_Common.h"

#if NN_BUILD_CONFIG_COMPILER_VC
#define strtoll _strtoi64
#define strtoull _strtoui64
#endif

namespace nn{ namespace npns{

namespace {
    const xmpp_log_level_t s_LogLevels[] = {
        XMPP_LEVEL_DEBUG, XMPP_LEVEL_INFO, XMPP_LEVEL_WARN, XMPP_LEVEL_ERROR
    };
#if NN_NPNS_ENABLE_LOG_XMPP
    const char * const s_LogLevelNames[4] = {
        "DEBUG", "INFO", "WARN", "ERROR"
    };
#endif
    void Logger(void * const userdata, const xmpp_log_level_t level,
            const char * const area, const char * const msg)
    {
#if NN_NPNS_ENABLE_LOG_XMPP
        xmpp_log_level_t filter_level = * static_cast<xmpp_log_level_t*>(userdata);
        if (level >= filter_level)
        {
            if (level >= XMPP_LEVEL_WARN)
            {
                NN_NPNS_WARN("%s %s %s\n", area, s_LogLevelNames[level], msg);
            }
            else
            {
                NN_NPNS_TRACE("%s %s %s\n", area, s_LogLevelNames[level], msg);
            }
        }
#endif
    }

    const xmpp_log_t s_Loggers[] = {
        { &Logger, (void*)&s_LogLevels[XMPP_LEVEL_DEBUG] },
        { &Logger, (void*)&s_LogLevels[XMPP_LEVEL_INFO]  },
        { &Logger, (void*)&s_LogLevels[XMPP_LEVEL_WARN]  },
        { &Logger, (void*)&s_LogLevels[XMPP_LEVEL_ERROR] }
    };

#if NN_NPNS_TEST_HEAP_XMPP
    struct XmppHeapTag;
    nn::bgsu::StaticHeapTemplate<XmppHeapTag, 128 * 1024, nn::mem::StandardAllocator> s_XmppHeap("xmpp");
    void *xmpp_mem_malloc(const size_t size, void * const userdata)
    {
        return s_XmppHeap.Allocate(size);
    }
    void xmpp_mem_free(void *p, void * const userdata)
    {
        s_XmppHeap.Free(p);
    }
    void *xmpp_mem_realloc(void *p, const size_t size, void * const userdata)
    {
        return s_XmppHeap.Reallocate(p, size);
    }
#else
    void *xmpp_mem_malloc(const size_t size, void * const userdata)
    {
        return malloc(size);
    }
    void xmpp_mem_free(void *p, void * const userdata)
    {
        free(p);
    }
    void *xmpp_mem_realloc(void *p, const size_t size, void * const userdata)
    {
        return realloc(p, size);
    }
#endif
    xmpp_mem_t s_MemoryConfig = {
        xmpp_mem_malloc, xmpp_mem_free, xmpp_mem_realloc, nullptr
    };
}

#define ENSURE_CONNECTED    do{\
        Result  _r = EnsureConnected(); \
        if (_r.IsFailure()) \
        { return _r; }  \
    }while (NN_STATIC_CONDITION(false))

#define THREAD_SAFE std::lock_guard<nn::os::Mutex> lock(m_mutex)

const char* Xmpp::PresenceAway = "away";
const char* Xmpp::PresenceChat = "chat";
const char* Xmpp::PresenceDoNotDisturb = "dnd";
const char* Xmpp::PresenceExtendedAway = "xa";

Xmpp::Xmpp(const char* pAgenaName, const char* pVersion)
    : m_mutex(true)
    , m_pContext(NULL)
    , m_pConnection(NULL)
    , m_pLog(NULL)
    , m_pAgentName(pAgenaName), m_pVersion(pVersion)
    , m_countAckRequiredMeesages(0)
    , m_bAuthFailureReceived(false)
{
}

void Xmpp::Initialize()
{
    THREAD_SAFE;

    NN_SDK_REQUIRES(m_pConnection == nullptr);
    NN_SDK_REQUIRES(m_pContext == nullptr);
#if NN_NPNS_TEST_HEAP_XMPP
    NN_SDK_REQUIRES(s_XmppHeap.IsClean());
#endif

    xmpp_initialize();

    m_bUsingProxy = false;
#if defined(NN_SDK_BUILD_DEBUG) && NN_NPNS_TEST_HEAP_XMPP == 1 && NN_NPNS_TEST_ALLOCATION_FAILURE == 1
    s_XmppHeap.EnableRandomFailure();
#endif
    /* create a context */
    m_pLog = &s_Loggers[XMPP_LEVEL_DEBUG];

    m_pContext = xmpp_ctx_new(&s_MemoryConfig, m_pLog);

    /* create a connection */
    m_pConnection = xmpp_conn_new(m_pContext);
}

void Xmpp::Finalize()
{
    THREAD_SAFE;

    NN_SDK_REQUIRES_NOT_NULL(m_pConnection);
    if (m_pConnection->state != XMPP_STATE_DISCONNECTED)
    {
        NN_NPNS_WARN("Connection state(%d) is not XMPP_STATE_DISCONNECTED. Force cleanup.\n", m_pConnection->state);
    }

    /* release our connection and context */
    xmpp_conn_release(m_pConnection);
    m_pConnection = nullptr;
    xmpp_ctx_free(m_pContext);
    m_pContext = nullptr;

    /* shutdown lib */
    xmpp_shutdown();

#if NN_NPNS_TEST_HEAP_XMPP
    if (!s_XmppHeap.IsClean())
    {
        s_XmppHeap.DumpAllocation();
        NN_SDK_ASSERT(s_XmppHeap.IsClean());
    }
#endif
}

void Xmpp::Reinitialize()
{
    THREAD_SAFE;

    Finalize();
    Initialize();
}

// public
Result Xmpp::Connect(const char* jid, const char* password)
{
    THREAD_SAFE;

    NN_SDK_REQUIRES_NOT_NULL(m_pConnection);
    NN_SDK_REQUIRES(m_pConnection->state == XMPP_STATE_DISCONNECTED);
    NN_SDK_REQUIRES(m_pConnection->tls == nullptr);

#if NN_NPNS_ENABLE_LOG_SOCKET_MEMORY
    static char statisticsBufffer[2048] = {};
    socket::GetResourceStatistics(socket::StatisticsType_Memory, statisticsBufffer, sizeof(statisticsBufffer), socket::StatisticsOption_None);
    NN_SDK_LOG("%s", statisticsBufffer);
#endif

    /* setup authentication information */
    xmpp_conn_set_jid(m_pConnection, jid);
    xmpp_conn_set_pass(m_pConnection, password);
    xmpp_conn_set_use_tls(m_pConnection, 1);
    m_pConnection->connect_timeout = NN_NPNS_TIMEOUT_XMPP * 1000;

#if NN_NPNS_TEST_USE_PROXY
    xmpp_conn_set_proxy(m_pConnection, NN_NPNS_TEST_PROXY_HOST, NN_NPNS_TEST_PROXY_PORT);
#if defined(NN_NPNS_TEST_PROXY_USER)
    xmpp_conn_set_proxy_auth(m_pConnection, NN_NPNS_TEST_PROXY_USER, NN_NPNS_TEST_PROXY_PASS);
#endif
#else // NN_NPNS_TEST_USE_PROXY
    {
        nifm::ProxySetting proxySetting;
        Result result = nifm::GetCurrentProxySetting(&proxySetting);
        if (result.IsSuccess())
        {
            if (proxySetting.isEnabled)
            {
                xmpp_conn_set_proxy(m_pConnection, proxySetting.proxy, proxySetting.port);
                if (proxySetting.authentication.isEnabled)
                {
                    xmpp_conn_set_proxy_auth(m_pConnection, proxySetting.authentication.username, proxySetting.authentication.password);
                }
            }
            else
            {
                xmpp_conn_set_proxy(m_pConnection, nullptr, 0);
            }
            m_bUsingProxy = proxySetting.isEnabled;
        }
        else
        {
            NN_NPNS_WARN("nifm::GetCurrentProxySetting failed.(%d)\n", result.GetInnerValueForDebug());
        }
    }
#endif // NN_NPNS_TEST_USE_PROXY

    // システムハンドラ登録
    m_bAuthFailureReceived = false;
    AddSystemHandler(AuthFailureHandlerEntry, "urn:ietf:params:xml:ns:xmpp-sasl", "failure", NULL);

    // SRV ルックアップを抑制するためにドメインを明示的に指定
    char* pDomain = xmpp_jid_domain(m_pContext, jid);
    int error = xmpp_connect_client(m_pConnection, pDomain, NN_NPNS_DEFAULT_PORT, ConnectionHandlerEntry, this);
    xmpp_free(m_pContext, pDomain);
    if (error < 0)
    {
        NN_NPNS_ERROR("xmpp_connect_client failed.(%d)\n", nn::socket::GetLastError());
        return ResultConnectionFailed();
    }

    m_pContext->loop_status = XMPP_LOOP_RUNNING;

    return ResultSuccess();
}

void Xmpp::Disconnect()
{
    OnBeforeDisconnect();

    THREAD_SAFE;
    NN_SDK_REQUIRES_NOT_NULL(m_pConnection);

    xmpp_disconnect(m_pConnection);

    Flush();
}

void Xmpp::RunOnce(unsigned long timeout)
{
    THREAD_SAFE;

    NN_SDK_ASSERT(!IsConnectionInProgress() || (IsConnectionInProgress() && m_pConnection->sock != -1));

    xmpp_run_once(m_pContext, timeout);
}

void Xmpp::Send(Stanza& stanza)
{
    THREAD_SAFE;

    stanza.Send(m_pConnection);
}

Result Xmpp::SendAndWait(Stanza* pReceiveStanza, Stanza& stanza, nn::os::Event* pEventInterrupt)
{
    //THREAD_SAFE;

    XmppAsyncManager::Cookie cookie;
    {
        char buffer[32];
        util::SNPrintf(buffer, sizeof(buffer), "%llu", cookie.GetId());
        stanza.SetId(buffer);
    }

    Result result;
    m_asyncManager.RegisterCookie(&cookie);
    {
        result = stanza.Send(m_pConnection);
        if (result.IsSuccess())
        {
            int ret = 0;
            if (pEventInterrupt)
            {
                ret = os::TimedWaitAny(nn::TimeSpan::FromSeconds(5),
                    cookie.GetEventBase(),          // 0
                    pEventInterrupt->GetBase());    // 1
            }
            else
            {
                // 0:成功, -1:タイムアウト
                ret = cookie.TimedWait(nn::TimeSpan::FromSeconds(5)) ? 0 : -1;
            }

            switch (ret)
            {
            case -1:
                result = ResultCommandTimeOut();
                break;
            case 0:
                result = cookie.GetResult();
                break;
            case 1:
                result = ResultInterruptByRequest();
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
    if (pReceiveStanza)
    {
        *pReceiveStanza = cookie.GetResultUserData();
    }
    m_asyncManager.UnregisterCookie(&cookie);
    return result;
}

void Xmpp::SendRawString(const char* pString)
{
    THREAD_SAFE;

    xmpp_send_raw_string(m_pConnection, pString);
}

void Xmpp::Flush()
{
    OnFlush();
}

void Xmpp::SetPresenceWithStatus(const char* pShowString, const char* pStatusString)
{
    NN_SDK_REQUIRES_NOT_NULL(pShowString);

    Stanza stanzaPresence(m_pContext, "presence");

    StanzaText stanzaShow(m_pContext, "show", pShowString);
    stanzaPresence.AddChild(stanzaShow);

    if (pStatusString)
    {
        StanzaText stanzaStatus(m_pContext, "status", pStatusString);
        stanzaPresence.AddChild(stanzaStatus);
    }

    Send(stanzaPresence);
}

void Xmpp::EnableStreamManagement()
{
    m_countAckRequiredMeesages = 0;
    SendRawString("<enable xmlns='urn:xmpp:sm:3'/>");
}

void Xmpp::MakePubsubStanza(Stanza& stanzaIq, Stanza& stanzaChild)
{
    Stanza stanzaPubsub(m_pContext, "pubsub", "http://jabber.org/protocol/pubsub");
    stanzaPubsub.AddChild(stanzaChild);

    char toBuffer[128];
    xmpp_snprintf(toBuffer, sizeof(toBuffer), "pubsub.%s", GetConnectedDomain());
    stanzaIq.SetAttribute("to", toBuffer);
    stanzaIq.AddChild(stanzaPubsub);
}

Result Xmpp::PubsubSubscribe(const char* pNodeName)
{
    ENSURE_CONNECTED;

    Stanza stanzaIq(m_pContext, "iq", "jabber:client");
    stanzaIq.SetType("set");

    Stanza stanzaSubscribe(m_pContext, "subscribe");
    stanzaSubscribe.SetAttribute("node", pNodeName);
    stanzaSubscribe.SetAttribute("jid", GetConnectedJid());

    MakePubsubStanza(stanzaIq, stanzaSubscribe);

    Result result = SendAndWait(stanzaIq);
    return result;
}

Result Xmpp::PubsubUnsubscribe(const char* pNodeName)
{
    ENSURE_CONNECTED;

    Stanza stanzaIq(m_pContext, "iq", "jabber:client");
    stanzaIq.SetType("set");

    Stanza stanzaUnsubscribe(m_pContext, "unsubscribe");
    stanzaUnsubscribe.SetAttribute("node", pNodeName);
    stanzaUnsubscribe.SetAttribute("jid", GetConnectedJid());

    MakePubsubStanza(stanzaIq, stanzaUnsubscribe);

    Result result = SendAndWait(stanzaIq);
    return result;
}

Result Xmpp::PubsubGet(const char* pNodeName, int32_t maxItems)
{
    ENSURE_CONNECTED;

    Stanza stanzaIq(m_pContext, "iq", "jabber:client");
    stanzaIq.SetType("get");

    Stanza stanzaItems(m_pContext, "items");
    stanzaItems.SetAttribute("node", pNodeName);
    stanzaItems.SetAttribute("max_items", maxItems);

    MakePubsubStanza(stanzaIq, stanzaItems);

    Result result = SendAndWait(stanzaIq);
    return result;
}

bool Xmpp::IsConnected() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pConnection);
    return m_pConnection->state == XMPP_STATE_CONNECTED && m_pConnection->authenticated;
}

bool Xmpp::IsConnectionInProgress() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pConnection);
    return (m_pConnection->state == XMPP_STATE_CONNECTING || m_pConnection->state == XMPP_STATE_CONNECTING_PROXY) && !m_pConnection->authenticated;
}

bool Xmpp::HasPendingData() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pConnection);
    return IsConnected() && m_pConnection->send_queue_head;
}

// helper
Result Xmpp::EnsureConnected()
{
    if (!IsConnected())
    {
        return ResultNotConnected();
    }
    return ResultSuccess();
}

Result Xmpp::GetLastError() const
{
    if (m_pConnection->tls_failed)
    {
        NN_NPNS_ERROR("TLS failed %d.\n", m_pConnection->tls_error);
        return ResultConnectionFailed();
    }
    if (m_pConnection->error != 0)
    {
        return ConvertErrorToResult(m_pConnection->error);
    }

    if (m_pConnection->authenticated == 0)
    {
        if (m_bAuthFailureReceived)
        {
            NN_NPNS_ERROR("Authentication failed.\n");
            return ResultAuthenticationFailed();
        }
        else
        {
            NN_NPNS_ERROR("Connection failed at authentication.\n");
            return ResultConnectionFailed();
        }
    }

    if (m_pConnection->state == XMPP_STATE_DISCONNECTED)
    {
        return ResultDisconnected();
    }
    return ResultSuccess();
}

bool Xmpp::IsUsingProxy() const
{
    return m_bUsingProxy;
}

int Xmpp::GetSocket() const
{
    if (m_pConnection && m_pConnection->state != XMPP_STATE_DISCONNECTED)
    {
        return m_pConnection->sock;
    }
    else
    {
        return -1;
    }
}

void Xmpp::ActivateKeepAlive()
{
    if (!IsConnected())
    {
        return;
    }

    int ret = SetTcpKeepAlive(true, NN_NPNS_TCP_KEEP_ALIVE_IDLE, NN_NPNS_TCP_KEEP_ALIVE_INTERVAL, NN_NPNS_TCP_KEEP_ALIVE_COUNT);
    if (ret != 0)
    {
        NN_NPNS_WARN("Xmpp::SetTcpKeepAlive failed. ret(%d)\n", ret);
        return;
    }
    NN_NPNS_TRACE("TCP KeepAlive is activated.\n");
}

void Xmpp::DeactivateKeepAlive()
{
    if (!IsConnected())
    {
        return;
    }

    // Keep-Alive のタイマを無効化したいができないので、失敗回数の制限をゆるくしておく
    int ret = SetTcpKeepAlive(false, NN_NPNS_TCP_KEEP_ALIVE_IDLE, NN_NPNS_TCP_KEEP_ALIVE_INTERVAL, 24 * 60);
    if (ret != 0)
    {
        NN_NPNS_WARN("Xmpp::SetTcpKeepAlive failed. ret(%d)\n", ret);
        return;
    }
    NN_NPNS_TRACE("TCP KeepAlive is deactivated.\n");
}

Result Xmpp::ConvertErrorToResult(int error)
{
    switch (error)
    {
    case ECONNABORTED:
        return ResultNotConnected();
    case ETIMEDOUT:
        return ResultConnectionTimeOut();
    case ECONNRESET:
        return ResultNotConnected();
    default:
        break;
    }
    return ResultUnknown();
}

Result Xmpp::ConvertErrorStanzaToResult(const Stanza & stanzaError)
{
    static const struct XmppErrorAndResultPair
    {
        const char* pType;
        Result result;
    }
    arrayResultMap[] =
    {
        { "bad-format",             ResultXmppBadFormat() },
        { "bad-request",            ResultXmppBadRequest() },
        { "conflict",               ResultXmppConflict() },
        { "feature-not-implemented",ResultXmppFeatureNotImplemented() },
        { "forbidden",              ResultXmppForbidden() },
        { "gone",                   ResultXmppGone() },
        { "internal-server-error",  ResultXmppInternalServerError() },
        { "item-not-found",         ResultXmppItemNotFound() },
        { "jid-malformed",          ResultXmppJidMalformed() },
        { "not-acceptable",         ResultXmppNotAcceptable() },
        { "not-allowed",            ResultXmppNotAllowed() },
        { "not-authorized",         ResultXmppNotAuthorized() },
        { "payment-required",       ResultXmppPaymentRequired() },
        { "recipient-unavailable",  ResultXmppRecipientUnavailable() },
        { "redirect",               ResultXmppRedirect() },
        { "registration-required",  ResultXmppRegistrationRequired() },
        { "remote-server-not-found",ResultXmppRemoteServerNotFound() },
        { "remote-server-timeout",  ResultXmppRemoteServerTimeout() },
        { "resource-constraint",    ResultXmppResourceConstraint() },
        { "service-unavailable",    ResultXmppServiceUnavailable() },
        { "subscription-required",  ResultXmppSubscriptionRequired() },
        { "unexpected-request",     ResultXmppUnexpectedRequest() },
    };

    const char* pStanzaName = stanzaError.GetName();
    for (int32_t i = 0; i < sizeof(arrayResultMap) / sizeof(arrayResultMap[0]); ++i)
    {
        const XmppErrorAndResultPair& pair = arrayResultMap[i];
        if (std::strcmp(pair.pType, pStanzaName) == 0)
        {
            return pair.result;
        }
    }

    return ResultXmppUnknown();
}

bool Xmpp::IsRunning() const
{
    return m_pContext && m_pContext->loop_status == XMPP_LOOP_RUNNING;
}

const char* Xmpp::GetConnectedDomain() const
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConnection->connectdomain);
    return m_pConnection->connectdomain;
}

const char* Xmpp::GetConnectedJid() const
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConnection->jid);
    return m_pConnection->jid;
}

xmpp_ctx_t* Xmpp::GetContext() const
{
    return m_pContext;
}

void Xmpp::AddHandler(xmpp_handler handler, const char * const ns, const char * const name, const char * const type)
{
    THREAD_SAFE;

    xmpp_handler_add(m_pConnection, handler, ns, name, type, this);
}

void Xmpp::AddSystemHandler(xmpp_handler handler, const char * const ns, const char * const name, const char * const type)
{
    THREAD_SAFE;

    handler_add(m_pConnection, handler, ns, name, type, this);
}

void Xmpp::AddUserHandler(xmpp_handler handler, const char * const ns, const char * const name, const char * const type, void * const userdata)
{
    THREAD_SAFE;

    xmpp_handler_add(m_pConnection, handler, ns, name, type, userdata);
}

void Xmpp::DeleteHandler(xmpp_handler handler)
{
    THREAD_SAFE;

    xmpp_handler_delete(m_pConnection, handler);
}

// callback
void Xmpp::ConnectionHandlerEntry(xmpp_conn_t * const, const xmpp_conn_event_t status,
          const int error, xmpp_stream_error_t * const stream_error,
          void * const userdata)
{
    static_cast<Xmpp*>(userdata)->OnChangeConnectionStatus(status, error, stream_error);
}

int Xmpp::VersionHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata)
{
    Stanza stanzaIncoming(stanza);
    static_cast<Xmpp*>(userdata)->OnReceiveVersion(stanzaIncoming);
    return HandlerResult_NotOneShot;
}

int Xmpp::MessageHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata)
{
    Stanza stanzaIncoming(stanza);
    static_cast<Xmpp*>(userdata)->OnReceiveMessage(stanzaIncoming);
    return HandlerResult_NotOneShot;
}

int Xmpp::ResultHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata)
{
    Stanza stanzaIncoming(stanza);
    static_cast<Xmpp*>(userdata)->OnReceiveResultAndError(stanzaIncoming);
    return HandlerResult_NotOneShot;
}

int Xmpp::ErrorHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata)
{
    Stanza stanzaIncoming(stanza);
    static_cast<Xmpp*>(userdata)->OnReceiveResultAndError(stanzaIncoming);
    return HandlerResult_NotOneShot;
}

int Xmpp::StreamManagementHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const, void * const userdata)
{
    static_cast<Xmpp*>(userdata)->OnAckNeeded();
    return HandlerResult_NotOneShot;
}

int Xmpp::AuthFailureHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata)
{
    Stanza stanzaIncoming(stanza);
    static_cast<Xmpp*>(userdata)->OnReceiveAuthFailure(stanzaIncoming);
    return HandlerResult_NotOneShot;
}

void Xmpp::OnChangeConnectionStatus(const xmpp_conn_event_t status,
        const int error, xmpp_stream_error_t * const stream_error)
{
    if (status == XMPP_CONN_CONNECT)
    {
        NN_NPNS_INFO("Connected to XMPP server.\n");
        AddHandler(VersionHandlerEntry, "jabber:iq:version", "iq", NULL);
        AddHandler(MessageHandlerEntry, NULL, "message", NULL);
        AddHandler(ResultHandlerEntry, "jabber:client", "iq", "result");
        AddHandler(ErrorHandlerEntry, "jabber:client", "iq", "error");
        AddHandler(StreamManagementHandlerEntry, "urn:xmpp:sm:3", "r", NULL);

        OnConnected();
    }
    else
    {
        if (error == 0)
        {
            NN_NPNS_INFO("Disconnected from XMPP server.\n");
        }
        else
        {
            NN_NPNS_WARN("Disconnected from XMPP server. (stream error=%d)\n", error);
        }
        xmpp_stop(m_pContext);

        if (stream_error && stream_error->stanza)
        {
            Stanza stanzaIncoming(stream_error->stanza);
            OnReceiveDisconnectError(stanzaIncoming);
        }

        OnDisconnected();
    }
}

void Xmpp::OnConnected()
{
    /* Send initial <presence/> so that we appear online to contacts */
    SetPresence("chat");

    EnableStreamManagement();
}

void Xmpp::OnDisconnected()
{
    // 非同期タスクをすべて終わらせる
    m_asyncManager.NotifyResultToAll(ResultDisconnected());
}

void Xmpp::OnBeforeDisconnect()
{
    DeleteHandler(StreamManagementHandlerEntry);
}

void Xmpp::OnFlush()
{
}

void Xmpp::OnReceiveVersion(const Stanza& stanzaIncoming)
{
    const char *ns;

    Stanza stanzaReply(m_pContext, "iq");
    stanzaReply.SetType("result");
    stanzaReply.SetId(stanzaIncoming.GetId());
    stanzaReply.SetAttribute("to", stanzaIncoming.GetAttribute("from"));

    Stanza stanzaQuery(m_pContext, "query");
    Stanza stanzaChild;
    if (stanzaIncoming.GetChildren(stanzaChild))
    {
        ns = stanzaChild.GetNs();
        if (ns) {
            stanzaQuery.SetNs(ns);
        }
    }

    {
        StanzaText stanzaName(m_pContext, "name", m_pAgentName);
        stanzaQuery.AddChild(stanzaName);

        StanzaText stanzaVersion(m_pContext, "version", m_pVersion);
        stanzaQuery.AddChild(stanzaVersion);
    }
    stanzaReply.AddChild(stanzaQuery);

    Send(stanzaReply);
}

void Xmpp::OnReceiveMessage(const Stanza& stanzaIncoming)
{

    if (stanzaIncoming.IsTypeError())
    {
        NN_NPNS_WARN("Ignored error message.\n");
        return;
    }

    // 個別
    {
        Stanza stanzaBody;
        if (stanzaIncoming.GetChildByName("body", stanzaBody))
        {
            Text text;
            if (stanzaBody.GetText(text))
            {
                OnReceiveMessageBody(text);
            }
            else
            {
                NN_NPNS_WARN("Failed to get body.\n");
            }
            return;
        }
    }

    // トピック
    {
        Stanza stanzaEvent, stanzaItems, stanzaItem;
        if (stanzaIncoming.GetChildByName("event", stanzaEvent)
            && stanzaEvent.GetChildByName("items", stanzaItems)
            && stanzaItems.GetChildren(stanzaItem))
        {
            do
            {
                Stanza stanzaEntry;
                if (stanzaItem.GetChildByName("entry", stanzaEntry))
                {
                    Text text;
                    if (stanzaEntry.GetText(text))
                    {
                        OnReceiveMessageBody(text);
                    }
                    else
                    {
                        NN_NPNS_WARN("Failed to get body.\n");
                    }
                }
            } while (stanzaItem.GetNext());
            return;
        }
    }
    NN_NPNS_WARN("Failed to parse a message.\n");
}

void Xmpp::OnReceiveResultAndError(const Stanza& stanzaIncoming)
{
    const char* pIdString = stanzaIncoming.GetId();
    if (!pIdString)
    {
        return;
    }

    uint64_t id = strtoull(pIdString, 0, 10);
    if (id == 0)
    {
        return;
    }

    bool found;
    if (stanzaIncoming.IsTypeError())
    {
        Result result;
        Stanza stanzaError, stanzaErrorDetail;
        if (stanzaIncoming.GetChildByName("error", stanzaError)
            && stanzaError.GetChildren(stanzaErrorDetail))
        {
            result = ConvertErrorStanzaToResult(stanzaErrorDetail);
        }
        else
        {
            result = ResultUnexpected();
        }
        found = m_asyncManager.NotifyResult(id, result);
    }
    else
    {
        Stanza stanzaChild;
        if (stanzaIncoming.GetChildren(stanzaChild))
        {
            found = m_asyncManager.NotifyResult(id, ResultSuccess(), stanzaChild);
        }
        else
        {
            found = m_asyncManager.NotifyResult(id, ResultSuccess());
        }
    }

    if (!found)
    {
        NN_NPNS_WARN("No cookie found for id=%s. (timeout?)\n", stanzaIncoming.GetId());
    }
}

void Xmpp::OnReceiveDisconnectError(const Stanza&)
{
}

void Xmpp::OnReceiveAuthFailure(const Stanza&)
{
    m_bAuthFailureReceived = true;
}

void Xmpp::OnAckNeeded()
{
    char buffer[64];
    util::SNPrintf(buffer, sizeof(buffer), "<a xmlns='urn:xmpp:sm:3' h='%ld'/>", ++m_countAckRequiredMeesages);
    SendRawString(buffer);
}

void Xmpp::OnReceiveMessageBody(const Text& text)
{
    return;
}

int Xmpp::SetTcpKeepAlive(bool enable, int idle, int interval, int count) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pConnection);

#ifndef NN_BUILD_CONFIG_OS_WIN
    int sock = m_pConnection->sock;
    int ret, optval;

    // SO_KEEPALIVE を変更してもタイマのタイミングは変わらないので先にパラメータをセットしておく
    ret = nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle, (void*)&idle, sizeof(idle));
    if (ret != 0) {
        return ret;
    }

    ret = nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl, (void*)&interval, sizeof(interval));
    if (ret != 0) {
        return ret;
    }

    ret = nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt, (void*)&count, sizeof(count));
    if (ret != 0) {
        return ret;
    }

    // NOTE: グローバル設定の always_keepalive が採用されるので実質無意味
    optval = enable ? 1 : 0;
    ret = nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_KeepAlive, (void*)&optval, sizeof(optval));
    if (ret != 0) {
        return ret;
    }
#else
    NN_UNUSED(idle); NN_UNUSED(interval); NN_UNUSED(count);
#endif
    return 0;
}

int Xmpp::SetTcpNoDelay(bool bValid) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pConnection);

#ifndef NN_BUILD_CONFIG_OS_WIN
    int sock = m_pConnection->sock;
    int ret;
    int optval = bValid ? 1 : 0;
    ret = nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, &optval, sizeof(optval));
    if (ret != 0)
    {
        NN_NPNS_ERROR("Failed to Xmpp::SetTcpNoDelay(%d) %d\n", bValid, ret);
        return ret;
    }
#else
    NN_UNUSED(bValid);
#endif
    return 0;
}

}}
