﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include "npns_Common.h"

namespace nn {
namespace npns {

template<typename StateT>
class StateMachineTemplate
{
public:
    StateMachineTemplate()
    {
    }
    virtual ~StateMachineTemplate() {}

    Result Initialize()
    {
        m_stateCurrent = GetInitialState();
        m_stateTarget  = GetInitialTargetState();
        return ResultSuccess();
    }

    virtual void Finalize()
    {
    }

    void SetTargetState(StateT stateTarget);
    StateT GetCurrentState() const;
    StateT GetTargetState() const;

    Result TransitState();

    bool IsTransitionComplete() const;

    virtual StateT GetInitialState() const = 0;
    virtual StateT GetInitialTargetState() const = 0;
    virtual const char* GetStateString(StateT) const
    {
        return "(unknown)";
    };

protected:
    virtual bool IsTargetState(StateT) const { return true; }
    virtual Result OnTransit() = 0;
    virtual Result OnEntering(StateT) = 0;
    virtual Result OnLeaving(StateT, StateT) { return ResultSuccess(); }
    virtual void OnEntered(StateT) {}
    virtual void OnTransitionComplete() {}
    virtual bool OnTransitionFailure(Result) { return false; }
    Result SetCurrentState(StateT StateTo);

private:
    StateT m_stateCurrent;
    StateT m_stateTarget;
};

template<typename StateT>
void StateMachineTemplate<StateT>::SetTargetState(StateT stateTarget)
{
    NN_SDK_ASSERT(IsTargetState(stateTarget), "[%s] is not target state.", GetStateString(stateTarget));

    if (m_stateTarget != stateTarget)
    {
        m_stateTarget = stateTarget;

        // m_stateTarget の変更の結果、
        // ステート遷移完了になることがあるのでハンドラを呼び出し
        if (IsTransitionComplete())
        {
            OnTransitionComplete();
        }
    }
}

template<typename StateT>
inline
StateT StateMachineTemplate<StateT>::GetCurrentState() const
{
    return m_stateCurrent;
}

template<typename StateT>
inline
StateT StateMachineTemplate<StateT>::GetTargetState() const
{
    NN_SDK_ASSERT(IsTargetState(m_stateTarget), "[%s] is not target state.", GetStateString(m_stateTarget));
    return m_stateTarget;
}

template<typename StateT>
Result StateMachineTemplate<StateT>::SetCurrentState(StateT state)
{
    Result result;

    // 移行先ステートが現在と違うステートであることをチェック
    if (m_stateCurrent != state)
    {
        // ハンドラの呼び出し
        result = OnLeaving(m_stateCurrent, state);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

        result = OnEntering(state);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

        // ステートを更新
        m_stateCurrent = state;

        // ステート変更ハンドラの呼び出し
        OnEntered(m_stateCurrent);

        // ステート遷移完了のハンドラを呼び出し
        if (IsTransitionComplete())
        {
            OnTransitionComplete();
        }
    }
    return ResultSuccess();
}

template<typename StateT>
inline bool StateMachineTemplate<StateT>::IsTransitionComplete() const
{
    return m_stateTarget == m_stateCurrent;
}

template<typename StateT>
Result StateMachineTemplate<StateT>::TransitState()
{
    Result result;

    result = OnTransit();
    if (result.IsFailure())
    {
        OnTransitionFailure(result);
    }
    return result;
}

}
}

