﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/npns/detail/npns_InternalTypes.h>

#include "npns_StateMachineTemplate.h"
#include "npns_Controller.h"

namespace nn {
namespace npns {

class StateMachine : public StateMachineTemplate<nn::npns::State>
{
public:
    // 型の定義
    typedef StateMachineTemplate<State> StateMachineBase;

    // コンストラクタ・デストラクタ
    StateMachine(Controller& controller, nn::os::Event& eventInterrupt);
    virtual ~StateMachine();

    // 最低限必要な初期化処理
    Result Initialize();
    // 最後の終了処理
    virtual void Finalize();

    bool IsInProgress() const;
    State GetTargetStateOfLastFailure() const;

    virtual State GetInitialState() const { return State_Initial; }
    virtual State GetInitialTargetState() const { return State_Suspend; }

    void SetTargetState(State stateTarget, bool bForce);
    virtual const char* GetStateString(State state) const;

    Result WaitAnyEvents();
    Result WaitAnyEvents(const TimeSpan& time);

    bool SetAutonomyEnabled(bool bEnable);
    bool IsAutonomyEnabled() const;

protected:
    virtual Result OnEntering(State stateTo);
    virtual Result OnLeaving(State stateFrom, State stateTo);
    virtual Result OnTransit();
    virtual void OnTransitionComplete();
    virtual bool OnTransitionFailure(Result result);
    virtual bool IsTargetState(State state) const;

    bool IsUserDriven() const;
    void SetTargetStateToBackoffWaiting(TimeSpan& time);
    bool IsBackgroundProcessingEnabled();
    bool IsEulaAccepted();
    TimeSpan GetSleepProcessingTimeout();

private:
    Controller&         m_Controller;
    os::Event&          m_eventInterrupt;
    TimeSpan            m_BackoffWait;
    State               m_stateLastTarget;
    int32_t             m_SerialErrorCount;
    bool                m_bUserDriven;
    bool                m_bError;
    bool                m_bAutonomy;
    bool                m_bEulaAcceptedOnce;

    void CalculateBackoffWait();
};

inline bool StateMachine::IsInProgress() const
{
    return !IsTransitionComplete();
}

inline bool StateMachine::IsUserDriven() const
{
    return m_bUserDriven;
}

inline State StateMachine::GetTargetStateOfLastFailure() const
{
    return m_stateLastTarget;
}

inline bool StateMachine::IsAutonomyEnabled() const
{
    return m_bAutonomy;
}

}
}

