﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_FormatString.h>
#include <nn/http.h>
#include <nn/http/http_JsonResponse.h>
#include "npns_Common.h"
#include "npns_ResourceConsumerBase.h"
#include "npns_DeviceTokenHolder.h"

namespace nn {
namespace npns {

const char* ResourceConsumerBase::ProductCode = "HAC";

ResourceConsumerBase::ResourceConsumerBase(http::ConnectionBroker& cb, DeviceTokenHolder& dth)
    : m_ConnectionBroker(cb)
    , m_DeviceTokenHolder(dth)
{
}

ResourceConsumerBase::~ResourceConsumerBase()
{
}

Result ResourceConsumerBase::SetupRequestHeaders(http::Request * pRequest)
{
    Result result;

    pRequest->SetFormatBufferSize(2048);
    pRequest->SetKeepAlive(false);

    result = pRequest->AddHeaderFormat("X-DeviceAuthToken: %s", m_DeviceTokenHolder.GetStorage().value);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = pRequest->AddHeaderFormat("X-ProductCode: %s", "HAC");
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return ResultSuccess();
}

Result ResourceConsumerBase::SendRequest(http::Request * pRequest, const http::JsonResponse::ParseRule parseRules[], const int64_t& resultCode)
{
    Result result;
    char JsonBuffer[64];
    http::JsonResponse res(JsonBuffer, sizeof(JsonBuffer));

    result = pRequest->GetResponse(&res);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = res.ReadAndParseWithRule(parseRules);

    // JSON パース失敗
    if (http::ResultJsonParseError::Includes(result)
        && (http::ResultHttpStatusClientError::Includes(res.GetStatusCodeAsResult())
            || http::ResultHttpStatusServerError::Includes(res.GetStatusCodeAsResult())))
    {
        NN_NPNS_WARN("Failed to parse server response(0x%08x). Server returns HTTP status %d\n", result.GetInnerValueForDebug(), res.GetStatusCode());
        return res.GetStatusCodeAsResult();
    }
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    // JSON パース成功時、ハンドリングしたい resultCode を個別にハンドリング
    if (resultCode == ResultCode_Success)
    {
        return ResultSuccess();
    }
    else if (ResultCodeRange_DeviceTokenError_Begin <= resultCode &&
             resultCode < ResultCodeRange_DeviceTokenError_End) // resultCode 10xx は DeviceToken 作り直し
    {
        m_DeviceTokenHolder.RequestRefresh(); // 次回トークン取得時にキャッシュを無視する
        return ResultDeviceTokenError();
    }

    // 上記以外の resultCode は StatusCode に従って Result を返す.
    // 必要なら呼び出し元で resultCode がハンドリングされる.
    if (res.GetStatusCode() != 200)
    {
        NN_NPNS_WARN("Server returns HTTP status %d\n", res.GetStatusCode());
        return res.GetStatusCodeAsResult();
    }

    return ResultSuccess();
}

}
}
