﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/npns/npns_INpns.sfdl.h>
#include <nn/npns/detail/npns_HipcPorts.h>

#include "npns_Common.h"
#include "npns_HipcServer.h"
#include "npns_ServiceImpl.h"

namespace nn { namespace npns {
namespace {

//-----------------------------------------------------------------------------
// 拡張ヒープを使用したアロケータを、HIPC プロキシ用に準備
struct  HipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, HipcTag> Allocator;

}   // namespace

IpcServerManager::IpcServerManager()
    : m_DeferredWorkerThread(*this)
#if NN_NPNS_SEPARATE_IPC_THREAD
    , m_IpcLoopThread(*this)
#endif
{
    Allocator::Initialize(nn::lmem::CreationOption_NoOption);
}

IpcServerManager::~IpcServerManager()
{
}

Result IpcServerManager::Initialize()
{
    Result result;

    for (int i = 0; i < PortIndex_Count; ++i)
    {
        result = InitializePort(i, SessionCountPerPort, PortNames[i]);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
    }

    result = m_DeferredWorkerThread.Initialize();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

#if NN_NPNS_SEPARATE_IPC_THREAD
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_IpcLoopThread.Initialize()
    );
#endif
    Start();
    return ResultSuccess();
}

void IpcServerManager::Finalize()
{
    RequestStop();
#if NN_NPNS_SEPARATE_IPC_THREAD
    m_IpcLoopThread.Finalize();
#endif
    m_DeferredWorkerThread.Finalize();
}

bool IpcServerManager::IsWorkerThreadContext() const
{
    return m_DeferredWorkerThread.IsWorkerThreadContext();
}

Result IpcServerManager::LoopAutoWithDefer()
{
    Result result;
    while (auto p = Wait())
    {
        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        case InvokeTag:
            {
                result = ProcessInvokeRequestWithDefer(p);
                if( result <= nn::sf::ResultProcessDeferred() )
                {
                    m_DeferredWorkerThread.DeferTask(p);
                    break;
                }
                NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
            }
            break;

        case AcceptTag:
            {
                result = ProcessAcceptRequest(p);
                NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    return ResultSuccess();
}

nn::Result IpcServerManager::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
{
    switch (portIndex)
    {
    case PortIndex_User:
    {
        sf::SharedPointer<INpnsUser>   pInterface = sf::ObjectFactory<Allocator::Policy>::CreateSharedEmplaced<INpnsUser,   ServiceImpl>();
        if (pInterface == nullptr)
        {
            return ResultOutOfMemory();
        }
        return AcceptImpl(pPort, pInterface);
    }
    case PortIndex_System:
    {
        sf::SharedPointer<INpnsSystem> pInterface = sf::ObjectFactory<Allocator::Policy>::CreateSharedEmplaced<INpnsSystem, ServiceImpl>();
        if (pInterface == nullptr)
        {
            return ResultOutOfMemory();
        }
        return AcceptImpl(pPort, pInterface);
    }
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#if NN_NPNS_SEPARATE_IPC_THREAD
void IpcServerManager::IpcLoopThread::ThreadBody()
{
    m_IpcServerManager.LoopAutoWithDefer();
}
#endif

}}
