﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/nn_Result.h>

#include <cstdlib>
#include <cstring>
#include <algorithm>
#include <mutex>

namespace nn{ namespace npns{

template<typename T>
class AsyncManager
{
    NN_DISALLOW_COPY( AsyncManager );
    NN_DISALLOW_MOVE( AsyncManager );

public:
    typedef uint64_t Id;
    class Cookie : public nn::util::IntrusiveListBaseNode<Cookie>
    {
    public:
        Cookie()
            : m_event(nn::os::EventClearMode_AutoClear)
            , m_result(ResultSuccess())
        {
            m_id = static_cast<Id>(AsyncManager::GetNextSerialId()) << 32u | reinterpret_cast<uintptr_t>(nn::os::GetCurrentThread());
        }

        ~Cookie()
        {
            NN_SDK_ASSERT(!nn::util::IntrusiveListBaseNode<Cookie>::IsLinked());
        }

        Id GetId() const
        {
            return m_id;
        }

        bool TimedWait(const nn::TimeSpan& timeout)
        {
            return m_event.TimedWait(timeout);
        }

        nn::os::EventType* GetEventBase()
        {
            return m_event.GetBase();
        }

        void NotifyResult(Result result)
        {
            m_result = result;
            m_event.Signal();
        }

        void SetResultUserData(const T& user)
        {
            m_user = user;
        }

        Result GetResult() const
        {
            return m_result;
        }

        T& GetResultUserData()
        {
            return m_user;
        }

    private:
        nn::os::Event m_event;
        Id            m_id;
        Result        m_result;
        T             m_user;
    };
    typedef nn::util::IntrusiveListBaseNodeTraits<Cookie> CookieTraits;
    typedef nn::util::IntrusiveList<Cookie, CookieTraits> CookieList;

    AsyncManager();
    ~AsyncManager();

    void RegisterCookie(Cookie* pCookie);
    void UnregisterCookie(Cookie* pCookie);
    bool NotifyResult(Id id, Result result);
    bool NotifyResult(Id id, Result result, const T& user);
    void NotifyResultToAll(Result result);

protected:

    static uint32_t GetNextSerialId();
    Cookie* FindCookieById(Id id);

private:
    CookieList m_cookieList;
    mutable nn::os::Mutex m_mutex;
    static std::atomic<uint32_t> s_lastSerialId;
};

template<typename T>
AsyncManager<T>::AsyncManager()
    : m_mutex(true)
{
}

template<typename T>
AsyncManager<T>::~AsyncManager()
{
    NN_SDK_ASSERT(m_cookieList.empty());
}

template<typename T>
uint32_t AsyncManager<T>::GetNextSerialId()
{
    return ++s_lastSerialId;
}

template<typename T>
typename AsyncManager<T>::Cookie* AsyncManager<T>::FindCookieById(Id id)
{
    for (auto& cookie : m_cookieList)
    {
        if (cookie.GetId() == id)
        {
            return &cookie;
        }
    }
    return nullptr;
}

template<typename T>
void AsyncManager<T>::RegisterCookie(Cookie* pCookie)
{
    std::lock_guard<nn::os::Mutex> lock(m_mutex);

    m_cookieList.push_back(*pCookie);
}

template<typename T>
void AsyncManager<T>::UnregisterCookie(Cookie* pCookie)
{
    std::lock_guard<nn::os::Mutex> lock(m_mutex);

    auto it = std::find_if(
        m_cookieList.begin(),
        m_cookieList.end(),
        [=](Cookie& e) { return &e == pCookie; });
    m_cookieList.erase(it);
}

template<typename T>
bool AsyncManager<T>::NotifyResult(Id id, Result result)
{
    std::lock_guard<nn::os::Mutex> lock(m_mutex);

    Cookie* pCookie = FindCookieById(id);
    if (pCookie)
    {
        pCookie->NotifyResult(result);
        return true;
    }
    else
    {
        return false;
    }
}

template<typename T>
bool AsyncManager<T>::NotifyResult(Id id, Result result, const T& user)
{
    std::lock_guard<nn::os::Mutex> lock(m_mutex);

    Cookie* pCookie = FindCookieById(id);
    if (pCookie)
    {
        pCookie->SetResultUserData(user);
        pCookie->NotifyResult(result);
        return true;
    }
    else
    {
        return false;
    }
}

template<typename T>
void AsyncManager<T>::NotifyResultToAll(Result result)
{
    std::lock_guard<nn::os::Mutex> lock(m_mutex);
    for (auto& cookie : m_cookieList)
    {
        cookie.NotifyResult(result);
    }
}

template<typename T>
std::atomic<uint32_t> AsyncManager<T>::s_lastSerialId(0);

}}


