﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <curl/curl.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/init.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nim/nim_Config.h>
#include <nn/nim/detail/nim_ServiceName.h>
#include <nn/nim/srv/nim_DeviceAccountStore.h>
#include <nn/nim/srv/nim_DeviceContext.h>
#include <nn/nim/srv/nim_ShopServiceManager.h>
#include <nn/nim/srv/nim_ShopServiceAccessServer.h>
#include <nn/nim/srv/nim_NetworkInstallManagerServer.h>
#include <nn/ntc/service/ntc_Service.h>
#include <nn/os/os_Thread.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/spl/spl_Api.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_ApiForSystem.h>
#include "ec_ShopServiceObject.h"

using namespace nn;

namespace
{
    // SIGLO-48863: Json DOM パースのため暫定的に 768KB から 1MB に増加。
    // 3.0.0 NUP では、768KB に戻し、SAX パースに変更する
    // SIGLO-72590: ShopServiceAccessor での用途( Curl / RapidJson )のため 1MiB のままとさせてください。
    NN_ALIGNAS(4096) char g_MallocBuffer[1 * 1024 * 1024];

    enum PortIndex
    {
        PortIndex_Nim,
        PortIndex_Shop,
        PortIndex_ShopServiceAccessor,
        PortIndex_Count
    };

    struct RequiredSessionCount
    {
        static const size_t ShopServiceManager = 16;
        static const size_t ShopServiceAccessServer = nim::srv::ShopServiceAccessServer::Configuration::RequiredSessionCount;
        static const size_t NetworkInstallManagerServer = 16;

        static const size_t RequiredMax = ShopServiceManager + ShopServiceAccessServer + NetworkInstallManagerServer;
    };

    // 非同期リクエストを受け取る分、必要セッションが増える
    class MyServerManager : public sf::HipcSimpleAllInOneServerManager<RequiredSessionCount::RequiredMax, PortIndex_Count>{};
    MyServerManager g_ServerManager;

    nim::srv::DeviceContext g_DeviceContext;
    nim::srv::DeviceAccountStore g_DeviceAccountStore;
    nim::srv::VirtualAccountStore g_VirtualAccountStore;
    sf::UnmanagedServiceObject<nim::detail::IShopServiceManager, nim::srv::ShopServiceManager> g_ShopServiceManager;
    sf::UnmanagedServiceObject<nim::detail::INetworkInstallManager, nim::srv::NetworkInstallManagerServer> g_NetworkInstallManagerServer;
    sf::UnmanagedServiceObject<nim::detail::IShopServiceAccessServerInterface, nim::srv::ShopServiceAccessServer::Interface> g_ShopServiceAccessServer;

    nn::socket::SystemConfigWithMemory <
        6,                                                              // tcpSocketCountMax
        0,                                                              // udpSocketCountMax
        nim::SocketSendBufferSize,                                     // tcpInitialSendBufferSize
        nn::socket::ConfigDefault::DefaultTcpInitialReceiveBufferSize,  // tcpInitialReceiveBufferSize
        nim::SocketSendBufferSize,                                     // tcpAutoSendBufferSizeMax
        nim::SocketReceiveBufferSize,                                  // tcpAutoReceiveBufferSizeMax
        0,                                                              // udpSendBufferSize
        0,                                                              // udpReceiveBufferSize
        2,                                                              // socketBufferEfficiency
        nn::socket::DefaultSocketAllocatorSize                          // allocatorPoolSize
    > g_SocketConfig(6);
    NN_STATIC_ASSERT(sizeof(g_SocketConfig) <= 2 * 1024 * 1024);

    //
    const int64_t   DeviceAccountSaveDataSize               = 0xc000; // 16KB + 32 KB
    const int64_t   DeviceAccountSaveDataJournalSize        = 0xc000; // 16KB + 32 KB
    const int       DeviceAccountSaveDataFlags              = 0;
    const int64_t   VirtualAccountSaveDataSize              = 0xc000; // 16KB + 32 KB
    const int64_t   VirtualAccountSaveDataJournalSize       = 0xc000; // 16KB + 32 KB
    const int       VirtualAccountSaveDataFlags             = fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData;
}

extern "C" void nninitStartup()
{
    init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain() NN_NOEXCEPT
{
    auto threadType = nn::os::GetCurrentThread();
    nn::os::ChangeThreadPriority(threadType, NN_SYSTEM_THREAD_PRIORITY(nim, MainIpcSession));
    nn::os::SetThreadNamePointer(threadType, NN_SYSTEM_THREAD_NAME(nim, MainIpcSession));
    nn::fs::InitializeWithMultiSessionForSystem();
    NN_ABORT_UNLESS_EQUAL(CURLE_OK, curl_global_init(CURL_GLOBAL_ALL));
    if(nn::pm::GetBootMode() == nn::pm::BootMode_Normal || nn::pm::GetBootMode() == nn::pm::BootMode_Maintenance)
    {
        es::Initialize();
        account::InitializeForSystemService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(ovln::InitializeSenderLibraryForOverlay());
    }
    ncm::Initialize();
    spl::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Initialize(g_SocketConfig));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::InitializeSystem());
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::InitializeForSystem());
    nn::ntc::service::StartServer();

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_DeviceContext.Initialize());

    if (pm::GetBootMode() == pm::BootMode_Normal)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_DeviceAccountStore.Initialize("nim_dac", 0x8000000000000073ull, DeviceAccountSaveDataSize, DeviceAccountSaveDataJournalSize, DeviceAccountSaveDataFlags));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_VirtualAccountStore.Initialize("nim_vac", 0x8000000000000075ull, VirtualAccountSaveDataSize, VirtualAccountSaveDataJournalSize, VirtualAccountSaveDataFlags));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ShopServiceManager.GetImpl().Initialize(&g_DeviceContext, &g_DeviceAccountStore, &g_VirtualAccountStore));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_ShopServiceManager.GetShared(), RequiredSessionCount::ShopServiceManager, nim::detail::ServiceNameShop));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_NetworkInstallManagerServer.GetImpl().Initialize(&g_DeviceContext, &g_DeviceAccountStore, false, true));
        ec::system::SetShopServiceObject(g_ShopServiceManager.GetShared());

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ShopServiceAccessServer.GetImpl().Initialize(&g_DeviceContext));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_ShopServiceAccessServer.GetShared(), RequiredSessionCount::ShopServiceAccessServer, nim::detail::ServiceNameShopServiceAccessor));
    }
    else if (pm::GetBootMode() == pm::BootMode_Maintenance)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_NetworkInstallManagerServer.GetImpl().Initialize(&g_DeviceContext, &g_DeviceAccountStore, true, true));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_NetworkInstallManagerServer.GetImpl().Initialize(&g_DeviceContext, &g_DeviceAccountStore, true, false));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_NetworkInstallManagerServer.GetShared(), RequiredSessionCount::NetworkInstallManagerServer, nim::detail::ServiceName));

    g_ServerManager.Start();
    g_ServerManager.LoopAuto();

    nn::ntc::service::StopServer();
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::Finalize());
    if(nn::pm::GetBootMode() == nn::pm::BootMode_Normal || nn::pm::GetBootMode() == nn::pm::BootMode_Maintenance)
    {
        ovln::FinalizeSenderLibraryForOverlay();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Finalize());
    ncm::Finalize();
    curl_global_cleanup();
}

// 以下は例外関連コードがリンクされてプロセスが肥大化するのを防ぐためのコード。

void* operator new(size_t size)
{
    return std::malloc(size);
}

void* operator new(size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete(void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}

void* operator new[](size_t size)
{
    return std::malloc(size);
}

void* operator new[](size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete[](void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}
