﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_Log.h>
#include <nn/htclow/detail/htclow_DebugApi.h>
#include <nn/htclow/detail/htclow_InternalApi.h>
#include <nn/htclow/server/htclow_Server.h>

#include <nn/socket.h>

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    const size_t ThreadStackSize = 16 * 1024;

    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_UserServerThreadStack[ThreadStackSize];
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_DebugServerThreadStack[ThreadStackSize];
}

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));

    // Start libnn_htclowServer
    nn::htclow::server::Initialize();

    // For DFC
    nn::htclow::detail::InitializeForUserApiWith(nn::htclow::server::GetUserServiceObject());
    nn::htclow::detail::InitializeForDebugApiWith(nn::htclow::server::GetDebugServiceObject());

    // Set default htclow driver
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::htclow::detail::OpenDriver(nn::htclow::detail::DriverType::Socket));

    nn::htclow::server::RegisterUserHipcServer();
    nn::htclow::server::RegisterDebugHipcServer();

    nn::os::ThreadType userServerThread;
    nn::os::ThreadType debugServerThread;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&userServerThread,
        [](void*)
        {
            nn::htclow::server::StartUserHipcServer();
            nn::htclow::server::LoopUserHipcServer();
        }
        , nullptr, g_UserServerThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(htc, UserIpcThread)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&debugServerThread,
        [](void*)
        {
            nn::htclow::server::StartDebugHipcServer();
            nn::htclow::server::LoopDebugHipcServer();
        }
        , nullptr, g_DebugServerThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(htc, DebugIpcThread)));

    nn::os::SetThreadNamePointer(&userServerThread, NN_SYSTEM_THREAD_NAME(htc, UserIpcThread));
    nn::os::SetThreadNamePointer(&debugServerThread, NN_SYSTEM_THREAD_NAME(htc, DebugIpcThread));

    nn::os::StartThread(&userServerThread);
    nn::os::StartThread(&debugServerThread);

    nn::os::WaitThread(&userServerThread);
    nn::os::WaitThread(&debugServerThread);

    nn::os::DestroyThread(&userServerThread);
    nn::os::DestroyThread(&debugServerThread);

    nn::socket::Finalize();
}
