﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nd/service/nd_Server.h>
#include <nn/ndd.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/bgtc.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/friends/service/friends_Server.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/fs/fs_SystemSaveData.h> // DisableAutoSaveDataCreation 用
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h> // nd サービスを稼動させるかの fwdbg settings チェック用。常時稼動させるようになったら消す。

#include "friends_PowerStateEventManager.h"

namespace
{
    // TORIAEZU:
    nn::Bit8 g_Heap[768 * 1024];
}

namespace
{
    // ソケットの同時利用数。（FG:2?, BG:1）
    const int SocketCount = 3;
    const int SocketConcurrencyCount = SocketCount;
    // MAX TCP sockets = 3, MAX UDP sockets = 0
    nn::socket::SystemConfigLightDefaultWithMemory<SocketCount, 0> g_SocketConfigWithMemory(SocketConcurrencyCount);

    bool IsNeighborDetectionServiceEnabled() NN_NOEXCEPT
    {
        bool enabled;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "nd", "service_enabled");
        if( size != sizeof(enabled) )
        {
            return false;
        }
        return enabled;
    }
}

void Initialize() NN_NOEXCEPT;
// void Finalize() NN_NOEXCEPT;

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_Heap, sizeof (g_Heap));
}

extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(friends, Main));

    Initialize();

    nn::friends::service::StartServer();

    if( IsNeighborDetectionServiceEnabled() )
    {
        nn::nd::service::StartServer();
    }

    PowerStateEventManager::LoopAuto();

    // PowerStateEventManager::LoopAuto は NO_RETURN なので、以下の関数は実行されない。
    // nn::friends::service::StopServer();
    // Finalize();
}

void Initialize() NN_NOEXCEPT
{
    nn::socket::Initialize(g_SocketConfigWithMemory);
    nn::nifm::InitializeSystem();
    nn::time::Initialize();
    nn::account::InitializeBaasAccessTokenAccessor();
    nn::account::InitializeForSystemService();
    nn::pdm::InitializeForQuery();
    nn::npns::InitializeForSystem();
    nn::bgtc::Initialize();
    nn::ovln::InitializeSenderLibraryForOverlay();
    if( IsNeighborDetectionServiceEnabled() )
    {
        nn::ndd::Initialize();
    }

    nn::fs::DisableAutoSaveDataCreation();
}

/*
void Finalize() NN_NOEXCEPT
{
    nn::ovln::FinalizeSenderLibraryForOverlay();
    nn::bgtc::Finalize();
    nn::npns::FinalizeForSystem();
    nn::pdm::FinalizeForQuery();
    nn::time::Finalize();
    nn::socket::Finalize();
}
*/

// 以下は例外関連コードがリンクされてプロセスが肥大化するのを防ぐためのコード。

void* operator new(size_t size)
{
    return std::malloc(size);
}

void* operator new(size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete(void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}

void* operator new[](size_t size)
{
    return std::malloc(size);
}

void* operator new[](size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete[](void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}
