﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fatal/detail/fatal_Log.h>
#include <nn/fatalsrv/fatalsrv_Server.h>
#include <nn/fatalsrv/fatalsrv_AbortObserver.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/init.h>
#include <nn/init/init_Malloc.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

#ifdef NN_FATAL_DUMP_ON_PRESS_VOLUME_UP
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/gpio/gpio.h>
#include <nn/svc/svc_Base.h>

namespace
{
    const int DumpThreadStackSize = 4 * 1024;
    NN_ALIGNAS(4096) nn::Bit8 g_DumpThreadStack[DumpThreadStackSize];

    nn::os::ThreadType g_DumpThread;

    // 音量ボタンを押した時に色々表示する
    void DumpThread(void*)
    {
        NN_DETAIL_FATAL_INFO("[fatal-dbg] start debug thread.\n");

        nn::gpio::Initialize();

        nn::gpio::GpioPadSession session;
        nn::gpio::OpenSession(&session, nn::gpio::GpioPadName_ButtonVolUp);
        nn::gpio::SetDirection(&session, nn::gpio::Direction_Input);

        int pushedCount = 0;
        while (true)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            if (nn::gpio::GetValue(&session) == nn::gpio::GpioValue_Low)
            {
                pushedCount++;
                NN_DETAIL_FATAL_INFO("[fatal-dbg] VolUp pressed.(%d)\n", pushedCount);
            }
            else
            {
                pushedCount = 0;
            }

            if (pushedCount == 5)
            {
                NN_DETAIL_FATAL_INFO("[fatal-dbg] Dump started.\n");
                pushedCount = 0;
                nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, -1ll, 0ll, 0ll); // All processes
                nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, -2ll, 0ll, 0ll); // Kernel threads
                nn::svc::KernelDebug(nn::svc::KernelDebugType_ThreadCallstack, -1ll, 0ll, 0ll);
                for(nn::Bit64 pid =  0; pid <   6; ++pid)
                {
                    nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, pid, 0ll, 0ll);
                }
                for(nn::Bit64 pid = 80; pid < 130; ++pid)
                {
                    nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, pid, 0ll, 0ll);
                }
            }
        }
    }

    void StartDumpThread()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(&g_DumpThread, DumpThread, nullptr,
                                 g_DumpThreadStack, DumpThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(fatalsrv, FatalDump))
            );
        nn::os::SetThreadNamePointer(&g_DumpThread, NN_SYSTEM_THREAD_NAME(fatalsrv, FatalDump));
        nn::os::StartThread(&g_DumpThread);
    }
}
#endif

namespace
{
    const int HeapSize = 2 * 1024 * 1024 + 100 * 1024; // 2MB + 68KB + 32KB(SIGLO-38637 ワークアラウンド)
    NN_ALIGNAS(4096) nn::Bit8 g_Heap[HeapSize];
}

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_Heap, sizeof(g_Heap));
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    nn::fatalsrv::InitializeAbortObserver();

    // fatal では fs shim レイヤでアボートしないように
    nn::fs::SetEnabledAutoAbort(false);
#ifdef NN_FATAL_DUMP_ON_PRESS_VOLUME_UP
    StartDumpThread();
#endif
    nn::fatalsrv::RegisterService();
    nn::fatalsrv::StartServer();
    NN_DETAIL_FATAL_TRACE("Server started.\n");
    nn::fatalsrv::WaitServerStop();
}
