﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstdlib>
#include <cstring>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/time.h>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/es/es_Configuration.h>
#include <nn/es/es_ETicketServiceHipcServer.h>
#include <nn/es/es_MountName.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

nn::Result MountEsSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id, int size, int journalSize)
{
    nn::fs::DisableAutoSaveDataCreation();
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, id))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, size, journalSize, 0));
            NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, id));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

namespace {
    // malloc の領域として静的な領域を確保
    const size_t MallocBufferSize = 128 * 1024;
    NN_ALIGNAS(4096) uint8_t g_MallocBuffer[ MallocBufferSize ];

    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_Allocator;
}   // anonymous

extern "C" void* malloc(size_t size)
{
    return Get(g_Allocator).Allocate(size);
}

extern "C" void free(void* p)
{
    if (p)
    {
        Get(g_Allocator).Free(p);
    }
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = std::malloc(sum);
    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    // メモリブロックのサイズを変更する
    return Get(g_Allocator).Reallocate(p, newSize);
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    return Get(g_Allocator).Allocate(size, alignment);
}

void* operator new(std::size_t size)
{
    return malloc(size);
}

void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void  operator delete(void* ptr) NN_NOEXCEPT
{
    free(ptr);
}

void* operator new[](std::size_t size)
{
    return malloc(size);
}

void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void  operator delete[](void* ptr) NN_NOEXCEPT
{
    free(ptr);
}

extern "C" void nninitStartup()
{
    // nninitStartup() では静的オブジェクトのコンストラクタは
    // まだ呼ばれていないため、placement new で明示的に呼ぶ。
    new( &Get(g_Allocator) ) nn::mem::StandardAllocator;

    // 静的に確保したバッファを malloc 用の領域として初期化
    Get(g_Allocator).Initialize(g_MallocBuffer, MallocBufferSize);
}

// この空定義がある場合は構造化ログが UART に出力される
#if 0
extern "C" void nndiagStartup()
{
}
#endif

extern "C" void nnMain()
{
    // 証明書用システムセーブデータをマウントする
    NN_ABORT_UNLESS_RESULT_SUCCESS(MountEsSystemSaveData(nn::es::CertificateDbMountName, nn::es::CertificateDbSystemSaveDataId, 200000, 200000));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::es::CertificateDbMountName);
    };

    // Common チケット用システムセーブデータをマウントする
    NN_ABORT_UNLESS_RESULT_SUCCESS(MountEsSystemSaveData(nn::es::CommonTicketDbMountName, nn::es::CommonTicketDbSystemSaveDataId, 10000000, 10000000));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::es::CommonTicketDbMountName);
    };

    // Personalized チケット用システムセーブデータをマウントする
    NN_ABORT_UNLESS_RESULT_SUCCESS(MountEsSystemSaveData(nn::es::PersonalizedTicketDbMountName, nn::es::PersonalizedTicketDbSystemSaveDataId, 70000000, 70000000));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::es::PersonalizedTicketDbMountName);
    };

    // チケットメタ記録用システムセーブデータをマウントする
    NN_ABORT_UNLESS_RESULT_SUCCESS(MountEsSystemSaveData(nn::es::TicketMetaRecordDbMountName, nn::es::TicketMetaRecordDbSystemSaveDataId, 304000, 304000));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::es::TicketMetaRecordDbMountName);
    };

    // eLicense アーカイブストア用システムセーブデータをマウントする
    NN_ABORT_UNLESS_RESULT_SUCCESS(MountEsSystemSaveData(nn::es::ELicenseArchiveStoreMountName, nn::es::ELicenseArchiveStoreSystemSaveDataId, 10000000, 10000000));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::es::ELicenseArchiveStoreMountName);
    };

    nn::spl::InitializeForEs();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    nn::es::SetConfigs();

    nn::es::InitializeETicketServer();
    nn::es::LoopETicketServer();
}
