﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/erpt/server/erpt_ServerApi.h>

namespace nn   {
namespace erpt {
namespace      {

const size_t ReportMemoryPoolSize = (192 * 1024);
uint8_t      g_ReportMemory[ReportMemoryPoolSize] NN_ALIGNAS(4096);

int MakeProductModelString(char* outString, size_t outStringSize, nn::settings::system::ProductModel value)
NN_NOEXCEPT
{
    switch( value )
    {
    case nn::settings::system::ProductModel_Invalid:
        return util::Strlcpy(outString, "Invalid", static_cast<int>(outStringSize));
    case nn::settings::system::ProductModel_Nx:
        return util::Strlcpy(outString, "NX", static_cast<int>(outStringSize));
    default:
        // Not to reveal the future ProductModel, just use the value as a number even when a new system::ProductModel is defined.
        return util::SNPrintf(outString, outStringSize, "%d", static_cast<int>(value));
    }
}

const char* RegionString(nn::settings::system::RegionCode value)
NN_NOEXCEPT
{
    switch (value)
    {
    case nn::settings::system::RegionCode_Japan:     return "Japan";
    case nn::settings::system::RegionCode_Usa:       return "Usa";
    case nn::settings::system::RegionCode_Europe:    return "Europe";
    case nn::settings::system::RegionCode_Australia: return "Australia";
    case nn::settings::system::RegionCode_China:     return "China";
    case nn::settings::system::RegionCode_Korea:     return "Korea";
    case nn::settings::system::RegionCode_Taiwan:    return "Taiwan";
    default:
        break;
    }
    return "RegionUnknown";
}

}}}

extern "C" void nninitInitializeAbortObserver()
{
}

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    nn::settings::system::SerialNumber    serialNumber    = {};
    nn::settings::system::FirmwareVersion firmwareVersion = {};

    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(erpt, Main));

    nn::Result result = nn::erpt::srv::Initialize(nn::erpt::g_ReportMemory, sizeof(nn::erpt::g_ReportMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::settings::system::GetSerialNumber(&serialNumber);
    nn::settings::system::GetFirmwareVersion(&firmwareVersion);

    char privateOsVersion[96];
    int l = nn::util::SNPrintf(privateOsVersion, sizeof(privateOsVersion), "%s (%.8s)", firmwareVersion.displayName, firmwareVersion.revision);
    NN_SDK_ASSERT_LESS(static_cast<size_t>(l), sizeof(privateOsVersion));
    NN_UNUSED(l);

    result = nn::erpt::srv::SetSerialNumberAndOsVersion(
                serialNumber.string,
                nn::util::Strnlen(serialNumber.string, sizeof(serialNumber.string) - 1) + 1,
                firmwareVersion.displayVersion,
                nn::util::Strnlen(firmwareVersion.displayVersion, sizeof(firmwareVersion.displayVersion) - 1) + 1,
                privateOsVersion,
                nn::util::Strnlen(privateOsVersion, sizeof(privateOsVersion) - 1) + 1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    char productModelString[16];
    l = nn::erpt::MakeProductModelString(productModelString, sizeof(productModelString), nn::settings::system::GetProductModel());
    NN_SDK_ASSERT_LESS(static_cast<size_t>(l), sizeof(productModelString));
    NN_UNUSED(l);

    result = nn::erpt::srv::SetProductModel(
                productModelString, static_cast<uint32_t>(std::strlen(productModelString)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::settings::system::RegionCode value;
    nn::settings::system::GetRegionCode(&value);
    auto regionString = nn::erpt::RegionString(value);

    result = nn::erpt::srv::SetRegionSetting(
                regionString, static_cast<uint32_t>(std::strlen(regionString)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::erpt::srv::InitializeAndStartService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::erpt::srv::Wait();
}
