﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/eclct/eclct.h>

namespace
{
#define HEAP_INITIALIZER_INITIALIZER {false}
struct HeapInitializer
{
    bool _initialized;
    nn::lmem::HeapHandle _handle;
    char _buffer[1024];

    void Initialize() NN_NOEXCEPT
    {
        _handle = nn::lmem::CreateExpHeap(_buffer, sizeof(_buffer), nn::lmem::CreationOption_NoOption);
        _initialized = true;
    }
    nn::lmem::HeapHandle GetHandle() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(_initialized);
        return _handle;
    }
} g_HeapInitializer = HEAP_INITIALIZER_INITIALIZER;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapInitializer.GetHandle(), size);
}

void Deallocate(void* ptr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapInitializer.GetHandle(), ptr);
}

NN_ALIGNAS(4096) char g_BufferForMemoryAllocator[(1 + 16) * 1024]; // 最低限管理用に 16KiB 必要

}

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_BufferForMemoryAllocator, sizeof(g_BufferForMemoryAllocator));
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    bool       analyticsOverrideFlag        = false;
    uint32_t   analyticsUpdatePeriodSeconds = (24 * 60 * 60);
    const char SettingName[]                = "eclct";
    const char SettingOverrideKey[]         = "analytics_override";
    const char SettingPollPeriodKey[]       = "analytics_pollperiod";

    bool       flag;
    uint32_t   period;

    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(eclct, Main));

    g_HeapInitializer.Initialize();
    nn::fs::SetAllocator(Allocate, Deallocate);

    if (nn::settings::system::GetQuestFlag())
    {
        analyticsOverrideFlag = true;
    }
    else
    {
        if (nn::settings::fwdbg::GetSettingsItemValue(
            &flag,
            sizeof(flag),
            SettingName,
            SettingOverrideKey) == sizeof(flag))
        {
            analyticsOverrideFlag = flag;
        }
    }

    if (nn::settings::fwdbg::GetSettingsItemValue(
            &period,
            sizeof(period),
            SettingName,
            SettingPollPeriodKey) == sizeof(period))
    {
        analyticsUpdatePeriodSeconds = period;
    }

    nn::eclct::InitializeAndRun(analyticsOverrideFlag, analyticsUpdatePeriodSeconds);
}
