﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/nn_SdkLog.h>

#include <nn/dmnt/detail/dmnt_Interface.h>
#include <nn/dmnt/dmnt_Api.h>

#include <nn/os/os_Thread.h>

#include "dmnt_ServerMain.h"
#include "dmnt_Server.h"
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include "..\..\TargetTools\SnapShotDumper\coredump\coredump_Format.h"
#include <nn/nn_SystemThreadDefinition.h>

#if DEBUG
#define DMNT_MEMORY_LOG(...) NN_SDK_LOG( "[dmnt:Memory] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define DMNT_MEMORY_LOG(...)
#endif

using namespace nn;

namespace
{
    // Just need a little for registers buffer, FS, and other things.
    const int ServerAllocationBufferSize = ((16 * 1024) * 2);// +0x10000;
    const int ServerThreadCount = 4;
    const int ServerThreadStackSize = 16 * 1024;

    class ServerManager : public nn::sf::HipcSimpleAllInOneServerManager<nn::dmnt::detail::MaxSessionCount, nn::dmnt::detail::MaxSessionCount>
    {
    private:
        virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT;
    };

    NN_ALIGNAS(64) nn::Bit8 s_DefaultServerAllocatorBuffer[ServerAllocationBufferSize];

    nn::sf::ExpHeapAllocator  s_DefaultServerAllocator;
    ServerManager g_ServerManager;
    nn::lmem::HeapHandle g_HeapHandle;

    NN_ALIGNAS(nn::os::StackRegionAlignment) nn::Bit8 s_Stack[nn::dmnt::detail::MaxSessionCount][ServerThreadStackSize];
    nn::os::ThreadType s_Thread[nn::dmnt::detail::MaxSessionCount];

    nn::Result ServerManager::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
    {
        NN_UNUSED(portIndex);

        nn::sf::SharedPointer<nn::dmnt::detail::IInterface> refInterface;

        refInterface = nn::sf::ObjectFactory<
            nn::sf::ExpHeapAllocator::Policy
        >::CreateSharedEmplaced<
            nn::dmnt::detail::IInterface,
            Server
        >(&s_DefaultServerAllocator);

        return this->AcceptImpl(pPort, refInterface);
    }

    void LoopFunction(void*) NN_NOEXCEPT
    {
        g_ServerManager.LoopAuto();
    }
}

//==============================================================================

void* Alloc( size_t size )
{
    void* p = nn::lmem::AllocateFromExpHeap( g_HeapHandle, size );
    if( p == nullptr )
    {
        NN_SDK_LOG( "[dmnt] Alloc() returned NULL trying to allocate %d bytes (%d bytes available)\n",  size, nn::lmem::GetExpHeapTotalFreeSize( g_HeapHandle ) );
    }
    DMNT_MEMORY_LOG("Alloc 0x%llx ( %d bytes, %d available)", p, size, nn::lmem::GetExpHeapTotalFreeSize(g_HeapHandle));
    return p;
}

//==============================================================================

void Dealloc( void* Ptr, size_t Size )
{
    (void)Size;
    nn::lmem::FreeToExpHeap( g_HeapHandle, Ptr );
    DMNT_MEMORY_LOG("Dealloc 0x%llx (%d bytes now available)", Ptr, nn::lmem::GetExpHeapTotalFreeSize(g_HeapHandle));
}

//==============================================================================

static bool sFSInitialized = false;

bool InitFS()
{
    if( sFSInitialized == false )
    {
        nn::Result res = nn::fs::MountHostRoot();
        //NN_SDK_LOG("[dmnt] - nn::fs::MountHostRoot() returned %x\n", res.GetInnerValueForDebug() );
        sFSInitialized = res.IsSuccess();
    }

    return sFSInitialized;
}

//==============================================================================

//==============================================================================
//  Redirect memory operations to our custom allocators/deallocators.
//==============================================================================

void* operator new ( size_t Size )
{
    void* pResult = Alloc( Size );
    return( pResult );
}

void* operator new [] ( size_t Size )
{
    void* pResult = Alloc( Size );
    return ( pResult );
}

void operator delete ( void* pMemory ) NN_NOEXCEPT
{
    Dealloc( pMemory, 0 );
}

void operator delete [] ( void* pMemory ) NN_NOEXCEPT
{
    Dealloc( pMemory, 0 );
}

void* malloc( size_t Size )
{
    return Alloc( Size );
}

void free( void* pMemory )
{
    Dealloc( pMemory, 0 );
}

void* calloc( size_t num, size_t size )
{
    size_t Size = num * size;
    void* p = Alloc( Size );

    if ( p != NULL )
    {
        memset( p, 0, Size );
    }
    return p;
}

void* realloc( void* p, size_t newSize )
{
    return NULL;
}

void *aligned_alloc(size_t alignment, size_t size )
{
    return Alloc( size );
}

size_t malloc_usable_size( const void* p )
{
    return 0;
}

//==============================================================================

void StartServer() NN_NOEXCEPT
{
    nn::fs::InitializeWithMultiSessionForSystem();

    g_HeapHandle = nn::lmem::CreateExpHeap(&s_DefaultServerAllocatorBuffer, sizeof(s_DefaultServerAllocatorBuffer), nn::lmem::CreationOption_NoOption);
    s_DefaultServerAllocator.Attach(g_HeapHandle);
    nn::fs::SetAllocator( Alloc, Dealloc );

    nn::Result result = g_ServerManager.InitializePort( 0, nn::dmnt::detail::MaxSessionCount, nn::dmnt::detail::PortName );
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    g_ServerManager.Start();

    for( int i = 0; i < nn::dmnt::detail::MaxSessionCount; ++i )
    {
        result = nn::os::CreateThread( &s_Thread[i], &LoopFunction, nullptr, s_Stack[i], sizeof(s_Stack[i]), NN_SYSTEM_THREAD_PRIORITY(dmnt, Ipc) );
        NN_ABORT_UNLESS( result.IsSuccess(), "CreateThread result=%08x", result );
        nn::os::SetThreadName( &s_Thread[i], NN_SYSTEM_THREAD_NAME(dmnt, Ipc) );
        nn::os::StartThread( &s_Thread[i] );
    }
}
