﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_Out.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/dbg/dbg_Api.h>

namespace coredump
{
    struct coredump_compiled_data;
}

enum
{
    //====================================================================
    // This is the size allocated to do memory reads & writes.
    // Tune to taste:  Bigger generally means faster.
    COREDUMP_MEMORY_SECTION_BUFFER_SIZE = 0x10000,//0x40000,

};


class Server
{
public:
    static void Initialize() NN_NOEXCEPT;

    nn::Result BreakDebugProcess( nn::svc::Handle debug ) NN_NOEXCEPT;
    nn::Result TerminateDebugProcess( nn::svc::Handle debug ) NN_NOEXCEPT;
    nn::Result CloseHandle( nn::svc::Handle debug ) NN_NOEXCEPT;

    nn::Result LoadImage( nn::sf::Out<uint32_t> outHandle, const nn::sf::InBuffer& FileName, const nn::sf::InBuffer& Args ) NN_NOEXCEPT;
    nn::Result AttachByProgramId( nn::sf::Out<uint32_t> outHandle, nn::sf::Out<nn::Bit64> pPID, nn::Bit64 processId ) NN_NOEXCEPT;
    nn::Result AttachOnLaunch( nn::sf::Out<uint32_t> outHandle, nn::sf::Out<nn::Bit64> pPID, nn::Bit64 processId ) NN_NOEXCEPT;
    nn::Result GetDebugMonitorProcessId( nn::sf::Out<nn::Bit64> out ) NN_NOEXCEPT;
    nn::Result GetProcessId( nn::sf::Out<nn::Bit64> out, nn::svc::Handle process ) NN_NOEXCEPT;
    nn::Result GetProcessHandle( nn::sf::Out<uint32_t> outHandle, nn::Bit64 processId ) NN_NOEXCEPT;
    nn::Result WaitSynchronization( nn::svc::Handle handle, nn::Bit64 ns ) NN_NOEXCEPT;
    nn::Result GetDebugEvent( const nn::sf::OutBuffer& pEvent, nn::svc::Handle process ) NN_NOEXCEPT;
    nn::Result GetProcessModuleInfo( nn::sf::Out<int> outCount, const nn::sf::OutBuffer& outModules, int num, nn::os::ProcessId pid ) NN_NOEXCEPT;
    nn::Result GetProcessList( nn::sf::Out<int32_t> pNumProcesses, const nn::sf::OutBuffer& outProcessIds ) NN_NOEXCEPT;
    nn::Result GetThreadList( nn::sf::Out<int32_t> pNumThreads, const nn::sf::OutBuffer& outThreads, nn::svc::Handle domain ) NN_NOEXCEPT;
    nn::Result GetDebugThreadContext( const nn::sf::OutBuffer& outContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags ) NN_NOEXCEPT;
    nn::Result GetAllDebugThreadInfo( nn::svc::Handle debug, const nn::sf::OutBuffer& outData, const nn::sf::InBuffer& threadInfo, nn::Bit64 arraySize ) NN_NOEXCEPT;
    nn::Result ContinueDebugEvent( nn::svc::Handle debug, nn::Bit32 flags, const nn::sf::InBuffer& threadIds, nn::Bit32 size ) NN_NOEXCEPT;
    nn::Result ReadDebugProcessMemory( const nn::sf::OutBuffer& buf, nn::svc::Handle debug, uint64_t addr, uint64_t size ) NN_NOEXCEPT;
    nn::Result WriteDebugProcessMemory( nn::svc::Handle debug, const nn::sf::InBuffer& buf, uint64_t addr, uint64_t size ) NN_NOEXCEPT;
    nn::Result SetDebugThreadContext( nn::svc::Handle debug, nn::Bit64 threadId, const nn::sf::InBuffer& context, nn::Bit32 controlFlags ) NN_NOEXCEPT;
    nn::Result GetDebugThreadParam( nn::sf::Out<nn::Bit64> pOut1, nn::sf::Out<nn::Bit32> pOut2, nn::svc::Handle debug, nn::Bit64 threadId, uint32_t select ) NN_NOEXCEPT;
    nn::Result InitializeThreadInfo( const nn::sf::OutBuffer& threadInfo, nn::svc::Handle debug, const nn::sf::InBuffer& createProcess, const nn::sf::InBuffer& createThread ) NN_NOEXCEPT;
    nn::Result QueryDebugProcessMemory( const nn::sf::OutBuffer&  pBlockInfo, const nn::sf::OutBuffer& pPageInfo, nn::svc::Handle debug, uint64_t addr) NN_NOEXCEPT;
    nn::Result GetProcessMemoryDetails( nn::sf::Out<int32_t> pNumMemoryBlocks, const nn::sf::OutBuffer& outData, nn::svc::Handle debugHandle ) NN_NOEXCEPT;

    nn::Result  SetHardwareBreakPoint( /*nn::svc::HardwareBreakPointRegisterName*/ uint32_t regNo,  nn::Bit64 control, nn::Bit64 value ) NN_NOEXCEPT;
    static void MultiCoreThread( void* pArg ) NN_NOEXCEPT;
    void*       SendMultiCoreRequest( void* pMultiCoreRequest ) NN_NOEXCEPT;
    nn::Result  GetJitDebugProcessList( nn::sf::Out<int32_t> pNumProcesses, const nn::sf::OutBuffer& outData ) NN_NOEXCEPT;

    nn::Result  CreateCoreDump( const nn::sf::InBuffer& FileName, const nn::sf::InBuffer Details, const nn::sf::InBuffer Modules, const nn::sf::InBuffer Threads, nn::svc::Handle debugHandle ) NN_NOEXCEPT;

    nn::Result  InitiateCoreDump( const nn::sf::InBuffer& FileName, const nn::sf::InBuffer Details, const nn::sf::InBuffer Modules, const nn::sf::InBuffer Threads, nn::svc::Handle debugHandle, nn::sf::Out<uint64_t> pDumpHandle ) NN_NOEXCEPT;
    nn::Result  ContinueCoreDump( uint64_t DumpHandle, nn::sf::Out<int32_t> pProgress ) NN_NOEXCEPT;
    nn::Result  AddTTYToCoreDump( const nn::sf::InBuffer Data, uint64_t DumpHandle) NN_NOEXCEPT;
    nn::Result  AddImageToCoreDump( const nn::sf::InBuffer Data, uint64_t DumpHandle) NN_NOEXCEPT;
    nn::Result  CloseCoreDump( uint64_t DumpHandle ) NN_NOEXCEPT;

    //==============================================================================
    // TargetIO

    //nn::Result  TargetIO_FileOpen  ( nn::sf::Out<nn::Bit64> pHandle, const nn::sf::InBuffer& FileName, uint32_t OpenMode  ) NN_NOEXCEPT;
    //nn::Result  TargetIO_FileSetSize( int64_t Size, const nn::sf::InBuffer& Handle ) NN_NOEXCEPT;
    //nn::Result  TargetIO_FileClose( const nn::sf::InBuffer& Handle ) NN_NOEXCEPT;
    //nn::Result  TargetIO_FileRead ( nn::sf::Out<int32_t> pAmountRead,    const nn::sf::InBuffer& Handle, int64_t offset, const nn::sf::OutBuffer& pBuffer, uint32_t AmountToRead ) NN_NOEXCEPT;
    //nn::Result  TargetIO_FileWrite( nn::sf::Out<int32_t> pAmountWritten, const nn::sf::InBuffer& Handle, int64_t offset, const nn::sf::OutBuffer& pBuffer, uint32_t AmountToWrite ) NN_NOEXCEPT;

    //nn::Result  TargetIO_FileCreate( const nn::sf::InBuffer& FileName ) NN_NOEXCEPT;
    //nn::Result  TargetIO_FileDelete( const nn::sf::InBuffer& FileName ) NN_NOEXCEPT;
    //nn::Result  TargetIO_FileRename( const nn::sf::InBuffer& FromName, const nn::sf::InBuffer& ToName ) NN_NOEXCEPT;

    //nn::Result  TargetIO_FileGetSize( const nn::sf::InBuffer& FileName, nn::sf::Out<int64_t> pSize ) NN_NOEXCEPT;

    //nn::Result  TargetIO_FileSetAttributes( const nn::sf::InBuffer& FromName, const nn::sf::InBuffer& pAttribs )  NN_NOEXCEPT;
    //nn::Result  TargetIO_FileSetTime( const nn::sf::InBuffer& FileName, uint64_t CreateTime, uint64_t AccessTime, uint64_t ModifyTime )  NN_NOEXCEPT;

    nn::Result TargetIO_FileOpen( const nn::sf::InBuffer& FileName, uint32_t OpenMode, int32_t CreationDisposition, const nn::sf::OutBuffer& pHandle ) NN_NOEXCEPT;
    nn::Result TargetIO_FileClose( const nn::sf::InBuffer& Handle ) NN_NOEXCEPT;
    nn::Result TargetIO_FileRead( const nn::sf::InBuffer& Handle, const nn::sf::OutBuffer& pBuffer, nn::sf::Out<int32_t> pNumberOfBytesRead, int64_t Offset ) NN_NOEXCEPT;
    nn::Result TargetIO_FileWrite( const nn::sf::InBuffer& Handle, const nn::sf::InBuffer& pBuffer, nn::sf::Out<int32_t> pNumberOfBytesWritten, int64_t Offset ) NN_NOEXCEPT;

    nn::Result TargetIO_FileSetAttributes( const nn::sf::InBuffer& FileName, const nn::sf::InBuffer& pAttribs ) NN_NOEXCEPT;

    nn::Result TargetIO_FileGetInformation( const nn::sf::InBuffer& FileName, nn::sf::Out<int32_t> pIsDirectory, const nn::sf::OutBuffer& pData ) NN_NOEXCEPT;
    nn::Result TargetIO_FileSetTime( const nn::sf::InBuffer& FileName, uint64_t CreateTime, uint64_t AccessTime, uint64_t ModifyTime ) NN_NOEXCEPT;

    nn::Result TargetIO_FileSetSize( const nn::sf::InBuffer& FileName, int64_t size ) NN_NOEXCEPT;
    nn::Result TargetIO_FileDelete( const nn::sf::InBuffer& FileName ) NN_NOEXCEPT;
    nn::Result TargetIO_FileMove( const nn::sf::InBuffer& FromName, const nn::sf::InBuffer& ToName ) NN_NOEXCEPT;

    nn::Result  TargetIO_DirectoryCreate( const nn::sf::InBuffer& DirectoryName )  NN_NOEXCEPT;
    nn::Result  TargetIO_DirectoryDelete( const nn::sf::InBuffer& DirectoryName )  NN_NOEXCEPT;
    nn::Result  TargetIO_DirectoryRename( const nn::sf::InBuffer& FromName, const nn::sf::InBuffer& ToName )  NN_NOEXCEPT;

    nn::Result  TargetIO_DirectoryGetCount( const nn::sf::InBuffer& DirectoryName, nn::sf::Out<int32_t> pNumberOfEntries )  NN_NOEXCEPT;
    nn::Result  TargetIO_DirectoryOpen( const nn::sf::InBuffer& DirectoryName, const nn::sf::OutBuffer& pHandle )  NN_NOEXCEPT;
    nn::Result  TargetIO_DirectoryGetNext( const nn::sf::InBuffer& Handle, const nn::sf::OutBuffer& pEntry )  NN_NOEXCEPT;
    nn::Result  TargetIO_DirectoryClose( const nn::sf::InBuffer& Handle )  NN_NOEXCEPT;

//    nn::Result  TargetIO_DirectoryGet( const nn::sf::InBuffer& DirectoryName, nn::sf::Out<int32_t> pNumberOfEntries, const nn::sf::OutBuffer& pEntries )  NN_NOEXCEPT;

    nn::Result  TargetIO_GetFreeSpace( const nn::sf::OutBuffer& pFreeSpace )  NN_NOEXCEPT;
    nn::Result  TargetIO_GetVolumeInformation( const nn::sf::OutBuffer& pVolumeName, const nn::sf::OutBuffer& pFileSystemName,
        nn::sf::Out<int32_t> pVolumeSerialNumber, nn::sf::Out<int32_t> pMaximumComponentLength, nn::sf::Out<int32_t> pFileSystemFlags )  NN_NOEXCEPT;
};
