﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/psc.h>

#include <nn/btm/btm_Result.h>
#include <nn/btm/btm_Types.h>
#include <btm_InternalTypes.h>
#include <btm_Worker.h>
#include <btm_Handler.h>
#include <btm_Utility.h>
#include <btm_Queue.h>

#include <nn/bluetooth/bluetooth_Api.h>

namespace nn { namespace btm {

namespace {

BtmState g_State;

//--------------------------------------------------
//Impl - Worker通信用のオブジェクト
//--------------------------------------------------
//SendCommand
ApiId g_CommandId;
uint8_t g_CommandBuffer[SIZE_OF_COMMAND_BUFFER];
nn::os::EventType   g_CommandEvent;

//ReceiveReport
nn::Result g_Result;
uint8_t g_ReportBuffer[SIZE_OF_REPORT_BUFFER];
nn::os::EventType   g_ReportEvent;

//通信可否制御
nn::os::MutexType g_Mutex;
bool g_IsAsyncWork;

//--------------------------------------------------
//MultiWait対象
//--------------------------------------------------
nn::os::SystemEventType g_SystemEventForBtCore;
nn::os::SystemEventType g_SystemEventForBtHid;
nn::os::SystemEventType g_SystemEventForBtBleCore;
//g_CommandEventは通信用オブジェクトとして記述しているが、MultiWait対象でもある
nn::psc::PmModule g_PmModule;

//--------------------------------------------------
//MultiWait用オブジェクト
//--------------------------------------------------
nn::os::MultiWaitType g_MultiWait;
nn::os::MultiWaitHolderType g_HolderForBtCore;
nn::os::MultiWaitHolderType g_HolderForBtHid;
nn::os::MultiWaitHolderType g_HolderForBtBleCore;
nn::os::MultiWaitHolderType g_HolderForApi;
nn::os::MultiWaitHolderType g_HolderForPsc;

}


BtmState GetState()
{
    return g_State;
}

bool TryEngageWorker()
{
    if(false == TryLockMutex(&g_Mutex))
    {
        return false;
    }

    //Workerが非同期動作している場合も、同期動作開始不可と判定する
    //具体例としては、Async系API（例：SetWlanMode）の指示を受け、Implとの同期動作を終えた後にTSI変更を行っているケース
    if(g_IsAsyncWork)
    {
        UnlockMutex(&g_Mutex);
        return false;
    }
    else
    {
        return true;
    }
}

void DisengageWorker()
{
    UnlockMutex(&g_Mutex);
}

//--------------------------------------------------
//Impl から Workerへの動作依頼
//
//Implはマルチスレッドで動作するため、ここでWorkerへの指示をMutexを用いて排他する
//以下内部動作。
//引数で取得したapiIdおよび、inBufferの内容が、この関数内でWorker管理のバッファにコピーされる。
//その後、g_CommandEventのシグナルによってWorkerが駆動する。
//Workerの処理が完了するとg_ReportEventがシグナルされる。
//Workerからの返答内容をoutBufferにコピーする
//mutex解放のタイミング上、少なくともout側のバッファコピーは必要
//--------------------------------------------------
nn::Result InvokeWorker(ApiId apiId, const void* pInBuffer, size_t inBufferSize, void* pOutBuffer, size_t outBufferSize)
{
    const uint8_t* inBuffer = static_cast<const uint8_t*>(pInBuffer);
    uint8_t* outBuffer = static_cast<uint8_t*>(pOutBuffer);

    NN_SDK_REQUIRES(inBufferSize <= SIZE_OF_COMMAND_BUFFER);
    NN_SDK_REQUIRES(outBufferSize <= SIZE_OF_REPORT_BUFFER);

    //ImplとWorkerの同期動作開始
    if(TryEngageWorker())
    {
        //SendCommand
        g_CommandId=apiId;
        memcpy(&g_CommandBuffer[0], &inBuffer[0], inBufferSize);
        g_Result = ResultSuccess();
        nn::os::SignalEvent(&g_CommandEvent);

        //ReceiveReport
        nn::os::WaitEvent(&g_ReportEvent);
        NN_ABORT_UNLESS_EQUAL(apiId, g_CommandId);
        memcpy(&outBuffer[0],&g_ReportBuffer[0],outBufferSize);
        auto apiResult = g_Result;//Mutex解除後に上書きされうるので、ローカルバッファに一度コピーしておく

        //ImplとWorkerの同期動作終了
        DisengageWorker();
        return apiResult;
    }
    else
    {
        return ResultBusyWorking();
    }
}


//--------------------------------------------------
//WorkerからImplへの動作完了通知
//--------------------------------------------------
void FinishWorking()
{
    nn::os::SignalEvent(&g_ReportEvent);
}


#ifdef BTM_PRINT_BUSY_TIME
nn::TimeSpan g_StartTimeSpan;
#endif

//--------------------------------------------------
//3種のイベント(Impl->Workerへの指示 , 下層からのレポート2種)をwait
//timeoutSecは、MultiWaitMode_TimedEvent以外では無視される
//--------------------------------------------------
MultiWaitId MultiWait(MultiWaitMode multiWaitMode, uint8_t timeoutSec)
{
    nn::os::MultiWaitHolderType* pHolder;
    MultiWaitId multiWaitId;

    //Holderを複数のMultiWaitに設定できないので、都度追加登録、削除するようにしている
    //[Todo]正しい実装方法を調査
    if(multiWaitMode == MultiWaitMode_Full)
    {
#ifdef BTM_PRINT_BUSY_TIME
        //計測終了、表示
        nn::os::Tick    tick = nn::os::GetSystemTick();
        nn::TimeSpan    finishTimeSpan = nn::os::ConvertToTimeSpan( tick );
        NN_UNUSED(finishTimeSpan);
        int64_t busyTime = finishTimeSpan.GetMilliSeconds() - g_StartTimeSpan.GetMilliSeconds();
        if(busyTime > 2500)
        {
            NN_DETAIL_BTM_ERROR("!!!! Long Delay !!!!");
        }
        NN_DETAIL_BTM_INFO("Previous busy time %16lld(msec)\n\n", busyTime);
#endif
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForApi );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForPsc );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtCore );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtHid );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtBleCore );
        pHolder = nn::os::WaitAny( &g_MultiWait );
        multiWaitId = static_cast<MultiWaitId>(nn::os::GetMultiWaitHolderUserData( pHolder ));
        nn::os::UnlinkMultiWaitHolder(&g_HolderForApi );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForPsc );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtCore );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtHid );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtBleCore);
#ifdef BTM_PRINT_BUSY_TIME
        //計測開始
        tick = nn::os::GetSystemTick();
        g_StartTimeSpan = nn::os::ConvertToTimeSpan( tick );
#endif
    }
    else if(multiWaitMode == MultiWaitMode_ApiPsc)
    {
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForApi );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForPsc );
        pHolder = nn::os::WaitAny( &g_MultiWait );
        multiWaitId = static_cast<MultiWaitId>(nn::os::GetMultiWaitHolderUserData( pHolder ));
        nn::os::UnlinkMultiWaitHolder(&g_HolderForApi );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForPsc );
    }
    else if(multiWaitMode == MultiWaitMode_Event)
    {
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtCore );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtHid );
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForBtBleCore);
        pHolder = nn::os::WaitAny( &g_MultiWait );
        multiWaitId = static_cast<MultiWaitId>(nn::os::GetMultiWaitHolderUserData( pHolder ));
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtCore );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtHid );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtBleCore);
    }
    else if(multiWaitMode == MultiWaitMode_TimedEvent)
    {
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtCore );
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtHid );
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForBtBleCore);
        pHolder = nn::os::TimedWaitAny( &g_MultiWait, nn::TimeSpan::FromSeconds(timeoutSec) );
        if(pHolder == NULL)
        {
            multiWaitId = MultiWaitId_Time;
        }
        else
        {
            multiWaitId = static_cast<MultiWaitId>(nn::os::GetMultiWaitHolderUserData( pHolder ));
        }
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtCore );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtHid );
        nn::os::UnlinkMultiWaitHolder(&g_HolderForBtBleCore);
    }
    else if(multiWaitMode == MultiWaitMode_Psc)
    {
        nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForPsc );
        pHolder = nn::os::WaitAny( &g_MultiWait );
        multiWaitId = static_cast<MultiWaitId>(nn::os::GetMultiWaitHolderUserData( pHolder ));
        nn::os::UnlinkMultiWaitHolder(&g_HolderForPsc );
    }
    else
    {
        BTM_LOG("Unhandled MultiWaitMode\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        NN_ABORT("Unhandled MultiWaitMode\n");//コンパイル時のMultiWaitId未初期化Warningを避けるため、ABORTも残す
    }

    switch(multiWaitId)
    {
       case MultiWaitId_BtCore:
       nn::os::ClearSystemEvent( &g_SystemEventForBtCore );
       break;

       case MultiWaitId_BtHid:
       nn::os::ClearSystemEvent( &g_SystemEventForBtHid );
       break;

       case MultiWaitId_BtBleCore:
       nn::os::ClearSystemEvent( &g_SystemEventForBtBleCore );
       break;

       case MultiWaitId_Api:
       nn::os::ClearEvent( &nn::btm::g_CommandEvent );
       break;

       case MultiWaitId_Psc:
       g_PmModule.GetEventPointer()->Clear();
       break;

       case MultiWaitId_Time:
       break;

       default:
       NN_SDK_REQUIRES(multiWaitId < MultiWaitId_End);
       break;
    }
    return multiWaitId;
}//NOLINT(impl/function_size)




void StateMachine()
{
    for(;;)
    {
        switch(g_State)
        {
            case BtmState_NotInitialized:
            {
                //基盤検査用の特殊ステート。復旧不能。下層からのイベントはブロックする
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_ApiPsc);
                if(multiWaitId == MultiWaitId_Api)
                {
                    if(g_CommandId == ApiId_AcquireDeviceConditionEvent ||
                            g_CommandId == ApiId_AcquireDiscoveryEvent ||
                            g_CommandId == ApiId_AcquireDeviceInfoEvent ||
                            g_CommandId == ApiId_AcquireAwakeReqEvent ||
                            g_CommandId == ApiId_AcquireRadioEvent ||
                            g_CommandId == ApiId_AcquireGamepadPairingEvent ||
                            g_CommandId == ApiId_AcquireLlrStateEvent ||
                            g_CommandId == ApiId_AcquireBleScanEvent ||
                            g_CommandId == ApiId_AcquireBleScanEvent ||
                            g_CommandId == ApiId_AcquireBleConnectionEvent ||
                            g_CommandId == ApiId_AcquireBlePairingEvent ||
                            g_CommandId == ApiId_AcquireBleSdpEvent ||
                            g_CommandId == ApiId_AcquireBleMtuConfigEvent
                        )
                    {
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                    else
                    {
                        g_Result = ResultInvalidStateNotInitialized();
                        FinishWorking();
                    }
                }
                else if(multiWaitId == MultiWaitId_Psc)
                {
                    BTM_LOG("PSC Event in production process.\n");
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    //基盤検査用特殊ステートでは、何もしない
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("Un-handled Event\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                }
                break;
            }
            case BtmState_Finalized:
            {
                //シャットダウン中の特殊ステート。復旧不能。上位層下位層からのイベントは、取り捨てせずにブロックする
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_Psc);
                if(multiWaitId == MultiWaitId_Psc)
                {
                    BTM_LOG("PSC Event in shutdown process.\n");
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    //何もしない
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("Un-handled Event\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                }
                break;
            }
            case BtmState_Initialized:
            {
                BTM_LOG("State Initialized\n");
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_Full);
                if(multiWaitId == MultiWaitId_Api)
                {
                    if(g_CommandId == ApiId_SetBurstMode ||
                       g_CommandId == ApiId_SetSlotMode ||
                       g_CommandId == ApiId_SetBluetoothMode ||
                       g_CommandId == ApiId_SetWlanMode
                       )
                    {
                        //非同期API
                        //Handlerの内部処理から直接Implのブロックを解く（FinishWorkingはHandler内で呼び出す）
                        //その後、処理が完了するまでの間は、g_IsAsyncWorkがtrueを維持する
                        g_IsAsyncWork = true;
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        //この間で次のAPI要求が来うるが、g_IsAsyncWorkがfalseに戻るまではBusyが返るため問題ない
                        g_IsAsyncWork = false;
                    }
                    else if(g_CommandId == ApiId_DisableRadio)
                    {
                        //同期API。ステート遷移あり
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_State = BtmState_RadioOff;
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_EnableRadio)
                    {
                        //RadioOn->RadioOn時の下層バグを回避するため、何も操作しない API
                        g_Result = ResultSuccess();
                        FinishWorking();
                    }
                    else
                    {
                        //同期API
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                }
                else if(multiWaitId == MultiWaitId_BtCore)
                {
                    HandleBtCoreReport();
                }
                else if(multiWaitId == MultiWaitId_BtHid)
                {
                    HandleBtHidReport();
                }
                else if (multiWaitId == MultiWaitId_BtBleCore)
                {
                    BTM_LOG("MultiWaitId_BtBleCore\n");
                    HandleBtBleCoreReport();
                }
                else if(multiWaitId == MultiWaitId_Psc)
                {
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    //StateMachineで通知を取った時点でFS系の処理は終わっている
                    if(pmState == nn::psc::PmState_MinimumAwake)
                    {
                        BTM_LOG("PmState_MinimumAwake\n");
                        HandleSleep();
                        g_State = BtmState_MinorSlept;
                    }
                    else if(pmState == nn::psc::PmState_ShutdownReady)
                    {
                        BTM_LOG("PmState_ShutdownReady\n");
                        HandleShutdown();
                        g_State = BtmState_Finalized;
                    }
                    else
                    {
                        BTM_LOG("Undefined power state transition.\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                    }
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("Un-handled Event\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                }
                break;
            }
            case BtmState_RadioOff:
            {
                BTM_LOG("State RadioOff\n");
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_Full);
                if(multiWaitId == MultiWaitId_Api)
                {
                    if(g_CommandId == ApiId_SetWlanMode ||
                       g_CommandId == ApiId_SetBluetoothMode
                      )
                    {
                        //非同期API
                        g_IsAsyncWork = true;
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_IsAsyncWork = false;
                    }
                    else if(g_CommandId == ApiId_EnableRadio)
                    {
                        //同期API。ステート遷移あり
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_State = BtmState_Initialized;
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_DisableRadio)
                    {
                        //RadioOff->RadioOff時の下層バグを回避するため、何も操作しない API
                        g_Result = ResultSuccess();
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_AcquireDeviceConditionEvent        ||
                            g_CommandId == ApiId_AcquireDiscoveryEvent              ||
                            g_CommandId == ApiId_AcquireDeviceInfoEvent             ||
                            g_CommandId == ApiId_AcquireAwakeReqEvent               ||
                            g_CommandId == ApiId_AcquireRadioEvent                  || g_CommandId == ApiId_DiscardRadioEvent                   ||
                            g_CommandId == ApiId_AcquireGamepadPairingEvent         || g_CommandId == ApiId_DiscardGamepadPairingEvent          ||
                            g_CommandId == ApiId_AcquireLlrStateEvent               ||
                            g_CommandId == ApiId_AcquireBleScanEvent                || g_CommandId == ApiId_DiscardBleScanEvent                 ||
                            g_CommandId == ApiId_AcquireBleConnectionEvent          || g_CommandId == ApiId_DiscardBleConnectionEvent           ||
                            g_CommandId == ApiId_AcquireBlePairingEvent             || g_CommandId == ApiId_DiscardBlePairingEvent              ||
                            g_CommandId == ApiId_AcquireBleSdpEvent                 || g_CommandId == ApiId_DiscardBleSdpEvent                  ||
                            g_CommandId == ApiId_AcquireBleMtuConfigEvent           || g_CommandId == ApiId_DiscardBleMtuConfigEvent            ||
                            g_CommandId == ApiId_StoreDeviceInfo                    ||
                            g_CommandId == ApiId_IncreaseDeviceInfoOrder            ||
                            g_CommandId == ApiId_EnableSlotSaving                   ||
                            g_CommandId == ApiId_DisableSlotSaving                  ||
                            g_CommandId == ApiId_ProtectDeviceInfo                  ||
                            g_CommandId == ApiId_BleRegisterUnconnectableScan       || g_CommandId == ApiId_BleUnRegisterUnconnectableScan      ||
                            g_CommandId == ApiId_BleRegisterSdUnconnectableScan     || g_CommandId == ApiId_BleUnRegisterSdUnconnectableScan    ||
                            g_CommandId == ApiId_BleRegisterWakeOnBle               || g_CommandId == ApiId_BleUnregisterWakeOnBle              ||
                            g_CommandId == ApiId_BleRegisterDataPath                || g_CommandId == ApiId_BleUnregisterDataPath               ||
                            g_CommandId == ApiId_ChangeUsecaseByBleDisconnect
                           )
                    {
                        //同期API
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_AddDeviceInfo ||
                            g_CommandId == ApiId_RemoveDeviceInfo ||
                            g_CommandId == ApiId_ClearGamepadPairingDatabase ||
                            g_CommandId == ApiId_StartGamepadPairing ||
                            g_CommandId == ApiId_CancelGamepadPairing)
                    {
                        //同期API。RadioOff時は特殊処理
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                    else
                    {
                        g_Result = ResultInvalidStateRadioOff();
                        FinishWorking();
                    }
                }
                else if(multiWaitId == MultiWaitId_BtCore)
                {
                    //フライトモード遷移前のイベントを一応処理
                    BTM_LOG("Event came in RadioOff\n");
                    HandleBtCoreReport();
                }
                else if(multiWaitId == MultiWaitId_BtHid)
                {
                    //フライトモード遷移前のイベントを一応処理
                    BTM_LOG("Event came in RadioOff\n");
                    HandleBtHidReport();
                }
                else if (multiWaitId == MultiWaitId_BtBleCore)
                {
                    //フライトモード遷移前のイベントを一応処理
                    BTM_LOG("[btm]Event came in RadioOff\n");
                    HandleBtBleCoreReport();
                }
                else if(multiWaitId == MultiWaitId_Psc)
                {
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    if(pmState == nn::psc::PmState_MinimumAwake)
                    {
                        BTM_LOG("PmState_MinimumAwake\n");
                        g_State = BtmState_RadioOffMinorSlept;
                    }
                    else if(pmState == nn::psc::PmState_ShutdownReady)
                    {
                        BTM_LOG("PmState_ShutdownReady\n");
                        HandleShutdown();
                        g_State = BtmState_Finalized;
                    }
                    else
                    {
                        BTM_LOG("Undefined power state transition.\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                    }
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("Un-handled Event\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                }
                break;
            }
            case BtmState_MinorSlept:
            {
                BTM_LOG("State MinorSlept\n");
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_Full);
                if(multiWaitId == MultiWaitId_Api)
                {
                    if(g_CommandId == ApiId_SetBurstMode ||
                       g_CommandId == ApiId_SetSlotMode ||
                       g_CommandId == ApiId_SetBluetoothMode ||
                       g_CommandId == ApiId_SetWlanMode)
                    {
                        //非同期API
                        g_IsAsyncWork = true;
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_IsAsyncWork = false;
                    }
                    else if(g_CommandId == ApiId_DisableRadio)
                    {
                        //同期API。ステート遷移あり
                        HandleApi(ApiId_DisableRadio_InMinorSlept, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_State = BtmState_RadioOffMinorSlept;
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_EnableRadio)
                    {
                        //RadioOn->RadioOn時の下層バグを回避するため、何も操作しない API
                        g_Result = ResultSuccess();
                        FinishWorking();
                    }
                    else
                    {
                        //同期API
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                }
                else if(multiWaitId == MultiWaitId_BtCore)
                {
                    BTM_LOG("Event came in MinorSlept\n");
                    HandleBtCoreReport();
                }
                else if(multiWaitId == MultiWaitId_BtHid)
                {
                    BTM_LOG("Event came in MinorSlept\n");
                    HandleBtHidReport();
                }
                else if (multiWaitId == MultiWaitId_BtBleCore)
                {
                    BTM_LOG("[btm]Event came in MinorSlept\n");
                    HandleBtBleCoreReport();
                }
                else if(multiWaitId == MultiWaitId_Psc)
                {
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    //StateMachineで通知を取った時点でFS系の処理は終わっている
                    if(pmState == nn::psc::PmState_SleepReady)
                    {
                        BTM_LOG("PmState_SleepReady\n");
                        HandleSleep();
                        g_State = BtmState_Slept;
                    }
                    else if(pmState == nn::psc::PmState_FullAwake)
                    {
                        BTM_LOG("PmState_FullAwake\n");
                        HandleAwake();
                        g_State = BtmState_Initialized;
                    }
                    else if(pmState == nn::psc::PmState_ShutdownReady)
                    {
                        BTM_LOG("PmState_ShutdownReady\n");
                        HandleShutdown();
                        g_State = BtmState_Finalized;
                    }
                    else
                    {
                        BTM_LOG("Undefined power state transition.\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                    }
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("Un-handled Event\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                }
                break;
            }
            case BtmState_RadioOffMinorSlept:
            {
                BTM_LOG("State RadioOffMinorSlept\n");
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_Full);
                if(multiWaitId == MultiWaitId_Api)
                {
                    if(g_CommandId == ApiId_SetWlanMode ||
                       g_CommandId == ApiId_SetBluetoothMode)
                    {
                        //非同期API
                        g_IsAsyncWork = true;
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_IsAsyncWork = false;
                    }
                    else if(g_CommandId == ApiId_EnableRadio)
                    {
                        //同期API。ステート遷移あり
                        HandleApi(ApiId_EnableRadio_InMinorSlept, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        g_State = BtmState_MinorSlept;
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_DisableRadio)
                    {
                        //RadioOff->RadioOff時の下層バグを回避するため、何も操作しない API
                        g_Result = ResultSuccess();
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_AcquireDeviceConditionEvent    ||
                            g_CommandId == ApiId_AcquireDiscoveryEvent          ||
                            g_CommandId == ApiId_AcquireDeviceInfoEvent         ||
                            g_CommandId == ApiId_AcquireAwakeReqEvent           ||
                            g_CommandId == ApiId_AcquireRadioEvent              || g_CommandId == ApiId_DiscardRadioEvent                   ||
                            g_CommandId == ApiId_AcquireGamepadPairingEvent     || g_CommandId == ApiId_DiscardGamepadPairingEvent          ||
                            g_CommandId == ApiId_AcquireLlrStateEvent           ||
                            g_CommandId == ApiId_AcquireBleScanEvent            || g_CommandId == ApiId_DiscardBleScanEvent                 ||
                            g_CommandId == ApiId_AcquireBleConnectionEvent      || g_CommandId == ApiId_DiscardBleConnectionEvent           ||
                            g_CommandId == ApiId_AcquireBlePairingEvent         || g_CommandId == ApiId_DiscardBlePairingEvent              ||
                            g_CommandId == ApiId_AcquireBleSdpEvent             || g_CommandId == ApiId_DiscardBleSdpEvent                  ||
                            g_CommandId == ApiId_AcquireBleMtuConfigEvent       || g_CommandId == ApiId_DiscardBleMtuConfigEvent            ||
                            g_CommandId == ApiId_StoreDeviceInfo                ||
                            g_CommandId == ApiId_IncreaseDeviceInfoOrder        ||
                            g_CommandId == ApiId_EnableSlotSaving               ||
                            g_CommandId == ApiId_DisableSlotSaving              ||
                            g_CommandId == ApiId_ProtectDeviceInfo              ||
                            g_CommandId == ApiId_BleRegisterUnconnectableScan   || g_CommandId == ApiId_BleUnRegisterUnconnectableScan      ||
                            g_CommandId == ApiId_BleRegisterSdUnconnectableScan || g_CommandId == ApiId_BleUnRegisterSdUnconnectableScan    ||
                            g_CommandId == ApiId_BleRegisterWakeOnBle           || g_CommandId == ApiId_BleUnregisterWakeOnBle              ||
                            g_CommandId == ApiId_BleRegisterDataPath            || g_CommandId == ApiId_BleUnregisterDataPath               ||
                            g_CommandId == ApiId_ChangeUsecaseByBleDisconnect
                        )
                    {
                        //同期API
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                    else if(g_CommandId == ApiId_AddDeviceInfo ||
                            g_CommandId == ApiId_RemoveDeviceInfo ||
                            g_CommandId == ApiId_ClearGamepadPairingDatabase ||
                            g_CommandId == ApiId_StartGamepadPairing ||
                            g_CommandId == ApiId_CancelGamepadPairing)
                    {
                        //同期API。RadioOff時は特殊処理
                        HandleApi(g_CommandId, g_CommandBuffer, SIZE_OF_COMMAND_BUFFER, g_ReportBuffer, SIZE_OF_REPORT_BUFFER, &g_Result);
                        FinishWorking();
                    }
                    else
                    {
                        g_Result = ResultInvalidStateRadioOff();
                        FinishWorking();
                    }
                }
                else if(multiWaitId == MultiWaitId_BtCore)
                {
                    //フライトモード遷移前のイベントを一応処理
                    BTM_LOG("Event came in RadioOffMinorSlept\n");
                    HandleBtCoreReport();
                }
                else if(multiWaitId == MultiWaitId_BtHid)
                {
                    //フライトモード遷移前のイベントを一応処理
                    BTM_LOG("Event came in RadioOffMinorSlept\n");
                    HandleBtHidReport();
                }
                else if(multiWaitId == MultiWaitId_Psc)
                {
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    if(pmState == nn::psc::PmState_FullAwake)
                    {
                        BTM_LOG("PmState_FullAwake\n");
                        g_State = BtmState_RadioOff;
                    }
                    else if(pmState == nn::psc::PmState_SleepReady)
                    {
                        BTM_LOG("PmState_SleepReady\n");
                        g_State = BtmState_RadioOffSlept;
                    }
                    else if(pmState == nn::psc::PmState_ShutdownReady)
                    {
                        BTM_LOG("PmState_ShutdownReady\n");
                        HandleShutdown();
                        g_State = BtmState_Finalized;
                    }
                    else
                    {
                        BTM_LOG("Undefined power state transition.\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                    }
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("Un-handled Event\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
                }
                break;
            }
            case BtmState_Slept:
            case BtmState_RadioOffSlept:
            {
                BTM_LOG("State Slept\n");
                //Sleptの方が遷移ステートとして深いので、上記どちらもSleptとして扱う
                MultiWaitId multiWaitId = MultiWait(MultiWaitMode_Psc);
                if(multiWaitId == MultiWaitId_Psc)
                {
                    nn::psc::PmState    pmState;
                    nn::psc::PmFlagSet  pmFlags;
                    auto result = g_PmModule.GetRequest(&pmState, &pmFlags);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    if(pmState == nn::psc::PmState_MinimumAwake)
                    {
                        BTM_LOG("PmState_MinimumAwake\n");
                        if(g_State == BtmState_Slept)
                        {
                            g_State = BtmState_MinorSlept;
                        }
                        else if(g_State == BtmState_RadioOffSlept)
                        {
                            g_State = BtmState_RadioOffMinorSlept;
                        }
                    }
                    else if(pmState == nn::psc::PmState_ShutdownReady)
                    {
                        BTM_LOG("PmState_ShutdownReady\n");
                        HandleShutdown();
                        g_State = BtmState_Finalized;
                    }
                    else
                    {
                        BTM_LOG("Ignorable PSC event.\n");//Essential系は無視する
                    }
                    g_PmModule.Acknowledge(pmState, nn::ResultSuccess());
                }
                else
                {
                    BTM_LOG("API or Event in Slept state.\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorInvalidOperationOnSleep());
                }
                break;
            }
            default:
            {
                BTM_LOG("Un-handled State\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
            }
        }
    }
}//NOLINT(impl/function_size)



void InitializeWorker()
{
    g_IsAsyncWork = false;
    BtmState initialState;
    initialState = InitializeHandler(&g_SystemEventForBtCore, &g_SystemEventForBtHid, &g_SystemEventForBtBleCore);
    g_State = initialState;

    //--------------------------------------------------
    //PSCからの通知を受け取るオブジェクトの初期化。pscへの登録
    //--------------------------------------------------
    const nn::psc::PmModuleId dependencies[] = {nn::psc::PmModuleId_Bluetooth, nn::psc::PmModuleId_Fs};
    auto result = g_PmModule.Initialize(
            nn::psc::PmModuleId_Btm,
            dependencies,
            sizeof(dependencies) / sizeof(dependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //--------------------------------------------------
    //Impl - Worker通信用オブジェクトの初期化
    //--------------------------------------------------
    nn::os::InitializeMutex(&nn::btm::g_Mutex, false, 0);
    nn::os::InitializeEvent(&nn::btm::g_CommandEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&nn::btm::g_ReportEvent, false, nn::os::EventClearMode_AutoClear);

    //--------------------------------------------------
    //Worker用マルチウェイトオブジェクトの初期化
    //--------------------------------------------------
    //以下をマルチウェイトする
      //Implからのイベントシグナル
      //下層からのシステムイベントシグナル
      //内部生成APIのイベントシグナル
      //PSCからのイベントシグナル
    nn::os::InitializeMultiWaitHolder(&g_HolderForBtCore, &g_SystemEventForBtCore);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForBtCore, MultiWaitId_BtCore);

    nn::os::InitializeMultiWaitHolder(&g_HolderForBtHid, &g_SystemEventForBtHid);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForBtHid, MultiWaitId_BtHid);

    nn::os::InitializeMultiWaitHolder(&g_HolderForBtBleCore, &g_SystemEventForBtBleCore);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForBtBleCore, MultiWaitId_BtBleCore);

    nn::os::InitializeMultiWaitHolder(&g_HolderForApi, &nn::btm::g_CommandEvent);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForApi, MultiWaitId_Api);

    nn::os::InitializeMultiWaitHolder(&g_HolderForPsc, g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(&g_HolderForPsc, MultiWaitId_Psc);//enum値が被りうるので、PmModuleId_Btmではなく内部定義の値を設定しておく

    nn::os::InitializeMultiWait( &g_MultiWait );
    //マルチウェイトしたいケースとそうでないケースがあるので、マルチウェイト直前で登録する
    //nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtCore );
    //nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForBtHid );
    //nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForApi );
    //nn::os::LinkMultiWaitHolder( &g_MultiWait, &g_HolderForPsc );

    InitializeQueue();
}

void FinalizeWorker()
{
    //--------------------------------------------------
    //Worker用マルチウェイトオブジェクトの末期化
    //--------------------------------------------------
    //nn::os::UnlinkMultiWaitHolder( &g_HolderForBtCore );
    //nn::os::UnlinkMultiWaitHolder( &g_HolderForBtHid );
    //nn::os::UnlinkMultiWaitHolder( &g_HolderForApi );
    //nn::os::UnlinkMultiWaitHolder( &g_HolderForPsc );
    nn::os::FinalizeMultiWaitHolder( &g_HolderForBtCore );
    nn::os::FinalizeMultiWaitHolder( &g_HolderForBtHid );
    nn::os::FinalizeMultiWaitHolder( &g_HolderForBtBleCore );
    nn::os::FinalizeMultiWaitHolder( &g_HolderForApi );
    nn::os::FinalizeMultiWaitHolder( &g_HolderForPsc );
    nn::os::FinalizeMultiWait( &g_MultiWait );

    //--------------------------------------------------
    //Impl - Worker通信用オブジェクトの末期化
    //--------------------------------------------------
    nn::os::FinalizeMutex(&nn::btm::g_Mutex);
    nn::os::FinalizeEvent(&nn::btm::g_CommandEvent);
    nn::os::FinalizeEvent(&nn::btm::g_ReportEvent);

    //--------------------------------------------------
    //Worker内部オブジェクトの末期化
    //--------------------------------------------------
    g_State = BtmState_NotInitialized;

    //--------------------------------------------------
    //PSCからの通知を受け取るオブジェクトの末期化。pscへの登録解除
    //--------------------------------------------------

    FinalizeQueue();

    FinalizeHandler(&g_SystemEventForBtCore, &g_SystemEventForBtHid, &g_SystemEventForBtBleCore);
}

void Worker()
{
    StateMachine();
}

}}
