﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/btm/btm_Result.h>
#include <nn/btm/btm_Types.h>
#include "btm_InternalTypes.h"
#include "btm_Utility.h"
#include "btm_Usecase.h"

namespace nn { namespace btm {

enum InternalUsecase
{
    InternalUsecase_Dynamic2Slot2,
    InternalUsecase_Dynamic2Slot6,
    InternalUsecase_Dynamic2Slot8,
    InternalUsecase_StaticJoy4,
    InternalUsecase_StaticJoy8,
    InternalUsecase_Active,
    InternalUsecase_Local8StaticJoy2,
    InternalUsecase_Local4StaticJoy4,
    InternalUsecase_WithBle1StaticJoy2,
    InternalUsecase_WithBle2StaticJoy2,
    InternalUsecase_WithBle3StaticJoy2,
    InternalUsecase_WithBle4StaticJoy2,
    InternalUsecase_WithBle1StaticJoy2Local8,       // Connection Capacity でのみ使用
    InternalUsecase_WithBle2StaticJoy2Local8,       // Connection Capacity でのみ使用
};

enum ExternalUsecase
{
    ExternalUsecase_Dynamic2Slot,
    ExternalUsecase_StaticJoy,
    ExternalUsecase_Active,
    ExternalUsecase_Local8,
    ExternalUsecase_Local4,
    ExternalUsecase_WithBle,
};


//--------------------------------------------------
//Usecase内の詳細処理
//--------------------------------------------------
enum TransitionType
{
    TransitionType_TransitionWithRestructure,           // ExternalUsecaseを跨いだ遷移
    TransitionType_Transition,                          // ExternalUsecaseを跨がない、InternalUsecaseの遷移
//    TransitionType_TransitionWithRestructureBleOnce,    // ExternalUsecase を跨ぐ遷移、かつ、BLE のConnection Parameter Update を 1 回要する遷移
    TransitionType_TransitionWithRestructureBleTwice,   // ExternalUsecase を跨ぐ遷移、かつ、BLE のConnection Parameter Update を 2 回要する遷移
    TransitionType_Stay,                                // 遷移しない
    TransitionType_Invalid,                             // 未定義の遷移
};
bool GetExternalUsecase(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, ExternalUsecase* pExternalUsecase);
bool GetInternalUsecase(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, InternalUsecase* pInternalUsecase);
bool GetInternalUsecaseForCc(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, InternalUsecase* pInternalUsecase);
void GetUsecaseDefinition(InternalUsecase usecase, SniffMode* pSniffMode, SlotMode* pSlotMode, CeLength* pCeLength);
void GetUsecaseDefinition(InternalUsecase usecase, uint8_t* pConnectionCapacity, uint8_t* pBleConnectionCapacity);
void GetUsecaseDefinition(InternalUsecase usecase, SniffMode* pSniffMode, SlotMode* pSlotMode, uint8_t* pConnectionCapacity, CeLength* pCeLength, uint8_t* pBleConnectionCapacity);
TransitionType GetTransitionType(InternalUsecase current, InternalUsecase next);
bool IsValidSlotMode(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, InternalUsecase internalUsecase, bool isViaRestructure);
bool IsValidSniffMode(const DeviceConditionList* pDcList, SniffMode sniffMode);

bool IsSlotModeMoreThan(const DeviceConditionList* pDcList, SlotMode slotMode, uint8_t* pCount, bool isCheckAllSlots);
bool IsSlotModeMoreThan(const DeviceConditionList* pDcList, SlotMode slotMode, bool isCheckAllSlots);
bool IsSlotModeEqualTo(const DeviceConditionList* pDcList, SlotMode slotMode, bool isCheckAllSlots);
uint8_t GetSlotModeValue(SlotMode slotMode);

bool IsCeLengthEqualTo(const GattClientConditionList* pGccList, CeLength ceLength, bool isCheckAllDevices);

//[Todo]ユースケース定義をデータ化する


void DebugPrintUsecase(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList)
{
    ExternalUsecase externalUsecase;
    InternalUsecase internalUsecase;
    bool isValidExternalUsecase;
    bool isValidInternalUsecase;

    isValidExternalUsecase = GetExternalUsecase(pDcList, pGccList, &externalUsecase);
    isValidInternalUsecase = GetInternalUsecase(pDcList, pGccList, &internalUsecase);


    //条件をPrint出力。文字列は btm_Types.h の enum と対応
    char stringWlanMode[4][32] = {
            "Local4",
            "Local8",
            "None",
            "User8",
    };
    NN_UNUSED(stringWlanMode);

    char stringBluetoothMode[2][32] = {
            "Active",
            "Auto",
    };
    NN_UNUSED(stringBluetoothMode);

    NN_SDK_LOG("%s,", &stringWlanMode[pDcList->wlanMode][0]);
    NN_SDK_LOG("%s,", &stringBluetoothMode[pDcList->bluetoothMode][0]);

    // Slot Saving For Pairing
    NN_SDK_LOG("%s,", pDcList->isSlotSavingForPairing ? "Yes" : "No");

    NN_SDK_LOG("%s,", pDcList->isSlotSaving ? "Yes" : "No");

    // 4 Slot Required
    NN_SDK_LOG("%s,", IsSlotModeMoreThan(pDcList, SlotMode_2, false) ? "Yes" : "No");

    // BLE Required
    NN_SDK_LOG("%s,", pDcList->isLargeSlotRequiredForBle ? "Yes" : "No");

    // BLE Device Count
    NN_SDK_LOG("%d,", pGccList->deviceCount);

    //UsecaseをPrint出力。文字列は btm_Usecase.cpp の enum と対応
    char stringExternalUsecase[6][32] = {
            "Dynamic2Slot",
            "StaticJoy",
            "Active",
            "Local8",
            "Local4",
            "WithBle",
    };
    NN_UNUSED(stringExternalUsecase);

    char stringInternalUsecase[12][32] = {
            "Dynamic2Slot2",
            "Dynamic2Slot6",
            "Dynamic2Slot8",
            "StaticJoy4",
            "StaticJoy8",
            "Active",
            "Local8StaticJoy2",
            "Local4StaticJoy4",
            "Local4StaticJoy2Ble1",
            "Local4StaticJoy2Ble2",
            "Local4StaticJoy2Ble3",
            "Local4StaticJoy2Ble4",
    };
    NN_UNUSED(stringInternalUsecase);

    if(isValidExternalUsecase)
    {
        NN_SDK_LOG("[%s],", &stringExternalUsecase[externalUsecase][0]);
    }
    else
    {
        NN_SDK_LOG("[Invalid],");
    }

    if(isValidInternalUsecase)
    {
        NN_SDK_LOG("[%s]", &stringInternalUsecase[internalUsecase][0]);
    }
    else
    {
        NN_SDK_LOG("[Invalid]");
    }

    NN_SDK_LOG("\n");
} //NOLINT(impl/function_size)

bool CheckUsecase(const DeviceConditionList* pCurrentDcList, const DeviceConditionList* pNextDcList,
                  const GattClientConditionList* pCurrentGccList, const GattClientConditionList* pNextGccList,
                  SniffMode* pRecommendedSniffMode, SlotMode* pDefaultSlotMode, CeLength* pDefaultCeLength,
                  bool* pIsRestructureAllSlotsNeeded, uint8_t* pRestructureBleSlotsCount)
{
    //DcListから現在のInternalUsecaseを確定
    InternalUsecase currentInternalUsecase;
    if(!GetInternalUsecase(pCurrentDcList, pCurrentGccList, &currentInternalUsecase))
    {
        BTM_LOG("Current Usecase is invalid.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorInvalidUsecase());
        return false;
    }
    BTM_LOG("Current InternalUsecase = enum(%d)\n",currentInternalUsecase);

    //DcListから次のInternalUsecaseを確定
    InternalUsecase nextInternalUsecase;
    if(!GetInternalUsecase(pNextDcList, pNextGccList, &nextInternalUsecase))
    {
        BTM_LOG("Next Usecase is invalid\n");
        return false;
    }
    BTM_LOG("Next InternalUsecase = enum(%d)\n",nextInternalUsecase);

    //次のInternalUsecaseにおける、デフォルトのSniffMode/SlotMode/CeLength を確定
    SniffMode sniffMode;
    SlotMode slotMode;
    CeLength ceLength;
    GetUsecaseDefinition(nextInternalUsecase, &sniffMode, &slotMode, &ceLength);

    //遷移タイプの確定。遷移タイプから、スロット再整形が必要か否かを決め、デフォルト設定と共に返す
    TransitionType transitionType;
    transitionType = GetTransitionType(currentInternalUsecase, nextInternalUsecase);
    if(transitionType == TransitionType_Invalid)
    {
        BTM_LOG("Usecase transition not permitted.\n");
        return false;
    }
    else if(transitionType == TransitionType_TransitionWithRestructure ||
            transitionType == TransitionType_TransitionWithRestructureBleTwice)
    {
        BTM_LOG("Usecase transition requires BR/EDR slot restruction for all connections. (%d)\n", transitionType);

        //スロット再整形を伴う遷移タイプでは、再整形後にも上位層から設定されたスロットモードを保てるか精査
        if(IsValidSlotMode(pNextDcList, pNextGccList, nextInternalUsecase, true))
        {
            *pIsRestructureAllSlotsNeeded   = true;
            *pDefaultSlotMode               = slotMode;
            *pRecommendedSniffMode          = sniffMode;
            *pDefaultCeLength               = ceLength;

            if (pNextGccList->deviceCount > 0 && transitionType == TransitionType_TransitionWithRestructureBleTwice)
            {
                *pRestructureBleSlotsCount = 2;
            }
            else if(pNextGccList->deviceCount > 0)
            {
                *pRestructureBleSlotsCount = 1;
            }
            else
            {
                *pRestructureBleSlotsCount = 0;
            }

            return true;
        }
        else
        {
            BTM_LOG("Slot Mode or CE Length is not valid.\n");
            return false;
        }
    }
    else if(transitionType == TransitionType_Transition ||
            transitionType == TransitionType_Stay)
    {
        BTM_LOG("Usecase transition requires BR/EDR slot restruction only for a single connection. (%d)\n", transitionType);

        //スロット再整形を伴わない遷移タイプでは、スロットモードがそのユースケースで許容されているかを精査
        if(IsValidSlotMode(pNextDcList, pNextGccList, nextInternalUsecase, false))
        {
            *pIsRestructureAllSlotsNeeded   = false;
            *pDefaultSlotMode               = slotMode;
            *pRecommendedSniffMode          = sniffMode;
            *pDefaultCeLength               = ceLength;

            if (pNextGccList->deviceCount > 0)
            {
                *pRestructureBleSlotsCount = 1;
            }
            else
            {
                *pRestructureBleSlotsCount = 0;
            }

            return true;
        }
        else
        {
            BTM_LOG("Slot Mode or CE Length is not valid.\n");
            return false;
        }
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        return false;
    }
}//NOLINT(impl/function_size)

bool CheckUsecase(const DeviceConditionList* pCurrentDcList, const DeviceConditionList* pNextDcList,
                  const GattClientConditionList* pCurrentGccList, const GattClientConditionList* pNextGccList,
                  SlotMode* pDefaultSlotMode, CeLength* pDefaultCeLength,
                  bool* pIsRestructureAllSlotsNeeded, uint8_t* pRestructureBleSlotsCount)
{
    SniffMode dummySniffMode;
    return CheckUsecase(pCurrentDcList, pNextDcList,
                        pCurrentGccList, pNextGccList,
                        &dummySniffMode, pDefaultSlotMode, pDefaultCeLength,
                        pIsRestructureAllSlotsNeeded, pRestructureBleSlotsCount);
}


TsiMode GetRecommendedTsi(SniffMode sniffMode, SlotMode slotMode)
{
    TsiMode tsiMode = TsiMode_Active;
    if(sniffMode == SniffMode_5ms)
    {
        if(slotMode == SlotMode_2)
        {
            //TSIの定義上、SlotMode2以外は下層の機能自体が無い
            tsiMode = TsiMode_1Fd1Td1Si5;
        }
        else if(slotMode == SlotMode_4)
        {
            BTM_LOG("Invalid TSI definition.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorInvalidTsi());
        }
        else if(slotMode == SlotMode_6)
        {
            BTM_LOG("Invalid TSI definition.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorInvalidTsi());
        }
        else if(slotMode == SlotMode_Active)
        {
            tsiMode = TsiMode_Active;
        }
    }
    else if(sniffMode == SniffMode_10ms)
    {
        if(slotMode == SlotMode_2)
        {
            tsiMode = TsiMode_9Fd1Td1Si10;
        }
        else if(slotMode == SlotMode_4)
        {
            tsiMode = TsiMode_2Fd1Td3Si10;
        }
        else if(slotMode == SlotMode_6)
        {
            tsiMode = TsiMode_0Fd3Td3Si10;
        }
        else if(slotMode == SlotMode_Active)
        {
            tsiMode = TsiMode_Active;
        }
    }
    else if(sniffMode == SniffMode_15ms)
    {
        if(slotMode == SlotMode_2)
        {
            tsiMode = TsiMode_10Fd1Td1Si15;
        }
        else if(slotMode == SlotMode_4)
        {
            tsiMode = TsiMode_7Fd1Td3Si15;
        }
        else if(slotMode == SlotMode_6)
        {
            tsiMode = TsiMode_3Fd1Td5Si15;
        }
        else if(slotMode == SlotMode_Active)
        {
            tsiMode = TsiMode_Active;
        }
    }
    else if(sniffMode == SniffMode_Active)
    {
        tsiMode = TsiMode_Active;
    }

    return tsiMode;
}

CeLength GetDefaultCeLength(const DeviceConditionList* pDeviceConditionList, const GattClientConditionList* pGattClientConditionList)
{
    InternalUsecase internalUsecase;
    CeLength        ceLength = CeLength_Invalid;

    if (GetInternalUsecase(pDeviceConditionList, pGattClientConditionList, &internalUsecase))
    {
        SniffMode       sniffMode;
        SlotMode        slotMode;

        GetUsecaseDefinition(internalUsecase, &sniffMode, &slotMode, &ceLength);

        NN_UNUSED(sniffMode);
        NN_UNUSED(slotMode);
    }

    return ceLength;
}

uint8_t GetConnectionCapacity(const DeviceConditionList* pCurrentDcList, const GattClientConditionList* pCurrentGccList)
{
    InternalUsecase internalUsecase;
    if(!GetInternalUsecaseForCc(pCurrentDcList, pCurrentGccList, &internalUsecase))
    {
        BTM_LOG("Current Usecase is invalid.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorInvalidUsecase());
        return false;
    }

    uint8_t capacity;
    uint8_t bleCapacity;
    GetUsecaseDefinition(internalUsecase, &capacity, &bleCapacity);
    return capacity;
}

uint8_t GetBleConnectionCapacity(const DeviceConditionList* pCurrentDcList, const GattClientConditionList* pCurrentGccList)
{
    InternalUsecase internalUsecase;
    if (!GetInternalUsecaseForCc(pCurrentDcList, pCurrentGccList, &internalUsecase))
    {
        BTM_LOG("Current Usecase is invalid.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorInvalidUsecase());
        return false;
    }

    uint8_t capacity;
    uint8_t bleCapacity;
    GetUsecaseDefinition(internalUsecase, &capacity, &bleCapacity);
    return bleCapacity;
}

//--------------------------------------------------
//Usecase内の詳細処理
//--------------------------------------------------

//DcListをもとに、ExternalUsecaseを取得
//上位層からの入力に対して、優先順、および禁則となる組み合わせで評価
bool GetExternalUsecase(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, ExternalUsecase* pExternalUsecase)
{
    // WithBle であるべきかを、BLE 接続による 4 スロット要求の有無で判定
    bool isWithBleRequired = (pDcList->isLargeSlotRequiredForBle);

    //StaticJoyであるべきかを、4スロット設定要求の有無で判定
    bool isStaticJoyRequired = false;

    if (!isWithBleRequired)
    {
        for (int i = 0; i < pDcList->deviceCount; i++)
        {
            if (pDcList->device[i].hidDeviceCondition.slotMode == SlotMode_4 &&
                pDcList->device[i].hidDeviceCondition.isLargeSlotRequired == true)
            {
                isStaticJoyRequired = true;
                break;
            }
        }
    }

    //禁則の組み合わせかどうかをまず確認
    if (pDcList->wlanMode == WlanMode_User8 && isWithBleRequired)
    {
        BTM_LOG("WlanMode_User8 is not allowed during withBle\n");
        return false;
    }
    if (pDcList->deviceCount > 2 && isWithBleRequired)
    {
        BTM_LOG("Device Count must be less then 2 during withBle\n");
        return false;
    }
    if (pDcList->bluetoothMode == BluetoothMode_Active && isWithBleRequired)
    {
        BTM_LOG("BluetoothMode_Active is not allowed during withBle\n");
        return false;
    }
    if (pGccList->isLargeCeLengthRequired && pGccList->deviceCount > 1)
    {
        BTM_LOG("When BLE device count is more than 1, large CE length can't be used\n");
        return false;
    }
    if(pDcList->wlanMode == WlanMode_User8 && pDcList->isSlotSavingForPairing)
    {
        BTM_LOG("WlanMode_User8 is not allowed during pairing\n");
        return false;
    }
    if(pDcList->wlanMode == WlanMode_User8 && pDcList->bluetoothMode == BluetoothMode_Active)
    {
        BTM_LOG("WlanMode_User8 is not allowed during BluetoothMode_Active\n");
        return false;
    }
    if(pDcList->wlanMode == WlanMode_Local8 && pDcList->isSlotSavingForPairing)
    {
        BTM_LOG("WlanMode_Local8 is not allowed during pairing\n");
        return false;
    }
    if(pDcList->wlanMode == WlanMode_Local8 && pDcList->bluetoothMode == BluetoothMode_Active)
    {
        BTM_LOG("WlanMode_Local8 is not allowed during BluetoothMode_Active\n");
        return false;
    }
    if(pDcList->wlanMode == WlanMode_Local4 && pDcList->isSlotSavingForPairing)
    {
        BTM_LOG("WlanMode_Local4 is not allowed during pairing\n");
        return false;
    }
    if(pDcList->wlanMode == WlanMode_Local4 && pDcList->bluetoothMode == BluetoothMode_Active)
    {
        BTM_LOG("WlanMode_Local4 is not allowed during BluetoothMode_Active\n");
        return false;
    }
    if(pDcList->bluetoothMode == BluetoothMode_Active && isStaticJoyRequired)
    {
        BTM_LOG("BluetoothMode_Active is not allowed during Static Joy\n");
        return false;
    }

    if (isWithBleRequired)
    {
        *pExternalUsecase = ExternalUsecase_WithBle;
        return true;
    }
    else if(pDcList->wlanMode == WlanMode_User8)
    {
        *pExternalUsecase = ExternalUsecase_StaticJoy;
        return true;
    }
    else if(pDcList->wlanMode == WlanMode_Local8)
    {
        *pExternalUsecase = ExternalUsecase_Local8;
        return true;
    }
    else if(pDcList->wlanMode == WlanMode_Local4)
    {
        *pExternalUsecase = ExternalUsecase_Local4;
        return true;
    }
    else if(pDcList->wlanMode == WlanMode_None)
    {
        if(pDcList->bluetoothMode == BluetoothMode_Active)
        {
            *pExternalUsecase = ExternalUsecase_Active;
            return true;
        }
        else if(pDcList->bluetoothMode == BluetoothMode_Auto)
        {
            if(pDcList->isSlotSavingForPairing)
            {
                *pExternalUsecase = ExternalUsecase_StaticJoy;
                return true;
            }
            else
            {
                if(pDcList->isSlotSaving)
                {
                    *pExternalUsecase = ExternalUsecase_StaticJoy;
                    return true;
                }
                else
                {
                    if(isStaticJoyRequired)
                    {
                        *pExternalUsecase = ExternalUsecase_StaticJoy;
                        return true;
                    }
                    else
                    {
                        *pExternalUsecase = ExternalUsecase_Dynamic2Slot;
                        return true;
                    }
                }
            }
        }
        else
        {
            return false;
        }
    }
    else
    {
        return false;
    }
} // NOLINT(impl/function_size)

//スロットモードを値で返す
//Activeは0, 定義外は255なので注意
uint8_t GetSlotModeValue(SlotMode slotMode)
{
    uint8_t slotModeValue;
    if(slotMode == SlotMode_2)
    {
        slotModeValue = 2;
    }
    else if(slotMode == SlotMode_4)
    {
        slotModeValue = 4;
    }
    else if(slotMode == SlotMode_6)
    {
        slotModeValue = 6;
    }
    else if(slotMode == SlotMode_Active)
    {
        slotModeValue = 0;
    }
    else
    {
        BTM_LOG("Invalid SlotMode.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        slotModeValue = 255;
    }
    return slotModeValue;
}

//デバイスのスロットモードに、指定のスロットモード（数）より大きいものがあるかを確認
//Activeはいつでも指定スロットモード以下として扱う
//[Todo]Activeは許容しないようにする
bool IsSlotModeMoreThan(const DeviceConditionList* pDcList, SlotMode slotMode, uint8_t* pCount, bool isCheckAllSlots)
{
    uint8_t count = 0;
    for(int i=0;i<pDcList->deviceCount;i++)
    {
        if(GetSlotModeValue(pDcList->device[i].hidDeviceCondition.slotMode) > GetSlotModeValue(slotMode))
        {
            if(isCheckAllSlots)
            {
                count++;
            }
            else
            {
                if(pDcList->device[i].hidDeviceCondition.isLargeSlotRequired)
                {
                    count++;
                }
            }
        }
    }

    *pCount = count;
    if(count > 0)
    {
        return true;
    }
    else
    {
        return false;
    }
}

bool IsSlotModeMoreThan(const DeviceConditionList* pDcList, SlotMode slotMode, bool isCheckAllSlots)
{
    uint8_t dummyCount;
    return IsSlotModeMoreThan(pDcList, slotMode, &dummyCount, isCheckAllSlots);
}

bool IsSlotModeEqualTo(const DeviceConditionList* pDcList, SlotMode slotMode, bool isCheckAllSlots)
{
    for(int i=0;i<pDcList->deviceCount;i++)
    {
        if(pDcList->device[i].hidDeviceCondition.slotMode != slotMode &&
           pDcList->device[i].hidDeviceCondition.slotMode != SlotMode_Active)
        {
            if(isCheckAllSlots)
            {
                BTM_LOG("%s: Found unexpected slot mode device.\n", NN_CURRENT_FUNCTION_NAME);
                return false;
            }
            else
            {
                if(pDcList->device[i].hidDeviceCondition.isLargeSlotRequired)
                {
                    BTM_LOG("%s: Found unexpected slot mode device.\n", NN_CURRENT_FUNCTION_NAME);
                    return false;
                }
            }
        }
    }
    return true;
}

bool IsCeLengthEqualTo(const GattClientConditionList* pGccList, CeLength ceLength, bool isCheckAllDevices)
{
    for (auto client : pGccList->gattClients)
    {
        if (client.connectedServer.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle &&
            client.maxCeLength != CeLength_Invalid &&
            client.maxCeLength != ceLength)
        {
            if (isCheckAllDevices)
            {
                BTM_LOG("%s: Found unexpected CE length device. Connection Handle = %d, CE Length = %d\n",
                    NN_CURRENT_FUNCTION_NAME, client.connectedServer.connectionHandle, client.maxCeLength);
                return false;
            }
            else
            {
                if (pGccList->isLargeCeLengthRequired)
                {
                    BTM_LOG("%s: Found unexpected CE length device. Connection Handle = %d, CE Length = %d\n",
                        NN_CURRENT_FUNCTION_NAME, client.connectedServer.connectionHandle, client.maxCeLength);
                    return false;
                }
            }
        }
    }

    return true;
}

//接続台数を最大化するようなInternalUsecaseを取得
//ConnectionCapacity取得用途向け
bool GetInternalUsecaseForCc(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, InternalUsecase* pInternalUsecase)
{
    ExternalUsecase externalUsecase;
    if(!GetExternalUsecase(pDcList, pGccList, &externalUsecase))
    {
        return false;
    }

    BTM_LOG("%s: External Usecase = %d\n", NN_CURRENT_FUNCTION_NAME, externalUsecase);

    switch(externalUsecase)
    {
    case ExternalUsecase_Dynamic2Slot:
    {
        if(pDcList->deviceCount <= 8)
        {
            *pInternalUsecase = InternalUsecase_Dynamic2Slot8;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_StaticJoy:
    {
        if(pDcList->deviceCount <= 8)
        {
            uint8_t count;
            IsSlotModeMoreThan(pDcList, SlotMode_2, &count, true);
            if(count <= 1)
            {
                *pInternalUsecase = InternalUsecase_StaticJoy8;
                return true;
            }
            else
            {
                if(pDcList->deviceCount <= 4)
                {
                    *pInternalUsecase = InternalUsecase_StaticJoy4;
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_Active:
    {
        if(pDcList->deviceCount <= 8)
        {
            *pInternalUsecase = InternalUsecase_Active;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_Local8:
    {
        if(pDcList->deviceCount <= 2)
        {
            *pInternalUsecase = InternalUsecase_Local8StaticJoy2;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_Local4:
    {
        if(pDcList->deviceCount <= 4)
        {
            *pInternalUsecase = InternalUsecase_Local4StaticJoy4;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_WithBle:
    {
        if (pDcList->wlanMode == WlanMode_Local8)
        {
            if (pDcList->deviceCount <= 1 && pGccList->deviceCount <= 1)
            {
                *pInternalUsecase = InternalUsecase_WithBle1StaticJoy2Local8;
                return true;
            }
            else if (pDcList->deviceCount == 0 && pGccList->deviceCount <= 2)
            {
                *pInternalUsecase = InternalUsecase_WithBle2StaticJoy2Local8;
                return true;
            }
        }
        else
        {
            if (pDcList->deviceCount <= 2 && pGccList->deviceCount <= 4)
            {
                *pInternalUsecase = InternalUsecase_WithBle4StaticJoy2;
                return true;
            }
        }

        return false;
        break;
    }
    default:
    {
        return false;
        break;
    }
    }

    BTM_LOG("Usecase handling error.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    return false;
} //NOLINT(impl/function_size)

//DcListをもとに、InternalUsecaseを取得
//btmが操作不可能なパラメータである接続台数をもとに、InternalUsecaseを一意に決定する
//DcListがそのUsecaseで許容されているかのチェックは、より後段で行う。
bool GetInternalUsecase(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, InternalUsecase* pInternalUsecase)
{
    ExternalUsecase externalUsecase;
    if(!GetExternalUsecase(pDcList, pGccList, &externalUsecase))
    {
        return false;
    }

    BTM_LOG("%s: External Usecase %d\n", NN_CURRENT_FUNCTION_NAME, externalUsecase);

    switch(externalUsecase)
    {
    case ExternalUsecase_Dynamic2Slot:
    {
        if(pDcList->deviceCount <= 2)
        {
            *pInternalUsecase = InternalUsecase_Dynamic2Slot2;
            return true;
        }
        else if(pDcList->deviceCount <= 6)
        {
            *pInternalUsecase = InternalUsecase_Dynamic2Slot6;
            return true;
        }
        else if(pDcList->deviceCount <= 8)
        {
            *pInternalUsecase = InternalUsecase_Dynamic2Slot8;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_StaticJoy:
    {
        if(pDcList->deviceCount <= 4)
        {
            *pInternalUsecase = InternalUsecase_StaticJoy4;
            return true;
        }
        else if(pDcList->deviceCount <= 8)
        {
            *pInternalUsecase = InternalUsecase_StaticJoy8;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_Active:
    {
        if(pDcList->deviceCount <= 8)
        {
            *pInternalUsecase = InternalUsecase_Active;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_Local8:
    {
        if(pDcList->deviceCount <= 2)
        {
            *pInternalUsecase = InternalUsecase_Local8StaticJoy2;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_Local4:
    {
        if(pDcList->deviceCount <= 4)
        {
            *pInternalUsecase = InternalUsecase_Local4StaticJoy4;
            return true;
        }
        else
        {
            return false;
        }
        break;
    }
    case ExternalUsecase_WithBle:
    {
        if (pDcList->wlanMode == WlanMode_Local8)
        {
            if (pDcList->deviceCount <= 1 && pGccList->deviceCount <= 1)
            {
                *pInternalUsecase = InternalUsecase_WithBle1StaticJoy2;
                return true;
            }
            else if (pDcList->deviceCount == 0 && pGccList->deviceCount == 2)
            {
                *pInternalUsecase = InternalUsecase_WithBle2StaticJoy2;
                return true;
            }
        }
        else
        {
            if (pDcList->deviceCount <= 2 && pGccList->deviceCount <= 1)
            {
                *pInternalUsecase = InternalUsecase_WithBle1StaticJoy2;
                return true;
            }
            else if (pDcList->deviceCount <= 2 && pGccList->deviceCount == 2)
            {
                *pInternalUsecase = InternalUsecase_WithBle2StaticJoy2;
                return true;
            }
            else if (pDcList->deviceCount <= 2 && pGccList->deviceCount == 3)
            {
                *pInternalUsecase = InternalUsecase_WithBle3StaticJoy2;
                return true;
            }
            else if (pDcList->deviceCount <= 2 && pGccList->deviceCount == 4)
            {
                *pInternalUsecase = InternalUsecase_WithBle4StaticJoy2;
                return true;
            }
        }

        return false;
        break;
    }
    default:
    {
        return false;
        break;
    }
    }

    BTM_LOG("Usecase handling error.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    return false;
}//NOLINT(impl/function_size)

//指定のユースケースにおいて、スロットモード設定が正しいかを確認する
bool IsValidSlotMode(const DeviceConditionList* pDcList, const GattClientConditionList* pGccList, InternalUsecase internalUsecase, bool isViaRestructure)
{
    bool isCheckAllSlots = !isViaRestructure;

    switch(internalUsecase)
    {
    case InternalUsecase_Dynamic2Slot2:
    case InternalUsecase_Dynamic2Slot6:
    case InternalUsecase_Dynamic2Slot8:
    {
        return !IsSlotModeMoreThan(pDcList, SlotMode_2, isCheckAllSlots);
    }
    case InternalUsecase_StaticJoy4:
    {
        return !IsSlotModeMoreThan(pDcList, SlotMode_4, isCheckAllSlots);
    }
    case InternalUsecase_StaticJoy8:
    {
        uint8_t count;
        if(IsSlotModeMoreThan(pDcList, SlotMode_4, isCheckAllSlots))
        {
            return false;
        }
        else if(IsSlotModeMoreThan(pDcList, SlotMode_2, &count, isCheckAllSlots))
        {
            if(count <= 1)
            {
                //スロット数4は1台のみ許容される
                return true;
            }
            else
            {
                return false;
            }
        }
        else
        {
            return true;
        }
    }
    case InternalUsecase_Active:
    {
        return IsSlotModeEqualTo(pDcList, SlotMode_Active, isCheckAllSlots);
    }
    case InternalUsecase_Local4StaticJoy4:
    {
        return IsSlotModeEqualTo(pDcList, SlotMode_4, isCheckAllSlots);
    }
    case InternalUsecase_Local8StaticJoy2:
    {
        return IsSlotModeEqualTo(pDcList, SlotMode_4, isCheckAllSlots);
    }
    case InternalUsecase_WithBle1StaticJoy2:
    {
        return (IsSlotModeEqualTo(pDcList, SlotMode_4, isCheckAllSlots) && IsCeLengthEqualTo(pGccList, CeLength_4, isCheckAllSlots));
    }
    case InternalUsecase_WithBle2StaticJoy2:
    {
        return (IsSlotModeEqualTo(pDcList, SlotMode_4, isCheckAllSlots) && IsCeLengthEqualTo(pGccList, CeLength_4, isCheckAllSlots));
    }
    case InternalUsecase_WithBle3StaticJoy2:
    case InternalUsecase_WithBle4StaticJoy2:
    {
        return (IsSlotModeEqualTo(pDcList, SlotMode_4, isCheckAllSlots) && IsCeLengthEqualTo(pGccList, CeLength_2, isCheckAllSlots));
    }
    default:
    {
        return false;
    }
    }
}


//遷移タイプを決定
TransitionType GetTransitionType(InternalUsecase current, InternalUsecase next)
{
    if(current == next)
    {
        return TransitionType_Stay;
    }

    switch(current)
    {
    case InternalUsecase_Dynamic2Slot2:
    {
        switch(next)
        {
        case InternalUsecase_Dynamic2Slot6:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_StaticJoy8:
        case InternalUsecase_Active:
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_Local8StaticJoy2:
        case InternalUsecase_WithBle1StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_Dynamic2Slot6:
    {
        switch(next)
        {
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_Dynamic2Slot8:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_StaticJoy8:
        case InternalUsecase_Active:
        case InternalUsecase_Local4StaticJoy4:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_Dynamic2Slot8:
    {
        switch(next)
        {
        case InternalUsecase_Dynamic2Slot6:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_StaticJoy8:
        case InternalUsecase_Active:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_StaticJoy4:
    {
        switch(next)
        {
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_Dynamic2Slot6:
        case InternalUsecase_Dynamic2Slot8:
        case InternalUsecase_Active:
            return TransitionType_TransitionWithRestructure;
        case InternalUsecase_StaticJoy8:
            return TransitionType_Transition;
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_Local8StaticJoy2:
        case InternalUsecase_WithBle1StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_StaticJoy8:
    {
        switch(next)
        {
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_Dynamic2Slot6:
        case InternalUsecase_Dynamic2Slot8:
        case InternalUsecase_Active:
            return TransitionType_TransitionWithRestructure;
        case InternalUsecase_StaticJoy4:
            return TransitionType_Transition;
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_Local8StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_Active:
    {
        switch(next)
        {
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_Dynamic2Slot6:
        case InternalUsecase_Dynamic2Slot8:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_StaticJoy8:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_Local8StaticJoy2:
    {
        switch(next)
        {
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_WithBle1StaticJoy2:
        case InternalUsecase_WithBle2StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_Local4StaticJoy4:
    {
        switch(next)
        {
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_Dynamic2Slot6:
        case InternalUsecase_Local8StaticJoy2:
        case InternalUsecase_WithBle1StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_WithBle1StaticJoy2:
    {
        switch (next)
        {
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_Local8StaticJoy2:
        case InternalUsecase_WithBle2StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_WithBle2StaticJoy2:
    {
        switch (next)
        {
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_Local8StaticJoy2:
        case InternalUsecase_WithBle1StaticJoy2:
        case InternalUsecase_WithBle3StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_WithBle3StaticJoy2:
    {
        switch (next)
        {
        case InternalUsecase_WithBle1StaticJoy2:
        case InternalUsecase_WithBle2StaticJoy2:
            return TransitionType_TransitionWithRestructureBleTwice;
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_WithBle4StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    case InternalUsecase_WithBle4StaticJoy2:
    {
        switch (next)
        {
        case InternalUsecase_WithBle1StaticJoy2:
        case InternalUsecase_WithBle2StaticJoy2:
            return TransitionType_TransitionWithRestructureBleTwice;
        case InternalUsecase_Dynamic2Slot2:
        case InternalUsecase_StaticJoy4:
        case InternalUsecase_Local4StaticJoy4:
        case InternalUsecase_WithBle3StaticJoy2:
            return TransitionType_TransitionWithRestructure;
        default:
            return TransitionType_Invalid;
        }
        break;
    }
    default:
    {
        BTM_LOG("Unhandled usecase.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        break;
    }
    }

    return TransitionType_Invalid;
}//NOLINT(impl/function_size)


void GetUsecaseDefinition(InternalUsecase usecase, SniffMode* pSniffMode, SlotMode* pSlotMode, uint8_t* pConnectionCapacity, CeLength* pCeLength, uint8_t* pBleConnectionCapacity)
{
    switch(usecase)
    {
    case InternalUsecase_Dynamic2Slot2:
    {
       *pSniffMode              = SniffMode_5ms;
       *pSlotMode               = SlotMode_2;
       *pConnectionCapacity     = 2;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_Dynamic2Slot6:
    {
       *pSniffMode              = SniffMode_10ms;
       *pSlotMode               = SlotMode_2;
       *pConnectionCapacity     = 6;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_Dynamic2Slot8:
    {
       *pSniffMode              = SniffMode_15ms;
       *pSlotMode               = SlotMode_2;
       *pConnectionCapacity     = 8;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_StaticJoy4:
    {
       *pSniffMode              = SniffMode_15ms;
       *pSlotMode               = SlotMode_2;
       *pConnectionCapacity     = 4;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_StaticJoy8:
    {
       *pSniffMode              = SniffMode_15ms;
       *pSlotMode               = SlotMode_2;
       *pConnectionCapacity     = 8;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_Active:
    {
       *pSniffMode              = SniffMode_Active;
       *pSlotMode               = SlotMode_Active;
       *pConnectionCapacity     = 8;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_Local8StaticJoy2:
    {
       *pSniffMode              = SniffMode_15ms;
       *pSlotMode               = SlotMode_4;
       *pConnectionCapacity     = 2;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_Local4StaticJoy4:
    {
       *pSniffMode              = SniffMode_15ms;
       *pSlotMode               = SlotMode_4;
       *pConnectionCapacity     = 4;
       *pCeLength               = CeLength_Invalid;
       *pBleConnectionCapacity  = 0;
       break;
    }
    case InternalUsecase_WithBle1StaticJoy2:
    {
        *pSniffMode             = SniffMode_15ms;
        *pSlotMode              = SlotMode_4;
        *pConnectionCapacity    = 2;
        *pCeLength              = CeLength_4;
        *pBleConnectionCapacity = 1;
        break;
    }
    case InternalUsecase_WithBle2StaticJoy2:
    {
        *pSniffMode             = SniffMode_15ms;
        *pSlotMode              = SlotMode_4;
        *pConnectionCapacity    = 2;
        *pCeLength              = CeLength_4;
        *pBleConnectionCapacity = 2;
        break;
    }
    case InternalUsecase_WithBle3StaticJoy2:
    {
        *pSniffMode             = SniffMode_15ms;
        *pSlotMode              = SlotMode_4;
        *pConnectionCapacity    = 2;
        *pCeLength              = CeLength_2;
        *pBleConnectionCapacity = 3;
        break;
    }
    case InternalUsecase_WithBle4StaticJoy2:
    {
        *pSniffMode             = SniffMode_15ms;
        *pSlotMode              = SlotMode_4;
        *pConnectionCapacity    = 2;
        *pCeLength              = CeLength_2;
        *pBleConnectionCapacity = 4;
        break;
    }
    case InternalUsecase_WithBle1StaticJoy2Local8:
    {
        *pSniffMode             = SniffMode_15ms;
        *pSlotMode              = SlotMode_4;
        *pConnectionCapacity    = 1;                // Local8 中は足して2台（既に1台つながっているので、追加で1台）
        *pCeLength              = CeLength_4;
        *pBleConnectionCapacity = 2;                // Local8 中は足して2台（既に1台つながっているので、追加で1台 = 2台）
        break;
    }
    case InternalUsecase_WithBle2StaticJoy2Local8:
    {
        *pSniffMode             = SniffMode_15ms;
        *pSlotMode              = SlotMode_4;
        *pConnectionCapacity    = 0;                // Local8 中は、足して2台（既に2台つながっているので、追加で0台）
        *pCeLength              = CeLength_4;
        *pBleConnectionCapacity = 2;                // Local8 中は、足して2台（既に2台つながっているので、追加で0台）
        break;
    }
    default:
    {
       BTM_LOG("Unhandled usecase.\n");
       NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
       break;
    }
    }

    BTM_LOG("InternalUsecase %d:\n", usecase);
    BTM_LOG("    Sniff Mode = %d, SlotMode = %d, ConnectionCapacity = %d, CeLength = %d, BleConnectionCapacity = %d\n",
            *pSniffMode, *pSlotMode, *pConnectionCapacity, *pCeLength, *pBleConnectionCapacity);
} //NOLINT(impl/function_size)

void GetUsecaseDefinition(InternalUsecase usecase, uint8_t* pConnectionCapacity, uint8_t* pBleConnectionCapacity)
{
    SniffMode tempSniffMode;
    SlotMode tempSlotMode;
    CeLength tempCeLength;
    GetUsecaseDefinition(usecase, &tempSniffMode, &tempSlotMode, pConnectionCapacity, &tempCeLength, pBleConnectionCapacity);
}

void GetUsecaseDefinition(InternalUsecase usecase, SniffMode* pSniffMode, SlotMode* pSlotMode, CeLength* pCeLength)
{
    uint8_t tempCc;
    uint8_t tempBleCc;
    GetUsecaseDefinition(usecase, pSniffMode, pSlotMode, &tempCc, pCeLength, &tempBleCc);
}

}}
