﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>

#include <nn/btm/btm_Result.h>
#include "btm_InternalTypes.h"
#include "btm_Queue.h"
#include "btm_Utility.h"

namespace nn { namespace btm {

namespace {
//--------------------------------------------------
//疑似クライアントのキュー
//--------------------------------------------------
nn::os::MessageQueueType g_Queue;
static const int COUNT_OF_QUEUE = 20;
uintptr_t g_QueueBuffer[ COUNT_OF_QUEUE ];

//--------------------------------------------------
//キューが扱うメッセージ
//--------------------------------------------------
struct MessageList{
    Message message[COUNT_OF_QUEUE];
};

MessageList g_MessageList;

//--------------------------------------------------
//送信時にメッセージ操作をする際のMutex
//--------------------------------------------------
nn::os::MutexType g_Mutex;
}


void InitializeQueue()
{
    nn::os::InitializeMessageQueue( &g_Queue, g_QueueBuffer, COUNT_OF_QUEUE );
    nn::os::InitializeMutex(&g_Mutex, false, 0);
    for(int i=0;i<COUNT_OF_QUEUE;i++)
    {
        g_MessageList.message[i].isUsing = false;
    }
}

void FinalizeQueue()
{
    nn::os::FinalizeMessageQueue(&g_Queue);
    nn::os::FinalizeMutex(&g_Mutex);
}

uint8_t SearchEmptyIndex()
{
    for(int i=0;i<COUNT_OF_QUEUE;i++)
    {
        if(g_MessageList.message[i].isUsing == false)
        {
            return i;
        }
    }

    NN_DETAIL_BTM_FATAL("Message full.\n");
    //キュー順を無視してメッセージを全て出力
    for(int i=0;i<COUNT_OF_QUEUE;i++)
    {
        NN_DETAIL_BTM_FATAL("%d use %d apiId %d\n",i , g_MessageList.message[i].isUsing, g_MessageList.message[i].apiId);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorQueueFull());
    return 0xFF;
}

bool IsCompensatingMessage(const Message* pMessageA, const Message* pMessageB)
{
    if(pMessageA->apiId == ApiId_StartGamepadPairing &&
       pMessageB->apiId == ApiId_CancelGamepadPairing)
    {
        return true;
    }
    if(pMessageA->apiId == ApiId_CancelGamepadPairing &&
       pMessageB->apiId == ApiId_StartGamepadPairing)
    {
        return true;
    }
    if(pMessageA->apiId == ApiId_EnableRadio &&
       pMessageB->apiId == ApiId_DisableRadio)
    {
        return true;
    }
    if(pMessageA->apiId == ApiId_DisableRadio &&
       pMessageB->apiId == ApiId_EnableRadio)
    {
        return true;
    }
    if (pMessageA->apiId == ApiId_BleConnect &&
        pMessageB->apiId == ApiId_BleConnect)
    {
        const CommandParam_BleConnect* paramA = reinterpret_cast<const CommandParam_BleConnect*>(pMessageA->buffer);
        const CommandParam_BleConnect* paramB = reinterpret_cast<const CommandParam_BleConnect*>(pMessageB->buffer);

        return (paramA->address == paramB->address);
    }

    return false;
}

//不要なMessageを適宜削除するDeQueue
void OptimizedDeQueue(Message* pMessage)
{
    for(;;)
    {
        uintptr_t pointerToData;

        //Message受信
        Message* pMessageA;
        nn::os::ReceiveMessageQueue(&pointerToData, &g_Queue);
        pMessageA = reinterpret_cast<Message*>(pointerToData);

        //後続のMessage有無確認
        bool isMessageB;
        Message* pMessageB;
        isMessageB = nn::os::TryPeekMessageQueue(&pointerToData, &g_Queue);
        pMessageB = reinterpret_cast<Message*>(pointerToData);

        if(isMessageB)//2つ取れた場合
        {
            if(IsCompensatingMessage(pMessageA, pMessageB))
            {
                //BがAを打ち消す。Aを捨てて再評価
                pMessageA->isUsing = false;
            }
            else
            {
                //打ち消さない。Bを残し、Aを返す
                *pMessage = *pMessageA;
                pMessageA->isUsing = false;//フラグ落とした時点で送信側から再利用されうるので、バッファコピー後に落とす
                break;
            }
        }
        else//1つ取れた場合
        {
            //Aを返す
            *pMessage = *pMessageA;
            pMessageA->isUsing = false;//フラグ落とした時点で送信側から再利用されうるので、バッファコピー後に落とす
            break;
        }
    }
}

//入力のisUsingは無視
void EnQueue(const Message* pMessage)
{
    LockMutex(&g_Mutex);

    uint8_t index;
    index = SearchEmptyIndex();

    g_MessageList.message[index] = *pMessage;
    g_MessageList.message[index].isUsing = true;
    nn::os::SendMessageQueue( &g_Queue, reinterpret_cast<uintptr_t>(&g_MessageList.message[index]));
    UnlockMutex(&g_Mutex);
}

//出力のisUsingは要無視
//読む側は常に一人なので、Mutexは取らない
void DeQueue(Message* pMessage)
{
    OptimizedDeQueue(pMessage);
    /*
    uintptr_t pointerToData;
    nn::os::ReceiveMessageQueue( &pointerToData, &g_Queue );

    Message* pReceivedMessage;
    pReceivedMessage = reinterpret_cast<Message*>(pointerToData);
    *pMessage = *pReceivedMessage;
    pReceivedMessage->isUsing = false;//フラグ落とした時点で送信側から再利用されうるので、バッファコピー後に落とす
    */
}

}}
