﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/btm/btm_IBtm.sfdl.h>
#include <nn/btm/btm_Result.h>

#include "btm_Impl.h"
#include "btm_Worker.h"
#include "btm_Queue.h"
#include "btm_Utility.h"

#include <nn/btm/btm_ServiceName.h>

//--------------------------------------------------------------------------
//SF機能
//--------------------------------------------------------------------------
namespace {

struct MyServerManagerOption
{
    static const size_t PointerTransferBufferSize = 4096;
};

class BtmServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<17, 4, MyServerManagerOption>
{
};

BtmServerManager*  g_pBtmServerManager;
std::aligned_storage<sizeof(BtmServerManager), NN_ALIGNOF(BtmServerManager)>::type g_BtmServerManagerStorage;


void InitializeBtmServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pBtmServerManager);
    g_pBtmServerManager = new (&g_BtmServerManagerStorage) BtmServerManager;

    // サービス名の登録とポートの初期化
    auto sessionCountMax = 6;
    typedef nn::sf::ObjectFactory<nn::sf::impl::StaticOneAllocationPolicy>  Factory;
    g_pBtmServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::btm::IBtm,         nn::btm::BtmImpl>(),        sessionCountMax,    nn::btm::BtmServiceName);
    g_pBtmServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::btm::IBtmSystem,   nn::btm::BtmSystemImpl>(),  sessionCountMax,    nn::btm::BtmSystemServiceName);
    g_pBtmServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::btm::IBtmUser,     nn::btm::BtmUserImpl>(),    sessionCountMax,    nn::btm::BtmUserServiceName);
    g_pBtmServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::btm::IBtmDebug,    nn::btm::BtmDebugImpl>(),   sessionCountMax,    nn::btm::BtmDebugServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は LoopAuto() 関数の呼び出しによって行なわれる。
    g_pBtmServerManager->Start();
}


void FinalizeBtmServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pBtmServerManager);

    // placement new でコンストラクトされているので明示的にデストラクタを呼ぶ
    g_pBtmServerManager->~BtmServerManager();
    g_pBtmServerManager = nullptr;
}


void LoopBtmServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pBtmServerManager);
    g_pBtmServerManager->LoopAuto();
}


void RequestStopBtmServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pBtmServerManager);
    g_pBtmServerManager->RequestStop();
}

} // namespace



//--------------------------------------------------------------------------
//スレッド オブジェクト
//--------------------------------------------------------------------------
namespace {
const size_t ThreadStackSize = 8192;

NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackA[ ThreadStackSize ];
nn::os::ThreadType  g_ThreadA;
char g_ThreadNameA[]="ThreadA";

NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackB[ ThreadStackSize ];
nn::os::ThreadType  g_ThreadB;
char g_ThreadNameB[]="ThreadB";

NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackQ[ ThreadStackSize ];
nn::os::ThreadType  g_ThreadQ;
char g_ThreadNameQ[]="ThreadQ";
}

//--------------------------------------------------------------------------
//SF サーバースレッド
//--------------------------------------------------------------------------
void BtmServerThread(void *arg)
{
    char* pThreadName = static_cast<char*>(arg);
    NN_UNUSED( pThreadName );
    //BTM_LOG("Started ServerThread %s\n",pThreadName);
    LoopBtmServer();
}

//--------------------------------------------------------------------------
//疑似サーバスレッド
//--------------------------------------------------------------------------
static const int QUASI_API_RETRY_WAIT_MSEC = 20;
void BtmQuasiAgentThread(void *arg)
{
    char* pThreadName = static_cast<char*>(arg);
    NN_UNUSED( pThreadName );

    nn::btm::Message message;
    nn::Result result;
    uint8_t dummyReportBuffer[nn::btm::SIZE_OF_REPORT_BUFFER];

    for(;;)
    {
        nn::btm::DeQueue(&message);
        for(;;)
        {
            result = nn::btm::InvokeWorker(message.apiId, &message.buffer[0], nn::btm::SIZE_OF_COMMAND_BUFFER, &dummyReportBuffer[0], nn::btm::SIZE_OF_REPORT_BUFFER);
            if(result.IsSuccess())
            {
                break;
            }
            else if(nn::btm::ResultBusy::Includes(result))
            {
                //Busyの時のみリトライを続ける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(QUASI_API_RETRY_WAIT_MSEC));
            }
            else
            {
                break;
            }
        }
    }
}

//--------------------------------------------------------------------------
// Main
//--------------------------------------------------------------------------
extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    //BTM_LOG("Bluetooth Manager nnMain() \n");
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(btm, Main));

#ifdef BTM_PRINT_BUSY_TIME
    nn::os::Tick startTick = nn::os::GetSystemTick();
#endif

    nn::btm::InitializeWorker();

#ifdef BTM_PRINT_BUSY_TIME
    nn::os::Tick mediumTick = nn::os::GetSystemTick();
#endif

    InitializeBtmServer();

#ifdef BTM_PRINT_BUSY_TIME
    nn::os::Tick endTick = nn::os::GetSystemTick();

    nn::TimeSpan startTs  = nn::os::ConvertToTimeSpan(startTick);
    nn::TimeSpan mediumTs = nn::os::ConvertToTimeSpan(mediumTick);
    nn::TimeSpan endTs    = nn::os::ConvertToTimeSpan(endTick);

    NN_DETAIL_BTM_INFO("InitializeWorker    %16lld(msec)\n", mediumTs.GetMilliSeconds() - startTs.GetMilliSeconds());
    NN_DETAIL_BTM_INFO("InitializeBtmServer %16lld(msec)\n", endTs.GetMilliSeconds() - mediumTs.GetMilliSeconds());
    NN_UNUSED(startTs);
    NN_UNUSED(mediumTs);
    NN_UNUSED(endTs);
#endif

    //SF サーバ用スレッドの作成
    nn::Result result;
    result = nn::os::CreateThread( &g_ThreadA, BtmServerThread, g_ThreadNameA, g_ThreadStackA, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(btm, IpcServerA) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&g_ThreadA, NN_SYSTEM_THREAD_NAME(btm, IpcServerA));
    nn::os::StartThread( &g_ThreadA );

    result = nn::os::CreateThread( &g_ThreadB, BtmServerThread, g_ThreadNameB, g_ThreadStackB, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(btm, IpcServerB) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&g_ThreadB, NN_SYSTEM_THREAD_NAME(btm, IpcServerB));
    nn::os::StartThread( &g_ThreadB );

    //疑似サーバー用スレッドの作成
    result = nn::os::CreateThread( &g_ThreadQ, BtmQuasiAgentThread, g_ThreadNameQ, g_ThreadStackQ, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(btm, QuasiServer) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&g_ThreadQ, NN_SYSTEM_THREAD_NAME(btm, QuasiServer));
    nn::os::StartThread( &g_ThreadQ );

    //Mainスレッドは以後Workerスレッドとして動作する
    nn::btm::Worker();

    //[Todo]終了処理対応
    NN_UNUSED( RequestStopBtmServer );
    FinalizeBtmServer();
    nn::btm::FinalizeWorker();
}
