﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_SdkLog.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator

#include <nn/btm/btm_Result.h>
#include <nn/btm/user/btm_UserResult.h>
#include <nn/btm/system/btm_SystemResult.h>

#include "btm_Impl.h"
#include "btm_InternalTypes.h"
#include "btm_Worker.h"
#include "btm_Handler.h"
#include "btm_Queue.h"
#include "btm_Utility.h"
#include "btm_BleScanFilters.h"

namespace nn { namespace btm {

static const int API_RETRY_WAIT_MSEC = 10;
static const int API_RETRY_COUNT     = 1000;

namespace {
struct  BtmSystemCoreByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BtmSystemCoreByHipcTag>   BtmSystemCoreAllocator;
struct  BtmUserCoreByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BtmUserCoreByHipcTag>     BtmUserCoreAllocator;

nn::Result ConvertToUserResult(nn::Result& result)
{
    if (user::ResultUser().Includes(result) || result.IsSuccess())
    {
        return result;
    }
    if (ResultInvalidArgument().Includes(result))
    {
        return user::ResultInvalidArgument();
    }
    else if (ResultBusyWorking().Includes(result))
    {
        return user::ResultBusyWorking();
    }
    else if (ResultBusy().Includes(result))
    {
        return user::ResultBusy();
    }
    else if (ResultFailureLowLayerGeneral().Includes(result))
    {
        return user::ResultFailureLowLayerGeneral();
    }
    else if (ResultFailureLowLayerCritical().Includes(result))
    {
        return user::ResultFailureLowLayerCritical();
    }
    else if (ResultFailureLowLayer().Includes(result))
    {
        return user::ResultFailureLowLayer();
    }
    else if (ResultInternalErrorTimeoutBleConfigScanFilterCondition().Includes(result))
    {
        return user::ResultBusy();
    }
    else if (ResultInvalidStateRadioOff().Includes(result))
    {
        return user::ResultBluetoothOff();
    }
    else if (ResultInvalidUsecase().Includes(result))
    {
        return user::ResultInvalidOperationForCurrentUsecase();
    }
    else if (ResultScanStartFailedFull().Includes(result))
    {
        return user::ResultScanStartFailedFull();
    }
    else if (ResultBleGattDataPathRegisterFailed().Includes(result))
    {
        return user::ResultBleGattDataPathRegisterFailed();
    }
    else
    {
        NN_SDK_LOG("[btm] Failed to convert btm result to btm user result. Unexpected result. Code = %d\n", result.GetDescription());
        return result;
    }
}
} // namespace

// BtmSystemCoreAllocator を静的コンストラクタで初期化するためのヘルパー
class BtmSystemCoreAllocatorInitializer
{
public:

    BtmSystemCoreAllocatorInitializer() NN_NOEXCEPT
    {
        BtmSystemCoreAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BtmSystemCoreAllocatorInitializer;

// BtmUserCoreAllocator を静的コンストラクタで初期化するためのヘルパー
class BtmUserCoreAllocatorInitializer
{
public:

    BtmUserCoreAllocatorInitializer() NN_NOEXCEPT
    {
        BtmUserCoreAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BtmUserCoreAllocatorInitializer;


void BtmImpl::GetHostDevicePropertyImpl(nn::sf::Out<nn::btm::HostDeviceProperty> pHostDeviceProperty) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    *pHostDeviceProperty = GetHostDeviceProperty();
}

void BtmImpl::GetStateImpl(nn::sf::Out<nn::btm::BtmState> pState) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部ステートを直接読み出す
    *pState = GetState();
}

bool BtmImpl::AcquireDeviceConditionEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireDeviceConditionEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireDeviceConditionEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

void BtmImpl::GetDeviceConditionImpl(nn::sf::Out<nn::btm::DeviceConditionList> pList) NN_NOEXCEPT
{
    *pList = GetDeviceConditionList();

    //カウントが壊れていた場合の、ロンチ向けワークアラウンド
    //[Todo]ワークアラウンドを抜き、下層が異常状態になったとしてもカウントが壊れないようにする
    if(pList->deviceCount > COUNT_OF_DC_LIST)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorOverConnection());//下層の問題修正確認のため、ABORTを追加
        pList->deviceCount = 0;
    }
}

nn::Result BtmImpl::SetBurstModeImpl(nn::btm::BdAddress bdAddress, bool isBurstMode) NN_NOEXCEPT
{
    CommandParam_SetBurstMode command;
    command.address = bdAddress;
    command.isBurstMode = isBurstMode;
    auto result = InvokeWorker(ApiId_SetBurstMode,&command,sizeof(command),nullptr,0);
    return result;
}

nn::Result BtmImpl::SetSlotModeImpl(const nn::btm::DeviceSlotModeList& deviceSlotModeList) NN_NOEXCEPT
{
    CommandParam_SetSlotMode command;
    command.list = deviceSlotModeList;
    auto result = InvokeWorker(ApiId_SetSlotMode,&command,sizeof(command),nullptr,0);
    return result;
}

nn::Result BtmImpl::SetBluetoothModeImpl(nn::btm::BluetoothMode bluetoothMode) NN_NOEXCEPT
{
    CommandParam_SetBluetoothMode command;
    command.bluetoothMode = bluetoothMode;
    auto result = InvokeWorker(ApiId_SetBluetoothMode,&command,sizeof(command),nullptr,0);
    return result;
}

nn::Result BtmImpl::SetWlanModeImpl(nn::btm::WlanMode wlanMode) NN_NOEXCEPT
{
    CommandParam_SetWlanMode command;
    command.wlanMode = wlanMode;
    auto result = InvokeWorker(ApiId_SetWlanMode,&command,sizeof(command),nullptr,0);
    return result;
}

bool BtmImpl::AcquireDeviceInfoEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireDeviceInfoEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireDeviceInfoEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

void BtmImpl::GetDeviceInfoImpl(nn::sf::Out<nn::btm::DeviceInfoList> pList) NN_NOEXCEPT
{
    *pList = GetDeviceInfoList();
}

void BtmImpl::AddDeviceInfoImpl(const nn::btm::DeviceInfo& info) NN_NOEXCEPT
{
    CommandParam_AddDeviceInfo command;
    command.info = info;

    Message message;
    message.apiId = ApiId_AddDeviceInfo;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);
}

void BtmImpl::RemoveDeviceInfoImpl(nn::btm::BdAddress address) NN_NOEXCEPT
{
    CommandParam_RemoveDeviceInfo command;
    command.address = address;

    Message message;
    message.apiId = ApiId_RemoveDeviceInfo;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);
}

void BtmImpl::ProtectDeviceInfoImpl(nn::btm::BdAddress address, bool isProtect) NN_NOEXCEPT
{
    CommandParam_ProtectDeviceInfo command;
    command.address = address;
    command.isProtect = isProtect;

    Message message;
    message.apiId = ApiId_ProtectDeviceInfo;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);
}

void BtmImpl::IncreaseDeviceInfoOrderImpl(nn::btm::BdAddress address) NN_NOEXCEPT
{
    CommandParam_IncreaseDeviceInfoOrder command;
    command.address = address;

    Message message;
    message.apiId = ApiId_IncreaseDeviceInfoOrder;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);
}

bool BtmImpl::AcquireLlrStateEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireLlrStateEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireLlrStateEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

bool BtmImpl::IsLlrStartedImpl() NN_NOEXCEPT
{
    return IsLlrStarted();
}

nn::Result BtmImpl::LlrNotifyImpl(nn::btm::BdAddress address) NN_NOEXCEPT
{
    CommandParam_LlrNotify command;
    command.address = address;
    auto result = InvokeWorker(ApiId_LlrNotify,&command,sizeof(command),nullptr,0);
    return result;
}

void BtmImpl::EnableRadioImpl() NN_NOEXCEPT
{
    //スリープ直前に呼ばれ、なおかつスリープ前に処理が必須のAPIのため、同期として処理する
    //API_RETRY_WAIT_MSEC 10ms * 1000 = 10sec をタイムアウトとし、タイムアウトした場合はABORTさせる
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_EnableRadio,nullptr,0,nullptr,0);
        if(result.IsSuccess())
        {
            return;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInvalidStateNotInitialized::Includes(result))
        {
            //生産工程向けステートでは何もしない
            return;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
}

void BtmImpl::DisableRadioImpl() NN_NOEXCEPT
{
    //スリープ直前に呼ばれ、なおかつスリープ前に処理が必須のAPIのため、同期として処理する
    //API_RETRY_WAIT_MSEC 10ms * 1000 = 10sec をタイムアウトとし、タイムアウトした場合はABORTさせる
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_DisableRadio,nullptr,0,nullptr,0);
        if(result.IsSuccess())
        {
            return;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInvalidStateNotInitialized::Includes(result))
        {
            //生産工程向けステートでは何もしない
            return;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
}

bool BtmImpl::AcquireAwakeReqEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireAwakeReqEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireAwakeReqEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

void BtmImpl::EnableSlotSavingImpl(bool isEnable) NN_NOEXCEPT
{
    Message message;
    if(isEnable)
    {
        message.apiId = ApiId_EnableSlotSaving;
    }
    else
    {
        message.apiId = ApiId_DisableSlotSaving;
    }
    EnQueue(&message);
}

void BtmImpl::HidDisconnectImpl(nn::btm::BdAddress bdAddress) NN_NOEXCEPT
{
    CommandParam_HidDisconnect command;
    command.address = bdAddress;

    Message message;
    message.apiId = ApiId_HidDisconnect;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);
}

nn::Result BtmImpl::HidSetRetransmissionModeImpl(nn::btm::BdAddress bdAddress, const nn::btm::ZeroRetransmissionList& zeroRetransmissionList) NN_NOEXCEPT
{
    CommandParam_HidSetRetransmissionMode command;
    command.address = bdAddress;
    command.zeroRetransmissionList = zeroRetransmissionList;
    auto result = InvokeWorker(ApiId_HidSetRetransmissionMode,&command,sizeof(command),nullptr,0);
    return result;
}

bool BtmImpl::AcquireBleScanEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleScanEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleScanEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
           return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmImpl::GetBleScanParameterGeneralImpl(nn::sf::Out<user::BleAdvFilterForGeneral> pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    if (!GetScanFilterParameter(pFilter.GetPointer(), parameterId))
    {
        return ResultInvalidArgument();
    }

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::GetBleScanParameterSmartDeviceImpl(nn::sf::Out<user::BleAdvFilterForSmartDevice> pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    if (!GetScanFilterParameter(pFilter.GetPointer(), parameterId))
    {
        return ResultInvalidArgument();
    }

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::StartBleScanForGeneralImpl(user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return ResultInvalidStateRadioOff();
    }

    CommandParam_StartBleScanForGeneral command;
    command.aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    command.condition = filter;
    auto result = InvokeWorker(ApiId_BleStartScanGeneral, &command, sizeof(command), nullptr, 0);
    return result;
}

nn::Result BtmImpl::StopBleScanForGeneralImpl() NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return ResultInvalidStateRadioOff();
    }

    CommandParam_StopBleScanForGeneral command;
    command.aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    auto result = InvokeWorker(ApiId_BleStopScanGeneral, &command, sizeof(command), nullptr, 0);
    return result;
}

void BtmImpl::GetBleScanResultsForGeneralImpl(nn::sf::OutArray<nn::btm::user::ScanResult> results, nn::sf::Out<uint8_t> pResultNum) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::ScanResult btmResults[nn::bluetooth::BleScanResultCountMax / 2];

    *pResultNum = GetScanResultGeneral(btmResults, NN_ARRAY_SIZE(btmResults), nn::applet::AppletResourceUserId::GetInvalidId());

    *pResultNum = (results.GetLength() < *pResultNum) ? results.GetLength() : *pResultNum;

    uint8_t num = (results.GetLength() < NN_ARRAY_SIZE(btmResults)) ? results.GetLength() : NN_ARRAY_SIZE(btmResults);

    for (int i = 0; i < num; ++i)
    {
        memcpy(&results[i], &btmResults[i], sizeof(user::ScanResult));
    }
}

nn::Result BtmImpl::StartBleScanForPairedDeviceImpl(user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return ResultInvalidStateRadioOff();
    }

    CommandParam_StartBleScanForPaired command;
    command.aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    command.condition = filter;
    auto result = InvokeWorker(ApiId_BleStartScanPaired, &command, sizeof(command), nullptr, 0);
    return result;
}

nn::Result BtmImpl::StopBleScanForPairedDeviceImpl() NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return ResultInvalidStateRadioOff();
    }

    CommandParam_StopBleScanForPaired command;
    command.aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    auto result = InvokeWorker(ApiId_BleStopScanPaired, &command, sizeof(command), nullptr, 0);
    return result;
}

nn::Result BtmImpl::StartBleScanForSmartDeviceImpl(user::BleAdvFilterForSmartDevice filter) NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return ResultInvalidStateRadioOff();
    }

    CommandParam_StartBleScanForSmartDevice command;
    command.aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    command.condition = filter;
    auto result = InvokeWorker(ApiId_BleStartScanSdGeneral, &command, sizeof(command), nullptr, 0);
    return result;
}

nn::Result BtmImpl::StopBleScanForSmartDeviceImpl() NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return ResultInvalidStateRadioOff();
    }

    CommandParam_StopBleScanForSmartDevice command;
    command.aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    auto result = InvokeWorker(ApiId_BleStopScanSdGeneral, &command, sizeof(command), nullptr, 0);
    return result;
}

void BtmImpl::GetBleScanResultsForSmartDeviceImpl(nn::sf::OutArray<nn::btm::user::ScanResult> results, nn::sf::Out<uint8_t> pResultNum) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::ScanResult btmResults[nn::bluetooth::BleScanResultCountMax / 2];

    *pResultNum = GetScanResultSmartDevice(btmResults, NN_ARRAY_SIZE(btmResults), nn::applet::AppletResourceUserId::GetInvalidId());

    *pResultNum = (results.GetLength() < *pResultNum) ? results.GetLength() : *pResultNum;

    uint8_t num = (results.GetLength() < NN_ARRAY_SIZE(btmResults)) ? results.GetLength() : NN_ARRAY_SIZE(btmResults);

    for (int i = 0; i < num; ++i)
    {
        memcpy(&results[i], &btmResults[i], sizeof(user::ScanResult));
    }
}

bool BtmImpl::AcquireBleConnectionEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleConnectionEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleConnectionEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmImpl::BleConnectImpl(nn::btm::BdAddress bdAddress) NN_NOEXCEPT
{
    if (IsRadioEnabled() == false)
    {
        return nn::btm::ResultInvalidStateRadioOff();
    }

    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    uint8_t connectionCount = GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    if (connectionCount == nn::bluetooth::BleConnectionCountMaxClient)
    {
        return nn::btm::ResultBleConnectionFailedFull();
    }

    for (auto state : connState)
    {
        if (state.address == bdAddress)
        {
            return nn::btm::ResultBleConnectionAlreadyExist();
        }
    }

    auto result = InvokeWorker(ApiId_BleTryConnect, nullptr, 0, nullptr, 0);

    if (result.IsFailure())
    {
        return result;
    }

    CommandParam_BleConnect command;
    command.aruid   = nn::applet::AppletResourceUserId::GetInvalidId();
    command.address = bdAddress;

    Message message;
    message.apiId = ApiId_BleConnect;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::BleOverrideConnectionImpl(uint32_t connectionHandle) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    bool connectionExist = false;

    for (auto state : connState)
    {
        if (state.connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return ResultBleConnectionNotFound();
    }

    CommandParam_BleOverrideConnection command;
    command.aruid               = nn::applet::AppletResourceUserId::GetInvalidId();
    command.connectionHandle    = connectionHandle;

    auto result = InvokeWorker(ApiId_BleOverrideConnection, &command, sizeof(command), nullptr, 0);

    return result;
}

nn::Result BtmImpl::BleDisconnectImpl(uint32_t connectionHandle) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    bool connectionExist = false;

    for (auto state : connState)
    {
        if (state.connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return ResultBleConnectionNotFound();
    }

    CommandParam_BleDisconnect command;
    command.connectionHandle = connectionHandle;

    Message message;
    message.apiId = ApiId_BleDisconnect;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

void BtmImpl::BleGetConnectionStateImpl(nn::sf::OutArray<nn::btm::user::BleClientConnState> pConnState, nn::sf::Out<uint8_t> pOutNum) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    *pOutNum = GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    *pOutNum = (pConnState.GetLength() < *pOutNum) ? pConnState.GetLength() : *pOutNum;

    int copiedIndex = 0;

    for (int i = 0; i < *pOutNum; ++i)
    {
        for (int j = copiedIndex; j < NN_ARRAY_SIZE(connState); ++j)
        {
            if (connState[j].connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
            {
                memcpy(&pConnState[i], &connState[j], sizeof(nn::btm::user::BleClientConnState));
                copiedIndex = j + 1;
                break;
            }
        }
    }

    for (int i = *pOutNum; i < pConnState.GetLength(); ++i)
    {
        memset(&pConnState[i].address, 0x00, sizeof(nn::bluetooth::Address));
        pConnState[i].connectionHandle = nn::bluetooth::BleInvalidConnectionHandle;
    }
}

void BtmImpl::BleGetGattClientConditionListImpl(nn::sf::Out<GattClientConditionList> pGattClientConditionList) NN_NOEXCEPT
{
    *pGattClientConditionList = GetGattClientConditionList();
}

bool BtmImpl::AcquireBlePairingEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBlePairingEvent buffer;
    for (int i = 0; i < API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBlePairingEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmImpl::BlePairDeviceImpl(uint32_t connectionHandle, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BLE_GATT_CLIENT_NUM_MAX];

    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));
    bool connectionExist = false;

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return ResultBleConnectionNotFound();
    }
    else if (!GetBlePairingSupported(connectionHandle))
    {
        return ResultBlePairingNotSupported();
    }
    else if (GetBlePairingOnGoing())
    {
        return ResultBlePairingOnGoing();
    }

    CommandParam_StartBlePairing command;

    command.connectionHandle = connectionHandle;
    command.condition = filter;
    command.pairing = true;

    Message message;
    message.apiId = ApiId_StartBlePairing;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::BleUnpairDeviceOnBothImpl(uint32_t connectionHandle, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));
    bool connectionExist = false;

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return ResultBleConnectionNotFound();
    }
    else if (!GetBlePairingSupported(connectionHandle))
    {
        return ResultBlePairingNotSupported();
    }
    else if (GetBlePairingOnGoing())
    {
        return ResultBlePairingOnGoing();
    }

    CommandParam_StartBlePairing command;

    command.connectionHandle = connectionHandle;
    command.condition = filter;
    command.pairing = false;

    Message message;
    message.apiId = ApiId_StartBlePairing;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::BleUnpairDeviceImpl(nn::btm::BdAddress bdAddress, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        // 接続が存在する場合は、双方からペアリング削除する
        if (connState[i].address == bdAddress)
        {
            auto result = BleUnpairDeviceOnBothImpl(connState[i].connectionHandle, filter);

            // 成功したら終わり
            if (result.IsSuccess())
            {
                return result;
            }
        }
    }

    // 失敗したら、本体からのみペアリングを削除
    if (GetBlePairingOnGoing())
    {
        return ResultBlePairingOnGoing();
    }

    CommandParam_BlePairingUpdateDatabase command;

    command.address = bdAddress;
    command.condition = filter;
    command.type = BlePairingAdvDataType_ManuCliSerId;
    command.pairing = false;

    Message message;
    message.apiId = ApiId_BlePairingUpdateDatabase;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

void BtmImpl::BleGetPairedAddressesImpl(nn::sf::OutArray<BdAddress> pAddresses, nn::sf::Out<uint8_t> pOutNum, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    *pOutNum = GetBlePairedDeviceAddress(pAddresses.GetData(), pAddresses.GetLength(), filter);

    return;
}

bool BtmImpl::AcquireBleServiceDiscoveryEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleSdpEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleSdpEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

void BtmImpl::GetGattServicesImpl(nn::sf::OutArray<user::GattService> pServices, nn::sf::Out<uint8_t> pOutNum, uint32_t connectionHandle) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    *pOutNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);

    *pOutNum = (pServices.GetLength() < *pOutNum) ? pServices.GetLength() : *pOutNum;

    for (int i = 0; i < *pOutNum; ++i)
    {
        memcpy(&pServices[i], &services[i], sizeof(user::GattService));
    }
}

bool BtmImpl::GetGattServiceImpl(nn::sf::Out<user::GattService> pService,
                                 uint32_t connectionHandle, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    uint8_t serviceNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);

    for (int i = 0; i < serviceNum; ++i)
    {
        if (services[i].uuid == uuid)
        {
            *pService = services[i];
            return true;
        }
    }

    return false;
}

bool BtmImpl::GetBelongingServiceImpl(nn::sf::Out<user::GattService> pService,
                                      uint32_t connectionHandle, uint16_t attributeHandle) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    uint8_t serviceNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);

    for (int i = 0; i < serviceNum; ++i)
    {
        if (services[i].handle < attributeHandle && attributeHandle <= services[i].endGroupHandle)
        {
            *pService = services[i];
            return true;
        }
    }

    return false;
}

void BtmImpl::GetGattIncludedServicesImpl(nn::sf::OutArray<user::GattService> pIncludedServices, nn::sf::Out<uint8_t> pOutNum,
                                          uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    uint8_t serviceNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);
    uint8_t belongingServiceIndex = nn::bluetooth::GattAttributeCountMaxClient + 1;
    uint8_t outNum = 0;

    for (int i = 0; i < serviceNum; ++i)
    {
        if (services[i].handle == serviceHandle)
        {
            belongingServiceIndex = i;
            break;
        }
    }

    if (belongingServiceIndex != nn::bluetooth::GattAttributeCountMaxClient + 1)
    {
        for (int i = 0; i < serviceNum; ++i)
        {
            if (services[belongingServiceIndex].handle < services[i].handle &&
                services[i].handle <= services[belongingServiceIndex].endGroupHandle &&
                outNum < pIncludedServices.GetLength())
            {
                memcpy(&pIncludedServices[outNum], &services[i], sizeof(user::GattService));
                pIncludedServices[outNum].type = nn::bluetooth::GattAttributeType_IncludedService;
                outNum++;
            }

            if (outNum == pIncludedServices.GetLength())
            {
                break;
            }
        }
    }
}

void BtmImpl::GetGattCharacteristicsImpl(nn::sf::OutArray<user::GattCharacteristic> pCharacteristics, nn::sf::Out<uint8_t> pOutNum,
                                         uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattCharacteristic chars[nn::bluetooth::GattAttributeCountMaxClient];

    *pOutNum = GetGattCharacteristics(chars, NN_ARRAY_SIZE(chars), connectionHandle, serviceHandle);

    *pOutNum = (pCharacteristics.GetLength() < *pOutNum) ? pCharacteristics.GetLength() : *pOutNum;

    for (int i = 0; i < *pOutNum; ++i)
    {
        memcpy(&pCharacteristics[i], &chars[i], sizeof(user::GattCharacteristic));
    }
}

void BtmImpl::GetGattDescriptorsImpl(nn::sf::OutArray<user::GattDescriptor> pDescriptors, nn::sf::Out<uint8_t> pOutNum,
                                     uint32_t connectionHandle, uint16_t charHandle) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattDescriptor descs[nn::bluetooth::GattAttributeCountMaxClient];

    *pOutNum = GetGattDescriptors(descs, NN_ARRAY_SIZE(descs), connectionHandle, charHandle);

    *pOutNum = (pDescriptors.GetLength() < *pOutNum) ? pDescriptors.GetLength() : *pOutNum;

    for (int i = 0; i < *pOutNum; ++i)
    {
        memcpy(&pDescriptors[i], &descs[i], sizeof(user::GattDescriptor));
    }
}

bool BtmImpl::AcquireBleMtuConfigEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleMtuConfigEvent buffer;
    for (int i = 0; i < API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleMtuConfigEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmImpl::ConfigureBleMtuImpl(uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT
{
    // BLE MTU の設定は同時に１つまで
    if (IsConfiguringBleMtu())
    {
        return ResultBusyProcessingExclusiveApi();
    }

    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    bool connectionExist = false;

    for (auto state : connState)
    {
        if (state.connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return nn::btm::ResultBleConnectionNotFound();
    }

    CommandParam_BleConfigureMtu command;
    command.connectionHandle = connectionHandle;
    command.mtu = mtu;

    Message message;
    message.apiId = ApiId_BleConfigureMtu;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

void BtmImpl::GetBleMtuImpl(nn::sf::Out<uint16_t> mtu, uint32_t connectionHandle) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    *mtu = GetBleMtu(connectionHandle);
}

nn::Result BtmImpl::RegisterBleGattDataPathImpl(user::BleDataPath path) NN_NOEXCEPT
{
    CommandParam_BleRegisterDataPath command;
    command.uuid = path.uuid;
    command.path = path.path;

    auto result = InvokeWorker(ApiId_BleRegisterDataPath, &command, sizeof(command), nullptr, 0);

    return result;
}

nn::Result BtmImpl::UnregisterBleGattDataPathImpl(user::BleDataPath path) NN_NOEXCEPT
{
    CommandParam_BleUnregisterDataPath command;
    command.uuid = path.uuid;
    command.path = path.path;

    auto result = InvokeWorker(ApiId_BleUnregisterDataPath, &command, sizeof(command), nullptr, 0);

    return result;
}

nn::Result BtmImpl::RegisterAppletResourceUserIdImpl(const nn::applet::AppletResourceUserId& aruid, uint32_t appletId) NN_NOEXCEPT
{
    RegisterAppletResourceUserId(aruid, appletId);

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::UnregisterAppletResourceUserIdImpl(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    UnregisterAppletResourceUserId(aruid);

    NN_RESULT_SUCCESS;
}

nn::Result BtmImpl::SetAppletResourceUserIdImpl(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    SetForeGroundAppletResourceUserId(aruid);

    NN_RESULT_SUCCESS;
}

void BtmSystemImpl::GetCoreImpl(nn::sf::Out<nn::sf::SharedPointer<nn::btm::IBtmSystemCore>> pOutService) NN_NOEXCEPT
{
    *pOutService = nn::sf::ObjectFactory<BtmSystemCoreAllocator::Policy>::CreateSharedEmplaced<nn::btm::IBtmSystemCore, nn::btm::BtmSystemCoreImpl>();
}

BtmSystemCoreImpl::BtmSystemCoreImpl() NN_NOEXCEPT
{
    m_GamepadPairingEventId = 0xFF;
    m_RadioEventId = 0xFF;
}

BtmSystemCoreImpl::~BtmSystemCoreImpl() NN_NOEXCEPT
{
    Message message;

    //クライアントが消失したら、GamepadPairingを止める
    message.apiId = ApiId_CancelGamepadPairing;
    EnQueue(&message);

    //クライアントが消失したら、システムイベントを破棄する
    if(m_GamepadPairingEventId != 0xFF)
    {
        BTM_LOG("Discard GppEvent. id = %d\n", m_GamepadPairingEventId);
        message.apiId = ApiId_DiscardGamepadPairingEvent;
        CommandParam_DiscardGamepadPairingEvent command;
        command.id = m_GamepadPairingEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    if(m_RadioEventId != 0xFF)
    {
        BTM_LOG("Discard RadioEvent. id = %d\n", m_RadioEventId);
        message.apiId = ApiId_DiscardRadioEvent;
        CommandParam_DiscardRadioEvent command;
        command.id = m_RadioEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }
}

nn::Result BtmSystemCoreImpl::StartGamepadPairingImpl() NN_NOEXCEPT
{
    //非同期型なので、コーナーケースではResultと実行結果が一致しない
    //暫定仕様として、Resultは停滞ステートにおける参考情報として返す
    //Success->実行失敗 は起こりうるが、Failureの場合はそもそも実行されない
    //[Todo]必要に応じて、ステートエラー時もキューイングし、該当するシステムイベントをシグナルさせる。
    //ただし、この場合はFailure->実行成功ケースが存在してしまうため、result自体を消す必要がある
    //[Todo]Result消す
    if(IsRadioEnabled() == false)
    {
        return nn::btm::system::ResultPairingFailureRadioOff();
    }

    DeviceConditionList dCList = GetDeviceConditionList();
    if(dCList.wlanMode != WlanMode_None)
    {
        return nn::btm::system::ResultPairingFailureLdnEnabled();
    }

    Message message;
    message.apiId = ApiId_StartGamepadPairing;
    EnQueue(&message);

    return ResultSuccess();
}

void BtmSystemCoreImpl::CancelGamepadPairingImpl() NN_NOEXCEPT
{
    Message message;
    message.apiId = ApiId_CancelGamepadPairing;
    EnQueue(&message);
}

void BtmSystemCoreImpl::ClearGamepadPairingDatabaseImpl() NN_NOEXCEPT
{
    Message message;
    message.apiId = ApiId_ClearGamepadPairingDatabase;
    EnQueue(&message);
}

void BtmSystemCoreImpl::GetPairedGamepadCountImpl(nn::sf::Out<std::uint8_t> pPairedGamepadCount) NN_NOEXCEPT
{
    DeviceInfoList deviceInfoList = GetDeviceInfoList();
    *pPairedGamepadCount = deviceInfoList.deviceCount;

    user::BleAdvFilterForGeneral filterCondition;
    BdAddress pairedBleAddress[nn::bluetooth::BlePairingCountMax];

    // Palma
    NN_ABORT_UNLESS(GetScanFilterParameter(&filterCondition, nn::bluetooth::BleScanParameterId_Palma));
    *pPairedGamepadCount += GetBlePairedDeviceAddress(pairedBleAddress, NN_ARRAY_SIZE(pairedBleAddress), filterCondition);
}

void BtmSystemCoreImpl::EnableRadioImpl() NN_NOEXCEPT
{
    Message message;
    message.apiId = ApiId_EnableRadio;
    EnQueue(&message);
}

void BtmSystemCoreImpl::DisableRadioImpl() NN_NOEXCEPT
{
    Message message;
    message.apiId = ApiId_DisableRadio;
    EnQueue(&message);
}

void BtmSystemCoreImpl::GetRadioOnOffImpl(nn::sf::Out<bool> pIsRadioOn) NN_NOEXCEPT
{
    *pIsRadioOn = IsRadioEnabled();
}

bool BtmSystemCoreImpl::AcquireRadioEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle)
{
    //システムイベント総数管理のため、1セッション1システムイベントに限定する
    if(m_RadioEventId != 0xFF)
    {
        BTM_LOG("over.\n");
        return false;
    }

    ReportParam_AcquireRadioEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireRadioEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            m_RadioEventId = buffer.id;
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            BTM_LOG("No resource.\n");
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

bool BtmSystemCoreImpl::AcquireGamepadPairingEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle)
{
    //システムイベント総数管理のため、1セッション1システムイベントに限定する
    if(m_GamepadPairingEventId != 0xFF)
    {
        BTM_LOG("over.\n");
        return false;
    }

    ReportParam_AcquireGamepadPairingEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireGamepadPairingEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            m_GamepadPairingEventId = buffer.id;
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            BTM_LOG("No resource.\n");
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

bool BtmSystemCoreImpl::IsGamepadPairingStartedImpl()
{
    return IsGamepadPairingStarted();
}


bool BtmDebugImpl::AcquireDiscoveryEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireDiscoveryEvent buffer;
    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_AcquireDiscoveryEvent,nullptr,0,&buffer,sizeof(buffer));
        if(result.IsSuccess())
        {
            handle.Set( nn::sf::NativeHandle( buffer.handle, false ) );
            return true;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if(ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmDebugImpl::StartDiscoveryImpl() NN_NOEXCEPT
{
    auto result = InvokeWorker(ApiId_StartDiscovery,nullptr,0,nullptr,0);
    return result;
}

nn::Result BtmDebugImpl::CancelDiscoveryImpl() NN_NOEXCEPT
{
    auto result = InvokeWorker(ApiId_CancelDiscovery,nullptr,0,nullptr,0);
    return result;
}

void BtmDebugImpl::GetDevicePropertyImpl(nn::sf::Out<nn::btm::DevicePropertyList> pList) NN_NOEXCEPT
{
    *pList = GetDevicePropertyList();
}

nn::Result BtmDebugImpl::CreateBondImpl(nn::btm::BdAddress bdAddress) NN_NOEXCEPT
{
    CommandParam_CreateBond command;
    command.address = bdAddress;
    auto result = InvokeWorker(ApiId_CreateBond,&command,sizeof(command),nullptr,0);
    return result;
}

nn::Result BtmDebugImpl::CancelBondImpl(nn::btm::BdAddress bdAddress) NN_NOEXCEPT
{
    CommandParam_CancelBond command;
    command.address = bdAddress;
    auto result = InvokeWorker(ApiId_CancelBond,&command,sizeof(command),nullptr,0);
    return result;
}

nn::Result BtmDebugImpl::SetTsiModeImpl(nn::btm::BdAddress bdAddress, nn::btm::TsiMode tsiMode) NN_NOEXCEPT
{
    CommandParam_SetTsiMode command;
    command.address = bdAddress;
    command.tsiMode = tsiMode;
    auto result = InvokeWorker(ApiId_ChangeTsiMode,&command,sizeof(command),nullptr,0);
    return result;
}

void BtmDebugImpl::GeneralTestImpl(int mode) NN_NOEXCEPT
{
    CommandParam_GeneralTest command;
    command.mode = mode;

    for(int i=0;i<API_RETRY_COUNT;i++)
    {
        auto result = InvokeWorker(ApiId_GeneralTest,&command,sizeof(command),nullptr,0);
        if(result.IsSuccess())
        {
            return;
        }
        else if(ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    BTM_LOG("Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
}

void BtmDebugImpl::GeneralGetImpl(int mode, nn::sf::Out<nn::btm::GeneralInfoList> pList) NN_NOEXCEPT
{
    *pList = GetGeneralInfo(mode);
}

void BtmDebugImpl::HidConnectImpl(nn::btm::BdAddress bdAddress) NN_NOEXCEPT
{
    CommandParam_HidConnect command;
    command.address = bdAddress;

    Message message;
    message.apiId = ApiId_HidConnect;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);
}

bool BtmDebugImpl::GetGattClientDisconnectionReasonImpl(nn::sf::Out<uint16_t> pOutReason,
                                                        uint32_t connectionHandle, BdAddress address) NN_NOEXCEPT
{
    uint16_t reason;

    bool IsReasonExisting = GetBleDisconnectionReason(&reason, connectionHandle, address);

    pOutReason.Set(reason);

    return IsReasonExisting;
}

bool BtmDebugImpl::GetBleConnectionParameterImpl(nn::sf::Out<uint16_t> pOutInterval, nn::sf::Out<uint16_t> pOutLatency, nn::sf::Out<uint16_t> pOutTimeout,
                                                 uint32_t connectionHandle) NN_NOEXCEPT
{
    return GetBleConnectionParameter(pOutInterval.GetPointer(), pOutLatency.GetPointer(), pOutTimeout.GetPointer(), connectionHandle);
}

bool BtmDebugImpl::GetBleConnectionParameterRequestImpl(nn::sf::Out<uint16_t> pOutIntervalMin, nn::sf::Out<uint16_t> pOutIntervalMax,
                                                        nn::sf::Out<uint16_t> pOutLatency, nn::sf::Out<uint16_t> pOutTimeout,
                                                        uint32_t connectionHandle) NN_NOEXCEPT
{
    return GetBleConnectionParameterRequest(pOutIntervalMin.GetPointer(), pOutIntervalMax.GetPointer(),
                                            pOutLatency.GetPointer(), pOutTimeout.GetPointer(),
                                            connectionHandle);
}


/**
 *  BtmUserCoreImpl
 */
void BtmUserImpl::GetCoreImpl(nn::sf::Out<nn::sf::SharedPointer<nn::btm::IBtmUserCore>> pOutService) NN_NOEXCEPT
{
    *pOutService = nn::sf::ObjectFactory<BtmUserCoreAllocator::Policy>::CreateSharedEmplaced<nn::btm::IBtmUserCore, nn::btm::BtmUserCoreImpl>();
}

/**
 *  BtmUserCoreImpl
 */

BtmUserCoreImpl::BtmUserCoreImpl() NN_NOEXCEPT
{
    m_BleConnectionOwner            = nn::applet::AppletResourceUserId::GetInvalidId();
    m_BleGeneralScanOwner           = nn::applet::AppletResourceUserId::GetInvalidId();
    m_BlePairedScanOwner            = nn::applet::AppletResourceUserId::GetInvalidId();
    m_BleSmartDeviceScanOwner       = nn::applet::AppletResourceUserId::GetInvalidId();

    m_BleScanEventId                = 0xFF;
    m_BleConnectionEventId          = 0xFF;
    m_BlePairingEventId             = 0xFF;
    m_BleServiceDiscoveryEventId    = 0xFF;
    m_BleMtuConfigEventId           = 0xFF;
}

BtmUserCoreImpl::~BtmUserCoreImpl() NN_NOEXCEPT
{
    Message message;

    //クライアントが消失したら、BLE Scan を止める
    if (m_BleGeneralScanOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        CommandParam_StopBleScanForGeneral command;
        command.aruid = m_BleGeneralScanOwner;
        message.apiId = ApiId_BleStopScanGeneral;
        memcpy(message.buffer, &command, sizeof(command));
        EnQueue(&message);
    }
    if(m_BlePairedScanOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        CommandParam_StopBleScanForPaired command;
        command.aruid = m_BlePairedScanOwner;
        message.apiId = ApiId_BleStopScanPaired;
        memcpy(message.buffer, &command, sizeof(command));
        EnQueue(&message);
    }
    if (m_BleSmartDeviceScanOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        CommandParam_StopBleScanForSmartDevice command;
        command.aruid = m_BleSmartDeviceScanOwner;
        message.apiId = ApiId_BleStopScanSdGeneral;
        memcpy(message.buffer, &command, sizeof(command));
        EnQueue(&message);
    }

    //クライアントが消失したら、システムイベントを破棄する
    if (m_BleScanEventId != 0xFF)
    {
        BTM_LOG("Discard BLE Scan Event. id = %d\n", m_BleScanEventId);
        message.apiId = ApiId_DiscardBleScanEvent;
        CommandParam_DiscardBleScanEvent command;
        command.id = m_BleScanEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    if (m_BleConnectionEventId != 0xFF)
    {
        BTM_LOG("Discard BLE Connection Event. id = %d\n", m_BleConnectionEventId);
        message.apiId = ApiId_DiscardBleConnectionEvent;
        CommandParam_DiscardBleConnectionEvent command;
        command.id = m_BleConnectionEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    if (m_BlePairingEventId != 0xFF)
    {
        BTM_LOG("Discard BLE Pairing Event. id = %d\n", m_BlePairingEventId);
        message.apiId = ApiId_DiscardBlePairingEvent;
        CommandParam_DiscardBlePairingEvent command;
        command.id = m_BlePairingEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    if (m_BleServiceDiscoveryEventId != 0xFF)
    {
        BTM_LOG("Discard BLE Service Discovery Event. id = %d\n", m_BleServiceDiscoveryEventId);
        message.apiId = ApiId_DiscardBleSdpEvent;
        CommandParam_DiscardBleSdpEvent command;
        command.id = m_BleServiceDiscoveryEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    if (m_BleMtuConfigEventId != 0xFF)
    {
        BTM_LOG("Discard BLE MTU Config Event. id = %d\n", m_BleMtuConfigEventId);
        message.apiId = ApiId_DiscardBleMtuConfigEvent;
        CommandParam_DiscardBleMtuConfigEvent command;
        command.id = m_BleMtuConfigEventId;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    // クライアントが消失したら、自身がオーナーであるBLE の接続を全て切る
    // クライアント向けのシステムイベント破棄後に切断するため、この位置でEnqueueする
    if (m_BleConnectionOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        message.apiId = ApiId_BleDisconnectAll;
        CommandParam_BleDisconnectAll command;
        command.aruid = m_BleConnectionOwner;
        memcpy(&message.buffer[0], &command, sizeof(command));
        EnQueue(&message);
    }

    // クライアントが消失したら、スキャン結果を全て破棄する
    {
        message.apiId = ApiId_BleClearScanResult;
        EnQueue(&message);
    }
}

bool BtmUserCoreImpl::AcquireBleScanEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleScanEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleScanEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            m_BleScanEventId = buffer.id;
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmUserCoreImpl::GetBleScanParameterGeneralImpl(nn::sf::Out<user::BleAdvFilterForGeneral> pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    if (!GetScanFilterParameter(pFilter.GetPointer(), parameterId))
    {
        return user::ResultInvalidArgument();
    }

    NN_RESULT_SUCCESS;
}

nn::Result BtmUserCoreImpl::GetBleScanParameterSmartDeviceImpl(nn::sf::Out<user::BleAdvFilterForSmartDevice> pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    if (!GetScanFilterParameter(pFilter.GetPointer(), parameterId))
    {
        return user::ResultInvalidArgument();
    }

    NN_RESULT_SUCCESS;
}

nn::Result BtmUserCoreImpl::StartBleScanForGeneralImpl(const nn::applet::AppletResourceUserId& aruid, nn::btm::user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return user::ResultBluetoothOff();
    }

    // General Scan のオーナーを取得
    m_BleGeneralScanOwner = aruid;

    CommandParam_StartBleScanForGeneral command;
    command.aruid = aruid;
    command.condition = filter;
    auto result = InvokeWorker(ApiId_BleStartScanGeneral, &command, sizeof(command), nullptr, 0);
    return ConvertToUserResult(result);
}

nn::Result BtmUserCoreImpl::StopBleScanForGeneralImpl() NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return user::ResultBluetoothOff();
    }
    else if (m_BleGeneralScanOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        CommandParam_StopBleScanForGeneral command;
        command.aruid = m_BleGeneralScanOwner;
        auto result = InvokeWorker(ApiId_BleStopScanGeneral, &command, sizeof(command), nullptr, 0);
        return ConvertToUserResult(result);
    }
    else
    {
        NN_RESULT_SUCCESS;
    }
}

void BtmUserCoreImpl::GetBleScanResultsForGeneralImpl(nn::sf::OutArray<nn::btm::user::ScanResult> results, nn::sf::Out<uint8_t> pResultNum,
                                                      const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::ScanResult btmResults[nn::bluetooth::BleScanResultCountMax / 2];

    *pResultNum = GetScanResultGeneral(btmResults, NN_ARRAY_SIZE(btmResults), aruid);

    *pResultNum = (results.GetLength() < *pResultNum) ? results.GetLength() : *pResultNum;

    uint8_t num = (results.GetLength() < NN_ARRAY_SIZE(btmResults)) ? results.GetLength() : NN_ARRAY_SIZE(btmResults);

    for (int i = 0; i < num; ++i)
    {
        memcpy(&results[i], &btmResults[i], sizeof(user::ScanResult));
    }
}

nn::Result BtmUserCoreImpl::StartBleScanForPairedDeviceImpl(const nn::applet::AppletResourceUserId& aruid, nn::btm::user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return user::ResultBluetoothOff();
    }

    // Paired Scan のオーナーを取得
    m_BlePairedScanOwner = aruid;

    CommandParam_StartBleScanForPaired command;
    command.aruid = aruid;
    command.condition = filter;
    auto result = InvokeWorker(ApiId_BleStartScanPaired, &command, sizeof(command), nullptr, 0);
    return ConvertToUserResult(result);
}

nn::Result BtmUserCoreImpl::StopBleScanForPairedDeviceImpl() NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return user::ResultBluetoothOff();
    }
    else if (m_BlePairedScanOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        CommandParam_StopBleScanForPaired command;
        command.aruid = m_BlePairedScanOwner;
        auto result = InvokeWorker(ApiId_BleStopScanPaired, &command, sizeof(command), nullptr, 0);
        return ConvertToUserResult(result);
    }
    else
    {
        NN_RESULT_SUCCESS;
    }
}

nn::Result BtmUserCoreImpl::StartBleScanForSmartDeviceImpl(const nn::applet::AppletResourceUserId& aruid, nn::btm::user::BleAdvFilterForSmartDevice filter) NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return user::ResultBluetoothOff();
    }

    // Smart Device Scan のオーナーを取得
    m_BleSmartDeviceScanOwner = aruid;

    CommandParam_StartBleScanForSmartDevice command;
    command.aruid = aruid;
    command.condition = filter;
    auto result = InvokeWorker(ApiId_BleStartScanSdGeneral, &command, sizeof(command), nullptr, 0);
    return ConvertToUserResult(result);
}

nn::Result BtmUserCoreImpl::StopBleScanForSmartDeviceImpl() NN_NOEXCEPT
{
    if (!IsRadioEnabled())
    {
        return user::ResultBluetoothOff();
    }
    else if (m_BleSmartDeviceScanOwner != nn::applet::AppletResourceUserId::GetInvalidId())
    {
        CommandParam_StopBleScanForSmartDevice command;
        command.aruid = m_BleSmartDeviceScanOwner;
        auto result = InvokeWorker(ApiId_BleStopScanSdGeneral, &command, sizeof(command), nullptr, 0);
        return ConvertToUserResult(result);
    }
    else
    {
        NN_RESULT_SUCCESS;
    }
}

void BtmUserCoreImpl::GetBleScanResultsForSmartDeviceImpl(nn::sf::OutArray<nn::btm::user::ScanResult> results, nn::sf::Out<uint8_t> pResultNum,
                                                          const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::ScanResult btmResults[nn::bluetooth::BleScanResultCountMax / 2];

    *pResultNum = GetScanResultSmartDevice(btmResults, NN_ARRAY_SIZE(btmResults), aruid);

    *pResultNum = (results.GetLength() < *pResultNum) ? results.GetLength() : *pResultNum;

    uint8_t num = (results.GetLength() < NN_ARRAY_SIZE(btmResults)) ? results.GetLength() : NN_ARRAY_SIZE(btmResults);

    for (int i = 0; i < num; ++i)
    {
        memcpy(&results[i], &btmResults[i], sizeof(user::ScanResult));
    }
}

//nn::Result BtmUserCoreImpl::RegisterBleStreetPassImpl(nn::btm::user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
//{
//    CommandParam_RegisterBleStreetPass command;
//    command.condition = filter;
//    auto result = InvokeWorker(ApiId_BleRegisterStreetPass, &command, sizeof(command), nullptr, 0);
//    return result;
//}
//
//void BtmUserCoreImpl::UnregisterBleStreetPassImpl(nn::btm::user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
//{
//    CommandParam_UnregisterBleStreetPass command;
//    command.condition = filter;
//    InvokeWorker(ApiId_BleUnRegisterStreetPass, &command, sizeof(command), nullptr, 0);
//}
//
//void BtmUserCoreImpl::GetBleStreetPassResultsImpl(
//    nn::sf::OutArray<nn::btm::user::ScanResult> results,
//    nn::sf::Out<uint8_t> pResultNum,
//    nn::btm::user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
//{
//    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
//
//}
//
//nn::Result BtmUserCoreImpl::RegisterBleSmartDeviceStreetPassImpl(nn::btm::user::BleAdvFilterForSmartDevice filter) NN_NOEXCEPT
//{
//    CommandParam_RegisterBleSmartDeviceStreetPass command;
//    command.condition = filter;
//    auto result = InvokeWorker(ApiId_BleRegisterSdStreetPass, &command, sizeof(command), nullptr, 0);
//    return result;
//}
//
//void BtmUserCoreImpl::UnregisterBleSmartDeviceStreetPassImpl(nn::btm::user::BleAdvFilterForSmartDevice filter) NN_NOEXCEPT
//{
//    CommandParam_UnregisterBleSmartDeviceStreetPass command;
//    command.condition = filter;
//    InvokeWorker(ApiId_BleUnRegisterSdStreetPass, &command, sizeof(command), nullptr, 0);
//}
//
//void BtmUserCoreImpl::GetBleSmartDeviceStreetPassResultsImpl(
//    nn::sf::OutArray<nn::btm::user::ScanResult> results,
//    nn::sf::Out<uint8_t> pResultNum,
//    nn::btm::user::BleAdvFilterForSmartDevice filter) NN_NOEXCEPT
//{
//    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
//
//}

bool BtmUserCoreImpl::AcquireBleConnectionEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleConnectionEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleConnectionEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            m_BleConnectionEventId = buffer.id;
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmUserCoreImpl::BleConnectImpl(const nn::applet::AppletResourceUserId& aruid, nn::btm::BdAddress bdAddress) NN_NOEXCEPT
{
    if (IsRadioEnabled() == false)
    {
        return nn::btm::user::ResultBluetoothOff();
    }

    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    uint8_t connectionCount = GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    if (connectionCount == nn::bluetooth::BleConnectionCountMaxClient)
    {
        return user::ResultBleConnectionFailedFull();
    }

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].address == bdAddress)
        {
            return user::ResultBleConnectionAlreadyExist();
        }
    }

    auto result = InvokeWorker(ApiId_BleTryConnect, nullptr, 0, nullptr, 0);

    if (result.IsFailure())
    {
        return ConvertToUserResult(result);
    }

    CommandParam_BleConnect command;
    command.aruid   = aruid;
    command.address = bdAddress;

    Message message;
    message.apiId = ApiId_BleConnect;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    // 接続のオーナーを取得
    m_BleConnectionOwner = aruid;

    NN_RESULT_SUCCESS;
}

nn::Result BtmUserCoreImpl::BleDisconnectImpl(uint32_t connectionHandle) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    bool connectionExist = false;

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return user::ResultBleConnectionNotFound();
    }

    CommandParam_BleDisconnect command;
    command.connectionHandle = connectionHandle;

    Message message;
    message.apiId = ApiId_BleDisconnect;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

void BtmUserCoreImpl::BleGetConnectionStateImpl(nn::sf::OutArray<nn::btm::user::BleClientConnState> pConnState, nn::sf::Out<uint8_t> pOutNum,
                                                const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    *pOutNum = GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    *pOutNum = (pConnState.GetLength() < *pOutNum) ? pConnState.GetLength() : *pOutNum;

    int copiedIndex = 0;

    for (int i = 0; i < *pOutNum; ++i)
    {
        for (int j = copiedIndex; j < NN_ARRAY_SIZE(connState); ++j)
        {
            if (connState[j].connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
            {
                memcpy(&pConnState[i], &connState[j], sizeof(nn::btm::user::BleClientConnState));
                copiedIndex = j + 1;
                break;
            }
        }
    }

    for (int i = *pOutNum; i < pConnState.GetLength(); ++i)
    {
        memset(&pConnState[i].address, 0x00, sizeof(nn::bluetooth::Address));
        pConnState[i].connectionHandle = nn::bluetooth::BleInvalidConnectionHandle;
    }
}

bool BtmUserCoreImpl::AcquireBlePairingEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBlePairingEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBlePairingEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            m_BlePairingEventId = buffer.id;
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmUserCoreImpl::BlePairDeviceImpl(uint32_t connectionHandle, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BLE_GATT_CLIENT_NUM_MAX];

    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));
    bool connectionExist = false;

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return user::ResultBleConnectionNotFound();
    }
    else if (!GetBlePairingSupported(connectionHandle))
    {
        return user::ResultBlePairingNotSupported();
    }
    else if (GetBlePairingOnGoing())
    {
        return user::ResultBlePairingOnGoing();
    }

    CommandParam_StartBlePairing command;

    command.connectionHandle = connectionHandle;
    command.condition = filter;
    command.pairing = true;

    Message message;
    message.apiId = ApiId_StartBlePairing;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

nn::Result BtmUserCoreImpl::BleUnpairDeviceOnBothImpl(uint32_t connectionHandle, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));
    bool connectionExist = false;

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return user::ResultBleConnectionNotFound();
    }
    else if (!GetBlePairingSupported(connectionHandle))
    {
        return user::ResultBlePairingNotSupported();
    }
    else if (GetBlePairingOnGoing())
    {
        return user::ResultBlePairingOnGoing();
    }

    CommandParam_StartBlePairing command;

    command.connectionHandle = connectionHandle;
    command.condition = filter;
    command.pairing = false;

    Message message;
    message.apiId = ApiId_StartBlePairing;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

nn::Result BtmUserCoreImpl::BleUnpairDeviceImpl(nn::btm::BdAddress bdAddress, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        // 接続が存在する場合は、双方からペアリング削除する
        if (connState[i].address == bdAddress)
        {
            auto result = BleUnpairDeviceOnBothImpl(connState[i].connectionHandle, filter);

            // 成功したら終わり
            if (result.IsSuccess())
            {
                return result;
            }
        }
    }

    // 失敗したら、本体からのみペアリングを削除
    if (GetBlePairingOnGoing())
    {
        return user::ResultBlePairingOnGoing();
    }

    CommandParam_BlePairingUpdateDatabase command;

    command.address     = bdAddress;
    command.condition   = filter;
    command.type        = BlePairingAdvDataType_ManuCliSerId;
    command.pairing     = false;

    Message message;
    message.apiId = ApiId_BlePairingUpdateDatabase;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

void BtmUserCoreImpl::BleGetPairedAddressesImpl(nn::sf::OutArray<BdAddress> pAddresses, nn::sf::Out<uint8_t> pOutNum, user::BleAdvFilterForGeneral filter) NN_NOEXCEPT
{
    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    *pOutNum = GetBlePairedDeviceAddress(pAddresses.GetData(), pAddresses.GetLength(), filter);

    return;
}

bool BtmUserCoreImpl::AcquireBleServiceDiscoveryEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleSdpEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleSdpEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            m_BleServiceDiscoveryEventId = buffer.id;
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

void BtmUserCoreImpl::GetGattServicesImpl(nn::sf::OutArray<user::GattService> pServices, nn::sf::Out<uint8_t> pOutNum,
                                          const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    *pOutNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);

    *pOutNum = (pServices.GetLength() < *pOutNum) ? pServices.GetLength() : *pOutNum;

    for (int i = 0; i < *pOutNum; ++i)
    {
        memcpy(&pServices[i], &services[i], sizeof(user::GattService));
    }
}

bool BtmUserCoreImpl::GetGattServiceImpl(nn::sf::Out<user::GattService> pService,
                                         const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    uint8_t serviceNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);

    for (int i = 0; i < serviceNum; ++i)
    {
        if (services[i].uuid == uuid)
        {
            *pService = services[i];
            return true;
        }
    }

    return false;
}

bool BtmUserCoreImpl::GetBelongingServiceImpl(nn::sf::Out<user::GattService> pService,
                                              const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle, uint16_t attributeHandle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    uint8_t serviceNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);

    for (int i = 0; i < serviceNum; ++i)
    {
        if (services[i].handle < attributeHandle && attributeHandle <= services[i].endGroupHandle)
        {
            *pService = services[i];
            return true;
        }
    }

    return false;
}

void BtmUserCoreImpl::GetGattIncludedServicesImpl(nn::sf::OutArray<user::GattService> pIncludedServices, nn::sf::Out<uint8_t> pOutNum,
                                                  const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    uint8_t serviceNum = GetGattServices(services, NN_ARRAY_SIZE(services), connectionHandle);
    uint8_t belongingServiceIndex = nn::bluetooth::GattAttributeCountMaxClient + 1;
    uint8_t outNum = 0;

    for (int i = 0; i < serviceNum; ++i)
    {
        if (services[i].handle == serviceHandle)
        {
            belongingServiceIndex = i;
            break;
        }
    }

    if(belongingServiceIndex != nn::bluetooth::GattAttributeCountMaxClient + 1)
    {
        for (int i = 0; i < serviceNum; ++i)
        {
            if (services[belongingServiceIndex].handle < services[i].handle &&
                services[i].handle <= services[belongingServiceIndex].endGroupHandle &&
                outNum < pIncludedServices.GetLength())
            {
                memcpy(&pIncludedServices[outNum], &services[i], sizeof(user::GattService));
                pIncludedServices[outNum].type = nn::bluetooth::GattAttributeType_IncludedService;
                outNum++;
            }

            if (outNum == pIncludedServices.GetLength())
            {
                break;
            }
        }
    }
}

void BtmUserCoreImpl::GetGattCharacteristicsImpl(nn::sf::OutArray<user::GattCharacteristic> pCharacteristics, nn::sf::Out<uint8_t> pOutNum,
                                                 const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattCharacteristic chars[nn::bluetooth::GattAttributeCountMaxClient];

    *pOutNum = GetGattCharacteristics(chars, NN_ARRAY_SIZE(chars), connectionHandle, serviceHandle);

    *pOutNum = (pCharacteristics.GetLength() < *pOutNum) ? pCharacteristics.GetLength() : *pOutNum;

    for (int i = 0; i < *pOutNum; ++i)
    {
        memcpy(&pCharacteristics[i], &chars[i], sizeof(user::GattCharacteristic));
    }
}

void BtmUserCoreImpl::GetGattDescriptorsImpl(nn::sf::OutArray<user::GattDescriptor> pDescriptors, nn::sf::Out<uint8_t> pOutNum,
                                             const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle, uint16_t charHandle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    user::GattDescriptor descs[nn::bluetooth::GattAttributeCountMaxClient];

    *pOutNum = GetGattDescriptors(descs, NN_ARRAY_SIZE(descs), connectionHandle, charHandle);

    *pOutNum = (pDescriptors.GetLength() < *pOutNum) ? pDescriptors.GetLength() : *pOutNum;

    for (int i = 0; i < *pOutNum; ++i)
    {
        memcpy(&pDescriptors[i], &descs[i], sizeof(user::GattDescriptor));
    }
}

bool BtmUserCoreImpl::AcquireBleMtuConfigEventImpl(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    ReportParam_AcquireBleMtuConfigEvent buffer;
    for (int i = 0; i<API_RETRY_COUNT; i++)
    {
        auto result = InvokeWorker(ApiId_AcquireBleMtuConfigEvent, nullptr, 0, &buffer, sizeof(buffer));
        if (result.IsSuccess())
        {
            handle.Set(nn::sf::NativeHandle(buffer.handle, false));
            m_BleMtuConfigEventId = buffer.id;
            return true;
        }
        else if (ResultBusy::Includes(result))
        {
            //Busyの時のみリトライを続ける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(API_RETRY_WAIT_MSEC));
        }
        else if (ResultInternalErrorNoResource::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return false;
        }
    }
    BTM_LOG("[btm]Sync API timeout.\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeOut());
    return false;
}

nn::Result BtmUserCoreImpl::ConfigureBleMtuImpl(const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    // BLE MTU の設定は同時に１つまで
    if (IsConfiguringBleMtu())
    {
        return user::ResultBusyProcessingExclusiveApi();
    }

    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];
    GetBleConnectionState(connState, NN_ARRAY_SIZE(connState));

    bool connectionExist = false;

    for (int i = 0; i < NN_ARRAY_SIZE(connState); ++i)
    {
        if (connState[i].connectionHandle == connectionHandle && connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            connectionExist = true;
            break;
        }
    }

    if (!connectionExist)
    {
        return user::ResultBleConnectionNotFound();
    }

    CommandParam_BleConfigureMtu command;
    command.connectionHandle    = connectionHandle;
    command.mtu                 = mtu;

    Message message;
    message.apiId = ApiId_BleConfigureMtu;
    memcpy(&message.buffer[0], &command, sizeof(command));
    EnQueue(&message);

    NN_RESULT_SUCCESS;
}

void BtmUserCoreImpl::GetBleMtuImpl(nn::sf::Out<uint16_t> mtu,
                                    const nn::applet::AppletResourceUserId& aruid, uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //Busy中でもステート読み出しできるように、Invokeせずに内部情報を直接読み出す
    *mtu = GetBleMtu(connectionHandle);
}

nn::Result BtmUserCoreImpl::RegisterBleGattDataPathImpl(const nn::applet::AppletResourceUserId& aruid, user::BleDataPath path) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    CommandParam_BleRegisterDataPath command;
    command.uuid = path.uuid;
    command.path = path.path;

    auto result = InvokeWorker(ApiId_BleRegisterDataPath, &command, sizeof(command), nullptr, 0);

    return ConvertToUserResult(result);
}

nn::Result BtmUserCoreImpl::UnregisterBleGattDataPathImpl(const nn::applet::AppletResourceUserId& aruid, user::BleDataPath path) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    CommandParam_BleUnregisterDataPath command;
    command.uuid = path.uuid;
    command.path = path.path;

    auto result = InvokeWorker(ApiId_BleUnregisterDataPath, &command, sizeof(command), nullptr, 0);

    return ConvertToUserResult(result);
}

}}  // namespace nn::btm
