﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/applet/applet_FundamentalTypes.h>

#include <nn/btm/btm_Types.h>
#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/btm_Result.h>
#include <nn/btm/user/btm_UserResult.h>
#include "btm_InternalTypes.h"
#include "btm_Worker.h"
#include "btm_Handler.h"
#include "btm_Usecase.h"
#include "btm_Utility.h"
#include "btm_Queue.h"
#include "btm_Devices.h"
#include "btm_AppletManager.h"

#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>

#include <nn/bluetooth/bluetooth_BleScanParameterIdPalma.h>

namespace nn { namespace btm {

namespace {
//btmの内部状態は以下オブジェクト群に保存し、都度、参照と更新を行う。
//Handlerはシングルスレッドで動作するため、逐次アクセスが可能。
DevicePropertyContainer  g_DpContainer;
DeviceConditionContainer g_DcContainer;
DeviceConditionContainer g_DcContainerForWorking;
DeviceInfoContainer      g_DiContainer;
HostConditionContainer   g_HcContainer;
SystemEventContainer g_SystemEventContainer;
BleGattClientConditionContainer g_BleGattClientContainer;
BleGattClientConditionContainer g_BleGattClientContainerForWorking;
BleScannerCondition             g_BleScannerCondition;
BlePairingInfoContainer         g_BlePairingInfoContainer;

AppletManager           g_AppletManager;
}


//--------------------------------------------------
//ファイル内汎用関数
//--------------------------------------------------
enum BreakCategory{
    BreakCategory_Core,
    BreakCategory_Hid,
    BreakCategory_Ble,
    BreakCategory_All,
};
struct RequiredReportType{
    BreakCategory breakCategory;
    uint8_t timeoutSec;//0はタイムアウト無し
    union
    {
        nn::bluetooth::EventType core;
        nn::bluetooth::HidEventType hid;
        nn::bluetooth::BleEventType ble;
    };
};
MultiWaitId MultiWaitAndHandleAsyncReport(RequiredReportType requiredReportType, void* pBufferBtp, size_t bufferSizeBtp);
MultiWaitId MultiWaitAndHandleAsyncReport(nn::bluetooth::EventType requiredCoreEventType, void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec);
MultiWaitId MultiWaitAndHandleAsyncReport(nn::bluetooth::HidEventType requiredHidEventType, void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec);
MultiWaitId MultiWaitAndHandleAsyncReport(nn::bluetooth::BleEventType requiredHidEventType, void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec);
MultiWaitId MultiWaitAndHandleAsyncReport(void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec);
void GetBtCoreReport(nn::bluetooth::EventType* pEventTypeBtp, void* pBufferBtp, size_t bufferSizeBtp);
void GetBtHidReport(nn::bluetooth::HidEventType* pHidEventTypeBtp, void* pBufferBtp, size_t bufferSizeBtp);
void GetBtBleCoreReport(nn::bluetooth::BleEventType* pBleCoreEventTypeBtp, void* pBufferBtp, size_t bufferSizeBtp);
void WaitForLlrFinish(void* pBufferBtp, size_t bufferSizeBtp);
void DisablePageScan(void* pBufferBtp, size_t bufferSizeBtp);
void EnablePageScan();
void DisconnectAllDevices(void* pBufferBtp, size_t bufferSizeBtp);
void DisconnectDevice(const BdAddress* pBdAddress, void* pBufferBtp, size_t bufferSizeBtp);
nn::Result EnableBleScan(void* pBufferBtp, size_t bufferSizeBtp);
nn::Result DisableBleScan(void* pBufferBtp, size_t bufferSizeBtp);
void CancelBleConnection(void* pBufferBtp, size_t bufferSizeBtp);
void DisconnectAllBleDevices(void* pBufferBtp, size_t bufferSizeBtp, bool forOwnedOnly, const nn::applet::AppletResourceUserId& owner);
bool UpdateUsecase(DeviceConditionList* pDeviceConditionList, GattClientConditionList* pGattClientConditionList, void* pBufferBtp, size_t bufferSizeBtp, bool isForceRestructure = false);
bool UpdateUsecase(DeviceConditionList* pDeviceConditionList, GattClientConditionList* pGattClientConditionList, nn::Result* pResult, void* pBufferBtp, size_t bufferSizeBtp);
void RestructureEachBtSlots(DeviceConditionList* pDeviceConditionList,
                            SniffMode sniffMode,
                            void* pBufferBtp, size_t bufferSizeBtp);
void RestructureAllBtSlots(DeviceConditionList* pDeviceConditionList, GattClientConditionList* pGattClientConditionList,
                           SniffMode sniffMode, SlotMode slotMode, CeLength ceLength, uint8_t bleConnectionParameterUpdateCount,
                           void* pBufferBtp, size_t bufferSizeBtp);
void RemoveOldDeviceInfo(const BdAddress* pBdAddress);
void DisableAutoPairing(void* pBufferBtp, size_t bufferSizeBtp);
void EnableAutoPairing(void* pBufferBtp, size_t bufferSizeBtp);
void Recovery();
nn::Result BleAddScanFilterCondition(uint8_t *bufferBtp, size_t bufferSizeBtp, const BleScannerCondition::ScanFilterCondition &condition);
nn::Result BleDeleteScanFilterCondition(uint8_t *bufferBtp, size_t bufferSizeBtp, const BleScannerCondition::ScanFilterCondition &condition);

//--------------------------------------------------
//Implスレッドから、Handlerの内部状態を直接取得するための関数
//マルチスレッドから同時に呼ばれうるが、すべてGet系の関数のため排他制御はしていない
//--------------------------------------------------
DevicePropertyList GetDevicePropertyList()
{
    return g_DpContainer.Get();
}

DeviceConditionList GetDeviceConditionList()
{
    return g_DcContainer.Get();
}

DeviceInfoList GetDeviceInfoList()
{
    return g_DiContainer.Get();
}

HostDeviceProperty GetHostDeviceProperty()
{
    return g_HcContainer.GetProperty();
}

bool GetScanFilterParameter(user::BleAdvFilterForGeneral* pFilter, uint16_t parameterId)
{
    return g_BleScannerCondition.GetScanFilter(pFilter, parameterId);
}

bool GetScanFilterParameter(user::BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId)
{
    return g_BleScannerCondition.GetScanFilter(pFilter, parameterId);
}

uint8_t GetScanResultGeneral(user::ScanResult *pResults, uint8_t inNum, const nn::applet::AppletResourceUserId& aruid)
{
    const BleScannerCondition::ScanResultList* results = g_BleScannerCondition.GetScanResultsGeneral();

    uint8_t outNum = 0;
    uint8_t num = (inNum < nn::bluetooth::BleScanResultCountMax / 2) ? inNum : nn::bluetooth::BleScanResultCountMax / 2;

    for (int i = 0; i < num; ++i)
    {
        if (results->results[i].aruid == aruid && results->results[i].result.rssi != 0)
        {
            memcpy(&pResults[i], &results->results[i].result, sizeof(user::ScanResult));
            outNum++;
        }
        else
        {
            pResults[i] = user::ScanResult();
        }
    }

    return outNum;
}

uint8_t GetScanResultSmartDevice(user::ScanResult *pResults, uint8_t inNum, const nn::applet::AppletResourceUserId& aruid)
{
    const BleScannerCondition::ScanResultList* results = g_BleScannerCondition.GetScanResultsSd();

    uint8_t outNum = 0;
    uint8_t num = (inNum < nn::bluetooth::BleScanResultCountMax / 2) ? inNum : nn::bluetooth::BleScanResultCountMax / 2;

    for (int i = 0; i < num; ++i)
    {
        if (results->results[i].aruid == aruid && results->results[i].result.rssi != 0)
        {
            memcpy(&pResults[i], &results->results[i].result, sizeof(user::ScanResult));
            outNum++;
        }
        else
        {
            pResults[i] = user::ScanResult();
        }
    }

    return outNum;
}

uint8_t GetScanResultUnconnectableScan(user::ScanResult *pResults, uint8_t inNum)
{
    return 0;
}

uint8_t GetScanResultSmartDeviceUnconnectableScan(user::ScanResult *pResults, uint8_t inNum)
{
    return 0;
}

uint8_t GetBleConnectionState(user::BleClientConnState *pConnState, uint8_t inNum)
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    uint8_t outNum = g_BleGattClientContainer.GetConnectionState(connState);
    outNum = (inNum < outNum) ? inNum : outNum;

    // 全部コピー
    uint8_t num = (inNum < nn::bluetooth::BleConnectionCountMaxClient) ? inNum : nn::bluetooth::BleConnectionCountMaxClient;

    for (int i = 0; i < num; ++i)
    {
        memcpy(&pConnState[i], &connState[i], sizeof(user::BleClientConnState));
    }

    return outNum;
}

GattClientConditionList GetGattClientConditionList()
{
    return *g_BleGattClientContainer.GetPtr();
}

uint8_t GetGattServices(user::GattService *pServices, uint8_t inNum, uint32_t connectionHandle)
{
    return g_BleGattClientContainer.GetGattServerServices(pServices, inNum, connectionHandle);
}

uint8_t GetGattCharacteristics(user::GattCharacteristic *pCharacteristics, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle)
{
    return g_BleGattClientContainer.GetGattServerCharacteristics(pCharacteristics, inNum, connectionHandle, serviceHandle);
}

uint8_t GetGattDescriptors(user::GattDescriptor *pDescriptors, uint8_t inNum, uint32_t connectionHandle, uint16_t charHandle)
{
    return g_BleGattClientContainer.GetGattServerDescritpors(pDescriptors, inNum, connectionHandle, charHandle);
}

uint16_t GetBleMtu(uint32_t connectionHandle)
{
    return g_BleGattClientContainer.GetMtu(connectionHandle);
}

bool    IsConfiguringBleMtu()
{
    return g_BleGattClientContainer.IsConfiguringMtu();
}

bool GetBlePairingSupported(uint32_t connectionHandle)
{
    return g_BleGattClientContainer.CheckIfSupportPairing(connectionHandle);
}

bool GetBlePairingOnGoing()
{
    return (g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_Init);
}

uint8_t GetBlePairedDeviceAddress(BdAddress* pAddresses, uint8_t inNum, const user::BleAdvFilterForGeneral& filter)
{
    uint8_t outNum = 0;

    const BlePairingInfoContainer::BlePairingInfo* pInfoList = g_BlePairingInfoContainer.GetInfoList();

    for (int i = 0; i < nn::bluetooth::BlePairingCountMax; ++i)
    {
        if (memcmp(pInfoList[i].advInfo.manufacturerId, filter.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize) == 0 &&
            memcmp(pInfoList[i].advInfo.clientId, filter.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize) == 0 &&
            memcmp(pInfoList[i].advInfo.serverId, filter.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize) == 0 &&
            outNum < inNum)
        {
            pAddresses[outNum++] = pInfoList[i].address;

            if (outNum == inNum)
            {
                break;
            }
        }
    }

    return outNum;
}

bool IsGamepadPairingStarted()
{
    return g_HcContainer.IsAutoPairing();
}

bool IsRadioEnabled()
{
    return g_HcContainer.IsRadio();
}

bool IsLlrStarted()
{
    return g_HcContainer.IsLlr();
}

GeneralInfoList GetGeneralInfo(int mode)
{
    GeneralInfoList giList;

    //デバッグ向け機能のため、giInfoListのサイズチェックはしない
    if(mode == 0)
    {
        HostConditionContainer::PseudoConnectionList list = g_HcContainer.GetPseudoConnectionList();
        for(int i=0;i<list.count;i++)
        {
            giList.info[i] = list.bdAddress[i].address[5];
        }
        giList.count = list.count;
    }

    return giList;
}

// Applet Manager にImpl から値をセットするメソッド群。Applet Manager 内で排他処理。
void RegisterAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid,
                                  uint32_t appletId)
{
    NN_ABORT_UNLESS(g_AppletManager.RegisterAppletResourceUser(aruid, appletId));
}

void UnregisterAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid)
{
    NN_ABORT_UNLESS(g_AppletManager.UnregisterAppletResourceUser(aruid));
}

void SetForeGroundAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid)
{
    NN_ABORT_UNLESS(g_AppletManager.SetInFocusedAppletResourceUser(aruid));

    CommandParam_BleDisconnectNoOwnerConnection command;
    command.aruid = aruid;

    Message message;
    message.apiId = ApiId_BleDisconnectNoOwnerConnection;
    memcpy(message.buffer, &command, sizeof(CommandParam_BleDisconnectNoOwnerConnection));
    EnQueue(&message);
}

bool GetBleDisconnectionReason(uint16_t* pOutReason, uint32_t connectionHandle, const BdAddress& address)
{
    debug::BleDisconnectionReason reason;

    bool IsReasonExisting = g_BleGattClientContainer.GetGattClientDisconnectionReason(&reason, connectionHandle, address);

    *pOutReason = static_cast<uint16_t>(reason);

    return IsReasonExisting;
}

bool GetBleConnectionParameter(uint16_t* pOutInterval, uint16_t* pOutLatency, uint16_t* pOutTimeout, uint32_t connectionHandle)
{
    return g_BleGattClientContainer.GetConnectionParameter(pOutInterval, pOutLatency, pOutTimeout, connectionHandle);
}

bool GetBleConnectionParameterRequest(uint16_t* pOutIntervalMin, uint16_t* pOutIntervalMax, uint16_t* pOutLatency, uint16_t* pOutTimeout,
                                      uint32_t connectionHandle)
{
    return g_BleGattClientContainer.GetConnectionParameterReq(pOutIntervalMin, pOutIntervalMax, pOutLatency, pOutTimeout, connectionHandle);
}

BtmState InitializeHandler(nn::os::SystemEventType* pCoreEvent, nn::os::SystemEventType* pHidEvent, nn::os::SystemEventType* pBleCoreEvent)
{
    nn::Result result;

    nn::bluetooth::InitializeBluetoothDriver();
    result = nn::bluetooth::InitializeBluetooth(pCoreEvent);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::bluetooth::InitializeHid(pHidEvent, 1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::bluetooth::InitializeBluetoothLe(pBleCoreEvent);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    if (nn::bluetooth::GetIsManufacturingMode())
    {
        // 工程検査モードではこれ以上なにもせず
        return BtmState_NotInitialized;
    }

    result=nn::bluetooth::EnableBluetooth();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //チップ情報を読み出し、btmに保持
    nn::bluetooth::AdapterProperty adapterProperty;
    result=nn::bluetooth::GetAdapterProperties(&adapterProperty);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    HostDeviceProperty hostDeviceProperty;
    memcpy(&hostDeviceProperty.bdName.name[0], &adapterProperty.bluetoothName.name[0], SIZE_OF_BDNAME);
    hostDeviceProperty.bdName.name[SIZE_OF_BDNAME - 1]= '\0';//NULL終端
    hostDeviceProperty.bdAddress = adapterProperty.bluetoothAddress;
    memcpy(&hostDeviceProperty.classOfDevice.cod[0], &adapterProperty.classOfDevice.cod[0], SIZE_OF_COD);
    hostDeviceProperty.featureSet = adapterProperty.featureSet;
    g_HcContainer.SetProperty(hostDeviceProperty);

    result = nn::bluetooth::EnableBluetoothLe();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //　Nintendo Gatt Pairing Service に関するGatt 操作の通知を登録
    result = nn::bluetooth::RegisterLeCoreDataPath(BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // GATT Client を4つ登録
    for (int i = 0; i < nn::bluetooth::BLE_GATT_CLIENT_NUM_MAX; ++i)
    {
        nn::bluetooth::GattAttributeUuid uuid = { nn::bluetooth::GattAttributeUuidLength_16, {(uint16_t)(i + 1)} };
        nn::bluetooth::RegisterLeClient(uuid);
    }

    // デフォルトのコネクションパラメーターを設定
    nn::bluetooth::LeConnectionParams connParam;
    connParam.minConnectionInterval = BleGattClientConditionContainer::DefaultConnectionIntervalMin;
    connParam.maxConnectionInterval = BleGattClientConditionContainer::DefaultConnectionIntervalMax;
    connParam.slaveLatency          = BleGattClientConditionContainer::DefaultSlaveLatency;
    connParam.supervisionTimeout    = BleGattClientConditionContainer::DefaultSupervisionTimeout;
    connParam.maxConnectionEventLen = static_cast<uint16_t>(BleGattClientConditionContainer::DefaultCeLengthMin);
    connParam.minConnectionEventLen = static_cast<uint16_t>(BleGattClientConditionContainer::DefaultCeLengthMax);

    result = nn::bluetooth::SetLeDefaultConnectionParameter(connParam);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // デフォルトのBLE Scan パラメータを設定
    result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_LowDuty.interval, BleScannerCondition::BleScanParameter_LowDuty.window);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // BLE Scan Filter を有効化（スキャン自体は開始しない）
    result = nn::bluetooth::EnableLeScanFilter(true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // BLE Scan Filter をクリア（念のため）
    result = nn::bluetooth::ClearLeScanFilters();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    g_BleScannerCondition.ClearFilters();

    // BLE ペアリング情報をSettings から読み出す
    g_BlePairingInfoContainer.Load();

    // ToDo: BLE すれ違い情報をSettings から読み出す

    result = nn::bluetooth::ExtSetMcMode(true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //Pairing情報をSettingsから読み出し、下層へ設定
    g_DiContainer.InitByNvm();
    const nn::settings::system::BluetoothDevicesSettings* pSettings = g_DiContainer.GetPtr();
    for(int i=0;i<g_DiContainer.GetCount();i++)
    {
        result = nn::bluetooth::HidAddPairedDevice(&pSettings[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    //Radio情報をSettingsから読み出し、下層へ設定
    if(nn::settings::system::IsBluetoothEnabled())
    {
        //起動時はPageScan無効のため、有効にする
        result = nn::bluetooth::ExtSetVisibility(false, true);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        g_HcContainer.SetRadio(true);

        // ToDo: BLE すれちがい通信のScan Filter を登録する

        // BLE すれちがい通信の登録がある場合、スキャンを開始する
        //if (g_BleScannerCondition.GetIsScanning())
        //{
        //    result = nn::bluetooth::StartLeScan();
        //    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        //}

        return BtmState_Initialized;
    }
    else
    {
        //起動時はPageScan無効のため、Radioモードのみ無効化する
        result = nn::bluetooth::ExtSetRadio(false);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        g_HcContainer.SetRadio(false);

        return BtmState_RadioOff;
    }
}

void FinalizeHandler(nn::os::SystemEventType* pCoreEvent, nn::os::SystemEventType* pHidEvent, nn::os::SystemEventType* pBleCoreEvent)
{
    //[Todo]システムイベント、その他グローバル変数の終了処理
    //BluetoothProcessおよび関連オブジェクトの末期化
    auto result = nn::bluetooth::ExtSetMcMode(false);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::bluetooth::DisableBluetoothLe();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::bluetooth::DisableBluetooth();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::bluetooth::CleanupHid();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::bluetooth::CleanupBluetoothLe();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::bluetooth::CleanupBluetooth();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::bluetooth::FinalizeBluetoothDriver();

    nn::os::DestroySystemEvent(pCoreEvent);
    nn::os::DestroySystemEvent(pHidEvent);
    nn::os::DestroySystemEvent(pBleCoreEvent);
}


//--------------------------------------------------
//BluetoothからのBtCoreレポートを処理。
//取得したレポートに応じて、btmの内部状態更新、クライアントプロセス向けのシステムイベントのシグナル、Bluetoothへの指示を行う。
//
//StateMachineのMultiWaitから直接呼ばれるパタン（後述のオーバーロード関数）と、HandleApiのMultiWaitAndHandleAsyncReportから呼ばれるパタン（本関数）がある。
//前者は外部起因のレポート、後者はAPI起因のレポート。
//後者向けに、特定のレポート取得を判別できる機能（requiredCoreEventTypeのイベントを取得すると、返り値がtrueとなる）を持つ。
//
//時間のかかる処理をここに書いてしまうとAPIの処理時間が延びうるため、
//そのような処理は擬似APIとしてキューイングし、StateMachineのMultiWaitで再評価するようにしている
//--------------------------------------------------
bool HandleBtCoreReport(nn::bluetooth::EventType requiredCoreEventType, void* pBufferBtp, size_t bufferSizeBtp)
{
    nn::bluetooth::EventType eventTypeBtp;
    GetBtCoreReport(&eventTypeBtp, pBufferBtp, bufferSizeBtp);

    BTM_LOG("Handle BtCoreReport Event: %d\n", eventTypeBtp);
    switch(eventTypeBtp)
    {
        case nn::bluetooth::EventFromAdapterPropertiesCallback:
        {
            BTM_LOG("AdapterProperties\n");
            break;
        }
        case nn::bluetooth::EventFromDiscoveryStateChangedCallback:
        {
            BTM_LOG("DiscoveryStateChanged\n");
            const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(pBufferBtp);
            if(pInfo->state == nn::bluetooth::BT_DISCOVERY_STOPPED)
            {
                BTM_LOG("Discovery stopped\n");
                g_HcContainer.SetDiscoverying(false);

                if(g_HcContainer.IsAutoPairing())
                {
                    BTM_LOG("Handling Auto Pairing\n");
                    BdAddress bdAddress;
                    bool isThereTarget;

                    for(int i=0;i<HostConditionContainer::APDN_COUNT;i++)
                    {
                        isThereTarget = g_DpContainer.SearchBdAddress(&bdAddress, &g_HcContainer.GetApdn(i)[0], g_HcContainer.GetApdnSize(i));
                        if(isThereTarget)
                        {
                            break;
                        }
                    }

                    if(isThereTarget)
                    {
                        //同期CreateBondとStartDiscoveryをAPIとして再キューイングする
                        //前者は時間のかかる処理のため
                        //後者は同期CreateBond完了後に、Discoveryを再開させるため
                        CommandParam_BlockingCreateBond command;
                        command.address = bdAddress;
                        Message message;//[Todo]毎回確保しないようにする
                        *reinterpret_cast<CommandParam_BlockingCreateBond*>(&message.buffer[0]) = command;
                        message.apiId = ApiId_BlockingCreateBond;
                        EnQueue(&message);

                        message.apiId = ApiId_ReStartDiscoveryForAutoPair;
                        EnQueue(&message);
                    }
                    else
                    {
                        g_DpContainer.Init();
                        nn::bluetooth::StartDiscovery();
                    }
                }
                else
                {
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Disc);
                }
                break;
            }
            else
            {
                BTM_LOG("Discovery started\n");
                g_HcContainer.SetDiscoverying(true);
            }
            break;
        }
        case nn::bluetooth::EventFromDeviceFoundCallback:
        {
            BTM_LOG("DeviceFound\n");
            const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromDeviceFoundCallback*>(pBufferBtp);
            DeviceProperty deviceProperty;

            memcpy(&deviceProperty.bdAddress.address[0], &pInfo->BdAddress.address[0], SIZE_OF_BDADDRESS);
            memcpy(&deviceProperty.classOfDevice.cod[0], &pInfo->ClassOfDevice[0], SIZE_OF_COD);
            memcpy(&deviceProperty.bdName.name[0], &pInfo->BdName.name[0], SIZE_OF_BDNAME);

            g_DpContainer.Add(deviceProperty);

            //ペアリングをなるべく早く完了させるために、対象デバイスが見つかり次第Discoveryをキャンセルし、
            //DiscoveryStateChangedコールバックからCreateBond処理へと進ませる。
            if(g_HcContainer.IsAutoPairing())
            {
                BdAddress dummyBdAddress;
                bool isThereTarget;

                for(int i=0;i<HostConditionContainer::APDN_COUNT;i++)
                {
                    isThereTarget = g_DpContainer.SearchBdAddress(&dummyBdAddress, &g_HcContainer.GetApdn(i)[0], g_HcContainer.GetApdnSize(i));
                    if(isThereTarget)
                    {
                        break;
                    }
                }

                if(isThereTarget)
                {
                    nn::bluetooth::CancelDiscovery();
                }
            }

            break;
        }
        case nn::bluetooth::EventFromSspRequestCallback:
        {
            BTM_LOG("SspRequest\n");
            const nn::bluetooth::InfoFromSspRequestCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromSspRequestCallback*>(pBufferBtp);

            for(int i=0;i<HostConditionContainer::APDN_COUNT;i++)
            {
                if(0 == memcmp(&g_HcContainer.GetApdn(i)[0], &pInfo->bluetoothName.name[0] ,g_HcContainer.GetApdnSize(i)))
                {
                    nn::bluetooth::SspReply(&pInfo->bluetoothAddress, nn::bluetooth::BT_SSP_VARIANT_PASSKEY_CONFIRMATION, true, 0);
                    break;
                }
            }
            break;
        }
        case nn::bluetooth::EventFromPinRequestCallback:
        {
            BTM_LOG("PinRequest\n");
            break;
        }
        case nn::bluetooth::EventFromBondStateChangedCallback:
        {
            BTM_LOG("BondStateChanged\n");
            const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromBondStateChangedCallback*>(pBufferBtp);
            if(pInfo->state == nn::bluetooth::BT_BOND_STATE_BONDED && pInfo->status == nn::bluetooth::BT_STATUS_SUCCESS)
            {
                nn::settings::system::BluetoothDevicesSettings deviceSetting;
                auto result = nn::bluetooth::HidGetPairedDevice(&pInfo->bluetoothAddress, &deviceSetting);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                NN_UNUSED(result);

                g_DiContainer.Add(deviceSetting);

                //Settings(Nvm)への書き込みは時間がかかるため、APIとして再キューイングする
                Message message;//[Todo]毎回確保しないようにする
                message.apiId = ApiId_StoreDeviceInfo;
                EnQueue(&message);

                g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            }
            //削除時にはBondStateChangedは来ない
            break;
        }
        default:
        {
            BTM_LOG("Un-handled System Event: 0x%X\n", eventTypeBtp);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorUnHandledCoreEvent());
        }
    }

    if(requiredCoreEventType == eventTypeBtp)
    {
        return true;
    }
    return false;
} //NOLINT(impl/function_size)

//--------------------------------------------------
//StateMachineのMultiWaitでReportを検出した場合に使われる関数
//レポート内容をクライアントプロセスに伝える必要が無いため、内部で作業用バッファを確保し、ハンドラー処理後は解放する
//--------------------------------------------------
void HandleBtCoreReport()
{
    //[Todo]バッファサイズ要調整。BUFFER_SIZE_OF_CORE_OUTで良いのでは？
    uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_HID_OUT;
    HandleBtCoreReport(nn::bluetooth::EventFromAdapterStateChangedCallback, bufferBtp, bufferSizeBtp);
}

//--------------------------------------------------
//BluetoothからのBtBleCore レポートを処理。
//取得したレポートに応じて、btmの内部状態更新、クライアントプロセス向けのシステムイベントのシグナル、Bluetoothへの指示を行う。
//
//StateMachineのMultiWaitから直接呼ばれるパタン（後述のオーバーロード関数）と、HandleApiのMultiWaitAndHandleAsyncReportから呼ばれるパタン（本関数）がある。
//前者は外部起因のレポート、後者はAPI起因のレポート。
//後者向けに、特定のレポート取得を判別できる機能（requiredCoreEventTypeのイベントを取得すると、返り値がtrueとなる）を持つ。
//
//時間のかかる処理をここに書いてしまうとAPIの処理時間が延びうるため、
//そのような処理は擬似APIとしてキューイングし、StateMachineのMultiWaitで再評価するようにしている
//--------------------------------------------------
bool HandleBtBleCoreReport(nn::bluetooth::BleEventType requiredCoreEventType, void* pBufferBtp, size_t bufferSizeBtp)
{
    nn::bluetooth::BleEventType eventTypeBtp;
    GetBtBleCoreReport(&eventTypeBtp, pBufferBtp, bufferSizeBtp);

    BTM_LOG("%s: Event: %d\n", __func__, eventTypeBtp);

    switch (eventTypeBtp)
    {
    case nn::bluetooth::EventFromLeClientStateChangedCallback:
    {
        const nn::bluetooth::InfoFromLeAppStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeAppStateChangedCallback*>(pBufferBtp);

        if (pInfo->status == nn::bluetooth::BT_OK)
        {
            if (pInfo->isRegistered)
            {
                BTM_LOG("%s: GATT Client Registered. Handle = %d\n", __func__, pInfo->clientIf);
                g_BleGattClientContainer.AddClient(pInfo->clientIf);
            }
            else
            {
                BTM_LOG("%s: GATT Client Unregistered. Handle = %d\n", __func__, pInfo->clientIf);
                g_BleGattClientContainer.RemoveClient(pInfo->clientIf);
            }
        }
    }
        break;
    case nn::bluetooth::EventFromLeServerStateChangedCallback:
        BTM_LOG("%s: EventFromLeServerStateChangedCallback. Currently not supported.\n", __func__);
        break;
    case nn::bluetooth::EventFromLeConnParamUpdateCallback:
    {
        const nn::bluetooth::InfoFromLeConnParamUpdateCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeConnParamUpdateCallback*>(pBufferBtp);

        if (pInfo->status == nn::bluetooth::BT_OK)
        {
            // 接続がないならなにもしないだけなので、結果は見ない
            g_BleGattClientContainer.UpdateConnectionParameters(pInfo->connId, pInfo->interval, pInfo->slaveLatency, pInfo->supervisionTimeout);
        }
    }
        break;
    case nn::bluetooth::EventFromLeConnParamUpdateReqCallback:
    {
        const nn::bluetooth::InfoFromLeConnParamUpdateReqCallback* pInfo = reinterpret_cast<const nn::bluetooth::InfoFromLeConnParamUpdateReqCallback*>(pBufferBtp);

        if (g_BleGattClientContainer.UpdateConnectionParameterRequest(pInfo->connId,
                                                                      pInfo->intervalMin, pInfo->intervalMax,
                                                                      pInfo->slaveLatency, pInfo->supervisionTimeout) == false)
        {
            // 接続が存在しないなら、これ以上何もしない
            break;
        }

        uint16_t currentInterval, currentLatency, currentTimeout;
        CeLength currentCeLength;
        NN_ABORT_UNLESS(g_BleGattClientContainer.GetConnectionParameters(pInfo->connId, &currentInterval, &currentLatency, &currentTimeout, &currentCeLength));

        nn::bluetooth::LeConnectionParams param;
        bool isMultipleOfDefault = false;

        param.bluetoothAddress = *g_BleGattClientContainer.GetGattServerAddress(pInfo->connId);

        isMultipleOfDefault = ((pInfo->intervalMin % BleGattClientConditionContainer::DefaultConnectionIntervalMin) == 0);

        param.minConnectionInterval = BleGattClientConditionContainer::DefaultConnectionIntervalMin;
        param.maxConnectionInterval = BleGattClientConditionContainer::DefaultConnectionIntervalMax;

        // 要求された最小値を超える 15 msec の倍数の最小値を設定する
        if (pInfo->intervalMin >= BleGattClientConditionContainer::DefaultConnectionIntervalMin)
        {
            int plusOne = (isMultipleOfDefault) ? 0 : 1;

            param.minConnectionInterval = ((pInfo->intervalMin / BleGattClientConditionContainer::DefaultConnectionIntervalMin) + plusOne) *
                                          BleGattClientConditionContainer::DefaultConnectionIntervalMin;
            param.maxConnectionInterval = param.minConnectionInterval;
        }

        DeviceConditionList deviceConditionList = g_DcContainer.Get();
        deviceConditionList.isLargeSlotRequiredForBle = true;

        CeLength ceLength = GetDefaultCeLength(&deviceConditionList, g_BleGattClientContainer.GetPtr());
        if (ceLength == CeLength_Invalid)
        {
            nn::bluetooth::LeClientDisconnect(pInfo->connId);
        }
        else
        {
            param.slaveLatency          = pInfo->slaveLatency;                  // 要求に従う
            param.supervisionTimeout    = pInfo->supervisionTimeout;            // 要求に従う
            param.minConnectionEventLen = static_cast<uint16_t>(CeLength_2);    // 常に

            if (g_BleGattClientContainer.IsLargeCeLengthRequired())
            {
                // 上位層から要求されていれば、現状維持
                param.maxConnectionEventLen = static_cast<uint16_t>(currentCeLength);
            }
            else
            {
                // 要求がなければ、デフォルト値を設定
                param.maxConnectionEventLen = static_cast<uint16_t>(ceLength);
            }
            param.preference = true;

            BTM_LOG("ConnId = %d, minInt = %d(req = %d), maxInt = %d(req = %d), lat = %d(req = %d), tout = %d(req = %d), CeLen = %d-%d\n",
                pInfo->connId,
                param.minConnectionInterval, pInfo->intervalMin,
                param.maxConnectionInterval, pInfo->intervalMax,
                param.slaveLatency, pInfo->slaveLatency,
                param.supervisionTimeout, pInfo->supervisionTimeout,
                param.minConnectionEventLen, param.maxConnectionEventLen);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::SetLeConnectionParameter(param));
        }

        break;
    }
    case nn::bluetooth::EventFromLeClientConnStateChangedCallback:
    {
        const nn::bluetooth::InfoFromLeConnStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeConnStateChangedCallback*>(pBufferBtp);

        switch (pInfo->connState)
        {
        case nn::bluetooth::BLE_CONN_STATE_CONNECTED:
        {
            BTM_LOG("Connected to %02X:%02X:%02X:%02X:%02X:%02X    Client If: %d    Conn Handle: %d\n",
                pInfo->address.address[0], pInfo->address.address[1], pInfo->address.address[2], pInfo->address.address[3], pInfo->address.address[4], pInfo->address.address[5],
                pInfo->clientIf, pInfo->connId);

            if (pInfo->status == nn::bluetooth::BT_OK)
            {
                nn::applet::AppletResourceUserId connectionTriggerAruid;
                bool IsConnectionRefusal = g_BleGattClientContainer.IsConnectionRefusal();

                // 接続全拒否状態、もしくは接続のトリガがフォアグラウンドでないなら切断
                if (IsConnectionRefusal)
                {
                    BTM_LOG("BLE connection is refused on connection event\n");
                    nn::bluetooth::LeClientDisconnect(pInfo->connId);
                    break;
                }
                else if(connectionTriggerAruid != nn::applet::AppletResourceUserId::GetInvalidId() &&
                        connectionTriggerAruid != g_AppletManager.GetInFocusedAppletResourceUser().aruid)
                {
                    BTM_LOG("BLE connection is refused because connection trigger is not in focus\n");
                    nn::bluetooth::LeClientDisconnect(pInfo->connId);
                    break;
                }

                DeviceConditionList deviceConditionList = g_DcContainer.Get();
                g_BleGattClientContainerForWorking  = g_BleGattClientContainer;

                if(g_BleGattClientContainerForWorking.ConnectClient(pInfo->clientIf, pInfo->connId, pInfo->address))
                {
                    deviceConditionList.isLargeSlotRequiredForBle = true;

                    SlotMode    defaultSlotMode;
                    CeLength    defaultCeLength;
                    bool        isRestructureAll;
                    uint8_t     bleRestructureCount;

                    if (CheckUsecase(g_DcContainer.GetPtr(), &deviceConditionList,
                                     g_BleGattClientContainer.GetPtr(), g_BleGattClientContainerForWorking.GetPtr(),
                                     &defaultSlotMode, &defaultCeLength, &isRestructureAll, &bleRestructureCount))
                    {
                        g_BleGattClientContainer    = g_BleGattClientContainerForWorking;

                        g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);       // 上位層に接続を早く伝えるために、イベントシグナル

                        CommandParam_ChangeUsecaseByBleConnect command;
                        command.defaultSlotMode = defaultSlotMode;
                        command.defaultCeLength = defaultCeLength;

                        Message message;
                        *reinterpret_cast<CommandParam_ChangeUsecaseByBleConnect*>(message.buffer) = command;
                        message.apiId = ApiId_ChangeUsecaseByBleConnect;
                        EnQueue(&message);

                        break;
                    }
                    else
                    {
                        nn::bluetooth::LeClientDisconnect(pInfo->connId);
                    }
                }
                else
                {
                    nn::bluetooth::LeClientDisconnect(pInfo->connId);
                    break;
                }
            }
        }
        break;
        case nn::bluetooth::BLE_CONN_STATE_CANCELED:
        case nn::bluetooth::BLE_CONN_STATE_DISCONNECTED:
        {
            BTM_LOG("Disconnected from %02X:%02X:%02X:%02X:%02X:%02X    reason: 0x%02X    Client If: %d    Conn Handle: %d\n",
                pInfo->address.address[0], pInfo->address.address[1], pInfo->address.address[2], pInfo->address.address[3], pInfo->address.address[4], pInfo->address.address[5],
                pInfo->reason, pInfo->clientIf, pInfo->connId);

            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            g_BleGattClientContainerForWorking  = g_BleGattClientContainer;

            if (g_BleGattClientContainerForWorking.DisconnectClient(pInfo->clientIf, pInfo->connId, pInfo->reason))
            {
                SlotMode    defaultSlotMode;
                CeLength    defaultCeLength;
                bool        isRestructureAll;
                uint8_t     bleRestructureCount;

                if (g_BleGattClientContainerForWorking.GetPtr()->deviceCount == 0)
                {
                    deviceConditionList.isLargeSlotRequiredForBle = false;
                }

                if (CheckUsecase(g_DcContainer.GetPtr(), &deviceConditionList,
                                 g_BleGattClientContainer.GetPtr(), g_BleGattClientContainerForWorking.GetPtr(),
                                 &defaultSlotMode, &defaultCeLength, &isRestructureAll, &bleRestructureCount))
                {
                    g_BleGattClientContainer    = g_BleGattClientContainerForWorking;

                    // UpdateUsecase は時間が掛かるので、ここでイベントシグナル
                    // 接続を受け付けない状態では、強制切断され、接続が発生したことを上位層はしらないので、切断も通知しない
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);

                    CommandParam_ChangeUsecaseByBleDisconnect command;
                    command.defaultSlotMode = defaultSlotMode;
                    command.defaultCeLength = defaultCeLength;

                    Message message;
                    *reinterpret_cast<CommandParam_ChangeUsecaseByBleDisconnect*>(message.buffer) = command;
                    message.apiId = ApiId_ChangeUsecaseByBleDisconnect;
                    EnQueue(&message);
                }
                else
                {
                    BTM_LOG("Invalid usecase by BLE disconnect\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorOverConnection());
                }
            }
        }
        break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    break;
    case nn::bluetooth::EventFromLeServerConnStateChangedCallback:
    {
        BTM_LOG("%s: EventFromLeServerConnStateChangedCallback. Currently not supported.\n", __func__);
        break;
    }
    case nn::bluetooth::EventFromLeScanStateChangedCallback:
    {
        const nn::bluetooth::InfoFromLeScanStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeScanStateChangedCallback*>(pBufferBtp);

        if (pInfo->status == nn::bluetooth::BT_OK)
        {
            if (pInfo->state == nn::bluetooth::BLE_SCAN_STATE_SCANNING)
            {
                BTM_LOG("%s: BLE Scan Started.\n", __func__);

                if (!g_BleScannerCondition.IsScanning())
                {
                    NN_SDK_LOG("[btm] Unexpected start event of BLE scan.\n");
                }
                g_BleScannerCondition.SetIsScanning(true);
            }
            else if (pInfo->state == nn::bluetooth::BLE_SCAN_STATE_COMPLETED)
            {
                BTM_LOG("%s: BLE Scan Stopped.\n", __func__);

                if (g_BleScannerCondition.IsScanning())
                {
                    NN_SDK_LOG("[btm] Unexpected stop event of BLE scan.\n");
                }
                g_BleScannerCondition.SetIsScanning(false);
            }
            else if (pInfo->state == nn::bluetooth::BLE_SCAN_STATE_FOUND_DEVICE)
            {
//                BTM_LOG("%s: BLE Scan found device.\n", __func__);

                // 接続されている or 接続を試行しているデバイスからのAdv は無視する
                nn::applet::AppletResourceUserId tempAruid;

                if (g_BleGattClientContainer.GetConnectionHandle(pInfo->address) != nn::bluetooth::BleInvalidConnectionHandle ||
                    g_BleGattClientContainer.IsPendingConnection(&tempAruid, pInfo->address))
                {
                    break;
                }

                if ((pInfo->pduType == nn::bluetooth::BleAdvertisePduType_AdvInd || pInfo->pduType == nn::bluetooth::BleAdvertisePduType_ScanRsp))
                {
                    user::ScanResult scanResult;

                    scanResult.address          = pInfo->address;
                    scanResult.adStructureNum   = pInfo->adStructureNum;
                    for (int i = 0; i < scanResult.adStructureNum; ++i)
                    {
                        scanResult.adStructures[i] = pInfo->adStructures[i];
                    }
                    scanResult.rssi             = pInfo->rssi;

                    // ToDo: Minimum Awake 状態かつ、Wake On BLE スキャンフィルタを検索し、一致するものがあれば、Full Awake する
                    // 接続までする？その場合、通知先は？通知しない？

                    if (pInfo->addressType == nn::bluetooth::BLE_ADV_ADDRESS_TYPE_PUBLIC)
                    {
                        nn::applet::AppletResourceUserId scanTrigger;
                        nn::bluetooth::BleAdvertiseFilter filterCondition;
                        BlePairingInfoContainer::BlePairingInfo info;
                        memcpy(info.address.address, scanResult.address.address, SIZE_OF_BDADDRESS);

                        // ペアリングされたAdvertise かをチェックして自動接続
                        if (g_BleScannerCondition.GetIfMatchScanFilter(&filterCondition, &scanTrigger, BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED, scanResult))
                        {
                            int index = 0;
                            BdAddress addressInAdvertise = { {0x00} };                      // Advertise Packet 中のペアリングアドレス
                            BdAddress ownAddress = g_HcContainer.GetProperty().bdAddress;   // 自分のアドレス

                            for (index = 0; index < scanResult.adStructureNum; ++index)
                            {
                                if (scanResult.adStructures[index].adType == nn::bluetooth::BleAdType_ManufactureSpecificData)
                                {
                                    info.type = BlePairingAdvDataType_ManuCliSerId;
                                    memcpy(info.advInfo.manufacturerId, &scanResult.adStructures[index].data[BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET], nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
                                    memcpy(info.advInfo.clientId, &scanResult.adStructures[index].data[BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET], nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
                                    memcpy(info.advInfo.serverId, &scanResult.adStructures[index].data[BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET], nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
                                    memcpy(addressInAdvertise.address, &scanResult.adStructures[index].data[BleScannerCondition::BLE_SCAN_CONDITION_MANU_PAIRED_ADDR_OFFSET], SIZE_OF_BDADDRESS);
                                    break;
                                }
                            }

                            if (index < scanResult.adStructureNum && g_BlePairingInfoContainer.IsPaired(info) &&
                                addressInAdvertise == ownAddress)
                            {
                                if (g_BleGattClientContainer.IsConnectionRefusal())
                                {
                                    // 接続できる状態でなければスキャン結果を破棄して break
                                    break;
                                }

                                Message message;
                                CommandParam_BleConnect command;
                                command.aruid = scanTrigger;
                                command.address = info.address;

                                // Usecase のチェックはこのApi の処理時に行われ、実際に接続するかを決定するため、ここではUsecase チェックしない
                                // 下層からのイベントハンドルを素早く抜けるため
                                *reinterpret_cast<CommandParam_BleConnect*>(&message.buffer[0]) = command;
                                message.apiId = ApiId_BleConnect;
                                EnQueue(&message);
                                break;
                            }
                        }
                    }

                    // 通常スキャンフィルタを検索し、一致するものが見つかれば、システムイベントをシグナル
                    if (g_BleScannerCondition.AddScanResultsGeneral(scanResult) || g_BleScannerCondition.AddScanResultsSd(scanResult))
                    {
                        g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleScan);
                        break;
                    }
                }
                else if (pInfo->pduType == nn::bluetooth::BleAdvertisePduType_AdvNonConnInd ||
                         pInfo->pduType == nn::bluetooth::BleAdvertisePduType_AdvScanInd)
                {
                    // ToDo: すれ違いスキャンフィルタを検索し、一致するものがあれば、保存
                }
            }
        }
    }
        break;
    case nn::bluetooth::EventFromLeScanFilterStateChangedCallback:
    {
        const nn::bluetooth::InfoFromLeScanFilterStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeScanFilterStateChangedCallback*>(pBufferBtp);

        if (pInfo->status == nn::bluetooth::BT_OK)
        {
            switch (pInfo->operation)
            {
            case nn::bluetooth::BLE_SCAN_FILTER_OP_COND_ADD:
                BTM_LOG("%s: BLE Scan Filter Added.\n", __func__);

                break;
            case nn::bluetooth::BLE_SCAN_FILTER_OP_COND_DELETE:
                BTM_LOG("%s: BLE Scan Filter Deleted.\n", __func__);
                break;
            case nn::bluetooth::BLE_SCAN_FILTER_OP_COND_CLEAR:
                BTM_LOG("%s: BLE Scan Filter Cleared.\n", __func__);
                g_BleScannerCondition.ClearFilters();
                break;
            case nn::bluetooth::BLE_SCAN_FILTER_OP_ENABLE:
                BTM_LOG("%s: BLE Scan Filter Enabled.\n", __func__);
                g_BleScannerCondition.SetIsScanFilterEnalbed(true);
                break;
            case nn::bluetooth::BLE_SCAN_FILTER_OP_DISABLE:
                BTM_LOG("%s: BLE Scan Filter Disabled.\n", __func__);
                g_BleScannerCondition.SetIsScanFilterEnalbed(false);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
        break;
    case nn::bluetooth::EventFromLeClientGattOpCallback:
    {
        BTM_LOG("EventFromLeClientGattOpCallback\n");

        Message message;
        const nn::bluetooth::InfoFromLeGattOperationCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeGattOperationCallback*>(pBufferBtp);

        // BLE ペアリング操作
        if (pInfo->serviceUuid == BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID)
        {
            BTM_LOG("BLE Pairing Operation callback\n");

            const BlePairingInfoContainer::BlePairingInfo *pCurrentPairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pCurrentPairingInfo == nullptr)
            {
                // ペアリング開始前
                BTM_LOG("BLE Pairing is not on going. Unexpected GATT operation.\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorFaultyDesign());
                break;
            }

            // GATT 処理でエラー発生
            if (pInfo->status != nn::bluetooth::BT_OK)
            {
                NN_SDK_LOG("[btm] BLE pairing aborted becase of GATT server request (%d).\n", pInfo->value[0]);
                message.apiId = ApiId_BlePairingWriteAbort;
                EnQueue(&message);
                break;
            }

            // Peripheral からのエラー通知
            if ((pInfo->operation == nn::bluetooth::GattOperationType_Notify ||
                 pInfo->operation == nn::bluetooth::GattOperationType_Indicate) &&
                pInfo->charcteristicUuid == BlePairingInfoContainer::NN_GATT_PAIRING_ERROR_HANDLER_CHARACTERISTIC_UUID)
            {
                BTM_LOG("Receive Nintendo Gatt Pairing Error Handler Notification.\n");

                if (pInfo->length != sizeof(BlePairingInfoContainer::BlePairingErrorCode))
                {
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }

                if (pInfo->value[0] == BlePairingInfoContainer::BlePairingErrorCode_Success && g_BlePairingInfoContainer.GetStage() == BlePairingInfoContainer::BlePairingStage_WriteComplete)
                {
                    BTM_LOG("Nintendo Gatt Pairing successfully completed.\n");

                    const BlePairingInfoContainer::BlePairingInfo* pPairingInfo = g_BlePairingInfoContainer.GetPairingDevice();

                    if (pPairingInfo == nullptr)
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorFaultyDesign());
                    }

                    CommandParam_BlePairingUpdateDatabase command;
                    command.type    = pPairingInfo->type;
                    command.address = pPairingInfo->address;
                    command.pairing = g_BlePairingInfoContainer.GetPairing();

                    if (command.type == BlePairingAdvDataType_ManuCliSerId)
                    {
                        memcpy(command.condition.manufacturerId, pPairingInfo->advInfo.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
                        memcpy(command.condition.clientId, pPairingInfo->advInfo.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
                        memcpy(command.condition.serverId, pPairingInfo->advInfo.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorFaultyDesign());
                    }

                    message.apiId = ApiId_BlePairingUpdateDatabase;
                    memcpy(&message.buffer[0], &command, sizeof(command));
                    EnQueue(&message);
                    break;
                }
                else
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase of GATT server request (%d).\n", pInfo->value[0]);
                    message.apiId = ApiId_BlePairingWriteAbort;
                    EnQueue(&message);
                    break;
                }
            }
            // Error Handler の有効化 -> Public Address の読み出し
            else if (pInfo->operation == nn::bluetooth::GattOperationType_WriteDescriptor &&
                pInfo->charcteristicUuid == BlePairingInfoContainer::NN_GATT_PAIRING_ERROR_HANDLER_CHARACTERISTIC_UUID &&
                pInfo->descriptorUuid == nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid)
            {
                BTM_LOG("BLE Pairing Operation callback. Enable error handler notification complete.\n");
                message.apiId = ApiId_BlePairingReadAddress;
                EnQueue(&message);
                break;
            }
            // Public Address の読み出し -> Adv Data の読み出し
            else if (pInfo->operation == nn::bluetooth::GattOperationType_ReadCharacteristic &&
                pInfo->charcteristicUuid == BlePairingInfoContainer::NN_GATT_PAIRING_ADDRESS_READER_CHARACTERISTIC_UUID)
            {
                BTM_LOG("BLE Pairing Operation callback. Read peer public address complete.\n");
                if (pInfo->length != SIZE_OF_BDADDRESS)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase read data size does not match at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    message.apiId = ApiId_BlePairingWriteAbort;
                    EnQueue(&message);
                    break;
                }

                BdAddress readAddress;
                BdAddress targetAddress = pCurrentPairingInfo->address;
                const BdAddress* pConnectedAddress = g_BleGattClientContainer.GetGattServerAddress(pInfo->connId);

                memcpy(readAddress.address, pInfo->value, SIZE_OF_BDADDRESS);

                if (pConnectedAddress == nullptr)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the connection is lost at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }
                if (*pConnectedAddress != readAddress)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the address is not public at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    message.apiId = ApiId_BlePairingWriteAbort;
                    EnQueue(&message);
                    break;
                }
                if (*pConnectedAddress != targetAddress)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the server is faked at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }

                message.apiId = ApiId_BlePairingReadAdvData;
                EnQueue(&message);
                break;
            }
            // Adv Data の読み出し → Public Address の書き込み
            else if (pInfo->operation == nn::bluetooth::GattOperationType_ReadCharacteristic &&
                pInfo->charcteristicUuid == BlePairingInfoContainer::NN_GATT_PAIRING_ADV_DATA_READER_CHARACTERISTIC_UUID)
            {
                if (pInfo->length != static_cast<uint16_t>(nn::bluetooth::BleAdStructurePayloadSizeMax) + 1 /* For Data Type Field */)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase read data size does not match at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    message.apiId = ApiId_BlePairingWriteAbort;
                    EnQueue(&message);
                    break;
                }

                BdAddress targetAddress = pCurrentPairingInfo->address;
                const BdAddress* pConnectedAddress = g_BleGattClientContainer.GetGattServerAddress(pInfo->connId);

                if (pConnectedAddress == nullptr)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the connection is lost at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }
                if (*pConnectedAddress != targetAddress)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the server is faked at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }

                if (pInfo->value[0] == static_cast<uint8_t>(BlePairingAdvDataType_ManuCliSerId))
                {
                    uint8_t readManufactureId[nn::bluetooth::BleNnAdvertiseManufacturerIdSize] = { 0x00 };
                    uint8_t readClientId[nn::bluetooth::BleNnAdvertiseManufactureClientIdSize] = { 0x00 };
                    uint8_t readServerId[nn::bluetooth::BleNnAdvertiseManufactureServerIdSize] = { 0x00 };

                    uint8_t offset = sizeof(BlePairingAdvDataType);
                    memcpy(readManufactureId, pInfo->value + offset, sizeof(readManufactureId));
                    offset += sizeof(readManufactureId);
                    memcpy(readClientId, pInfo->value + offset, sizeof(readClientId));
                    offset += sizeof(readClientId);
                    memcpy(readServerId, pInfo->value + offset, sizeof(readServerId));
                    offset += sizeof(readServerId);

                    if (memcmp(readManufactureId, pCurrentPairingInfo->advInfo.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize) != 0 ||
                        memcmp(readClientId, pCurrentPairingInfo->advInfo.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize) != 0 ||
                        memcmp(readServerId, pCurrentPairingInfo->advInfo.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize) != 0)
                    {
                        NN_SDK_LOG("[btm] BLE pairing aborted becase the server's advertise data does not match at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                        message.apiId = ApiId_BlePairingWriteAbort;
                        EnQueue(&message);
                        break;
                    }
                }
                else
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the server's advertise data type does not match at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    message.apiId = ApiId_BlePairingWriteAbort;
                    EnQueue(&message);
                    break;
                }

                message.apiId = ApiId_BlePairingWriteAddress;
                EnQueue(&message);
                break;
            }
            // Public Address の書き込み → Complete Message の書き込み
            else if (pInfo->operation == nn::bluetooth::GattOperationType_WriteCharacteristic &&
                pInfo->charcteristicUuid == BlePairingInfoContainer::NN_GATT_PAIRING_ADDRESS_WRITER_CHARACTERISTIC_UUID)
            {
                BdAddress targetAddress = pCurrentPairingInfo->address;
                const BdAddress* pConnectedAddress = g_BleGattClientContainer.GetGattServerAddress(pInfo->connId);

                if (pConnectedAddress == nullptr)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the connection is lost at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }
                if (*pConnectedAddress != targetAddress)
                {
                    NN_SDK_LOG("[btm] BLE pairing aborted becase the server is faked at stage %d.\n", g_BlePairingInfoContainer.GetStage());
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }

                message.apiId = ApiId_BlePairingWriteComplete;
                EnQueue(&message);
                break;
            }
            // Complete Message の書き込み → データベースのアップデート
            else if (pInfo->operation == nn::bluetooth::GattOperationType_WriteCharacteristic &&
                pInfo->charcteristicUuid == BlePairingInfoContainer::NN_GATT_PAIRING_COMMAND_CHARACTERISTIC_UUID)
            {
                if (g_BlePairingInfoContainer.GetStage() == BlePairingInfoContainer::BlePairingStage_WriteComplete)
                {
                    // 何もしない
                    break;
                }
                // Abort したら、この時点でSignal する
                else if(g_BlePairingInfoContainer.GetStage() == BlePairingInfoContainer::BlePairingStage_WriteAbort)
                {
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                    break;
                }
            }
            else
            {
                NN_SDK_LOG("[btm] Unexpected GATT opereration during BLE pairing.\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorFaultyDesign());
            }
        }
    }
        break;
    case nn::bluetooth::EventFromLeClientServiceDiscoveryCallback:
    {
        const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback*>(pBufferBtp);

        for (int i = 0; i < pInfo->numAttributes; ++i)
        {
            const nn::bluetooth::GattSdpAttribute *pAttribute = &pInfo->attributes[i];

            switch (pInfo->attributes[i].type)
            {
            case nn::bluetooth::GattAttributeType_Service:
            {
                user::GattService service;

                service.type                = nn::bluetooth::GattAttributeType_Service;
                service.handle              = pAttribute->handle;
                service.instanceId          = pAttribute->id.instanceId;
                service.uuid                = pAttribute->id.uuid;
                service.isPrimaryService    = pAttribute->isPrimaryService;
                service.endGroupHandle      = pAttribute->endGroupHandle;

                if (!g_BleGattClientContainer.AddGattServerService(pInfo->connId, service))
                {
                    BTM_LOG("Failed to add service to internal database. No more space.\n");
                }
                // Service が存在すれば、ペアリングをサポートしている
                if (service.uuid == BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID)
                {
                    g_BleGattClientContainer.SupportPairing(pInfo->connId);
                }
            }
            break;
            case nn::bluetooth::GattAttributeType_IncludedService:
            {
                user::GattService service;

                service.type                = nn::bluetooth::GattAttributeType_IncludedService;
                service.handle              = pAttribute->handle;
                service.instanceId          = pAttribute->id.instanceId;
                service.uuid                = pAttribute->id.uuid;
                service.isPrimaryService    = pAttribute->isPrimaryService;
                service.endGroupHandle      = pAttribute->endGroupHandle;

                if (!g_BleGattClientContainer.AddGattServerService(pInfo->connId, service))
                {
                    BTM_LOG("Failed to add service to internal database. No more space.\n");
                }
            }
                break;
            case nn::bluetooth::GattAttributeType_Characteristic:
            {
                user::GattCharacteristic characteristic;

                characteristic.type         = nn::bluetooth::GattAttributeType_Characteristic;
                characteristic.handle       = pAttribute->handle;
                characteristic.instanceId   = pAttribute->id.instanceId;
                characteristic.uuid         = pAttribute->id.uuid;
                characteristic.property     = pAttribute->property;
                characteristic.permission   = 0x0000;

                if (!g_BleGattClientContainer.AddGattServerCharacteristic(pInfo->connId, characteristic))
                {
                    BTM_LOG("Failed to add characteristic to internal database. No more space.\n");
                }
            }
                break;
            case nn::bluetooth::GattAttributeType_Descriptor:
            {
                user::GattDescriptor descriptor;

                descriptor.type         = nn::bluetooth::GattAttributeType_Descriptor;
                descriptor.handle       = pAttribute->handle;
                descriptor.uuid         = pAttribute->id.uuid;
                descriptor.permission   = 0x0000;

                if(!g_BleGattClientContainer.AddGattServerDescriptor(pInfo->connId, descriptor))
                {
                    BTM_LOG("Failed to add descriptor to internal database. No more space.\n");
                }
            }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
        break;
    case nn::bluetooth::EventFromLeClientServiceDiscoveryStateChangedCallback:
    {
        BTM_LOG("%s: EventFromLeClientServiceDiscoveryStateChangedCallback.\n", NN_CURRENT_FUNCTION_NAME);
        const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback*>(pBufferBtp);

        NN_UNUSED(pInfo);

        g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleSdp);
    }
        break;
    case nn::bluetooth::EventFromLeClientConfigureMtuCallback:
    {
        const nn::bluetooth::InfoFromLeClientMtuConfigurationCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeClientMtuConfigurationCallback*>(pBufferBtp);

        g_BleGattClientContainer.UpdateMtu(pInfo->connId, pInfo->mtu);

        g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleMtuConfig);
    }
        break;
    case nn::bluetooth::EventFromLeServerProfileChangedCallback:
        BTM_LOG("%s: EventFromLeServerProfileChangedCallback. Currently not supported.\n", __func__);
        break;
    case nn::bluetooth::EventFromLeServerGattReqCallback:
        BTM_LOG("%s: EventFromLeServerGattReqCallback. Currently not supported.\n", __func__);
        break;
    default:
        BTM_LOG("Un-Handle BLE Core Event: 0x%X\n", eventTypeBtp);
        NN_ABORT();
    }

    if (requiredCoreEventType == eventTypeBtp)
    {
        return true;
    }

    return false;
} //NOLINT(impl/function_size)

//--------------------------------------------------
//StateMachineのMultiWaitでReportを検出した場合に使われる関数
//レポート内容をクライアントプロセスに伝える必要が無いため、内部で作業用バッファを確保し、ハンドラー処理後は解放する
//--------------------------------------------------
void HandleBtBleCoreReport()
{
    uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT];
    size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT;
    HandleBtBleCoreReport(nn::bluetooth::EventFromLeClientStateChangedCallback, bufferBtp, bufferSizeBtp);
}

//--------------------------------------------------
//下層からのBtHidレポートを処理
//HandleBtCoreReportと基本は同じ
//--------------------------------------------------
bool HandleBtHidReport(nn::bluetooth::HidEventType requiredHidEventType, void* pBufferBtp, size_t bufferSizeBtp)
{
    nn::bluetooth::HidEventType hidEventTypeBtp;
    GetBtHidReport(&hidEventTypeBtp, pBufferBtp, bufferSizeBtp);
    BTM_LOG("Handle BtHidReport Event: %d\n", hidEventTypeBtp);

    switch(hidEventTypeBtp)
    {
        case nn::bluetooth::EventFromConnectionStateCallback:
        {
            BTM_LOG("ConnectionState\n");
            const nn::bluetooth::InfoFromConnectionStateCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromConnectionStateCallback*>(pBufferBtp);
            if(pInfo->state == nn::bluetooth::BTHH_CONN_STATE_CONNECTED)
            {
                BTM_LOG("BTHH_CONN_STATE_CONNECTED ");
                PrintByteDataAsHex(&pInfo->bluetoothAddress.address[0], 6);
                bool isPairedInfoExist;
                DeviceInfo deviceInfo;
                isPairedInfoExist = g_DiContainer.SearchDeviceInfo(&deviceInfo, pInfo->bluetoothAddress);
                NN_SDK_REQUIRES(isPairedInfoExist);

                //[Todo]階層減らす
                if(g_HcContainer.IsConnectionRefusal())
                {
                    nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                }
                else
                {
                    g_DcContainerForWorking = g_DcContainer;
                    bool isAddedDeviceCondition;
                    isAddedDeviceCondition = g_DcContainerForWorking.Add(deviceInfo);

                    if(isAddedDeviceCondition)
                    {
                        //ユースケース内ならば、dcリストを更新する。 ユースケース外ならば、disconnectを発行する
                        bool isValidUsecase;
                        bool isRestructureAll;
                        uint8_t restructureBleSlotsCount;
                        SlotMode defaultSlotMode;
                        CeLength defaultCeLength;
                        // BLE のDeviceCondition は変化なしなので、現在のものを渡す
                        isValidUsecase = CheckUsecase(g_DcContainer.GetPtr(), g_DcContainerForWorking.GetPtr(),
                                                      g_BleGattClientContainer.GetPtr(), g_BleGattClientContainer.GetPtr(),
                                                      &defaultSlotMode, &defaultCeLength,
                                                      &isRestructureAll, &restructureBleSlotsCount);
                        if(isValidUsecase)
                        {
                            //dcリストの更新
                            g_DcContainer = g_DcContainerForWorking;

                            //スロット再整形をキューイング。シグナルはスロット再整形開始直前および完了時に行われる。
                            CommandParam_ChangeUsecaseByConnect command;
                            command.list.deviceCount = 1;
                            command.isRestructureAll = isRestructureAll;
                            command.list.device[0].bdAddress = pInfo->bluetoothAddress;
                            command.list.device[0].slotMode  = defaultSlotMode;
                            Message message;//[Todo]毎回確保しないようにする
                            *reinterpret_cast<CommandParam_ChangeUsecaseByConnect*>(&message.buffer[0]) = command;
                            message.apiId = ApiId_ChangeUsecaseByConnect;
                            EnQueue(&message);

                            //デバイスのペアリング情報の生存優先度を最大にする
                            //接続毎にSettingsを更新するのはNvmの劣化を招くため、ここではメモリ上の値のみを更新する
                            //btmは基本的にメモリ上の値を参照して動くため、Settingsの更新はシャットダウン時で問題ない
                            g_DiContainer.MoveToTail(pInfo->bluetoothAddress);

                            //MinorSlept中の接続は、起床要因としてシグナル
                            //[Todo]StateはWorkerスコープの情報のため、判定方法、もしくは箇所を変える
                            if(GetState() == BtmState_MinorSlept)
                            {
                                g_SystemEventContainer.Signal(SystemEventContainer::EventType_AwakeReq);
                            }
                        }
                        else
                        {
                            nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                        }
                    }
                    else
                    {
                        nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                        //バグ洗い出しのため、あえてABORTさせる
                        BTM_LOG("9th device connected.\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorOverConnection());
                    }
                }

                //接続系イベントは、現状Llrの終了条件の一つ。Llr終了直後に上位層がDcListを参照して結果判断する可能性があるため、DcListの更新よりも後段にシグナルさせる
                if(g_HcContainer.IsLlr(pInfo->bluetoothAddress))
                {
                    g_HcContainer.SetLlrDisabled();
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Llr);
                }
            }
            else if(pInfo->state == nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED)
            {
                BTM_LOG("BTHH_CONN_STATE_DISCONNECTED ");
                PrintByteDataAsHex(&pInfo->bluetoothAddress.address[0], 6);
                g_DcContainerForWorking = g_DcContainer;
                bool isDelete;
                isDelete = g_DcContainerForWorking.Remove(pInfo->bluetoothAddress);

                if(isDelete)
                {
                    //btmのユースケース判定による自動切断の場合は、対象エントリがDcListに登録されていないため、isDelete=falseとなり、このパスには来ない
                    bool isValidUsecase;
                    bool isRestructureAll;
                    uint8_t restructureBleSlotsCount;
                    SlotMode defaultSlotMode;
                    CeLength defaultCeLength;
                    // BLE のDeviceCondition は変化なしなので、現在のものを渡す
                    isValidUsecase = CheckUsecase(g_DcContainer.GetPtr(), g_DcContainerForWorking.GetPtr(),
                                                  g_BleGattClientContainer.GetPtr(), g_BleGattClientContainer.GetPtr(),
                                                  &defaultSlotMode, &defaultCeLength,
                                                  &isRestructureAll, &restructureBleSlotsCount);
                    if(isValidUsecase)
                    {
                        //dcリストの更新
                        g_DcContainer = g_DcContainerForWorking;
                        g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);

                        //スロット再整形をキューイング。スロット再整形完了時に再度シグナルされる
                        CommandParam_ChangeUsecaseByDisconnect command;
                        command.isRestructureAll = isRestructureAll;
                        Message message;//[Todo]毎回確保しないようにする
                        *reinterpret_cast<CommandParam_ChangeUsecaseByDisconnect*>(&message.buffer[0]) = command;
                        message.apiId = ApiId_ChangeUsecaseByDisconnect;
                        EnQueue(&message);
                    }
                    else
                    {
                        BTM_LOG("Invalid usecase by disconnect.\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorOverConnection());
                    }
                }

                //Disconnectは、Llr終了条件の一つの可能性があるが、確定はしていないため既存の動作を保つ
                //[Todo]確定次第削除か追加
                /*if(g_HcContainer.IsLlr(pInfo->bluetoothAddress))
                {
                    g_HcContainer.SetLlrDisabled();
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Llr);
                }*/
            }
            else if(pInfo->state == nn::bluetooth::BTHH_CONN_STATE_FAILED_GENERIC)
            {
                BTM_LOG("BTHH_CONN_STATE_FAILED_GENERIC\n");
                //汎用エラーは、Llr終了条件の一つ
                if(g_HcContainer.IsLlr(pInfo->bluetoothAddress))
                {
                    g_HcContainer.SetLlrDisabled();
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Llr);
                }
            }
            else
            {
                BTM_LOG("Un-handled connection event.%d\n",pInfo->state);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorUnHandledHidEvent());
            }
            break;
        }
        case nn::bluetooth::EventFromGetReportCallback:
        {
            BTM_LOG("GetReport\n");
            break;
        }
        case nn::bluetooth::EventFromExtensionCallbacks:
        {
            const nn::bluetooth::InfoFromExtensionCallbacks* pInfo = reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(pBufferBtp);
            BTM_LOG("Extension. eventType = enum(%d)\n", pInfo->eventType);
            if(pInfo->eventType == nn::bluetooth::EventFromSetZeroRetranCallback && pInfo->status == nn::bluetooth::BTHH_OK)
            {
                //[Todo]下層のイベントに情報を入れてもらい、きちんと反映させる
                ZeroRetransmissionList zRList;
                zRList.enabledReportIdCount = 0;
                g_DcContainer.Update(pInfo->bluetoothAddress, zRList);
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);
            }

            //[Todo]BurstModeのイベントを引っかけて、DeviceConditionの更新とシステムイベントのシグナルをするようにする。TSIの更新は不要
            break;
        }
        default:
        {
            BTM_LOG("Un-handled Hid System Event: 0x%X\n", hidEventTypeBtp);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorUnHandledHidEvent());
        }
    }

    if(requiredHidEventType == hidEventTypeBtp)
    {
        return true;
    }
    return false;
} //NOLINT(impl/function_size)

//--------------------------------------------------
//StateMachineのMultiWaitでReportを検出した場合に使われる関数
//レポート内容をクライアントプロセスに伝える必要が無いため、内部で作業用バッファを確保し、ハンドラー処理後は解放する
//--------------------------------------------------
void HandleBtHidReport()
{
    uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_HID_OUT;
    HandleBtHidReport(nn::bluetooth::EventFromConnectionStateCallback, bufferBtp, bufferSizeBtp);
}





//--------------------------------------------------
//スリープするために、BTを沈静化し、LLRモードを有効にする
//[Todo]StateMachineからの直接呼出しではなく、HandleApiとして受け付ける。
//[Todo]その際、Impl, Quasiスレッドとの排他に注意。MutexとBusyフラグによるアクセス制限ができないため、通信用バッファを上書きしうる
//--------------------------------------------------
void HandleSleep()
{
    //内部でMultiWaitする場合のバッファ。イベントのタイプ以外はCore/Hid/BLE共用
    uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    const size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_HID_OUT;

    NN_ABORT_UNLESS_RESULT_SUCCESS(DisableBleScan(&bufferBtp[0], bufferSizeBtp));
    CancelBleConnection(&bufferBtp[0], bufferSizeBtp);

    DisableAutoPairing(&bufferBtp[0], bufferSizeBtp);
    WaitForLlrFinish(&bufferBtp[0], bufferSizeBtp);
    DisablePageScan(&bufferBtp[0], bufferSizeBtp);
    DisconnectAllDevices(&bufferBtp[0], bufferSizeBtp);

    DisconnectAllBleDevices(&bufferBtp[0], bufferSizeBtp, false, nn::applet::AppletResourceUserId::GetInvalidId());

    auto result = nn::bluetooth::ExtStartLlrMode();//ブロッキングAPI
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //スリープ時は、ペアリング情報のNvmへの保存はしない。そのままシャットダウンするにしても、一度起床するはずなので。
}

//--------------------------------------------------
//スリープ復帰のために、LLRモードを無効にし、BTの沈静化を解除する
//[Todo]StateMachineからの直接呼出しではなく、HandleApiとして受け付ける
//--------------------------------------------------
void HandleAwake()
{
    //内部でMultiWaitする場合のバッファ。イベントのタイプ以外はCore/Hid/BLE共用
    uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    const size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_HID_OUT;

    auto result = nn::bluetooth::ExtExitLlrMode();//ブロッキングAPI
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    EnablePageScan();

    // 起床時にフィルタが存在していれば、再開する
    if (g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
    {
        BTM_LOG("Filter condition exists. Restart BLE scan.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(EnableBleScan(bufferBtp, bufferSizeBtp));
    }
}

//--------------------------------------------------
//安全にシャットダウンするために、BTチップのリセットピンを制御する
//[Todo]StateMachineからの直接呼出しではなく、HandleApiとして受け付ける
//--------------------------------------------------
void HandleShutdown()
{
    uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    const size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_HID_OUT;

    NN_ABORT_UNLESS_RESULT_SUCCESS(DisableBleScan(&bufferBtp[0], bufferSizeBtp));
    CancelBleConnection(&bufferBtp[0], bufferSizeBtp);

    DisableAutoPairing(&bufferBtp[0], bufferSizeBtp);
    WaitForLlrFinish(&bufferBtp[0], bufferSizeBtp);
    DisablePageScan(&bufferBtp[0], bufferSizeBtp);
    DisconnectAllDevices(&bufferBtp[0], bufferSizeBtp);

    DisconnectAllBleDevices(&bufferBtp[0], bufferSizeBtp, false, nn::applet::AppletResourceUserId::GetInvalidId());

    auto result = nn::bluetooth::DisableBluetooth();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    //[Todo]必要に応じてイベント待ち
    g_DiContainer.WriteToNvm();
}


//--------------------------------------------------
//BluetoothのAPI呼び出しと、目的のレポート取得までを逐次処理。
//目的外のレポートが来た場合も、MultiWaitAndHandleAsyncReport()内部でハンドラ処理が行われる
//[Todo]バッファサイズのチェック
//Bluetooth用バッファは内部で確保。Impl向けバッファは引数で取得
//--------------------------------------------------
void HandleApi(ApiId commandId, const void* pCommand, size_t commandSize,
        void* pReport, size_t reportSize, nn::Result* pResult)
{
    BTM_LOG("ApiId = %d\n",commandId);

    nn::Result result = ResultSuccess();
    *pResult = ResultSuccess();

    //内部でMultiWaitする場合のバッファ。イベントのタイプ以外はCore/Hid/Ble共用
    //[Todo]バッファの取り回し決める
    static uint8_t bufferBtp[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    static const size_t bufferSizeBtp = nn::bluetooth::BUFFER_SIZE_OF_HID_OUT;

    switch(commandId)
    {
        case ApiId_AcquireDiscoveryEvent:
        {
            BTM_LOG("ApiId_AcquireDiscoveryEvent\n");
            ReportParam_AcquireDiscoveryEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireDiscoveryEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_Disc);
            break;
        }
        case ApiId_AcquireDeviceConditionEvent:
        {
            BTM_LOG("ApiId_AcquireDeviceConditionEvent\n");
            ReportParam_AcquireDeviceConditionEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireDeviceConditionEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_Cdc);
            break;
        }
        case ApiId_StartDiscovery:
        {
            BTM_LOG("ApiId_StartDiscovery\n");
            //[Todo]連続で呼ばれた場合、以前の結果が消えることになるが良いか？
            g_DpContainer.Init();
            result = nn::bluetooth::StartDiscovery();
            *pResult = ConvertResult(result);
            break;
        }
        case ApiId_CancelDiscovery:
        {
            BTM_LOG("ApiId_CancelDiscovery\n");
            result = nn::bluetooth::CancelDiscovery();
            *pResult = ConvertResult(result);
            break;
        }
        case ApiId_CreateBond:
        {
            BTM_LOG("ApiId_CreateBond\n");
            const BdAddress* pAddress = &reinterpret_cast<const CommandParam_CreateBond*>(pCommand)->address;// NOLINT

            //ペアリング数が最大だった場合、1件先に消す（後消しは処理が複雑になるので、試行前に消してしまう）
            if(g_DiContainer.GetCount() == COUNT_OF_DI_LIST && !g_DiContainer.SearchDeviceInfo(nullptr, *pAddress))
            {
                //暫定対処：DisconnectEventを取得するため
                //[Todo]暫定対処を抜く
                CancelBleConnection(bufferBtp, bufferSizeBtp);
                DisablePageScan(&bufferBtp[0], bufferSizeBtp);
                DisconnectDevice(pAddress, &bufferBtp[0], bufferSizeBtp);

                RemoveOldDeviceInfo(pAddress);

                //暫定対処：DisconnectEventを取得するため
                //[Todo]暫定対処を抜く
                EnablePageScan();

                g_DiContainer.WriteToNvm();//[Todo]シャットダウン時に移行
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            }

            result = nn::bluetooth::CreateBond(pAddress,nn::bluetooth::NoPreference);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            //CreateBondを投げたら即返す。SspReqイベントはHandleCoreReport内部で即返答
            break;
        }
        case ApiId_CancelBond:
        {
            BTM_LOG("ApiId_CancelBond\n");
            const BdAddress* pAddress = &reinterpret_cast<const CommandParam_CancelBond*>(pCommand)->address;// NOLINT
            result = nn::bluetooth::CancelBond(pAddress);
            *pResult = ConvertResult(result);
            break;
        }
        case ApiId_ChangeTsiMode:
        {
            BTM_LOG("ApiId_ChangeTsiMode\n");
            const CommandParam_SetTsiMode* pChangeTsiMode = reinterpret_cast<const CommandParam_SetTsiMode*>(pCommand);
            char tsi = static_cast<char>(pChangeTsiMode->tsiMode);
            result = nn::bluetooth::ExtSetTsi(&pChangeTsiMode->address, tsi);
            *pResult = ConvertResult(result);
            break;
        }
        case ApiId_SetBurstMode:
        {
            BTM_LOG("ApiId_SetBurstMode\n");
            const CommandParam_SetBurstMode* pApiCommand = reinterpret_cast<const CommandParam_SetBurstMode*>(pCommand);
            result = nn::bluetooth::ExtSetBurstMode(&pApiCommand->address, pApiCommand->isBurstMode);
            *pResult = ConvertResult(result);
            FinishWorking();
            //[Todo]ユースケース遷移を必要に応じて入れる
            break;
        }
        case ApiId_SetSlotMode:
        {
            BTM_LOG("ApiId_SetSlotMode\n");
            const CommandParam_SetSlotMode* pApiCommand = reinterpret_cast<const CommandParam_SetSlotMode*>(pCommand);
            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();

            //変更要望となるローカルDeviceConditionListを作成
            for(int i=0;i<deviceConditionList.deviceCount;i++)
            {
                for(int j=0;j<pApiCommand->list.deviceCount;j++)
                {
                    if(IsSameBdAddress(&deviceConditionList.device[i].bdAddress,
                            &pApiCommand->list.device[j].bdAddress))
                    {
                        BTM_LOG("slotMode = enum(%d)\n", pApiCommand->list.device[j].slotMode);
                        deviceConditionList.device[i].hidDeviceCondition.slotMode = pApiCommand->list.device[j].slotMode;
                        if(pApiCommand->list.device[j].slotMode == SlotMode_4)
                        {
                            deviceConditionList.device[i].hidDeviceCondition.isLargeSlotRequired = true;//現状4スロット設定のみ担保する。2スロット,Activeは各種要因で変化しうる
                        }
                        else
                        {
                            deviceConditionList.device[i].hidDeviceCondition.isLargeSlotRequired = false;
                        }
                    }
                }
            }

            bool isSuccess;
            isSuccess = UpdateUsecase(&deviceConditionList, &gattClientConditionList, pResult, &bufferBtp[0], bufferSizeBtp);

            //ローカル通信中に2Slot設定が失敗する件のワークアラウンド。
            //IsSlotModeRequiredフラグを落としたうえで、4スロットは維持する。
            //[Todo]全体仕様で対応する。その場合、Hidが指定するのはスロットモードではなく、IR/NFC Enableという形に抽象化する。
            if(!isSuccess)
            {
                deviceConditionList = g_DcContainer.Get();
                for(int i=0;i<deviceConditionList.deviceCount;i++)
                {
                    for(int j=0;j<pApiCommand->list.deviceCount;j++)
                    {
                        if(IsSameBdAddress(&deviceConditionList.device[i].bdAddress,
                                &pApiCommand->list.device[j].bdAddress))
                        {
                            if(pApiCommand->list.device[j].slotMode != SlotMode_4)
                                deviceConditionList.device[i].hidDeviceCondition.isLargeSlotRequired = false;//要求フラグのみを落とす
                        }
                    }
                }
                g_DcContainer.Update(deviceConditionList);
                //g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);//[Todo]hid層のスロット制御が最適化されたら有効にする
            }
            break;
        }
        case ApiId_SetBluetoothMode:
        {
            BTM_LOG("ApiId_SetBluetoothMode\n");
            const CommandParam_SetBluetoothMode* pApiCommand = reinterpret_cast<const CommandParam_SetBluetoothMode*>(pCommand);
            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();
            deviceConditionList.bluetoothMode = pApiCommand->bluetoothMode;

            UpdateUsecase(&deviceConditionList, &gattClientConditionList, pResult, &bufferBtp[0], bufferSizeBtp);
            break;
        }
        case ApiId_SetWlanMode:
        {
            BTM_LOG("ApiId_SetWlanMode\n");
            const CommandParam_SetWlanMode* pApiCommand = reinterpret_cast<const CommandParam_SetWlanMode*>(pCommand);

            //[Todo]UpdateUsecase成功後にDisableする
            if(pApiCommand->wlanMode != WlanMode_None)
            {
               DisableAutoPairing(&bufferBtp[0], bufferSizeBtp);

               if (g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
               {
                   NN_ABORT_UNLESS_RESULT_SUCCESS(EnableBleScan(bufferBtp, bufferSizeBtp));
               }
            }

            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();
            deviceConditionList.wlanMode = pApiCommand->wlanMode;

            UpdateUsecase(&deviceConditionList, &gattClientConditionList, pResult, &bufferBtp[0], bufferSizeBtp);
            break;
        }
        case ApiId_AcquireDeviceInfoEvent:
        {
            BTM_LOG("ApiId_AcquireDeviceInfoEvent\n");
            ReportParam_AcquireDeviceInfoEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireDeviceInfoEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_Rdi);
            break;
        }
        case ApiId_AddDeviceInfo:
        {
            BTM_LOG("ApiId_AddDeviceInfo\n");
            const CommandParam_AddDeviceInfo* pApiCommand = reinterpret_cast<const CommandParam_AddDeviceInfo*>(pCommand);

            //ペアリング数が最大だった場合、1件先に消す（後消しは処理が複雑になるので、試行前に消してしまう）
            if(g_DiContainer.GetCount() == COUNT_OF_DI_LIST && !g_DiContainer.SearchDeviceInfo(nullptr, pApiCommand->info.bdAddress))
            {
                //暫定対処：DisconnectEventを取得するため
                //[Todo]暫定対処を抜く
                if(g_HcContainer.IsRadio())
                {
                    CancelBleConnection(bufferBtp, bufferSizeBtp);
                    DisablePageScan(&bufferBtp[0], bufferSizeBtp);
                    DisconnectDevice(&pApiCommand->info.bdAddress, &bufferBtp[0], bufferSizeBtp);
                }

                RemoveOldDeviceInfo(&pApiCommand->info.bdAddress);

                //暫定対処：DisconnectEventを取得するため
                //[Todo]暫定対処を抜く
                if(g_HcContainer.IsRadio())
                {
                    EnablePageScan();
                }
            }

            nn::settings::system::BluetoothDevicesSettings additionalSet;
            DiDsUtility::Convert(&additionalSet, pApiCommand->info);
            result = nn::bluetooth::HidAddPairedDevice(&additionalSet);
            if(result.IsSuccess())
            {
                g_DiContainer.Add(additionalSet);
                g_DiContainer.WriteToNvm();//[Todo]シャットダウン時に書きこむ
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            }
            else if(nn::bluetooth::ResultNoResources::Includes(result))
            {
                //NoResources時は何もしない
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            break;
        }
        case ApiId_RemoveDeviceInfo:
        {
            BTM_LOG("ApiId_RemoveDeviceInfo\n");
            BdAddress bdAddress = reinterpret_cast<const CommandParam_RemoveDeviceInfo*>(pCommand)->address;

            //暫定対処：DisconnectEventを取得するため
            //[Todo]暫定対処を抜く
            if(g_HcContainer.IsRadio())
            {
                CancelBleConnection(bufferBtp, bufferSizeBtp);
                DisablePageScan(&bufferBtp[0], bufferSizeBtp);
                DisconnectDevice(&bdAddress, &bufferBtp[0], bufferSizeBtp);
            }

            result = nn::bluetooth::RemoveBond(&bdAddress);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            g_DiContainer.Remove(bdAddress);
            g_DiContainer.WriteToNvm();//[Todo]シャットダウン時に書きこむ

            //暫定対処：DisconnectEventを取得するため
            //[Todo]暫定対処を抜く
            if(g_HcContainer.IsRadio())
            {
                EnablePageScan();
            }

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            break;
        }
        case ApiId_ProtectDeviceInfo:
        {
            BTM_LOG("ApiId_ProtectDeviceInfo\n");
            const CommandParam_ProtectDeviceInfo* pApiCommand = reinterpret_cast<const CommandParam_ProtectDeviceInfo*>(pCommand);
            if(pApiCommand->isProtect)
            {
                g_HcContainer.AddPseudoConnection(pApiCommand->address);
            }
            else
            {
                g_HcContainer.RemovePseudoConnection(pApiCommand->address);
            }
            break;
        }
        case ApiId_IncreaseDeviceInfoOrder:
        {
            BTM_LOG("ApiId_IncreaseDeviceInfoOrder\n");
            BdAddress bdAddress = reinterpret_cast<const CommandParam_IncreaseDeviceInfoOrder*>(pCommand)->address;
            g_DiContainer.MoveToTail(bdAddress);
            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            break;
        }
        case ApiId_AcquireLlrStateEvent:
        {
            BTM_LOG("ApiId_AcquireLlrStateEvent\n");
            ReportParam_AcquireLlrStateEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireLlrStateEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_Llr);
            break;
        }
        case ApiId_LlrNotify:
        {
            BTM_LOG("ApiId_LlrNotify\n");
            if(g_HcContainer.IsLlr())
            {
                //[Todo]Result適切なものに変える
                *pResult = ResultFailureLowLayerGeneral();
            }
            else
            {
                BdAddress bdAddress = reinterpret_cast<const CommandParam_LlrNotify*>(pCommand)->address;
                result = nn::bluetooth::HidWakeController(&bdAddress);
                *pResult = ConvertResult(result);
                if(pResult->IsSuccess())
                {
                    PrintByteDataAsHex(&bdAddress.address[0], 6);
                    g_HcContainer.SetLlrEnabled(bdAddress);
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Llr);
                }
            }
            break;
        }
        case ApiId_DisableRadio:
        {
            BTM_LOG("ApiId_DisableRadio\n");

            NN_ABORT_UNLESS_RESULT_SUCCESS(DisableBleScan(&bufferBtp[0], bufferSizeBtp));
            CancelBleConnection(bufferBtp, bufferSizeBtp);

            DisableAutoPairing(&bufferBtp[0], bufferSizeBtp);
            WaitForLlrFinish(&bufferBtp[0], bufferSizeBtp);
            DisablePageScan(&bufferBtp[0], bufferSizeBtp);
            DisconnectAllDevices(&bufferBtp[0], bufferSizeBtp);

            DisconnectAllBleDevices(&bufferBtp[0], bufferSizeBtp, false, nn::applet::AppletResourceUserId::GetInvalidId());

            result = nn::bluetooth::ExtSetRadio(false);//同期API
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            g_HcContainer.SetRadio(false);

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Radio);
            break;
        }
        case ApiId_DisableRadio_InMinorSlept:
        {
            BTM_LOG("ApiId_DisableRadio_InMinorSlept\n");
            result = nn::bluetooth::ExtExitLlrMode();//同期API
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            result = nn::bluetooth::ExtSetRadio(false);//同期API
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            g_HcContainer.SetRadio(false);

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Radio);
            break;
        }
        case ApiId_EnableRadio:
        {
            BTM_LOG("ApiId_EnableRadio\n");
            result = nn::bluetooth::ExtSetRadio(true);//同期API
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            EnablePageScan();

            if (g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(EnableBleScan(bufferBtp, bufferSizeBtp));
            }

            g_HcContainer.SetRadio(true);
            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Radio);
            break;
        }
        case ApiId_EnableRadio_InMinorSlept:
        {
            BTM_LOG("ApiId_EnableRadio_InMinorSlept\n");
            result = nn::bluetooth::ExtSetRadio(true);//同期API
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            result = nn::bluetooth::ExtStartLlrMode();//同期API
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            g_HcContainer.SetRadio(true);
            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Radio);
            break;
        }
        case ApiId_AcquireAwakeReqEvent:
        {
            BTM_LOG("ApiId_AcquireAwakeReqEvent\n");
            ReportParam_AcquireAwakeReqEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireAwakeReqEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_AwakeReq);
            break;
        }
        case ApiId_EnableSlotSaving:
        {
            BTM_LOG("ApiId_EnableSlotSaving\n");
            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();
            deviceConditionList.isSlotSaving = true;
            UpdateUsecase(&deviceConditionList, &gattClientConditionList, &bufferBtp[0], bufferSizeBtp);
            break;
        }
        case ApiId_DisableSlotSaving:
        {
            BTM_LOG("ApiId_DisableSlotSaving\n");
            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();
            deviceConditionList.isSlotSaving = false;
            UpdateUsecase(&deviceConditionList, &gattClientConditionList, &bufferBtp[0], bufferSizeBtp);
            break;
        }
        case ApiId_HidConnect:
        {
            BTM_LOG("ApiId_HidConnect\n");
            const BdAddress* pAddress = &reinterpret_cast<const CommandParam_HidConnect*>(pCommand)->address;// NOLINT
            result = nn::bluetooth::HidConnect(pAddress);
            *pResult = ConvertResult(result);
            break;
        }
        case ApiId_HidDisconnect:
        {
            BTM_LOG("ApiId_HidDisconnect\n");
            const BdAddress* pAddress = &reinterpret_cast<const CommandParam_HidDisconnect*>(pCommand)->address;// NOLINT
            result = nn::bluetooth::HidDisconnect(pAddress);
            //[Todo]要Result分岐
            //NN_ABORT_UNLESS_RESULT_SUCCESS(result);//非接続に発行した際にAbortし得るので、Resultは見ない
            break;
        }
        case ApiId_HidSetRetransmissionMode:
        {
            BTM_LOG("ApiId_SetRetransmissionMode\n");
            const CommandParam_HidSetRetransmissionMode* pApiCommand = reinterpret_cast<const CommandParam_HidSetRetransmissionMode*>(pCommand);
            result = nn::bluetooth::ExtSetZeroRetran(&pApiCommand->address, pApiCommand->zeroRetransmissionList.enabledReportIdCount,
                    const_cast<uint8_t*>(&pApiCommand->zeroRetransmissionList.enabledReportId[0]));
            //*pResult = ConvertResult(result);//非接続デバイスへの変更時はエラーが返ってくるため、上位層には結果を通知しない
            break;
        }
        case ApiId_StartGamepadPairing:
        {
            BTM_LOG("ApiId_StartGamepadPairing\n");

            NN_ABORT_UNLESS_RESULT_SUCCESS(DisableBleScan(bufferBtp, bufferSizeBtp));
            EnableAutoPairing(&bufferBtp[0], bufferSizeBtp);
            break;
        }
        case ApiId_CancelGamepadPairing:
        {
            BTM_LOG("ApiId_CancelGamepadPairing\n");
            DisableAutoPairing(&bufferBtp[0], bufferSizeBtp);

            if (g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
            {
                BTM_LOG("BLE scan filter condition exists. Resume BLE scan.\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(EnableBleScan(bufferBtp, bufferSizeBtp));
            }

            break;
        }
        case ApiId_ClearGamepadPairingDatabase:
        {
            //[Todo]デバイス名でのフィルタ機能を入れる
            BTM_LOG("ApiId_ClearGamepadPairingDatabase\n");

            // BLE ペアリングが進行中であれば、再キューイング
            if (g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_Init)
            {
                Message message;
                message.apiId = ApiId_ClearGamepadPairingDatabase;
                EnQueue(&message);
            }

            //暫定対処：DisconnectEventを取得するため
            //[Todo]暫定対処を抜く
            if(g_HcContainer.IsRadio())
            {
                CancelBleConnection(bufferBtp, bufferSizeBtp);

                DisablePageScan(&bufferBtp[0], bufferSizeBtp);
                DisconnectAllDevices(&bufferBtp[0], bufferSizeBtp);

                g_BleGattClientContainer.SetConnectionRefusal(true);
                // システムがオーナーである接続のみを同期的に全切断
                DisconnectAllBleDevices(&bufferBtp, bufferSizeBtp, true, nn::applet::AppletResourceUserId::GetInvalidId());
            }

            const DeviceInfoList deviceInfoList = g_DiContainer.Get();

            for(int i = 0; i < deviceInfoList.deviceCount; i++)
            {
                result = nn::bluetooth::RemoveBond(&deviceInfoList.device[i].bdAddress);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            g_DiContainer.Remove();
            g_DiContainer.WriteToNvm();//[Todo]シャットダウン時に書く

            user::BleAdvFilterForGeneral            filterCondition;
            BlePairingInfoContainer::BlePairingInfo pairingInfo;
            BdAddress                               pairedAddress[nn::bluetooth::BlePairingCountMax];
            uint8_t                                 pairedDeviceNum = 0;

            // Palma
            GetScanFilterParameter(&filterCondition, nn::bluetooth::BleScanParameterId_Palma);
            pairedDeviceNum = GetBlePairedDeviceAddress(pairedAddress, NN_ARRAY_SIZE(pairedAddress), filterCondition);

            for (int i = 0; i < pairedDeviceNum; ++i)
            {
                pairingInfo.address = pairedAddress[i];
                pairingInfo.advInfo = filterCondition;
                pairingInfo.type    = BlePairingAdvDataType_ManuCliSerId;

                g_BlePairingInfoContainer.RemovePairingInfo(pairingInfo);
            }

            g_BlePairingInfoContainer.Save();

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
            g_BlePairingInfoContainer.InitializeBlePairingState();

            //暫定対処：DisconnectEventを取得するため
            //[Todo]暫定対処を抜く
            if(g_HcContainer.IsRadio())
            {
                EnablePageScan();
                g_BleGattClientContainer.SetConnectionRefusal(false);
            }

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            break;
        }
        case ApiId_AcquireRadioEvent:
        {
            BTM_LOG("ApiId_AcquireRadioEvent\n");
            ReportParam_AcquireRadioEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireRadioEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_Radio, &p_ReportParam->id);
            break;
        }
        case ApiId_AcquireGamepadPairingEvent:
        {
            BTM_LOG("ApiId_AcquireGamepadPairingEvent\n");
            ReportParam_AcquireGamepadPairingEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireGamepadPairingEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_Gpp, &p_ReportParam->id);
            break;
        }
        case ApiId_ChangeUsecaseByConnect:
        {
            //システムイベントのシグナルの仕方がApiId_SetSlotModeとは異なるため、Api自体を分けている。
            //ユースケース再評価にSlotModeの入力が必要な理由は、RestructureEachBtSlotsdeで、新たに接続されたデバイスのBdAddressとデフォルトスロット値が必要となるため。
            BTM_LOG("ApiId_ChangeUsecaseByConnect\n");
            g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);//接続切断を上位層にすばやく伝えるためにBusyとなった段階で1度シグナル。ユースケース遷移は時間がかかるため。

            const CommandParam_ChangeUsecaseByConnect* pApiCommand = reinterpret_cast<const CommandParam_ChangeUsecaseByConnect*>(pCommand);
            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();

            //変更要望となるローカルDeviceConditionListを作成
            //[Todo]スロットモードの指定は、下のユースケース判定時のデフォルトスロットを設定したほうが、正しそう。この場合、SlotModeのパラメータは不要でBdAddressだけがあればよい
            for(int i=0;i<deviceConditionList.deviceCount;i++)
            {
                for(int j=0;j<pApiCommand->list.deviceCount;j++)
                {
                    if(IsSameBdAddress(&deviceConditionList.device[i].bdAddress,
                            &pApiCommand->list.device[j].bdAddress))
                    {
                        deviceConditionList.device[i].hidDeviceCondition.slotMode = pApiCommand->list.device[j].slotMode;
                    }
                }
            }

            UpdateUsecase(&deviceConditionList, &gattClientConditionList, &bufferBtp[0], bufferSizeBtp, pApiCommand->isRestructureAll);
            break;
        }
        case ApiId_ChangeUsecaseByBleConnect:
        {
            BTM_LOG("ApiId_ChangeUsecaseByBleConnect\n");

            const CommandParam_ChangeUsecaseByBleConnect* pApiCommand = reinterpret_cast<const CommandParam_ChangeUsecaseByBleConnect*>(pCommand);

            DeviceConditionList dcList      = *g_DcContainer.GetPtr();
            GattClientConditionList gccList = *g_BleGattClientContainer.GetPtr();

            dcList.isLargeSlotRequiredForBle = true;
            for (int i = 0; i < dcList.deviceCount; ++i)
            {
                if (!dcList.device[i].hidDeviceCondition.isLargeSlotRequired)
                {
                    dcList.device[i].hidDeviceCondition.slotMode = pApiCommand->defaultSlotMode;
                }
            }
            for (auto& client : gccList.gattClients)
            {
                if (client.connectedServer.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
                {
                    client.maxCeLength = pApiCommand->defaultCeLength;
                }
            }

            UpdateUsecase(&dcList, &gccList, &bufferBtp[0], bufferSizeBtp, true);   // BLE の接続、切断は常にRestructureSlotAll

            break;
        }
        case ApiId_ChangeUsecaseByDisconnect:
        {
            BTM_LOG("ApiId_ChangeUsecaseByDisconnect\n");
            const CommandParam_ChangeUsecaseByDisconnect* pApiCommand = reinterpret_cast<const CommandParam_ChangeUsecaseByDisconnect*>(pCommand);
            DeviceConditionList deviceConditionList = g_DcContainer.Get();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();

            UpdateUsecase(&deviceConditionList, &gattClientConditionList, &bufferBtp[0], bufferSizeBtp, pApiCommand->isRestructureAll);
            break;
        }
        case ApiId_ChangeUsecaseByBleDisconnect:
        {
            BTM_LOG("ApiId_ChageUsecaseByBleDisconnect\n");

            const CommandParam_ChangeUsecaseByBleDisconnect* pApiCommand = reinterpret_cast<const CommandParam_ChangeUsecaseByBleDisconnect*>(pCommand);

            DeviceConditionList deviceConditionList         = *g_DcContainer.GetPtr();
            GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();

            if (gattClientConditionList.deviceCount == 0)
            {
                deviceConditionList.isLargeSlotRequiredForBle = false;
            }
            for (auto& client : gattClientConditionList.gattClients)
            {
                if (client.connectedServer.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
                {
                    client.maxCeLength = pApiCommand->defaultCeLength;
                }
            }

            UpdateUsecase(&deviceConditionList, &gattClientConditionList, &bufferBtp[0], bufferSizeBtp, true);   // BLE の接続、切断は常にRestructureSlotAll

            break;
        }
        case ApiId_StoreDeviceInfo:
        {
            BTM_LOG("ApiId_StoreDeviceInfo\n");
            g_DiContainer.WriteToNvm();
            break;
        }
        case ApiId_BlockingCreateBond:
        {
            BTM_LOG("ApiId_BlockingCreateBond\n");
            const BdAddress* pAddress = &reinterpret_cast<const CommandParam_BlockingCreateBond*>(pCommand)->address;// NOLINT

            //ペアリング数が最大だった場合、1件先に消す（後消しは処理が複雑になるので、試行前に消してしまう）
            if(g_DiContainer.GetCount() == COUNT_OF_DI_LIST && !g_DiContainer.SearchDeviceInfo(nullptr, *pAddress))
            {
                //暫定対処：DisconnectEventを取得するため
                //[Todo]暫定対処を抜く
                CancelBleConnection(bufferBtp, bufferSizeBtp);
                DisablePageScan(&bufferBtp[0], bufferSizeBtp);
                DisconnectDevice(pAddress, &bufferBtp[0], bufferSizeBtp);

                RemoveOldDeviceInfo(pAddress);

                //暫定対処：DisconnectEventを取得するため
                //[Todo]暫定対処を抜く
                EnablePageScan();

                g_DiContainer.WriteToNvm();//[Todo]シャットダウン時に移行
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
            }

            //同期型としてBonding処理を行う
            //各イベント取得に応じた操作はイベントハンドラに任せ、ここでは処理終了のイベントが来ることのみを確認する
            result = nn::bluetooth::CreateBond(pAddress, nn::bluetooth::NoPreference);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            if(result.IsSuccess())
            {
                for(;;)
                {
                    MultiWaitId multiWaitId;
                    multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromBondStateChangedCallback, &bufferBtp[0], bufferSizeBtp, 10);
                    if(multiWaitId == MultiWaitId_Time)
                    {
                        //NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeoutBlockingCreateBond());
                        NN_DETAIL_BTM_ERROR("Workaround in ApiId_BlockingCreateBond.\n");
                        break;
                    }
                    else
                    {
                        const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromBondStateChangedCallback*>(&bufferBtp[0]);
                        if(IsSameBdAddress(&pInfo->bluetoothAddress, pAddress))
                        {
                            if(pInfo->state == nn::bluetooth::BT_BOND_STATE_NONE || pInfo->state == nn::bluetooth::BT_BOND_STATE_BONDED)
                            {
                                break;
                            }
                        }
                    }
                }
            }

            break;
        }
        case ApiId_ReStartDiscoveryForAutoPair:
        {
            BTM_LOG("ApiId_ReStartDiscoveryForAutoPair\n");
            if(g_HcContainer.IsAutoPairing())
            {
                g_DpContainer.Init();
                nn::bluetooth::StartDiscovery();
            }
            break;
        }
        case ApiId_DiscardRadioEvent:
        {
            BTM_LOG("ApiId_DiscardRadioEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardRadioEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_Radio, id);
            break;
        }
        case ApiId_DiscardGamepadPairingEvent:
        {
            BTM_LOG("ApiId_DiscardGamepadPairingEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardGamepadPairingEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_Gpp, id);
            break;
        }
        case ApiId_Recovery:
        {
            BTM_LOG("ApiId_Recovery\n");
            Recovery();
            break;
        }
        case ApiId_GeneralTest:
        {
            BTM_LOG("ApiId_Test\n");
            const CommandParam_GeneralTest* pApiCommand = reinterpret_cast<const CommandParam_GeneralTest*>(pCommand);
            if(pApiCommand->mode == 0)
            {
                HandleShutdown();
            }
            else if(pApiCommand->mode == 1)
            {
                BTM_LOG("ApiId_Test ExtSetVisibility(false, false)\n");
                result = nn::bluetooth::ExtSetVisibility(false, false);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            else if(pApiCommand->mode == 2)
            {
                BTM_LOG("ApiId_Test ExtStartLlrMode\n");
                result = nn::bluetooth::ExtStartLlrMode();
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            else if(pApiCommand->mode == 3)
            {
                BTM_LOG("ApiId_Test ExtExitLlrMode\n");
                result = nn::bluetooth::ExtExitLlrMode();
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            else if(pApiCommand->mode == 4)
            {
                BTM_LOG("ApiId_Test ExtSetVisibility(false, true)\n");
                result = nn::bluetooth::ExtSetVisibility(false, true);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            else if(pApiCommand->mode == 7)
            {
                BTM_LOG("ApiId_Test HandleSleep()\n");
                HandleSleep();
            }
            else if(pApiCommand->mode == 8)
            {
                BTM_LOG("ApiId_Test HandleAwake()\n");
                HandleAwake();
            }
            else if(pApiCommand->mode == 9)
            {
                BTM_LOG("ApiId_Test Recovery\n");
                Message message;//[Todo]毎回確保しないようにする
                message.apiId = ApiId_Recovery;
                EnQueue(&message);
            }
            else if(pApiCommand->mode == 10)
            {
                BTM_LOG("ApiId_Test Usecase\n");
                DeviceConditionList dcList;
                GattClientConditionList gccList;
                g_DcContainerForWorking.Init();
                dcList = g_DcContainerForWorking.Get();

                for (int wlanMode = 0; wlanMode < 4; wlanMode++)
                {
                    for (int bluetoothMode = 0; bluetoothMode < 2; bluetoothMode++)
                    {
                        for (int slotSavingForPairing = 0; slotSavingForPairing < 2; slotSavingForPairing++)
                        {
                            for (int slotSaving = 0; slotSaving < 2; slotSaving++)
                            {
                                for (int largeSlotRequired = 0; largeSlotRequired < 2; largeSlotRequired++)
                                {
                                    for (int largeSlotRequiredForBle = 0; largeSlotRequiredForBle < 2; ++largeSlotRequiredForBle)
                                    {
                                        for (int bleDeviceCount = 0; bleDeviceCount < nn::bluetooth::BleConnectionCountMaxClient; ++bleDeviceCount)
                                        {
                                            dcList.wlanMode                                         = static_cast<WlanMode>(wlanMode);
                                            dcList.bluetoothMode                                    = static_cast<BluetoothMode>(bluetoothMode);
                                            dcList.isSlotSavingForPairing                           = (slotSavingForPairing == 1);
                                            dcList.isSlotSaving                                     = (slotSaving == 1);
                                            dcList.deviceCount                                      = 1;
                                            dcList.device[0].hidDeviceCondition.isLargeSlotRequired = (largeSlotRequired == 1);
                                            dcList.device[0].hidDeviceCondition.slotMode            = static_cast<SlotMode>(largeSlotRequired);
                                            dcList.isLargeSlotRequiredForBle                        = (largeSlotRequiredForBle == 1);

                                            gccList.deviceCount                                     = bleDeviceCount;

                                            DebugPrintUsecase(&dcList, &gccList);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(pApiCommand->mode == 11)
            {
                BTM_LOG("ApiId_Test reset PseudoConnection\n");
                HostConditionContainer::PseudoConnectionList list = g_HcContainer.GetPseudoConnectionList();
                for(int i=0;i<list.count;i++)
                {
                    g_HcContainer.RemovePseudoConnection(list.bdAddress[i]);
                }
            }
            break;
        }
        case ApiId_AcquireBleScanEvent:
        {
            BTM_LOG("ApiId_AcquireBleScanEvent\n");
            ReportParam_AcquireBleScanEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireBleScanEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_BleScan, &p_ReportParam->id);
            break;
        }
        case ApiId_DiscardBleScanEvent:
        {
            BTM_LOG("ApiId_DiscardBleScanEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardBleScanEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_BleScan, id);
            break;
        }
        case ApiId_BleClearScanResult:
        {
            BTM_LOG("ApiId_BleClearScanResult\n");

            g_BleScannerCondition.ClearScanResultsGeneral();
            g_BleScannerCondition.ClearScanResultsSd();

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleScan);

            break;
        }
        case ApiId_BleStartScanGeneral:
        {
            BTM_LOG("ApiId_BleStartScanGeneral\n");

            if (!g_BleScannerCondition.IsScanning() && !g_HcContainer.IsAutoPairing())
            {
                BTM_LOG("First BLE scan filter condition. Start BLE scan.\n");

                result = EnableBleScan(bufferBtp, bufferSizeBtp);
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
            }

            const CommandParam_StartBleScanForGeneral* pApiCommand = reinterpret_cast<const CommandParam_StartBleScanForGeneral*>(pCommand);
            uint8_t version = 0x01;

            BleScannerCondition::ScanFilterCondition condition;
            memset(&condition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            condition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_GENERAL;
            condition.condition.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;

            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET), pApiCommand->condition.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_VERSION_OFFSET), &version, sizeof(uint8_t));
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET), pApiCommand->condition.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET), pApiCommand->condition.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            condition.condition.structure.length = BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET + 1;    // for length field

            for (int i = 0; i < BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET; ++i)
            {
                condition.condition.mask[i] = 0xFF;
            }
            condition.condition.maskeLength = BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET;
            condition.aruid = pApiCommand->aruid;

            *pResult = BleAddScanFilterCondition(bufferBtp, bufferSizeBtp, condition);

            if (pResult->IsSuccess())
            {
                g_BleScannerCondition.ClearScanResultsGeneral();

                // Scan Parameter を設定。最低速なので、強制上書き。
                result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_LowDuty.interval, BleScannerCondition::BleScanParameter_LowDuty.window);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }

            break;
        }
        case ApiId_BleStopScanGeneral:
        {
            BTM_LOG("ApiId_BleStopScanGeneral\n");

            const CommandParam_StopBleScanForGeneral* pApiCommand = reinterpret_cast<const CommandParam_StopBleScanForGeneral*>(pCommand);
            BleScannerCondition::ScanFilterCondition conditions[BleScannerCondition::BLE_SCAN_CONDITION_MAX_FOR_GENERAL];
            int conditionCount = g_BleScannerCondition.GetScanFilterCondition(conditions, NN_ARRAY_SIZE(conditions), BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_GENERAL);

            for (int i = 0; i < conditionCount; ++i)
            {
                if (conditions[i].inUse && conditions[i].aruid == pApiCommand->aruid)
                {
                    *pResult = BleDeleteScanFilterCondition(bufferBtp, bufferSizeBtp, conditions[i]);

                    if(pResult->IsFailure())
                    {
                        NN_SDK_LOG("[btm] Failed to delete scan filter condition for general. result = %d-%d\n", result.GetModule(), result.GetDescription());
                        break;
                    }
                }
            }

            if (pResult->IsSuccess())
            {
                if (g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_GENERAL)       == 0 &&
                    g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED)        == 0 &&
                    g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL)    != 0)
                {
                    result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_HighDuty.interval, BleScannerCondition::BleScanParameter_HighDuty.window);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }

                if (!g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
                {
                    BTM_LOG("Final BLE scan filter condition. Stop BLE scan.\n");

                    result = nn::bluetooth::ClearLeScanFilters();
                    if (result.IsFailure())
                    {
                        *pResult = ResultFailureLowLayerGeneral();
                        break;
                    }
                    // Result は見ない
                    DisableBleScan(bufferBtp, bufferSizeBtp);
                }
            }

            break;
        }
        case ApiId_BleStartScanPaired:
        {
            BTM_LOG("ApiId_BleStartScanPaired\n");

            if (!g_BleScannerCondition.IsScanning() && !g_HcContainer.IsAutoPairing())
            {
                BTM_LOG("First BLE scan filter condition. Start BLE scan.\n");

                result = EnableBleScan(bufferBtp, bufferSizeBtp);
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
            }

            const CommandParam_StartBleScanForPaired* pApiCommand = reinterpret_cast<const CommandParam_StartBleScanForPaired*>(pCommand);
            uint8_t version = 0x01;
            uint8_t wakeOnBle = 0x00;
            BdAddress address = g_HcContainer.GetProperty().bdAddress;

            BleScannerCondition::ScanFilterCondition condition;
            memset(&condition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            condition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED;
            condition.condition.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;

            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET), pApiCommand->condition.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_VERSION_OFFSET), &version, sizeof(uint8_t));
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET), pApiCommand->condition.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET), pApiCommand->condition.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET), &wakeOnBle, sizeof(uint8_t));
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_PAIRED_ADDR_OFFSET), address.address, SIZE_OF_BDADDRESS);
            condition.condition.structure.length = BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET + 1;    // for length field

            for (int i = 0; i < BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET; ++i)
            {
                condition.condition.mask[i] = 0xFF;
            }
            // Wake On BLE に対応しているかは無視する
            condition.condition.mask[BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET] = 0x00;
            condition.condition.maskeLength = BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET;
            condition.aruid = pApiCommand->aruid;

            *pResult = BleAddScanFilterCondition(bufferBtp, bufferSizeBtp, condition);

            if (pResult->IsSuccess())
            {
                // Scan Parameter を設定。最低速なので、強制上書き。
                result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_LowDuty.interval, BleScannerCondition::BleScanParameter_LowDuty.window);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }

            break;
        }
        case ApiId_BleStopScanPaired:
        {
            BTM_LOG("ApiId_BleStopScanPaired\n");

            const CommandParam_StopBleScanForPaired* pApiCommand = reinterpret_cast<const CommandParam_StopBleScanForPaired*>(pCommand);
            BleScannerCondition::ScanFilterCondition conditions[BleScannerCondition::BLE_SCAN_CONDITION_MAX_FOR_PAIRED];
            int conditionCount = g_BleScannerCondition.GetScanFilterCondition(conditions, NN_ARRAY_SIZE(conditions), BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED);

            for (int i = 0; i < conditionCount; ++i)
            {
                if (conditions[i].inUse && conditions[i].aruid == pApiCommand->aruid)
                {
                    *pResult = BleDeleteScanFilterCondition(bufferBtp, bufferSizeBtp, conditions[i]);

                    if(pResult->IsFailure())
                    {
                        NN_SDK_LOG("[btm] Failed to delete scan filter condition for paired. result = %d-%d\n", result.GetModule(), result.GetDescription());
                        break;
                    }
                }
            }

            if (pResult->IsSuccess())
            {
                if (g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_GENERAL)       == 0 &&
                    g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED)        == 0 &&
                    g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL)    != 0)
                {
                    result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_HighDuty.interval, BleScannerCondition::BleScanParameter_HighDuty.window);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
                if (!g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
                {
                    BTM_LOG("Final BLE scan filter condition. Stop BLE scan.\n");

                    result = nn::bluetooth::ClearLeScanFilters();
                    if (result.IsFailure())
                    {
                        *pResult = ResultFailureLowLayerGeneral();
                        break;
                    }
                    // Result は見ない
                    DisableBleScan(bufferBtp, bufferSizeBtp);
                }
            }

            break;
        }
        case ApiId_BleStartScanSdGeneral:
        {
            BTM_LOG("ApiId_BleStartScanSdGeneral\n");

            if (!g_BleScannerCondition.IsScanning() && !g_HcContainer.IsAutoPairing())
            {
                BTM_LOG("First BLE scan filter condition. Start BLE scan.\n");

                result = EnableBleScan(bufferBtp, bufferSizeBtp);
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
            }

            const CommandParam_StartBleScanForSmartDevice* pApiCommand = reinterpret_cast<const CommandParam_StartBleScanForSmartDevice*>(pCommand);

            BleScannerCondition::ScanFilterCondition condition;
            memset(&condition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            condition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL;
            condition.condition.structure.adType = nn::bluetooth::BleAdType_CompleteListOf128bitUuid;

            uint8_t offset = 0;
            memcpy((condition.condition.structure.data + offset), pApiCommand->condition.uu.uuid128, nn::bluetooth::GattAttributeUuidLength_128);
            offset += nn::bluetooth::GattAttributeUuidLength_128;
            condition.condition.structure.length = offset + 1;    // for length field

            for (int i = 0; i < offset; ++i)
            {
                condition.condition.mask[i] = 0xFF;
            }
            condition.condition.maskeLength = offset;
            condition.aruid = pApiCommand->aruid;

            *pResult = BleAddScanFilterCondition(bufferBtp, bufferSizeBtp, condition);

            if (pResult->IsSuccess())
            {
                g_BleScannerCondition.ClearScanResultsSd();

                if (g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_GENERAL)       == 0 &&
                    g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED)        == 0 &&
                    g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL)    != 0)
                {
                    result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_HighDuty.interval, BleScannerCondition::BleScanParameter_HighDuty.window);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            }

            break;
        }
        case ApiId_BleStopScanSdGeneral:
        {
            BTM_LOG("ApiId_BleStopScanSdGeneral\n");
            const CommandParam_StopBleScanForSmartDevice* pApiCommand = reinterpret_cast<const CommandParam_StopBleScanForSmartDevice*>(pCommand);
            BleScannerCondition::ScanFilterCondition conditions[BleScannerCondition::BLE_SCAN_CONDITION_MAX_FOR_SD_GENERAL];
            int conditionCount = g_BleScannerCondition.GetScanFilterCondition(conditions, NN_ARRAY_SIZE(conditions), BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL);

            for (int i = 0; i < conditionCount; ++i)
            {
                if (conditions[i].inUse && conditions[i].aruid == pApiCommand->aruid)
                {
                    *pResult = BleDeleteScanFilterCondition(bufferBtp, bufferSizeBtp, conditions[i]);

                    if(pResult->IsFailure())
                    {
                        NN_SDK_LOG("[btm] Failed to delete scan filter condition for smart device. result = %d-%d\n", result.GetModule(), result.GetDescription());
                        break;
                    }
                }
            }

            if (pResult->IsSuccess())
            {
                if ((g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_GENERAL)  != 0 ||
                     g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_PAIRED)   != 0) &&
                     g_BleScannerCondition.GetFilterConditionCount(BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL) == 0)
                {
                    result = nn::bluetooth::SetLeScanParameter(BleScannerCondition::BleScanParameter_LowDuty.interval, BleScannerCondition::BleScanParameter_LowDuty.window);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
                if (!g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
                {
                    BTM_LOG("Final BLE scan filter condition. Stop BLE scan.\n");

                    result = nn::bluetooth::ClearLeScanFilters();
                    if (result.IsFailure())
                    {
                        *pResult = ResultFailureLowLayerGeneral();
                        break;
                    }
                    // Result は見ない
                    DisableBleScan(bufferBtp, bufferSizeBtp);
                }
            }

            break;
        }
        case ApiId_BleRegisterUnconnectableScan:
        {
            BTM_LOG("ApiId_BleRegisterUnconnectableScan\n");

            if (!g_BleScannerCondition.IsScanning() && !g_HcContainer.IsAutoPairing())
            {
                result = EnableBleScan(bufferBtp, bufferSizeBtp);
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
            }

            const CommandParam_RegisterBleUnconnectableScan* pApiCommand = reinterpret_cast<const CommandParam_RegisterBleUnconnectableScan*>(pCommand);
            uint8_t version = 0x01;

            BleScannerCondition::ScanFilterCondition condition;
            memset(&condition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            condition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_UNCONNECTABLE_SCAN;
            condition.condition.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;

            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET), pApiCommand->condition.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_VERSION_OFFSET), &version, sizeof(uint8_t));
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET), pApiCommand->condition.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET), pApiCommand->condition.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            condition.condition.structure.length = BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET + 1;    // for length field

            for (int i = 0; i < BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET; ++i)
            {
                condition.condition.mask[i] = 0xFF;
            }
            condition.condition.maskeLength = BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET;
            condition.aruid = nn::applet::AppletResourceUserId::GetInvalidId();

            *pResult = BleAddScanFilterCondition(bufferBtp, bufferSizeBtp, condition);

            // ToDo: Scan Parameter の設定

            break;
        }
        case ApiId_BleUnRegisterUnconnectableScan:
        {
            BTM_LOG("ApiId_BleUnRegisterUnconnectableScan\n");
            const CommandParam_UnregisterBleUnconnectableScan* pApiCommand = reinterpret_cast<const CommandParam_UnregisterBleUnconnectableScan*>(pCommand);
            uint8_t version = 0x01;

            BleScannerCondition::ScanFilterCondition filterCondition;
            memset(&filterCondition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            filterCondition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_UNCONNECTABLE_SCAN;
            filterCondition.condition.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;

            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET), pApiCommand->condition.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_VERSION_OFFSET), &version, sizeof(uint8_t));
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET), pApiCommand->condition.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET), pApiCommand->condition.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            filterCondition.condition.structure.length = BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET + 1;    // for length field

            for (int i = 0; i < BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET; ++i)
            {
                filterCondition.condition.mask[i] = 0xFF;
            }
            filterCondition.condition.maskeLength = BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET;
            filterCondition.aruid = nn::applet::AppletResourceUserId::GetInvalidId();

            *pResult = BleDeleteScanFilterCondition(bufferBtp, bufferSizeBtp, filterCondition);

            if (pResult->IsSuccess() && !g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
            {
                result = nn::bluetooth::ClearLeScanFilters();
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
                // Result は見ない
                DisableBleScan(bufferBtp, bufferSizeBtp);
            }

            // ToDo: Scan Parameter の設定


            break;
        }
        case ApiId_BleRegisterSdUnconnectableScan:
        {
            BTM_LOG("ApiId_BleRegisterSdUnconnectableScan\n");

            if (!g_BleScannerCondition.IsScanning() && !g_HcContainer.IsAutoPairing())
            {
                result = EnableBleScan(bufferBtp, bufferSizeBtp);
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
            }

            const CommandParam_RegisterBleSmartDeviceUnconnectableScan* pApiCommand = reinterpret_cast<const CommandParam_RegisterBleSmartDeviceUnconnectableScan*>(pCommand);

            BleScannerCondition::ScanFilterCondition condition;
            memset(&condition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            condition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_UNCONNECTABLE_SCAN;
            condition.condition.structure.adType = nn::bluetooth::BleAdType_CompleteListOf128bitUuid;

            uint8_t offset = 0;
            memcpy((condition.condition.structure.data + offset), pApiCommand->condition.uu.uuid128, nn::bluetooth::GattAttributeUuidLength_128);
            offset += nn::bluetooth::GattAttributeUuidLength_128;
            condition.condition.structure.length = offset + 1;    // for length field

            for (int i = 0; i < offset; ++i)
            {
                condition.condition.mask[i] = 0xFF;
            }
            condition.condition.maskeLength = offset;
            condition.aruid = nn::applet::AppletResourceUserId::GetInvalidId();

            *pResult = BleAddScanFilterCondition(bufferBtp, bufferSizeBtp, condition);

            // ToDo: Scan Parameter の設定

            break;
        }
        case ApiId_BleUnRegisterSdUnconnectableScan:
        {
            BTM_LOG("ApiId_BleUnRegisterSdUnconnectableScan\n");
            const CommandParam_UnregisterBleSmartDeviceUnconnectableScan* pApiCommand = reinterpret_cast<const CommandParam_UnregisterBleSmartDeviceUnconnectableScan*>(pCommand);

            BleScannerCondition::ScanFilterCondition filterCondition;
            memset(&filterCondition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            filterCondition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_SD_UNCONNECTABLE_SCAN;
            filterCondition.condition.structure.adType = nn::bluetooth::BleAdType_CompleteListOf128bitUuid;

            uint8_t offset = 0;
            memcpy((filterCondition.condition.structure.data + offset), pApiCommand->condition.uu.uuid128, nn::bluetooth::GattAttributeUuidLength_128);
            offset += nn::bluetooth::GattAttributeUuidLength_128;
            filterCondition.condition.structure.length = offset + 1;    // for length field

            for (int i = 0; i < offset; ++i)
            {
                filterCondition.condition.mask[i] = 0xFF;
            }
            filterCondition.condition.maskeLength = offset;
            filterCondition.aruid = nn::applet::AppletResourceUserId::GetInvalidId();

            *pResult = BleDeleteScanFilterCondition(bufferBtp, bufferSizeBtp, filterCondition);

            if (pResult->IsSuccess() && !g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
            {
                result = nn::bluetooth::ClearLeScanFilters();
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
                // Result は見ない
                DisableBleScan(bufferBtp, bufferSizeBtp);
            }

            // ToDo: Scan Parameter の設定

            break;
        }
        case ApiId_BleRegisterWakeOnBle:
        {
            BTM_LOG("ApiId_BleRegisterWakeOnBle\n");

            if (!g_BleScannerCondition.IsScanning() && !g_HcContainer.IsAutoPairing())
            {
                result = EnableBleScan(bufferBtp, bufferSizeBtp);
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
            }

            const CommandParam_RegisterWakeOnBleScan* pApiCommand = reinterpret_cast<const CommandParam_RegisterWakeOnBleScan*>(pCommand);
            uint8_t version = 0x01;
            uint8_t wakeOnBle = 0x01;
            BdAddress address = g_HcContainer.GetProperty().bdAddress;

            BleScannerCondition::ScanFilterCondition condition;
            memset(&condition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            condition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_WAKE_ON;
            condition.condition.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;

            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET), pApiCommand->condition.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_VERSION_OFFSET), &version, sizeof(uint8_t));
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET), pApiCommand->condition.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET), pApiCommand->condition.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET), &wakeOnBle, sizeof(uint8_t));
            memcpy((condition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_PAIRED_ADDR_OFFSET), address.address, SIZE_OF_BDADDRESS);
            condition.condition.structure.length = BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET + 1;    // for length field

            for (int i = 0; i < BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET; ++i)
            {
                condition.condition.mask[i] = 0xFF;
            }

            condition.condition.maskeLength = BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET;
            condition.aruid = nn::applet::AppletResourceUserId::GetInvalidId();

            *pResult = BleAddScanFilterCondition(bufferBtp, bufferSizeBtp, condition);

            // ToDo: Scan Parameter の設定

            break;
        }
        case ApiId_BleUnregisterWakeOnBle:
        {
            BTM_LOG("ApiId_BleUnregisterWakeOnBle\n");
            const CommandParam_UnregisterWakeOnBleScan* pApiCommand = reinterpret_cast<const CommandParam_UnregisterWakeOnBleScan*>(pCommand);
            uint8_t version = 0x01;
            uint8_t wakeOnBle = 0x01;
            BdAddress address = g_HcContainer.GetProperty().bdAddress;

            BleScannerCondition::ScanFilterCondition filterCondition;
            memset(&filterCondition, 0x00, sizeof(BleScannerCondition::ScanFilterCondition));
            filterCondition.condition.filterIndex = BleScannerCondition::BLE_SCAN_FILTER_INDEX_FOR_WAKE_ON;
            filterCondition.condition.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;

            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_ID_OFFSET), pApiCommand->condition.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_VERSION_OFFSET), &version, sizeof(uint8_t));
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET), pApiCommand->condition.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET), pApiCommand->condition.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_WOLE_OFFSET), &wakeOnBle, sizeof(uint8_t));
            memcpy((filterCondition.condition.structure.data + BleScannerCondition::BLE_SCAN_CONDITION_MANU_PAIRED_ADDR_OFFSET), address.address, SIZE_OF_BDADDRESS);
            filterCondition.condition.structure.length = BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET + 1;    // for length field

            for (int i = 0; i < BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET; ++i)
            {
                filterCondition.condition.mask[i] = 0xFF;
            }
            filterCondition.condition.maskeLength = BleScannerCondition::BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET;
            filterCondition.aruid = nn::applet::AppletResourceUserId::GetInvalidId();

            *pResult = BleDeleteScanFilterCondition(bufferBtp, bufferSizeBtp, filterCondition);

            if (pResult->IsSuccess() && !g_BleScannerCondition.IsFilterConditionExisting() && !g_HcContainer.IsAutoPairing())
            {
                result = nn::bluetooth::ClearLeScanFilters();
                if (result.IsFailure())
                {
                    *pResult = ResultFailureLowLayerGeneral();
                    break;
                }
                // Result は見ない
                DisableBleScan(bufferBtp, bufferSizeBtp);
            }

            // ToDo: Scan Parameter の設定

            break;
        }
        case ApiId_AcquireBleConnectionEvent:
        {
            BTM_LOG("ApiId_AcquireBleConnectionEvent\n");
            ReportParam_AcquireBleConnectionEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireBleConnectionEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_BleConnection, &p_ReportParam->id);
            break;
        }
        case ApiId_DiscardBleConnectionEvent:
        {
            BTM_LOG("ApiId_DiscardBleConnectionEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardBleConnectionEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_BleConnection, id);
            break;
        }
        case ApiId_BleTryConnect:
        {
            BTM_LOG("ApiId_BleTryConnect\n");

            DeviceConditionList     dcList = *g_DcContainer.GetPtr();
            GattClientConditionList gccList = *g_BleGattClientContainer.GetPtr();
            SniffMode   sniffMode;
            SlotMode    slotMode;
            CeLength    ceLength;
            bool        isRestructureAllSlotsNeeded;
            uint8_t     restructureBleSlotsCount;

            dcList.isLargeSlotRequiredForBle = true;
            gccList.deviceCount++;

            bool isBleAvailable = CheckUsecase(g_DcContainer.GetPtr(), &dcList,
                                               g_BleGattClientContainer.GetPtr(), &gccList,
                                               &sniffMode, &slotMode, &ceLength,
                                               &isRestructureAllSlotsNeeded, &restructureBleSlotsCount);

            if (!isBleAvailable)
            {
                *pResult = ResultInvalidUsecase();
            }

            break;
        }
        case ApiId_BleConnect:
        {
            BTM_LOG("ApiId_BleConnect\n");

            DeviceConditionList     dcList  = *g_DcContainer.GetPtr();
            GattClientConditionList gccList = *g_BleGattClientContainer.GetPtr();
            SniffMode   sniffMode;
            SlotMode    slotMode;
            CeLength    ceLength;
            bool        isRestructureAllSlotsNeeded;
            uint8_t     restructureBleSlotsCount;

            const CommandParam_BleConnect* p_CommandParam = reinterpret_cast<const CommandParam_BleConnect*>(pCommand);

            bool IsConnectionRefusal = g_BleGattClientContainer.IsConnectionRefusal();
            if (IsConnectionRefusal)
            {
                BTM_LOG("BLE connection is refused on request\n");
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
                break;
            }
            else if(p_CommandParam->aruid != nn::applet::AppletResourceUserId::GetInvalidId() &&
                    p_CommandParam->aruid != g_AppletManager.GetInFocusedAppletResourceUser().aruid)
            {
                BTM_LOG("BLE connection is refused because connection trigger is not in focus\n");
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
                break;
            }

            nn::applet::AppletResourceUserId tempAruid;
            if (g_BleGattClientContainer.IsPendingConnection(&tempAruid, p_CommandParam->address))
            {
                // すでに試行中の接続なら、何もせずBreak
                BTM_LOG("Connection is already ongoing. Ignore.\n");
                break;
            }
            NN_UNUSED(tempAruid);

            uint8_t gattClientHandle = g_BleGattClientContainer.GetConnectableClientHandle();
            if (gattClientHandle == nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE)
            {
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
                break;
            }

            // CheckUsecase() して、BLE の接続ができるかどうかを調べる
            // ペアリング済みデバイスに対する自動接続時には、直接ここに来るので、ここでもCheckUsecase する。
            dcList.isLargeSlotRequiredForBle = true;
            gccList.deviceCount++;

            bool isBleAvailable = CheckUsecase(g_DcContainer.GetPtr(), &dcList,
                                               g_BleGattClientContainer.GetPtr(), &gccList,
                                               &sniffMode, &slotMode, &ceLength,
                                               &isRestructureAllSlotsNeeded, &restructureBleSlotsCount);

            // 非同期API もしくは、内部処理で来るので、Result は返さない。
            // クライアント向けには、ApiId_BleTryConnect の処理でResult を返している。
            if (!isBleAvailable)
            {
                BTM_LOG("BLE is not available in current Usecase.\n");
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
            }
            else
            {
                auto result = nn::bluetooth::LeClientConnect(p_CommandParam->aruid, gattClientHandle, &p_CommandParam->address, true);

                if (result.IsFailure())
                {
                    // 下層のAPI 呼び出しに失敗したら、イベントをシグナルする
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
                }
                else
                {
                    g_BleGattClientContainer.AddPendingConnection(gattClientHandle, p_CommandParam->address, p_CommandParam->aruid);
                }
            }

            break;
        }
        case ApiId_BleOverrideConnection:
        {
            BTM_LOG("ApiId_BleOverrideConnection\n");

            const CommandParam_BleOverrideConnection* p_CommandParam = reinterpret_cast<const CommandParam_BleOverrideConnection*>(pCommand);

            if (!g_BleGattClientContainer.OverrideConnection(p_CommandParam->connectionHandle, p_CommandParam->aruid))
            {
                *pResult = nn::btm::ResultBleConnectionNotFound();
            }

            break;
        }
        case ApiId_BleDisconnect:
        {
            BTM_LOG("ApiId_BleDisconnect\n");

            const CommandParam_BleDisconnect* p_CommandParam = reinterpret_cast<const CommandParam_BleDisconnect*>(pCommand);

            uint8_t gattClientHandle = g_BleGattClientContainer.GetClientHandle(p_CommandParam->connectionHandle);

            if (gattClientHandle == nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE)
            {
                // 接続が存在しないなら、イベントをシグナルして、何もしない
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
                break;
            }

            auto result = nn::bluetooth::LeClientDisconnect(p_CommandParam->connectionHandle);

            if (result.IsFailure())
            {
                // 下層のAPI 呼び出しに失敗したら、イベントをシグナルする
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleConnection);
                break;
            }

            break;
        }
        case ApiId_BleDisconnectNoOwnerConnection:
        {
            BTM_LOG("ApiId_BleDisconnectNoOwnerConnection\n");

            const CommandParam_BleDisconnectNoOwnerConnection* p_CommandParam = reinterpret_cast<const CommandParam_BleDisconnectNoOwnerConnection*>(pCommand);

            user::BleClientConnState connectionState[nn::bluetooth::BleConnectionCountMaxClient];
            g_BleGattClientContainer.GetConnectionState(connectionState);

            for (auto state : connectionState)
            {
                const nn::applet::AppletResourceUserId* aruid = g_BleGattClientContainer.GetConnectionOwner(state.connectionHandle);

                if (aruid == nullptr || *aruid == nn::applet::AppletResourceUserId::GetInvalidId())
                {
                    continue;
                }

                if (*aruid != p_CommandParam->aruid)
                {
                    CommandParam_BleDisconnect command;
                    command.connectionHandle = state.connectionHandle;

                    Message message;
                    message.apiId = ApiId_BleDisconnect;
                    memcpy(message.buffer, &command, sizeof(CommandParam_BleDisconnect));

                    EnQueue(&message);
                }
            }

            break;
        }
        case ApiId_BleDisconnectAll:
        {
            BTM_LOG("ApiId_BleDisconnectAll\n");

            const CommandParam_BleDisconnectAll* p_CommandParam = reinterpret_cast<const CommandParam_BleDisconnectAll*>(pCommand);

            user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

            g_BleGattClientContainer.GetConnectionState(connState);

            Message message;
            message.apiId = ApiId_BleDisconnect;

            for (const auto& connectionState : connState)
            {
                auto connectionOwner = g_BleGattClientContainer.GetConnectionOwner(connectionState.connectionHandle);

                if (connectionOwner != nullptr &&
                    *connectionOwner == p_CommandParam->aruid)
                {
                    CommandParam_BleDisconnect command;
                    command.connectionHandle = connectionState.connectionHandle;
                    memcpy(&message.buffer[0], &command, sizeof(command));

                    EnQueue(&message);
                }
            }

            break;
        }
        case ApiId_AcquireBlePairingEvent:
        {
            BTM_LOG("ApiId_AcquireBlePairingEvent\n");
            ReportParam_AcquireBlePairingEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireBlePairingEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_BlePairing, &p_ReportParam->id);
            break;
        }
        case ApiId_DiscardBlePairingEvent:
        {
            BTM_LOG("ApiId_DiscardBlePairingEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardBlePairingEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_BlePairing, id);
            break;
        }
        case ApiId_StartBlePairing:
        {
            BTM_LOG("ApiId_StartBlePairing\n");

            const CommandParam_StartBlePairing* p_CommandParam = reinterpret_cast<const CommandParam_StartBlePairing*>(pCommand);

            const BdAddress *pGattServerAddress = g_BleGattClientContainer.GetGattServerAddress(p_CommandParam->connectionHandle);
            if (pGattServerAddress == nullptr)
            {
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                break;
            }

            BlePairingInfoContainer::BlePairingInfo pairingInfo;
            pairingInfo.address     = *pGattServerAddress;
            pairingInfo.type        = BlePairingAdvDataType_ManuCliSerId;
            pairingInfo.advInfo     = p_CommandParam->condition;

            if (!g_BlePairingInfoContainer.SetPairingDevice(pairingInfo))
            {
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                break;
            }

            g_BlePairingInfoContainer.SetPairing(p_CommandParam->pairing);
            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_StartPairing);

            Message message;
            message.apiId = ApiId_BleEnablePairingErrorHandler;
            EnQueue(&message);

            break;
        }
        case ApiId_BleEnablePairingErrorHandler:
        {
            BTM_LOG("ApiId_BleEnablePairingErrorHandler\n");

            const BlePairingInfoContainer::BlePairingInfo* pairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pairingInfo == nullptr || g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_StartPairing)
            {
                NN_SDK_LOG("[btm] pairing info not found or invalid BLE pairing sequence at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();
                break;
            }

            uint32_t connectionHandle = g_BleGattClientContainer.GetConnectionHandle(pairingInfo->address);
            if (connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                NN_SDK_LOG("[btm] Connection lost at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();
                break;
            }

            const user::GattService*        pPairingService = g_BleGattClientContainer.GetGattServerService(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
            const user::GattCharacteristic* pErrorHandlerChar = g_BleGattClientContainer.GetGattServerCharacteristic(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_ERROR_HANDLER_CHARACTERISTIC_UUID);
            const user::GattDescriptor*     pCccDescriptor = g_BleGattClientContainer.GetGattServerDescritpor(connectionHandle, nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid);
            if (pPairingService == nullptr || pErrorHandlerChar == nullptr || pCccDescriptor == nullptr)
            {
                NN_SDK_LOG("[btm] Attribute not found at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();
                break;
            }

            nn::bluetooth::GattId serviceId, charId, descId;
            uint8_t data[2] = { 0x00 };

            //!< Indication が使えるなら優先
            if ((pErrorHandlerChar->property & nn::bluetooth::GattAttributeProperty_Indicate) != 0)
            {
                BTM_LOG("Nintendo Gatt Pairing Error Handler supports indication.\n");
                data[0] = 0x02;
            }
            else if ((pErrorHandlerChar->property & nn::bluetooth::GattAttributeProperty_Notify) != 0)
            {
                BTM_LOG("Nintendo Gatt Pairing Error Handler supports notification.\n");
                data[0] = 0x01;
            }
            else
            {
                NN_SDK_LOG("[btm] Nintendo Gatt Pairing Error Handler doesn't support either indication nor notification. property = %d\n", pErrorHandlerChar->property);
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();

                break;
            }

            serviceId.instanceId    = pPairingService->instanceId;
            serviceId.uuid          = pPairingService->uuid;
            charId.instanceId       = pErrorHandlerChar->instanceId;
            charId.uuid             = pErrorHandlerChar->uuid;
            descId.instanceId       = 0x00;
            descId.uuid             = pCccDescriptor->uuid;

            result = nn::bluetooth::LeClientRegisterNotification(connectionHandle, serviceId, pPairingService->isPrimaryService, charId);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[btm] Failed to register notification at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();

                break;
            }

            result = nn::bluetooth::LeClientWriteDescriptor(connectionHandle, serviceId, pPairingService->isPrimaryService, charId, descId, data, sizeof(data), 0);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[btm] Failed to write descriptor at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();

                break;
            }

            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_EnableErrorHandler);

            break;
        }
        case ApiId_BlePairingReadAddress:
        {
            BTM_LOG("ApiId_BlePairingReadAddress\n");

            Message message;
            message.apiId = ApiId_BlePairingWriteAbort;

            const BlePairingInfoContainer::BlePairingInfo* pairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pairingInfo == nullptr || g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_EnableErrorHandler)
            {
                NN_SDK_LOG("[btm] pairing info not found or invalid BLE pairing sequence at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            uint32_t connectionHandle = g_BleGattClientContainer.GetConnectionHandle(pairingInfo->address);
            if (connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                NN_SDK_LOG("[btm] Connection lost at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            const user::GattService*        pPairingService = g_BleGattClientContainer.GetGattServerService(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
            const user::GattCharacteristic* pReadAddressChar = g_BleGattClientContainer.GetGattServerCharacteristic(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_ADDRESS_READER_CHARACTERISTIC_UUID);
            if (pPairingService == nullptr || pReadAddressChar == nullptr)
            {
                NN_SDK_LOG("[btm] Attribute not found at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            nn::bluetooth::GattId serviceId, charId;

            serviceId.instanceId    = pPairingService->instanceId;
            serviceId.uuid          = pPairingService->uuid;
            charId.instanceId       = pReadAddressChar->instanceId;
            charId.uuid             = pReadAddressChar->uuid;

            result = nn::bluetooth::LeClientReadCharacteristic(connectionHandle, serviceId, pPairingService->isPrimaryService, charId, 0);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[btm] Failed to read characteristic at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            NN_UNUSED(message);
            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_ReadPublicAddress);

            break;
        }
        case ApiId_BlePairingReadAdvData:
        {
            BTM_LOG("ApiId_BlePairingReadAdvData\n");

            Message message;
            message.apiId = ApiId_BlePairingWriteAbort;

            const BlePairingInfoContainer::BlePairingInfo* pairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pairingInfo == nullptr || g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_ReadPublicAddress)
            {
                NN_SDK_LOG("[btm] pairing info not found or invalid BLE pairing sequence at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            uint32_t connectionHandle = g_BleGattClientContainer.GetConnectionHandle(pairingInfo->address);
            if (connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                NN_SDK_LOG("[btm] Connection lost at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            const user::GattService*        pPairingService = g_BleGattClientContainer.GetGattServerService(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
            const user::GattCharacteristic* pReadAdvDataChar = g_BleGattClientContainer.GetGattServerCharacteristic(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_ADV_DATA_READER_CHARACTERISTIC_UUID);
            if (pPairingService == nullptr || pReadAdvDataChar == nullptr)
            {
                NN_SDK_LOG("[btm] Attribute not found at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            nn::bluetooth::GattId serviceId, charId;

            serviceId.instanceId    = pPairingService->instanceId;
            serviceId.uuid          = pPairingService->uuid;
            charId.instanceId       = pReadAdvDataChar->instanceId;
            charId.uuid             = pReadAdvDataChar->uuid;

            result = nn::bluetooth::LeClientReadCharacteristic(connectionHandle, serviceId, pPairingService->isPrimaryService, charId, 0);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[btm] Failed to read characteristic at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            NN_UNUSED(message);
            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_ReadAdvData);

            break;
        }
        case ApiId_BlePairingWriteAddress:
        {
            BTM_LOG("ApiId_BlePairingWriteAddress\n");

            Message message;
            message.apiId = ApiId_BlePairingWriteAbort;

            const BlePairingInfoContainer::BlePairingInfo* pairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pairingInfo == nullptr || g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_ReadAdvData)
            {
                NN_SDK_LOG("[btm] pairing info not found or invalid BLE pairing sequence at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            uint32_t connectionHandle = g_BleGattClientContainer.GetConnectionHandle(pairingInfo->address);
            if (connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                NN_SDK_LOG("[btm] Connection lost at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            const user::GattService*        pPairingService = g_BleGattClientContainer.GetGattServerService(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
            const user::GattCharacteristic* pWriteAddressChar = g_BleGattClientContainer.GetGattServerCharacteristic(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_ADDRESS_WRITER_CHARACTERISTIC_UUID);
            if (pPairingService == nullptr || pWriteAddressChar == nullptr)
            {
                NN_SDK_LOG("[btm] Attribute not found at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            nn::bluetooth::GattId serviceId, charId;

            serviceId.instanceId    = pPairingService->instanceId;
            serviceId.uuid          = pPairingService->uuid;
            charId.instanceId       = pWriteAddressChar->instanceId;
            charId.uuid             = pWriteAddressChar->uuid;

            BdAddress address;

            if (g_BlePairingInfoContainer.GetPairing())
            {
                address = g_HcContainer.GetProperty().bdAddress;
            }
            else
            {
                memset(address.address, 0x00, SIZE_OF_BDADDRESS);
            }

            result = nn::bluetooth::LeClientWriteCharacteristic(connectionHandle, serviceId, pPairingService->isPrimaryService, charId, address.address, SIZE_OF_BDADDRESS, 0, true);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[btm] Failed to write characteristic at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            NN_UNUSED(message);
            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_WritePublicAddress);

            break;
        }
        case ApiId_BlePairingWriteComplete:
        {
            BTM_LOG("ApiId_BlePairingWriteComplete\n");

            Message message;
            message.apiId = ApiId_BlePairingWriteAbort;

            const BlePairingInfoContainer::BlePairingInfo* pairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pairingInfo == nullptr || g_BlePairingInfoContainer.GetStage() != BlePairingInfoContainer::BlePairingStage_WritePublicAddress)
            {
                NN_SDK_LOG("[btm] pairing info not found or invalid BLE pairing sequence at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            uint32_t connectionHandle = g_BleGattClientContainer.GetConnectionHandle(pairingInfo->address);
            if (connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                NN_SDK_LOG("[btm] Connection lost at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            const user::GattService*        pPairingService = g_BleGattClientContainer.GetGattServerService(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
            const user::GattCharacteristic* pWriteCompleteChar = g_BleGattClientContainer.GetGattServerCharacteristic(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_COMMAND_CHARACTERISTIC_UUID);
            if (pPairingService == nullptr || pWriteCompleteChar == nullptr)
            {
                NN_SDK_LOG("[btm] Attribute not found at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            nn::bluetooth::GattId serviceId, charId;

            serviceId.instanceId    = pPairingService->instanceId;
            serviceId.uuid          = pPairingService->uuid;
            charId.instanceId       = pWriteCompleteChar->instanceId;
            charId.uuid             = pWriteCompleteChar->uuid;

            uint8_t command = BlePairingInfoContainer::BlePairingCommand_CompleteOperation;

            result = nn::bluetooth::LeClientWriteCharacteristic(connectionHandle, serviceId, pPairingService->isPrimaryService, charId, &command, sizeof(command), 0, false);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[btm] Failed to write characteristic at BLE pairing stage %d\n", g_BlePairingInfoContainer.GetStage());
                EnQueue(&message);
                break;
            }

            NN_UNUSED(message);
            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_WriteComplete);

            break;
        }
        case ApiId_BlePairingWriteAbort:
        {
            BTM_LOG("ApiId_BlePairingWriteAbort\n");

            Message message;
            message.apiId = ApiId_BlePairingWriteAbort;

            const BlePairingInfoContainer::BlePairingInfo* pairingInfo = g_BlePairingInfoContainer.GetPairingDevice();
            if (pairingInfo == nullptr)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorFaultyDesign());
            }

            uint32_t connectionHandle = g_BleGattClientContainer.GetConnectionHandle(pairingInfo->address);
            if (connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();
                break;
            }

            const user::GattService*        pPairingService = g_BleGattClientContainer.GetGattServerService(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID);
            const user::GattCharacteristic* pWriteCompleteChar = g_BleGattClientContainer.GetGattServerCharacteristic(connectionHandle, BlePairingInfoContainer::NN_GATT_PAIRING_COMMAND_CHARACTERISTIC_UUID);
            if (pPairingService == nullptr || pWriteCompleteChar == nullptr)
            {
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                g_BlePairingInfoContainer.InitializeBlePairingState();
                break;
            }

            nn::bluetooth::GattId serviceId, charId;

            serviceId.instanceId = pPairingService->instanceId;
            serviceId.uuid = pPairingService->uuid;
            charId.instanceId = pWriteCompleteChar->instanceId;
            charId.uuid = pWriteCompleteChar->uuid;

            uint8_t command = BlePairingInfoContainer::BlePairingCommand_AbortOperation;

            result = nn::bluetooth::LeClientWriteCharacteristic(connectionHandle, serviceId, pPairingService->isPrimaryService, charId, &command, sizeof(command), 0, false);
            if (result.IsFailure())
            {
                static uint8_t retryCount = 0;
                retryCount++;

                if (retryCount < BlePairingInfoContainer::abortPairingRetryCountMax)
                {
                    EnQueue(&message);      // 再キューイング
                }
                else
                {
                    g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
                    g_BlePairingInfoContainer.InitializeBlePairingState();
                }
                break;
            }

            NN_UNUSED(message);
            g_BlePairingInfoContainer.SetStage(BlePairingInfoContainer::BlePairingStage_WriteAbort);

            break;
        }
        case ApiId_BlePairingUpdateDatabase:
        {
            const CommandParam_BlePairingUpdateDatabase* p_CommandParam = reinterpret_cast<const CommandParam_BlePairingUpdateDatabase*>(pCommand);

            BlePairingInfoContainer::BlePairingInfo pairingInfo;

            pairingInfo.type    = p_CommandParam->type;
            pairingInfo.address = p_CommandParam->address;
            pairingInfo.advInfo = p_CommandParam->condition;

            if (p_CommandParam->pairing)
            {
                // 現状、ペアリング情報の追加に失敗するパターンはないので、Abort させる
                NN_ABORT_UNLESS(g_BlePairingInfoContainer.AddPairingInfo(pairingInfo));
            }
            else
            {
                // 削除するパターンは、データベースに存在しないパターンがありえるので、Abort しない
                if (g_BlePairingInfoContainer.RemovePairingInfo(pairingInfo))
                {
                    BTM_LOG("Failed to remove pairing info. Pairing info does not exist.\n");
                }
            }

            g_BlePairingInfoContainer.Save();

            g_SystemEventContainer.Signal(SystemEventContainer::EventType_BlePairing);
            g_BlePairingInfoContainer.InitializeBlePairingState();

            break;
        }
        case ApiId_AcquireBleSdpEvent:
        {
            BTM_LOG("ApiId_AcquireBleSdpEvent\n");
            ReportParam_AcquireBleSdpEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireBleSdpEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_BleSdp, &p_ReportParam->id);
            break;
        }
        case ApiId_DiscardBleSdpEvent:
        {
            BTM_LOG("ApiId_DiscardBleSdpEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardBleSdpEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_BleSdp, id);
            break;
        }
        case ApiId_AcquireBleMtuConfigEvent:
        {
            BTM_LOG("ApiId_AcquireBleMtuConfigEvent\n");
            ReportParam_AcquireBleMtuConfigEvent* p_ReportParam = reinterpret_cast<ReportParam_AcquireBleMtuConfigEvent*>(pReport);
            *pResult = g_SystemEventContainer.Acquire(&p_ReportParam->handle, SystemEventContainer::EventType_BleMtuConfig, &p_ReportParam->id);
            break;
        }
        case ApiId_DiscardBleMtuConfigEvent:
        {
            BTM_LOG("ApiId_DiscardBleMtuConfigEvent\n");
            const uint8_t id = reinterpret_cast<const CommandParam_DiscardBleMtuConfigEvent*>(pCommand)->id;
            g_SystemEventContainer.Discard(SystemEventContainer::EventType_BleMtuConfig, id);
            break;
        }
        case ApiId_BleConfigureMtu:
        {
            BTM_LOG("ApiId_BleConfigureMtu\n");
            const CommandParam_BleConfigureMtu* p_CommandParam = reinterpret_cast<const CommandParam_BleConfigureMtu*>(pCommand);

            g_BleGattClientContainer.StartConfigureMtu(p_CommandParam->connectionHandle);

            auto result = nn::bluetooth::LeClientConfigureMtu(p_CommandParam->connectionHandle, p_CommandParam->mtu);

            if (result.IsFailure())
            {
                // 下層のAPI 呼び出しに失敗したら、システムイベントをシグナル
                g_SystemEventContainer.Signal(SystemEventContainer::EventType_BleMtuConfig);
                // MTU の更新開始を終了
                g_BleGattClientContainer.UpdateMtu(p_CommandParam->connectionHandle, g_BleGattClientContainer.GetMtu(p_CommandParam->connectionHandle));
                break;
            }

            break;
        }
        case ApiId_BleRegisterDataPath:
        {
            BTM_LOG("ApiId_BleRegisterDataPath\n");
            const CommandParam_BleRegisterDataPath* p_CommandParam = reinterpret_cast<const CommandParam_BleRegisterDataPath*>(pCommand);

            switch (p_CommandParam->path)
            {
            case nn::btm::user::BleDataPathType::BLE_DATA_PATH_GENERAL:
                result = nn::bluetooth::RegisterLeDataPath(p_CommandParam->uuid);
                break;
            case nn::btm::user::BleDataPathType::BLE_DATA_PATH_HID:
                result = nn::bluetooth::RegisterLeHidDataPath(p_CommandParam->uuid);
                break;
            default:
                BTM_LOG("Unsupported GATT Data path. &d\n", p_CommandParam->path);
                *pResult = ResultInvalidArgument();
            }

            if (result.IsFailure())
            {
                *pResult = ResultBleGattDataPathRegisterFailed();
            }

            break;
        }
        case ApiId_BleUnregisterDataPath:
        {
            BTM_LOG("ApiId_BleUnregisterDataPath\n");
            const CommandParam_BleUnregisterDataPath* p_CommandParam = reinterpret_cast<const CommandParam_BleUnregisterDataPath*>(pCommand);

            switch (p_CommandParam->path)
            {
            case nn::btm::user::BleDataPathType::BLE_DATA_PATH_GENERAL:
                result = nn::bluetooth::UnregisterLeDataPath(p_CommandParam->uuid);
                break;
            case nn::btm::user::BleDataPathType::BLE_DATA_PATH_HID:
                result = nn::bluetooth::UnregisterLeHidDataPath(p_CommandParam->uuid);
                break;
            default:
                BTM_LOG("Unsupported GATT Data path. &d\n", p_CommandParam->path);
                *pResult = ResultInvalidArgument();
            }

            if (result.IsFailure())
            {
                *pResult = ResultBleGattDataPathRegisterFailed();
            }

            break;
        }
        default:
        {
            BTM_LOG("Un-handled ApiId: %d\n", commandId);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorUnHandledApiId());
            break;
        }
    }
} //NOLINT(impl/function_size)











//--------------------------------------------------
//目的のレポート取得まで、ループを継続。
//HandleApiにて、API結果のレポートを待ちつつ、別のレポートの処理を止めないための処理関数。
//途中で目的外のレポートが来た場合は、逐次処理（イベントタイプに応じたHandleReportの呼び出し）する。
//--------------------------------------------------
MultiWaitId MultiWaitAndHandleAsyncReport(RequiredReportType requiredReportType, void* pBufferBtp, size_t bufferSizeBtp)
{
    bool isBreakEnabledForTime;
    if(requiredReportType.timeoutSec > 0)
    {
        isBreakEnabledForTime = true;
    }
    else
    {
        isBreakEnabledForTime = false;
    }

    for(;;)
    {
        MultiWaitId multiWaitId;
        if(isBreakEnabledForTime)
        {
            multiWaitId = MultiWait(MultiWaitMode_TimedEvent, requiredReportType.timeoutSec);
        }
        else
        {
            multiWaitId = MultiWait(MultiWaitMode_Event);
        }

        //シグナルしたイベント種別に適したイベントハンドラを呼び出す
        //Break要求に合致していた場合、処理を抜ける
        bool isBreaked;
        if(multiWaitId == MultiWaitId_Api)
        {
            BTM_LOG("Double command\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        }
        else if(multiWaitId == MultiWaitId_BtCore)
        {
            isBreaked = HandleBtCoreReport(requiredReportType.core, pBufferBtp, bufferSizeBtp);
            if(requiredReportType.breakCategory == BreakCategory_All)
            {
                return multiWaitId;
            }
            else if(requiredReportType.breakCategory == BreakCategory_Core && isBreaked)
            {
                return multiWaitId;
            }
            else
            {
                //Break条件に合致しないため、イベント待ちを続ける
            }
        }
        else if(multiWaitId == MultiWaitId_BtHid)
        {
            isBreaked = HandleBtHidReport(requiredReportType.hid, pBufferBtp, bufferSizeBtp);
            if(requiredReportType.breakCategory == BreakCategory_All)
            {
                return multiWaitId;
            }
            else if(requiredReportType.breakCategory == BreakCategory_Hid && isBreaked)
            {
                return multiWaitId;
            }
            else
            {
                //Break条件に合致しないため、イベント待ちを続ける
            }
        }
        else if (multiWaitId == MultiWaitId_BtBleCore)
        {
            isBreaked = HandleBtBleCoreReport(requiredReportType.ble, pBufferBtp, bufferSizeBtp);
            if (requiredReportType.breakCategory == BreakCategory_All)
            {
                return multiWaitId;
            }
            else if (requiredReportType.breakCategory == BreakCategory_Ble && isBreaked)
            {
                return multiWaitId;
            }
            else
            {
                //Break条件に合致しないため、イベント待ちを続ける
            }
        }
        else if(multiWaitId == MultiWaitId_Time)
        {
            if(isBreakEnabledForTime)
            {
                return multiWaitId;
            }
            else
            {
                BTM_LOG("Un-handled Time Event\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
            }
        }
        else
        {
            BTM_LOG("Un-handled Event\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        }
    }
}

//HandleApiの記述量を減らすため、Core/Hid/Ble用の関数を同名で用意しておく
//timeoutSec == 0の場合は、タイムアウト無し
MultiWaitId MultiWaitAndHandleAsyncReport(nn::bluetooth::EventType requiredCoreEventType, void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec)
{
    RequiredReportType requiredReportType;
    requiredReportType.breakCategory = BreakCategory_Core;
    requiredReportType.timeoutSec = timeoutSec;
    requiredReportType.core = requiredCoreEventType;
    return MultiWaitAndHandleAsyncReport(requiredReportType, pBufferBtp, bufferSizeBtp);
}
MultiWaitId MultiWaitAndHandleAsyncReport(nn::bluetooth::HidEventType requiredHidEventType, void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec)
{
    RequiredReportType requiredReportType;
    requiredReportType.breakCategory = BreakCategory_Hid;
    requiredReportType.timeoutSec = timeoutSec;
    requiredReportType.hid = requiredHidEventType;
    return MultiWaitAndHandleAsyncReport(requiredReportType, pBufferBtp, bufferSizeBtp);
}

MultiWaitId MultiWaitAndHandleAsyncReport(nn::bluetooth::BleEventType requiredBleEventType, void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec)
{
    RequiredReportType requiredReportType;
    requiredReportType.breakCategory = BreakCategory_Ble;
    requiredReportType.timeoutSec = timeoutSec;
    requiredReportType.ble = requiredBleEventType;
    return MultiWaitAndHandleAsyncReport(requiredReportType, pBufferBtp, bufferSizeBtp);
}

MultiWaitId MultiWaitAndHandleAsyncReport(void* pBufferBtp, size_t bufferSizeBtp, uint8_t timeoutSec)
{
    RequiredReportType requiredReportType;
    requiredReportType.breakCategory = BreakCategory_All;
    requiredReportType.timeoutSec = timeoutSec;
    return MultiWaitAndHandleAsyncReport(requiredReportType, pBufferBtp, bufferSizeBtp);
}


//--------------------------------------------------
//下層からのBtCore, BtHid, BtBleCore レポートの内容を、指定のバッファにコピー
//--------------------------------------------------
void GetBtCoreReport(nn::bluetooth::EventType* pEventTypeBtp, void* pBufferBtp, size_t bufferSizeBtp)
{
    NN_SDK_REQUIRES(bufferSizeBtp >= nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
    uint8_t* pBuf = static_cast<uint8_t*>(pBufferBtp);
    nn::bluetooth::GetEventInfo(pEventTypeBtp, pBuf, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
}

void GetBtHidReport(nn::bluetooth::HidEventType* pHidEventTypeBtp, void* pBufferBtp, size_t bufferSizeBtp)
{
    NN_SDK_REQUIRES(bufferSizeBtp >= nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);
    uint8_t* pBuf = static_cast<uint8_t*>(pBufferBtp);
    nn::bluetooth::HidGetEventInfo(pHidEventTypeBtp, pBuf, nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);
}

void GetBtBleCoreReport(nn::bluetooth::BleEventType* pBleCoreEventTypeBtp, void* pBufferBtp, size_t bufferSizeBtp)
{
    NN_SDK_REQUIRES(bufferSizeBtp >= nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);
    uint8_t* pBuf = static_cast<uint8_t*>(pBufferBtp);
    nn::bluetooth::GetLeCoreEventInfo(pBleCoreEventTypeBtp, pBuf, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
}

void WaitForLlrFinish(void* pBufferBtp, size_t bufferSizeBtp)
{
    //10秒間イベントを監視し、IsLlr = falseを待つ
    for(int i=0;i<10;i++)
    {
        if(g_HcContainer.IsLlr())
        {
            BTM_LOG("wait for Llr finish.\n");
        }
        else
        {
            return;
        }

        //1secタイムアウト付きでイベントを待つ。forの繰り返し回数が10のため、最長は約10sec
        MultiWaitId multiWaitId;
        multiWaitId = MultiWaitAndHandleAsyncReport(pBufferBtp, bufferSizeBtp, 1);
        if(multiWaitId == MultiWaitId_Time)
        {
            BTM_LOG("1 sec timeout.\n");
        }
        else if(multiWaitId == MultiWaitId_BtCore)
        {
            BTM_LOG("core event.\n");
        }
        else if(multiWaitId == MultiWaitId_BtHid)
        {
            BTM_LOG("hid event.\n");
        }
        else if (multiWaitId == MultiWaitId_BtBleCore)
        {
            BTM_LOG("BLE event.\n");
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        }
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeoutWaitForLlrFinish());
}

//--------------------------------------------------
//PageScanの停止
//下層のキューを取りきることで、PageScan停止前後での試行中の接続結果まで同期する
//--------------------------------------------------
void DisablePageScan(void* pBufferBtp, size_t bufferSizeBtp)
{
    nn::Result result;
    result = nn::bluetooth::ExtSetVisibility(false, false);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //試行中の接続数が0になるまで、キューを取り続ける
    //PendingConnection数が0に減らないケースを検出するために、100ms x 100回　= 10secのタイムアウトを設ける
    uint8_t pendingConnectionCount = 0xFF;
    for(int i=0;i<100;i++)
    {
        if(pendingConnectionCount == 0)
        {
            return;
        }
        else
        {
            result = nn::bluetooth::ExtGetPendingConnections();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        for(;;)
        {
            MultiWaitId multiWaitId;
            multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromExtensionCallbacks, pBufferBtp, bufferSizeBtp, 10);
            if(multiWaitId == MultiWaitId_Time)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeoutDisablePageScan());
            }
            else
            {
                const nn::bluetooth::InfoFromExtensionCallbacks* pInfo = reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(pBufferBtp);
                if(pInfo->eventType == nn::bluetooth::EventFromGetPendingConnectionsCallback)
                {
                    pendingConnectionCount = pInfo->connections.numPending;
                    BTM_LOG("PendingConnectionCount = %d\n", pendingConnectionCount);
                    break;
                }
            }
        }

        if(pendingConnectionCount != 0)
        {
            //試行中の接続があった場合、
            //GetPendingConnectionとイベント取得が高速で回ることを避けるためSleepさせる
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeoutDisablePageScan());
}

//--------------------------------------------------
//PageScanの開始
//--------------------------------------------------
void EnablePageScan()
{
    auto result = nn::bluetooth::ExtSetVisibility(false, true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------
//全デバイスの切断
//--------------------------------------------------
void DisconnectAllDevices(void* pBufferBtp, size_t bufferSizeBtp)
{
    DeviceConditionList dcList = g_DcContainer.Get();
    uint8_t commandCount = dcList.deviceCount;

    //全デバイスに対して切断を発行する。下層では切断済みだが、キューを取りきれていない可能性があるため、DisconnectのResultは見ない
    //上記のケースにおいても、DisconnectEventはdeviceCount分返って来ることが正しい（イベントがキューにたまっていることが原因のため）。
    for(int i=0;i<commandCount;i++)
    {
        nn::bluetooth::HidDisconnect(&dcList.device[i].bdAddress);
    }
    for(int i=0;i<commandCount;i++)
    {
        for(;;)
        {
            MultiWaitId multiWaitId;
            multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromConnectionStateCallback, pBufferBtp, bufferSizeBtp, 10);
            if(multiWaitId == MultiWaitId_Time)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeoutDisconnectAllDevices());
            }
            else
            {
                const nn::bluetooth::InfoFromConnectionStateCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromConnectionStateCallback*>(pBufferBtp);
                if(pInfo->state == nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED)
                {
                    break;
                }
            }
        }
    }
}

//--------------------------------------------------
//特定デバイスの切断
//--------------------------------------------------
void DisconnectDevice(const BdAddress* pBdAddress, void* pBufferBtp, size_t bufferSizeBtp)
{
    DeviceCondition dummyDeviceCondition;
    if(g_DcContainer.SearchDeviceCondition(&dummyDeviceCondition, *pBdAddress))
    {
        //対象のデバイスを切断する。
        //下層的には切断済みだが、切断イベントがキューから取り出されていないケースがあるため、resultは見ない
        nn::bluetooth::HidDisconnect(pBdAddress);
        for(;;)
        {
            MultiWaitId multiWaitId;
            multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromConnectionStateCallback, pBufferBtp, bufferSizeBtp, 10);
            if(multiWaitId == MultiWaitId_Time)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorTimeoutDisconnectDevice());
            }
            else
            {
                const nn::bluetooth::InfoFromConnectionStateCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromConnectionStateCallback*>(pBufferBtp);
                if(pInfo->state == nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED && IsSameBdAddress(&pInfo->bluetoothAddress, pBdAddress))
                {
                    break;
                }
            }
        }
    }
}

//--------------------------------------------------
//g_DeviceConditionListと入力値のDeviceConditionListをもとにユースケースを再評価し、
//それに応じたTSIをデバイスに設定しなおす。
//g_DeviceConditionList、pDeviceConditionListは設定後の値で更新される。
//
//接続と切断を起因としたユースケース判定は、g_DcContainerの更新タイミングとユースケース評価タイミングにズレがあるため、更新タイミングでCheckUsecaseしたRestructureAll結果を入力できるようにしている
//--------------------------------------------------
bool UpdateUsecase(DeviceConditionList* pDeviceConditionList, GattClientConditionList* pGattClientConditionList, void* pBufferBtp, size_t bufferSizeBtp, bool isForceRestructure)
{
    bool isValidUsecase;
    SniffMode nextSniffMode;
    SlotMode nextDefaultSlotMode;
    CeLength nextDefaultCeLength        = CeLength_2;
    bool isRestructureAllSlotsNeeded    = false;
    uint8_t restructureBleSlotCount     = 0;

    isValidUsecase = CheckUsecase(g_DcContainer.GetPtr(), pDeviceConditionList,
                                  g_BleGattClientContainer.GetPtr(), pGattClientConditionList,
                                  &nextSniffMode, &nextDefaultSlotMode, &nextDefaultCeLength,
                                  &isRestructureAllSlotsNeeded, &restructureBleSlotCount);
    if(isValidUsecase)
    {
           if(isRestructureAllSlotsNeeded || isForceRestructure || restructureBleSlotCount > 0)
           {
               RestructureAllBtSlots(pDeviceConditionList, pGattClientConditionList,
                                     nextSniffMode, nextDefaultSlotMode, nextDefaultCeLength, restructureBleSlotCount,
                                     pBufferBtp, bufferSizeBtp);
#ifdef BTM_CHECK_SNIFF_HISTORY
               HostConditionContainer::SniffModeHistory smh;
               smh.time = GetCurrentTimeMs();
               smh.sniffMode = nextSniffMode;
               g_HcContainer.AddSniffModeHistory(smh);
#endif
           }
           else
           {
               RestructureEachBtSlots(pDeviceConditionList,
                                      nextSniffMode,
                                      pBufferBtp, bufferSizeBtp);
           }
           g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);
           return true;
    }
    else
    {
        BTM_LOG("Invalid usecase.\n");
        return false;
    }
}

//--------------------------------------------------
//機能はUpdateUsecaseと同様。
//ユースケース評価完了時点でFinishWorking()を呼び、サーバスレッドのブロックを開放する
//--------------------------------------------------
bool UpdateUsecase(DeviceConditionList* pDeviceConditionList, GattClientConditionList* pGattClientConditionList, nn::Result* pResult, void* pBufferBtp, size_t bufferSizeBtp)
{
    bool isValidUsecase;
    SniffMode nextSniffMode;
    SlotMode nextDefaultSlotMode;
    CeLength nextDefaultCeLength        = CeLength_2;
    bool isRestructureAllSlotsNeeded    = false;
    uint8_t restructureBleSlotCount     = 0;

    isValidUsecase = CheckUsecase(g_DcContainer.GetPtr(), pDeviceConditionList,
                                  g_BleGattClientContainer.GetPtr(), pGattClientConditionList,
                                  &nextSniffMode, &nextDefaultSlotMode, &nextDefaultCeLength,
                                  &isRestructureAllSlotsNeeded, &restructureBleSlotCount);
    if(isValidUsecase)
    {
           *pResult = ResultSuccess();
           FinishWorking();
           if(isRestructureAllSlotsNeeded || restructureBleSlotCount > 0)
           {
               RestructureAllBtSlots(pDeviceConditionList, pGattClientConditionList,
                                     nextSniffMode, nextDefaultSlotMode, nextDefaultCeLength, restructureBleSlotCount,
                                     pBufferBtp, bufferSizeBtp);
#ifdef BTM_CHECK_SNIFF_HISTORY
               HostConditionContainer::SniffModeHistory smh;
               smh.time = GetCurrentTimeMs();
               smh.sniffMode = nextSniffMode;
               g_HcContainer.AddSniffModeHistory(smh);
#endif
           }
           else
           {
               RestructureEachBtSlots(pDeviceConditionList,
                                      nextSniffMode,
                                      pBufferBtp, bufferSizeBtp);
           }
           g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);
           return true;
    }
    else
    {
        BTM_LOG("Invalid usecase.\n");
        *pResult = ResultInvalidUsecase();
        FinishWorking();
        return false;
    }
}

//--------------------------------------------------
//g_DeviceConditionListと入力値のDeviceConditionListを比べ、SlotModeに差分がある物のみ、
//入力値（当該SlotMode および sniffMode）から計算されるTSIで設定しなおす。
//g_DeviceConditionList、pDeviceConditionListは設定後の値で更新される。
//--------------------------------------------------
void RestructureEachBtSlots(DeviceConditionList* pDeviceConditionList,
                            SniffMode sniffMode,
                            void* pBufferBtp, size_t bufferSizeBtp)
{
    nn::Result result;

    //変更対象のローカルDcListのindexを決める
    DeviceConditionContainer::DiffIndex diffIndex;
    diffIndex = g_DcContainer.SearchDifferentSlotModeIndex(*pDeviceConditionList);
    BTM_LOG("targetIndexCount = %d\n",diffIndex.count);

    //スロット整形中に接続イベントを取得してしまい、定義外のユースケースに入ってしまうことを防ぐために、都度切断のフラグを立てておく
    //整形前にユースケースを更新しておく対策も可能だが、イベントシグナルとは非同期にDeviceConditionを取得している上位層がモード遷移完了を誤って判定し得るため、このようにしている
    //[Todo]現在のユースケース判定で定義外だった場合の処理を定義し、そこでカバーする案（例：定義外だったらデバイスを全切断）を考える
    g_HcContainer.SetConnectionRefusal(true);
    g_BleGattClientContainer.SetConnectionRefusal(true);

    //TargetIndexのデバイスのみActiveに一度戻す
    for(int i = 0; i < diffIndex.count; i++)
    {
        BTM_LOG("SetTsi = 255\n");
        result = nn::bluetooth::ExtSetTsi(&pDeviceConditionList->device[diffIndex.index[i]].bdAddress,0xFF);
        BTM_LOG("API returned\n");
        if(result.IsSuccess())
        {
            for(;;)
            {
                MultiWaitId multiWaitId;
                multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromExtensionCallbacks, pBufferBtp, bufferSizeBtp, 10);
                if(multiWaitId == MultiWaitId_Time)
                {
                    NN_DETAIL_BTM_ERROR("Workaround in RestructureEachBtSlots. To Active.\n");
                    nn::bluetooth::HidDisconnect(&pDeviceConditionList->device[diffIndex.index[i]].bdAddress);
                    break;
                }
                else
                {
                    const nn::bluetooth::InfoFromExtensionCallbacks* pInfo = reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(pBufferBtp);
                    if(pInfo->eventType == nn::bluetooth::EventFromTsiExitCallback &&
                            IsSameBdAddress(&pInfo->bluetoothAddress,&pDeviceConditionList->device[diffIndex.index[i]].bdAddress))
                    {
                        if(pInfo->status == nn::bluetooth::BTHH_OK)
                        {
                            break;
                        }
                        else
                        {
                            BTM_LOG("TSI change failure. status = enum(%d)\n", pInfo->status);
                            nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                            break;
                        }
                    }
                }
            }
        }
        //[Todo]失敗するケースが、「すでに設定済み」以外に無いかを要確認
    }

    //TargetIndexのデバイスのみTSIを再設定
    for(int i = 0; i < diffIndex.count; i++)
    {
        TsiMode tsiMode;
        tsiMode = GetRecommendedTsi(sniffMode, pDeviceConditionList->device[diffIndex.index[i]].hidDeviceCondition.slotMode);
        BTM_LOG("SetTsi = enum(%d)\n",tsiMode);
        result = nn::bluetooth::ExtSetTsi(&pDeviceConditionList->device[diffIndex.index[i]].bdAddress, tsiMode);
        BTM_LOG("API returned\n");
        if(result.IsSuccess())
        {
            for(;;)
            {
                MultiWaitId multiWaitId;
                multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromExtensionCallbacks, pBufferBtp, bufferSizeBtp, 10);
                if(multiWaitId == MultiWaitId_Time)
                {
                    NN_DETAIL_BTM_ERROR("Workaround in RestructureEachBtSlots. To Tsi.\n");
                    nn::bluetooth::HidDisconnect(&pDeviceConditionList->device[diffIndex.index[i]].bdAddress);
                    break;
                }
                else
                {
                    const nn::bluetooth::InfoFromExtensionCallbacks* pInfo = reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(pBufferBtp);
                    if(pInfo->eventType == nn::bluetooth::EventFromTsiSetCallback &&
                            IsSameBdAddress(&pInfo->bluetoothAddress,&pDeviceConditionList->device[diffIndex.index[i]].bdAddress))
                    {
                        if(pInfo->status == nn::bluetooth::BTHH_OK)
                        {
                            //TSI設定が完了したため、SniffModeを更新する
                            //SlotModeはローカルDcListに値がすでに入っているので、ここでは更新しない
                            pDeviceConditionList->device[diffIndex.index[i]].hidDeviceCondition.sniffMode = sniffMode;
                            break;
                        }
                        else
                        {
                            BTM_LOG("TSI change failure. status = enum(%d)\n", pInfo->status);
                            nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                            break;
                        }
                    }
                }
            }
        }
    }

    g_HcContainer.SetConnectionRefusal(false);
    g_BleGattClientContainer.SetConnectionRefusal(false);

    g_DcContainer.Update(*pDeviceConditionList);

    uint8_t count;
    count = GetConnectionCapacity(g_DcContainer.GetPtr(), g_BleGattClientContainer.GetPtr());
    g_DcContainer.Update(count);
}

//--------------------------------------------------
//pDeviceConditionListのデバイスを、入力値（sniffMode, slotMode）から定まるTSIで設定しなおす。
//g_DeviceConditionList、pDeviceConditionListは設定後の値で更新される。
//--------------------------------------------------
void RestructureAllBtSlots(DeviceConditionList* pDeviceConditionList, GattClientConditionList* pGattClientConditionList,
                           SniffMode sniffMode, SlotMode slotMode, CeLength ceLength, uint8_t bleConnectionParameterUpdateCount,
                           void* pBufferBtp, size_t bufferSizeBtp)
{
    nn::Result result;

    //スロット整形中に接続イベントを取得してしまい、定義外のユースケースに入ってしまうことを防ぐために、都度切断のフラグを立てておく
    g_HcContainer.SetConnectionRefusal(true);
    g_BleGattClientContainer.SetConnectionRefusal(true);

    //デフォルトSlotModeに要求スロットモードを上書きしたDcListを作成
    for(int i = 0; i < pDeviceConditionList->deviceCount; i++)
    {
        pDeviceConditionList->device[i].hidDeviceCondition.sniffMode = sniffMode;
        if(pDeviceConditionList->device[i].hidDeviceCondition.isLargeSlotRequired == false)
        {
            pDeviceConditionList->device[i].hidDeviceCondition.slotMode = slotMode;
        }
        //isLargeSlotRequiredな場合は、元のSlotModeを保つ
    }

    // デフォルトCeLength に要求CeLength を上書きしたGccList を作成
    for (int i = 0; i < NN_ARRAY_SIZE(pGattClientConditionList->gattClients); ++i)
    {
        if (pGattClientConditionList->gattClients[i].connectedServer.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
        {
            continue;
        }

        // isLargeCeLengthRequired な場合は、元のCeLength を保つ
        if (!pGattClientConditionList->isLargeCeLengthRequired)
        {
            pGattClientConditionList->gattClients[i].maxCeLength = ceLength;
        }
    }

    //ローカルDeviceConditionListの通りに下層の設定を変更
    //全台Activeに一度戻す
    for(int i = 0; i < pDeviceConditionList->deviceCount; i++)
    {
        BTM_LOG("%s: SetTsi = 255\n", NN_CURRENT_FUNCTION_NAME);
        result = nn::bluetooth::ExtSetTsi(&pDeviceConditionList->device[i].bdAddress,0xFF);

        if(result.IsSuccess())
        {
            for(;;)
            {
                MultiWaitId multiWaitId;
                multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromExtensionCallbacks, pBufferBtp, bufferSizeBtp, 10);
                if(multiWaitId == MultiWaitId_Time)
                {
                    NN_DETAIL_BTM_ERROR("Workaround in RestructureAllBtSlots. To Active.\n");
                    nn::bluetooth::HidDisconnect(&pDeviceConditionList->device[i].bdAddress);
                    break;
                }
                else
                {
                    const nn::bluetooth::InfoFromExtensionCallbacks* pInfo = reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(pBufferBtp);
                    if(pInfo->eventType == nn::bluetooth::EventFromTsiExitCallback &&
                            IsSameBdAddress(&pInfo->bluetoothAddress,&pDeviceConditionList->device[i].bdAddress))
                    {
                        if(pInfo->status == nn::bluetooth::BTHH_OK)
                        {
                            break;
                        }
                        else
                        {
                            BTM_LOG("TSI change failure. status = enum(%d)\n", pInfo->status);
                            nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                            break;
                        }
                    }
                }
            }
        }
        //[Todo]失敗するケースが、「すでに設定済み」以外に無いかを要確認
    }

    uint8_t bleConnectionParameterUpdateRetryCount = 0;

    // BLE のコネクションパラメータアップデートを実施
    for (int i = 0; i < bleConnectionParameterUpdateCount; ++i)
    {
        for (int j = 0; j < NN_ARRAY_SIZE(pGattClientConditionList->gattClients); ++j)
        {
            if (pGattClientConditionList->gattClients[j].connectedServer.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
            {
                continue;
            }

            nn::bluetooth::LeConnectionParams param;
            param.bluetoothAddress      = pGattClientConditionList->gattClients[j].connectedServer.address;
            param.minConnectionInterval = pGattClientConditionList->gattClients[j].connectionInterval;
            param.minConnectionEventLen = static_cast<uint16_t>(CeLength_2);
            param.maxConnectionInterval = pGattClientConditionList->gattClients[j].connectionInterval;
            param.maxConnectionEventLen = static_cast<uint16_t>(pGattClientConditionList->gattClients[j].maxCeLength);
            param.slaveLatency          = pGattClientConditionList->gattClients[j].slaveLatency;
            param.supervisionTimeout    = pGattClientConditionList->gattClients[j].supervisionTimeout;
            param.preference            = true;

            BTM_LOG("%s: Set CE Length = %d\n", NN_CURRENT_FUNCTION_NAME, param.maxConnectionEventLen);
            result = nn::bluetooth::SetLeConnectionParameter(param);

            if (result.IsSuccess())
            {
                while(NN_STATIC_CONDITION(true))
                {
                    MultiWaitId multiWaitId;
                    multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeConnParamUpdateCallback, pBufferBtp, bufferSizeBtp, 3);
                    if (multiWaitId == MultiWaitId_Time)
                    {
                        NN_SDK_LOG("[btm] %s: BLE Connection Parameter Update timeouted.\n", NN_CURRENT_FUNCTION_NAME);
                        nn::bluetooth::LeClientDisconnect(pGattClientConditionList->gattClients[j].connectedServer.connectionHandle);
                        break;
                    }
                    else
                    {
                        const nn::bluetooth::InfoFromLeConnParamUpdateCallback* pInfo = reinterpret_cast<const nn::bluetooth::InfoFromLeConnParamUpdateCallback*>(pBufferBtp);

                        if (pInfo->status == nn::bluetooth::BT_OK)
                        {
                            // BTM が抱えているパラメータはイベントハンドラで更新されている
                            if (pInfo->connId == pGattClientConditionList->gattClients[j].connectedServer.connectionHandle)
                            {
                                bleConnectionParameterUpdateRetryCount = 0;
                                break;
                            }
                            else
                            {
                                // 違う接続に対するイベントである場合は、再度待ちに入る
                                BTM_LOG("BLE connection param updated for an unexpected connection (%d).Keep waiting for %d.\n",
                                    pInfo->connId, pGattClientConditionList->gattClients[j].connectedServer.connectionHandle);
                                continue;
                            }
                        }
                        else
                        {
                            NN_SDK_LOG("[btm] %s: BLE Connection Parameter Update failed. Status = %d.\n", NN_CURRENT_FUNCTION_NAME, pInfo->status);
                            // ビジーな場合は、リトライする。
                            if (pInfo->status == nn::bluetooth::BT_ERR_BUSY && bleConnectionParameterUpdateRetryCount < 5)
                            {
                                bleConnectionParameterUpdateRetryCount++;
                                NN_SDK_LOG("[btm] %s: Retry. Count = %d\n", NN_CURRENT_FUNCTION_NAME, bleConnectionParameterUpdateRetryCount);
                                j--;
                            }
                            // それ以外は切断
                            else
                            {
                                bleConnectionParameterUpdateRetryCount = 0;
                                nn::bluetooth::LeClientDisconnect(pGattClientConditionList->gattClients[j].connectedServer.connectionHandle);
                            }
                            break;
                        }
                    }
                }
            }
            else
            {
                NN_SDK_LOG("[btm] %s: BLE Connection Parameter Update failed. Result = %d.\n", NN_CURRENT_FUNCTION_NAME, result.GetDescription());
            }
        }
    }

    //全台にTSIを再設定
    TsiMode tsiMode;
    for(int i = 0; i < pDeviceConditionList->deviceCount; i++)
    {
        tsiMode = GetRecommendedTsi(sniffMode, pDeviceConditionList->device[i].hidDeviceCondition.slotMode);
        BTM_LOG("%s: SetTsi = enum(%d)\n", NN_CURRENT_FUNCTION_NAME, tsiMode);
        result = nn::bluetooth::ExtSetTsi(&pDeviceConditionList->device[i].bdAddress, tsiMode);

        if(result.IsSuccess())
        {
            for(;;)
            {
                MultiWaitId multiWaitId;
                multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromExtensionCallbacks, pBufferBtp, bufferSizeBtp, 10);
                if(multiWaitId == MultiWaitId_Time)
                {
                    NN_DETAIL_BTM_ERROR("Workaround in RestructureAllBtSlots. To Tsi.\n");
                    nn::bluetooth::HidDisconnect(&pDeviceConditionList->device[i].bdAddress);
                    break;
                }
                else
                {
                    const nn::bluetooth::InfoFromExtensionCallbacks* pInfo = reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(pBufferBtp);
                    if(pInfo->eventType == nn::bluetooth::EventFromTsiSetCallback &&
                            IsSameBdAddress(&pInfo->bluetoothAddress,&pDeviceConditionList->device[i].bdAddress))
                    {
                        if(pInfo->status == nn::bluetooth::BTHH_OK)
                        {
                            break;
                        }
                        else
                        {
                            BTM_LOG("TSI change failure. status = enum(%d)\n", pInfo->status);
                            nn::bluetooth::HidDisconnect(&pInfo->bluetoothAddress);
                            break;
                        }
                    }
                }
            }
        }
    }

    g_HcContainer.SetConnectionRefusal(false);
    g_BleGattClientContainer.SetConnectionRefusal(false);

    g_DcContainer.Update(*pDeviceConditionList);
    g_BleGattClientContainer.Update(pGattClientConditionList);

    uint8_t count;
    count = GetConnectionCapacity(g_DcContainer.GetPtr(), g_BleGattClientContainer.GetPtr());
    g_DcContainer.Update(count);

    uint8_t bleDeviceCountCapacity;
    bleDeviceCountCapacity = GetBleConnectionCapacity(g_DcContainer.GetPtr(), g_BleGattClientContainer.GetPtr());
    g_BleGattClientContainer.Update(bleDeviceCountCapacity);
} //NOLINT(impl/function_size)

//--------------------------------------------------
//指定のBdAddressのDeviceInfoを削除する。
//上記がない場合、切断済かつもっとも昔に接続したDeviceInfoを削除する。
//--------------------------------------------------
void RemoveOldDeviceInfo(const BdAddress* pBdAddress)
{
    nn::Result result;
    DeviceInfo deviceInfo;
    bool isThereDevice;
    isThereDevice = g_DiContainer.SearchDeviceInfo(&deviceInfo, *pBdAddress);
    if(isThereDevice)
    {
        //対象のBdAddressのペアリング情報を削除
        result = nn::bluetooth::RemoveBond(&deviceInfo.bdAddress);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        //[Todo]非同期だが完了取得手段が無い
        g_DiContainer.Remove(deviceInfo.bdAddress);
        return;
    }
    else
    {
        //切断済かつもっとも昔に接続したペアリング情報を削除
        int diCount = g_DiContainer.GetCount();
        DeviceCondition dummyDeviceCondition;
        for(int i=0;i<diCount;i++)
        {
            BdAddress bdAddress = g_DiContainer.Get(i).bdAddress;
            bool isConnected = g_DcContainer.SearchDeviceCondition(&dummyDeviceCondition, bdAddress);
            bool isPseudoConnected = g_HcContainer.IsPseudoConnection(bdAddress);
            if(!isConnected && !isPseudoConnected)
            {
                result = nn::bluetooth::RemoveBond(&bdAddress);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                //[Todo]非同期だが完了取得手段が無い
                g_DiContainer.Remove(bdAddress);
                return;
            }
        }

        //削除に適当なDeviceInfoが無かった場合、PseudoConnectionを無視して削除
        for(int i=0;i<diCount;i++)
        {
            BdAddress bdAddress = g_DiContainer.Get(i).bdAddress;
            bool isConnected = g_DcContainer.SearchDeviceCondition(&dummyDeviceCondition, bdAddress);
            if(!isConnected)
            {
                result = nn::bluetooth::RemoveBond(&bdAddress);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                //[Todo]非同期だが完了取得手段が無い
                g_DiContainer.Remove(bdAddress);
                return;
            }
        }

        //現プラットフォームでは、ペアリング数 > 接続数 の関係にあるため、ここには来ない
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }
}

//--------------------------------------------------
//自動ペアリングの停止
//--------------------------------------------------
void DisableAutoPairing(void* pBufferBtp, size_t bufferSizeBtp)
{
    //自動ペアリングは複数クライアントから多重に無効化され得るため、
    //再試行しないようにフラグを確認する
    if(g_HcContainer.IsAutoPairing() && g_HcContainer.IsRadio())
    {
        DeviceConditionList deviceConditionList = g_DcContainer.Get();
        GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();
        deviceConditionList.isSlotSavingForPairing = false;

        bool isValidUsecase;
        isValidUsecase = UpdateUsecase(&deviceConditionList, &gattClientConditionList, pBufferBtp, bufferSizeBtp);

        if(isValidUsecase)
        {
            g_HcContainer.SetAutoPairing(false);

            auto result = nn::bluetooth::CancelDiscovery();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            MultiWaitId multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromDiscoveryStateChangedCallback, pBufferBtp, bufferSizeBtp, 1);

            if (multiWaitId == MultiWaitId_Time)
            {
                if (g_HcContainer.IsDiscoverying())
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutStopGamepadPairing());
                }
            }
            else
            {
                const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(pBufferBtp);
                NN_ABORT_UNLESS(pInfo->state == nn::bluetooth::BT_DISCOVERY_STOPPED);
            }
        }
    }

    //状態変更の有無にかかわらず毎回シグナルする。
    //理由は、EnableAutoPairing()に記述。
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Gpp);
}

//--------------------------------------------------
//自動ペアリングの開始
//--------------------------------------------------
void EnableAutoPairing(void* pBufferBtp, size_t bufferSizeBtp)
{
    //自動ペアリングは複数クライアントから多重に有効化され得るため、
    //再試行しないようにフラグを確認する
    if(!g_HcContainer.IsAutoPairing() && g_HcContainer.IsRadio())
    {
        DeviceConditionList deviceConditionList = g_DcContainer.Get();
        GattClientConditionList gattClientConditionList = *g_BleGattClientContainer.GetPtr();
        deviceConditionList.isSlotSavingForPairing = true;

        bool isValidUsecase;
        isValidUsecase = UpdateUsecase(&deviceConditionList, &gattClientConditionList, pBufferBtp, bufferSizeBtp);

        if(isValidUsecase)
        {
            g_HcContainer.SetAutoPairing(true);
            g_DpContainer.Init();
            auto result = nn::bluetooth::StartDiscovery();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    //状態変更の有無にかかわらず毎回シグナルする。
    //理由：
    //上位層向けの本APIはResultを返すが、そのResult判定後にAPIをキューイングしているため厳密なResultにはなっていない。厳密さを担保するために、本システムイベントがある。
    //ResultSuccessを返したのにもかかわらず上述のUpdateUsecase()失敗でペアリングが弾かれるケースが問題なため、チェック結果にかかわらず毎回シグナルすることで、最新の状態を通知するようにしている。
    //ResultSuccessが返ったのにもかかわらず最新状態としてはペアリング停止状態となることがありうるが、それはペアリング開始直後に別要因（例：ローカル通信の開始）によってペアリングが自動停止したパターンと上位層には見える。
    //コーナーケース（例：ローカル通信終了直後での試行）で不必要にキューイングが弾かれてResultFailureが返るパターンについては、機会損失ではあるが問題ではない
    //[Todo]APIのResultを削除し、変化時のみSignalするようにする。ただし、ローカル通信中は実行が弾かれるため、そのケースをイベントシグナルでどう表現するかは要調整
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Gpp);
}

//--------------------------------------------------
//Bluetoothの再初期化を行う
//--------------------------------------------------
//[Todo]機能が安定したら、Init/Finとマージする
void Recovery()
{
    //[Todo]リカバリ内のFatalは特殊Fatalにして他とは区別する
    //[Todo]生産工程のケア
    //[Todo]ペアリング時の処理


    //DcContainerはdevice情報以外は全て初期値であることを期待
    //初期値以外が設定されている場合は復旧が難しいため、ひとまず一律ABORT
    DeviceConditionList dcList = g_DcContainer.Get();
    if(dcList.wlanMode != nn::btm::WlanMode_None)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorRecoveryPrecondition());
    }
    if(dcList.bluetoothMode != nn::btm::BluetoothMode_Auto)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorRecoveryPrecondition());
    }
    if(dcList.isSlotSavingForPairing)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorRecoveryPrecondition());
    }

    //RadioOff中は同上の理由でABORT。そもそも、RadioOff中に問題に遭遇するケースは少ない
    if(!g_HcContainer.IsRadio())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorRecoveryPrecondition());
    }

    auto result = nn::bluetooth::DisableBluetooth();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[Todo]専用Resultに置き換え

    result = nn::bluetooth::EnableBluetooth();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[Todo]専用Resultに置き換え

    result = nn::bluetooth::ExtSetMcMode(true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[Todo]専用Resultに置き換え

    //Conteiner類のリセット
    g_DpContainer.Init();//リカバリでは基本影響無いが、Discovery中のゴミが残る可能性があるので、一応初期化
    g_DcContainer.Init();//強制的に初期化。初期化不可なケースは事前条件確認でABORTしている
    //g_DiContainer;//リカバリでは影響無し
    //HcContainerは状態系を初期化
    g_HcContainer.SetAutoPairing(false);
    g_HcContainer.SetRadio(true);
    g_HcContainer.SetConnectionRefusal(false);
    g_HcContainer.SetLlrDisabled();
    //g_SystemEventContainer;//リカバリでは影響無し

    //Pairing情報を下層へ設定
    const nn::settings::system::BluetoothDevicesSettings* pSettings = g_DiContainer.GetPtr();
    for(int i=0;i<g_DiContainer.GetCount();i++)
    {
        result = nn::bluetooth::HidAddPairedDevice(&pSettings[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[Todo]専用Resultに置き換え
    }

    //Radio情報を、下層へ設定。起動時はPageScan無効のため、有効にする
    //RadioOffはAbortにしたため、処理分岐しない
    //[Todo]StateMachineのRadio系ステートとの整合がずれないか確認
    result = nn::bluetooth::ExtSetVisibility(false, true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[Todo]専用Resultに置き換え

    //上位層に更新を伝える
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Cdc);
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Disc);
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Rdi);
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Gpp);
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Radio);
    g_SystemEventContainer.Signal(SystemEventContainer::EventType_Llr);

    //[Todo]エラーレポートに、起動後のリカバリ回数を含める
}

nn::Result BleAddScanFilterCondition(uint8_t *bufferBtp, size_t bufferSizeBtp, const BleScannerCondition::ScanFilterCondition &condition)
{
    NN_ABORT_UNLESS_NOT_NULL(bufferBtp);
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSizeBtp, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

    // すでに登録済みなら、何もせずResultSuccess()
    if (g_BleScannerCondition.GetIsScanConditionRegistered(condition.condition))
    {
        return nn::ResultSuccess();
    }

    // フィルタ数が最大に達しているなら、return
    if (g_BleScannerCondition.GetFilterConditionCount(condition.condition.filterIndex) >=
        g_BleScannerCondition.GetFilterConditionCountMax(condition.condition.filterIndex))
    {
        NN_SDK_LOG("[btm] BLE Scan filter condition for %d filter is already full (%d).\n",
                    condition.condition.filterIndex,
                    g_BleScannerCondition.GetFilterConditionCount(condition.condition.filterIndex));
        return ResultScanStartFailedFull();
    }

    auto result = nn::bluetooth::AddLeScanFilterCondition(&condition.condition);

    if (result.IsFailure())
    {
        return ResultFailureLowLayerGeneral();;
    }

    MultiWaitId multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeScanFilterStateChangedCallback, &bufferBtp[0], bufferSizeBtp, 1);

    if (multiWaitId == MultiWaitId_Time)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleConfigScanFilterCondition());
    }
    else
    {
        const nn::bluetooth::InfoFromLeScanFilterStateChangedCallback *pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeScanFilterStateChangedCallback*>(&bufferBtp[0]);

        if (pInfo->status != nn::bluetooth::BT_OK)
        {
            return ResultFailureLowLayerGeneral();;
        }
        else
        {
            g_BleScannerCondition.AddFilterCondition(condition);
        }
    }

    return result;
}

nn::Result BleDeleteScanFilterCondition(uint8_t *bufferBtp, size_t bufferSizeBtp, const BleScannerCondition::ScanFilterCondition &condition)
{
    NN_ABORT_UNLESS_NOT_NULL(bufferBtp);
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSizeBtp, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

    nn::Result result = ResultSuccess();

    const nn::applet::AppletResourceUserId* aruid = g_BleScannerCondition.GetFilterConditionOwner(condition.condition);

    if (aruid == nullptr || !g_BleScannerCondition.GetIsScanConditionRegistered(condition.condition) || *aruid != condition.aruid)
    {
        // フィルタが存在しないので、成功
        return result;
    }

    result = nn::bluetooth::DeleteLeScanFilterCondition(&condition.condition);

    if (result.IsFailure())
    {
        return ResultFailureLowLayerGeneral();;
    }

    MultiWaitId multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeScanFilterStateChangedCallback, &bufferBtp[0], bufferSizeBtp, 1);

    if (multiWaitId == MultiWaitId_Time)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleConfigScanFilterCondition());
    }
    else
    {
        const nn::bluetooth::InfoFromLeScanFilterStateChangedCallback *pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeScanFilterStateChangedCallback*>(&bufferBtp[0]);

        if (pInfo->status != nn::bluetooth::BT_OK)
        {
            return ResultFailureLowLayerGeneral();;
        }
        else
        {
            g_BleScannerCondition.DeleteFilterCondition(condition);
        }
    }

    return result;
}

nn::Result EnableBleScan(void* pBufferBtp, size_t bufferSizeBtp)
{
    NN_ABORT_UNLESS_NOT_NULL(pBufferBtp);
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSizeBtp, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

    if (g_BleScannerCondition.IsScanning())
    {
        return nn::ResultSuccess();
    }

    auto result = nn::bluetooth::StartLeScan();

    if (result.IsSuccess())
    {
        g_BleScannerCondition.SetIsScanning(true);

        MultiWaitId multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeScanStateChangedCallback, pBufferBtp, bufferSizeBtp, 10);

        if (multiWaitId == MultiWaitId_Time)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleConfigScanFilterCondition());
        }
        else
        {
            const nn::bluetooth::InfoFromLeScanStateChangedCallback* pInfo = reinterpret_cast<const nn::bluetooth::InfoFromLeScanStateChangedCallback*>(pBufferBtp);

            if (pInfo->state != nn::bluetooth::BleScanState::BLE_SCAN_STATE_SCANNING)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleConfigScanFilterCondition());
            }
        }
    }

    return result;
}

nn::Result DisableBleScan(void* pBufferBtp, size_t bufferSizeBtp)
{
    NN_ABORT_UNLESS_NOT_NULL(pBufferBtp);
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSizeBtp, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

    if (!g_BleScannerCondition.IsScanning())
    {
        return nn::ResultSuccess();
    }

    auto result = nn::bluetooth::StopLeScan();

    if (result.IsSuccess())
    {
        g_BleScannerCondition.SetIsScanning(false);

        int waitCount = 0;
        // デバイス発見のイベントが上がってくる可能性もあるので、5 msec * 10 回待つ
        while (waitCount < 10)
        {
            MultiWaitId multiWaitId;
            multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeScanStateChangedCallback, pBufferBtp, bufferSizeBtp, 10);
            if (multiWaitId == MultiWaitId_Time)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleConfigScanFilterCondition());
                break;
            }
            else
            {
                const nn::bluetooth::InfoFromLeScanStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeScanStateChangedCallback*>(pBufferBtp);
                if (pInfo->status == nn::bluetooth::BT_OK && pInfo->state == nn::bluetooth::BLE_SCAN_STATE_COMPLETED)
                {
                    return nn::ResultSuccess();
                }
                else
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
                }
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleConfigScanFilterCondition());
    }

    return result;
}

void CancelBleConnection(void* pBufferBtp, size_t bufferSizeBtp)
{
    int waitCount = 0;

    while (waitCount < 10)
    {
        BdAddress   pendingConnectionAddress;
        uint8_t     pendingClientHandle = nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE;

        int pendingConnectionCount = g_BleGattClientContainer.GetPendingConnection(&pendingConnectionAddress, 1);

        if (pendingConnectionCount == 0)
        {
            BTM_LOG("BLE pending connection doesn't exist.\n");
            return;
        }
        else
        {
            BTM_LOG("BLE pending connection exists.\n");
            pendingClientHandle = g_BleGattClientContainer.GetClientHandle(pendingConnectionAddress);
        }

        if (pendingClientHandle != nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE)
        {
            auto result = nn::bluetooth::LeClientCancelConnection(pendingClientHandle, &pendingConnectionAddress, true);

            if (result.IsSuccess())
            {
                MultiWaitId multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeClientConnStateChangedCallback, pBufferBtp, bufferSizeBtp, 10);
                if (multiWaitId == MultiWaitId_Time)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutCancelBleConnection());
                }
            }
        }

        waitCount++;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutCancelBleConnection());
}

void DisconnectAllBleDevices(void* pBufferBtp, size_t bufferSizeBtp, bool forOwnedOnly, const nn::applet::AppletResourceUserId& owner)
{
    user::BleClientConnState connState[nn::bluetooth::BleConnectionCountMaxClient];

    g_BleGattClientContainer.GetConnectionState(connState);

    for (const auto& connection : connState)
    {
        auto connectionOwner = g_BleGattClientContainer.GetConnectionOwner(connection.connectionHandle);

        if((!forOwnedOnly ||
           (forOwnedOnly && connectionOwner != nullptr && *connectionOwner == owner)) &&
           connection.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            nn::bluetooth::LeClientDisconnect(connection.connectionHandle);

            while (NN_STATIC_CONDITION(true))
            {
                MultiWaitId multiWaitId = MultiWaitAndHandleAsyncReport(nn::bluetooth::EventFromLeClientConnStateChangedCallback, pBufferBtp, bufferSizeBtp, 5);

                if (multiWaitId == MultiWaitId_Time)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInternalErrorTimeoutBleDisconnectAllDevices());
                }
                else
                {
                    const nn::bluetooth::InfoFromLeConnStateChangedCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromLeConnStateChangedCallback*>(pBufferBtp);
                    if (pInfo->connState == nn::bluetooth::BleConnectionState::BLE_CONN_STATE_DISCONNECTED)
                    {
                        break;
                    }
                }
            }
        }
    }
}

}}
