﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#pragma once

#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/debug/btm_DebugApi.h>

#include "btm_InternalTypes.h"

namespace nn { namespace btm {

class DevicePropertyContainer {
public:
    DevicePropertyContainer();
    ~DevicePropertyContainer();
    void Init();
    DevicePropertyList Get() const;

    //devicePropertyを一つ追加する。COUNT_OF_DP_LIST以上の追加は行われない
    void Add(const DeviceProperty& deviceProperty);

    //指定のbdNameをもつBdAddressを一つ返す
    bool SearchBdAddress(BdAddress* pBdAddress, const char bdName[], uint8_t bdNameSize) const;

private:
    DevicePropertyList m_DevicePropertyList;

};


class DeviceConditionContainer {
public:
    struct DiffIndex{
        uint8_t count;
        uint8_t index[COUNT_OF_DC_LIST];
    };

    DeviceConditionContainer();
    ~DeviceConditionContainer();
    void Init();
    DeviceConditionList Get() const;
    const DeviceConditionList* GetPtr() const;

    //DeviceInfoの情報を元に、エントリを追加
    bool Add(const DeviceInfo& deviceInfo);

    //BdAddressが一致するエントリを削除する。true:削除完了。false:指定のBdAddressを持つエントリが存在しない
    bool Remove(const BdAddress& bdAddress);

    //BdAddressが一致するエントリの再送モードを上書きする。BdAddressが一致しない場合は何もしない
    void Update(const BdAddress& bdAddress, const ZeroRetransmissionList& zeroRetransmissionList);

    //WlanMode,BluetoothModeを上書きする。また、BdAddressが一致するエントリを上書きする
    void Update(const DeviceConditionList& deviceConditionList);

    //DeviceCountCapacityを上書きする
    void Update(uint8_t deviceCountCapacity);

    //BdAddress一致かつスロットモード不一致なエントリのindexを返す。返すindexは入力値のもの
    DiffIndex SearchDifferentSlotModeIndex(const DeviceConditionList& deviceConditionList) const;

    //BdAddressをもとに、DeviceConditionを取得する
    bool SearchDeviceCondition(DeviceCondition* pDeviceCondition, const BdAddress& bdAddress) const;

    void SetLargeSlotModeRequiredForBle(bool required);
private:
    DeviceConditionList m_DeviceConditionList;

    bool SearchDeviceConditionIndex(uint8_t* pIndex, const BdAddress& bdAddress) const;
};


//DeviceInfoContainerは、m_DeviceInfoListをベースとして動作する。
//m_DeviceSettingListは特定のAPIをトリガとして、一時的に正しい値が設定される
class DeviceInfoContainer {
public:
    DeviceInfoContainer();
    ~DeviceInfoContainer();

    //Nvm(Settings)の内容をもとに初期化する
    void InitByNvm();

    //Nvm(Settings)に内容を書き込む
    void WriteToNvm();

    DeviceInfoList Get() const;

    //入力値で指定したDeviceInfoを取得。指定したエントリが有効か否かを評価しない点注意
    DeviceInfo Get(uint8_t index) const;
    const nn::settings::system::BluetoothDevicesSettings* GetPtr();
    void Add(const DeviceInfo& deviceInfo);
    void Add(const nn::settings::system::BluetoothDevicesSettings& setting);
    void Remove(const BdAddress& bdAddress);
    void Remove();

    //BdAddressをもとに、DeviceInfoを取得する
    bool SearchDeviceInfo(DeviceInfo* pDeviceInfo, const BdAddress& bdAddress) const;

    //m_DeviceInfoListのエントリ数 = m_DeviceSettingListのエントリ数 を返す
    uint8_t GetCount() const;

    //指定エントリを最後尾のindexに回し、前詰めする
    void MoveToTail(const BdAddress& bdAddress);

private:
    DeviceInfoList m_DeviceInfoList;
    nn::settings::system::BluetoothDevicesSettings m_DeviceSettingList[nn::settings::system::BluetoothDevicesSettingsCountMax];

    //m_DeviceSettingListをm_DeviceInfoListの内容で初期化する
    void InitDeviceSettingList();
};


class DiDsUtility {
public:
    DiDsUtility();
    ~DiDsUtility();

    static void Reset(nn::settings::system::BluetoothDevicesSettings* pSetting);
    static bool SearchIndex(uint8_t* pIndex, const DeviceInfoList& deviceInfoList, const BdAddress& bdAddress);
    static void Convert(nn::settings::system::BluetoothDevicesSettings* pSetting, const DeviceInfo& deviceInfo);
    static void Convert(DeviceInfo* pDeviceInfo, const nn::settings::system::BluetoothDevicesSettings& setting);
    static void ConvertList(nn::settings::system::BluetoothDevicesSettings buffer[], const DeviceInfoList& deviceInfoList);
    static void ConvertList(DeviceInfoList* pDeviceInfoList, const nn::settings::system::BluetoothDevicesSettings buffer[], uint8_t bufferCount);
};


//[Todo]Device情報ではないので、同種のクラスが増えてきたらファイル分割する
class SystemEventContainer
{
public:
    static const int COUNT_OF_EVENT_TYPE = 12;
    enum EventType
    {
        EventType_Disc                      = 0,
        EventType_Cdc                       = 1,
        EventType_Rdi                       = 2,
        EventType_AwakeReq                  = 3,
        EventType_Radio                     = 4,
        EventType_Gpp                       = 5,
        EventType_Llr                       = 6,
        EventType_BleScan                   = 7,
        EventType_BleConnection             = 8,
        EventType_BlePairing                = 9,
        EventType_BleSdp                    = 10,
        EventType_BleMtuConfig              = 11,
    };
    SystemEventContainer();
    nn::Result Acquire(nn::os::NativeHandle* pHandle, EventType eventType);
    nn::Result Acquire(nn::os::NativeHandle* pHandle, EventType eventType, uint8_t* pId);
    void Discard(EventType eventType, uint8_t id);
    void Signal(EventType eventType);

private:
    struct EventUnit{
        bool isInUse;
        nn::os::SystemEventType event;
    };
    static const int COUNT_OF_EVENT_UNIT = 3;
    struct EventList{
        EventUnit eventUnit[COUNT_OF_EVENT_UNIT];
    };

    //COUNT_OF_EVENT_LISTの各内訳
    //Disc:                 (test)
    //Cdc:                  hid, ldn, (test)
    //Rdi:                  hid, (test)
    //AwakeReq:             spsm
    //Radio:                Ocean本体設定, Oceanクイックメニュー, (test)
    //Gpp:                  Oceanコンサポ, (test)
    //Llr:                  hid, (test)
    //BleScan:              application
    //BleConnection:        application, hid
    //BlePairing:           application
    //BleSdp:               application, hid
    //BleMtuConfig:         application

    EventList m_EventList[COUNT_OF_EVENT_TYPE];

    void Validate(EventType eventType) const;
};


class HostConditionContainer{
public:
    static const int APDN_COUNT = 4;//Auto Pairing DeviceName Count

    HostConditionContainer();
    void SetProperty(HostDeviceProperty& hdp);
    HostDeviceProperty GetProperty() const;
    const char* GetApdn(uint8_t index) const;
    uint8_t GetApdnSize(uint8_t index) const;
    void SetDiscoverying(bool isDiscoverying);
    void SetAutoPairing(bool isEnabled);
    void SetRadio(bool isEnabled);
    void SetConnectionRefusal(bool isEnabled);
    void SetLlrEnabled(const BdAddress& bdAddress);
    void SetLlrDisabled();
    bool IsDiscoverying() const;
    bool IsAutoPairing() const;
    bool IsRadio() const;
    bool IsConnectionRefusal() const;
    bool IsLlr() const;//LlrNotifyが動作中か
    bool IsLlr(const BdAddress& bdAddress) const;//指定BdAddressのLlrNotifyが動作中か

    struct SniffModeHistory
    {
        int64_t time;
        nn::btm::SniffMode sniffMode;
    };
    void AddSniffModeHistory(const SniffModeHistory& smh);

    static const int COUNT_OF_PSEUDO_CONNECTION = COUNT_OF_DI_LIST;
    struct PseudoConnectionList
    {
        uint8_t count;
        nn::btm::BdAddress bdAddress[COUNT_OF_PSEUDO_CONNECTION];
    };
    void AddPseudoConnection(const BdAddress& bdAddress);
    void RemovePseudoConnection(const BdAddress& bdAddress);
    bool IsPseudoConnection(const BdAddress& bdAddress) const;
    PseudoConnectionList GetPseudoConnectionList() const;

private:
    char m_Apdn[APDN_COUNT][SIZE_OF_BDNAME] = {"NintendoGamepad", "Joy-Con", "Pro Controller", "Lic Pro Controller"};
    uint8_t m_ApdnSize[APDN_COUNT] = {15, 7, 14};
    HostDeviceProperty m_HostDeviceProperty;
    bool m_IsDiscoverying;
    bool m_IsAutoPairing;
    bool m_IsRadio;
    bool m_IsConnectionRefusal;
    bool m_IsLlr;
    BdAddress m_LlrAddress;
    static const int COUNT_OF_SMH = 2;
    SniffModeHistory m_Smh[COUNT_OF_SMH];
    bool IsRapidChange(const uint64_t& previousTimeMs, const uint64_t& newestTimeMs, const uint64_t& thresholdTimeMs);
    bool IsSlowChange(const uint64_t& previousTimeMs, const uint64_t& newestTimeMs, const uint64_t& thresholdTimeMs);
    PseudoConnectionList m_PseudoConnectionList;

    void ValidatePseudoConnectionList() const;
    uint8_t SearchPseudoConnectionIndex(const BdAddress& bdAddress) const;
};

class BleGattClientConditionContainer
{
private:
    // サイズが大きすぎて、Usecase では使えないので再定義する。
    struct GattServerWithProfile : GattServer
    {
        nn::applet::AppletResourceUserId    connectionOwnerApplet;

        user::GattService   serviceList[nn::bluetooth::GattAttributeCountMaxClient];
        uint8_t             serviceNum;

        user::GattCharacteristic    charList[nn::bluetooth::GattAttributeCountMaxClient];
        uint8_t                     charNum;

        user::GattDescriptor    descList[nn::bluetooth::GattAttributeCountMaxClient];
        uint8_t                 descNum;

        bool    supportPairing;
        bool    isPendingConnection;
    };

    struct GattClientInternal
    {
        uint8_t                 handle;
        GattServerWithProfile   connectedServer;

        uint16_t    connectionInterval;
        uint16_t    slaveLatency;
        uint16_t    supervisionTimeout;
        uint16_t    connectionIntervalReqMin;
        uint16_t    connectionIntervalReqMax;
        uint16_t    slaveLatencyReq;
        uint16_t    supervisionTimeoutReq;
        uint16_t    mtu;
        CeLength    maxCeLength;    //!< 2, 4, or 8

        bool        isConfiguringMtu;
    };

    struct GattClientConditionListInternal
    {
        GattClientInternal gattClients[nn::bluetooth::BLE_GATT_CLIENT_NUM_MAX];

        uint8_t deviceCount;
        uint8_t deviceCountCapacity;
        uint8_t pendingConnectionCount;

        bool        isLargeCeLengthRequired;
    };

    GattClientConditionList         m_GattClientConditionList;
    GattClientConditionListInternal m_GattClientConditionListInternal;

    struct GattClientConnectionHistory
    {
        BdAddress                       address;
        uint32_t                        connectionHandle;
        debug::BleDisconnectionReason   disconnectionReason;
    };

    GattClientConnectionHistory m_GattClientConnectionHistroyList[nn::bluetooth::BLE_GATT_CLIENT_NUM_MAX * 2];

    void Clear(int index);

    int GetClientIndex(uint32_t connHandle) const;
    int GetServiceIndex(uint32_t connHandle, uint16_t serviceHandle) const;
    int GetCharacteristicIndex(uint32_t connHandle, uint16_t charHandle) const;
    int GetDescriptorIndex(uint32_t connHandle, uint16_t descHandle) const;

    bool m_ConnectionRefusal;

    // Gatt Client の接続履歴を追加します。リストが埋まっている場合は、最も古いデータを上書きします。
    void AddGattClientConnectionHistory(uint32_t connectionHandle, const BdAddress& address, debug::BleDisconnectionReason reason);
public:
    // 接続直後、UpdateUsecase() 前のコネクションパラメータ
    static const uint16_t DefaultConnectionIntervalMin  = 12;
    static const uint16_t DefaultConnectionIntervalMax  = 12;
    static const uint16_t DefaultSlaveLatency           = 0;
    static const uint16_t DefaultSupervisionTimeout     = 1000;
    static const CeLength DefaultCeLengthMin            = CeLength_2;
    static const CeLength DefaultCeLengthMax            = CeLength_2;

    const GattClientConditionList* GetPtr();

    // m_GattClientConditionInternal を更新する
    void Update(const GattClientConditionList* gattClientConditionList);

    // m_GattClientConditionInternal.deviceCountCapacity を更新する
    void Update(uint8_t connectionCapacity);

    BleGattClientConditionContainer();
    // 新しいGattClient を登録。空きが無ければfalse。
    bool AddClient(uint8_t handle);
    // 既存のGattClient を削除。見つからなければfalse。
    bool RemoveClient(uint8_t handle);

    // Gatt Client の接続を登録。見つからなければfalse。
    bool ConnectClient(uint8_t handle, uint32_t connHandle, BdAddress address);
    // Gatt Client の切断を登録。見つからなければfalse。
    bool DisconnectClient(uint8_t handle, uint32_t connHandle, uint16_t reason);
    // 既存の接続の connectionOwnerApllet を上書きします
    bool OverrideConnection(uint32_t connectionHandle, const nn::applet::AppletResourceUserId& aruid);

    void AddPendingConnection(uint8_t handle, const BdAddress& address, const nn::applet::AppletResourceUserId& aruid);
    int GetPendingConnection(BdAddress* pAddress, int addressCount) const;
    bool IsPendingConnection(nn::applet::AppletResourceUserId* pAruid, const BdAddress& address);

    // BLE の接続拒否状態を設定する。BTM の内的要因による設定の場合のみ、isForceUpdate = true とする。
    // isForceUpdate の場合、BleConnectionRefusalState_RefuseAll からでも強制的に更新できる
    void SetConnectionRefusal(bool refusal);
    bool IsConnectionRefusal() const;

    // Gatt Client のうち、接続を持たないClient のHandle を返す。見つからなければ、nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE。
    uint8_t GetConnectableClientHandle() const;
    // 指定されたConn Handle をもつClient のHandle を返す。見つからなければ、nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE。
    uint8_t GetClientHandle(uint32_t connHandle) const;
    // 指定された BD Address の GATT Server と繋がっている（もしくは接続試行中である） Client の Handle を返す。見つからなければ、nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE。
    uint8_t GetClientHandle(const BdAddress& address) const;

    // 指定されたBD Address のGATT Server とつながっているClient のConn Handle を返す。見つからなければ、nn::bluetooth::BLE_INVALID_CONNECTION_HANDLE。
    uint32_t GetConnectionHandle(BdAddress address) const;

    // 指定されたConn Handle を持つClient と繋がっているGatt Server のAddress を返す。見つからなければ、nullptr。
    const BdAddress* GetGattServerAddress(uint32_t connHandle) const;

    // 指定されたConn Handle のオーナーであるアプレットのARUID を返す。見つからなければ、nullptr。
    const nn::applet::AppletResourceUserId* GetConnectionOwner(uint32_t connHandle);

    // 指定されたConn Handle を持つClient の接続パラメータを更新する。見つからなければ、false。
    bool UpdateConnectionParameters(uint32_t connHandle, uint16_t connectionInterval, uint16_t slaveLatency, uint16_t supervisionTimeout);
    // 指定されたConn Handle を持つClient の接続パラメータを取得する。見つからなければ、false。
    bool GetConnectionParameters(uint32_t connHandle, uint16_t* pConnectionInterval, uint16_t* pSlaveLatency, uint16_t* pSupervisionTimeout, CeLength* pCeLengthMax) const;
    // 指定されたConn Handle を持つClient の接続パラメータリクエストを更新する。見つからなければ、false。
    bool UpdateConnectionParameterRequest(uint32_t connHandle,
                                          uint16_t connectionIntervalMin, uint16_t connectionIntervalMax,
                                          uint16_t slaveLatency, uint16_t supervisionTimeout);
    bool IsLargeCeLengthRequired();

    // 指定されたConn Handle を持つClient のMTU の更新を開始する。見つからなければ、false。
    bool StartConfigureMtu(uint32_t connHandle);
    // 指定されたConn Handle を持つClient のMTU を更新する。見つからなければ、false。
    bool UpdateMtu(uint32_t connHandle, uint16_t mtu);
    // 指定されたConn Handle を持つClient のMTU を取得する。
    uint16_t GetMtu(uint32_t connHandle) const;
    // 全接続について、MTU 設定中かを取得する
    bool IsConfiguringMtu() const;

    // 指定されたConn Handle を持つClient のGATT Server がペアリングをサポートするかを登録する。見つからなければ、false。
    bool SupportPairing(uint32_t connHandle);
    // 指定されたConn Handle を持つClient のGATT Server がペアリングをサポートするかを返す。
    bool CheckIfSupportPairing(uint32_t connHandle) const;

    // Gatt Client の接続情報を返す
    int GetConnectionState(user::BleClientConnState(&connState)[nn::bluetooth::BleConnectionCountMaxClient]) const;

    // 指定されたConn Handle を持つClient のGatt Server のProfile にGATT Service を追加する。空きが無ければfalse。
    bool AddGattServerService(uint32_t connHandle, user::GattService service);
    // 指定されたConn Handle を持つCleint のGatt Server のGatt Service 一覧を取得する。
    uint8_t GetGattServerServices(user::GattService* pServices, uint8_t inNum, uint32_t connHandle) const;
    // 指定されたConn Handle を持つClient のGatt Server のGatt Service のうち、指定されたUUID を持つものを返す。なければnullptr。
    const user::GattService* GetGattServerService(uint32_t connHandle, const nn::bluetooth::GattAttributeUuid& uuid) const;

    // 指定されたConn Handle を持つClient のGatt Server のProfile にGATT Characteristic を追加する。空きが無ければfalse。
    bool AddGattServerCharacteristic(uint32_t connHandle, user::GattCharacteristic characteristic);
    // 指定されたConn Handle を持つCleint のGatt Server の指定されたUUID をもつGatt Service に属するGatt Characteristic 一覧を取得する。
    uint8_t GetGattServerCharacteristics(user::GattCharacteristic* pCharacteristic, uint8_t inNum, uint32_t connHandle, uint16_t sertiveHandle) const;
    // 指定されたConn Handle を持つClient のGatt Server のGatt Characteristic のうち、指定されたUUID を持つものを返す。なければnullptr。
    const user::GattCharacteristic* GetGattServerCharacteristic(uint32_t connHandle, const nn::bluetooth::GattAttributeUuid& uuid) const;

    // 指定されたConn Handle を持つClient のGatt Server のProfile にGATT Descriptor を追加する。空きが無ければfalse。
    bool AddGattServerDescriptor(uint32_t connHandle, user::GattDescriptor descriptor);
    // 指定されたConn Handle を持つCleint のGatt Server の指定されたUUID をもつGatt Characteristic に属するGatt Descriptor 一覧を取得する。
    uint8_t GetGattServerDescritpors(user::GattDescriptor* pDescriptors, uint8_t inNum, uint32_t connHandle, uint16_t charHandle) const;
    // 指定されたConn Handle を持つClient のGatt Server のGatt Descriptor のうち、指定されたUUID を持つものを返す。なければnullptr。
    const user::GattDescriptor* GetGattServerDescritpor(uint32_t connHandle, const nn::bluetooth::GattAttributeUuid& uuid) const;

    // Gatt Client の切断理由を取得します。最も新しい履歴を取得します。履歴が存在しない場合は、False。
    bool GetGattClientDisconnectionReason(debug::BleDisconnectionReason* pOutReason, uint32_t connectionHandle, const BdAddress& address);

    // 指定されたConn Handle を持つClient の接続パラメータを取得します。接続がなければ 0 に設定し、False。
    bool GetConnectionParameter(uint16_t* pInterval, uint16_t *pLatency, uint16_t* pTimeout, uint32_t connHandle) const;
    // 指定されたConn Handle を持つClient がServer から要求された接続パラメータを取得します。接続がなければ 0 に設定し、False。
    bool GetConnectionParameterReq(uint16_t* pIntervaMin, uint16_t* pIntervalMax, uint16_t *pLatency, uint16_t* pTimeout, uint32_t connHandle) const;
};

class BleScannerCondition
{
public:
    BleScannerCondition();

    struct BleScanParameter
    {
        uint16_t interval;
        uint16_t window;
    };

    static const BleScanParameter BleScanParameter_LowDuty;
    static const BleScanParameter BleScanParameter_HighDuty;

    static const uint8_t BLE_SCAN_CONDITION_MANU_ID_OFFSET          = 0;
    static const uint8_t BLE_SCAN_CONDITION_MANU_VERSION_OFFSET     = BLE_SCAN_CONDITION_MANU_ID_OFFSET + static_cast<uint8_t>(nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
    static const uint8_t BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET   = BLE_SCAN_CONDITION_MANU_VERSION_OFFSET + sizeof(uint8_t);
    static const uint8_t BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET   = BLE_SCAN_CONDITION_MANU_CLIENT_ID_OFFSET + static_cast<uint8_t>(nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
    static const uint8_t BLE_SCAN_CONDITION_MANU_WOLE_OFFSET        = BLE_SCAN_CONDITION_MANU_SERVER_ID_OFFSET + static_cast<uint8_t>(nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
    static const uint8_t BLE_SCAN_CONDITION_MANU_PAIRED_ADDR_OFFSET = BLE_SCAN_CONDITION_MANU_WOLE_OFFSET + sizeof(uint8_t);
    static const uint8_t BLE_SCAN_CONDITION_MANU_ARBIT_DATA_OFFSET  = BLE_SCAN_CONDITION_MANU_PAIRED_ADDR_OFFSET + SIZE_OF_BDADDRESS;

    static const uint8_t BLE_SCAN_CONDITION_MAX                             = 48;
    static const uint8_t BLE_SCAN_CONDITION_MAX_FOR_GENERAL                 = 5;
    static const uint8_t BLE_SCAN_CONDITION_MAX_FOR_PAIRED                  = 5;
    static const uint8_t BLE_SCAN_CONDITION_MAX_FOR_UNCONNECTABLE_SCAN      = 10;
    static const uint8_t BLE_SCAN_CONDITION_MAX_FOR_SD_GENERAL              = 5;
    static const uint8_t BLE_SCAN_CONDITION_MAX_FOR_SD_UNCONNECTABLE_SCAN   = 5;
    static const uint8_t BLE_SCAN_CONDITION_MAX_FOR_WAKE_ON                 = 10;
    // Note:
    // BLE_GENERAL_SCAN_FILTER_MAX + BLE_PAIRED_SCAN_FILTER_MAX + BLE_SCAN_CONDITION_MAX_FOR_UNCONNECTABLE_SCAN + BLE_SMART_DEVICE_GENERAL_SCAN_FILTER_MAX + BLE_SCAN_CONDITION_MAX_FOR_SD_UNCONNECTABLE_SCAN
    // は48 以下である必要がある。（現在、5 + 10 + 10 + 5 + 5 = 35）

    // Note:
    // BLE_SCAN_CONDITION_MAX_FOR_UNCONNECTABLE_SCAN + BLE_SCAN_CONDITION_MAX_FOR_SD_UNCONNECTABLE_SCAN + BLE_WAKE_ON_BLE_SCAN_FILTER_MAX
    // は48 以下である必要がある。（現在、10 + 5 + 10 = 25）

    static const uint8_t BLE_SCAN_FILTER_INDEX_FOR_GENERAL                  = 0;
    static const uint8_t BLE_SCAN_FILTER_INDEX_FOR_PAIRED                   = 1;
    static const uint8_t BLE_SCAN_FILTER_INDEX_FOR_WAKE_ON                  = 2;
    static const uint8_t BLE_SCAN_FILTER_INDEX_FOR_UNCONNECTABLE_SCAN       = 3;
    static const uint8_t BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL               = 4;    // Service UUID
    static const uint8_t BLE_SCAN_FILTER_INDEX_FOR_SD_UNCONNECTABLE_SCAN    = 5;

    static const uint16_t BLE_SCAN_RESULT_TIMEOUT = 5000;   // Scan 結果を破棄するタイムアウト時間。msec。

    // Scan 中であるかを返す
    bool IsScanning() const { return m_IsScanning; }
    // Scan 中であることをセットする
    void SetIsScanning(bool isScanning) { m_IsScanning = isScanning; }
    // Scan Filter が有効であるかを返す
    bool GetIsScanFilterEnalbed() const { return m_IsScanFilterEnabled; }
    // Scan Filter が有効であるかを設定する
    void SetIsScanFilterEnalbed(bool isEnabled) { m_IsScanFilterEnabled = isEnabled; }

    struct ScanFilterCondition
    {
        nn::applet::AppletResourceUserId    aruid;
        nn::bluetooth::BleAdvertiseFilter   condition;
        bool inUse;
    };

    struct ScanFilter
    {
        ScanFilterCondition conditions[BLE_SCAN_CONDITION_MAX];
        uint8_t             conditionNum;
        uint8_t             conditionNumMax;
        uint8_t             filterIndex;
    };

    // Scan Filter をすべてClear する
    void ClearFilters();
    // 指定したFilter Index をもつScan Filter のCondition をクリアする
    void DeleteFilter(uint8_t filterIndex);
    // 指定したCondition が有効になっているかを返す
    bool GetIsScanConditionRegistered(const nn::bluetooth::BleAdvertiseFilter& condition);
    // 指定したFilter Index をもつScan Filter にCondition を追加する。空きがない、もしくはすでに存在するならfalse。
    bool AddFilterCondition(const ScanFilterCondition& condition);
    // 指定したFilter Index をもつScan Filter からCondition を削除する。見つからなければfalse。
    bool DeleteFilterCondition(const ScanFilterCondition& condition);
    // ScanFilterCondition のリストを取得する
    int GetScanFilterCondition(ScanFilterCondition* pOutConditions, int inNum, uint8_t filterIndex);
    // 指定したCondition のオーナーを取得する
    const nn::applet::AppletResourceUserId* GetFilterConditionOwner(const nn::bluetooth::BleAdvertiseFilter& condition);
    // 指定したFilter Index を持つScan Filter が持つ Condition の数。
    uint8_t GetFilterConditionCount(uint8_t filterIndex);
    // 指定したFilter Index を持つScan Filter が持ちうる Condition の数。
    uint8_t GetFilterConditionCountMax(uint8_t filterIndex);
    // 全 Scan Filter について、Condition を持つかどうか。
    bool IsFilterConditionExisting();

    struct TimedScanResult
    {
        nn::applet::AppletResourceUserId    aruid;
        user::ScanResult                    result;
        nn::os::Tick                        scannedTick;
    };

    struct ScanResultList
    {
        uint8_t count;
        TimedScanResult results[nn::bluetooth::BleScanResultCountMax / 2];
    };

    // General Scan のスキャン結果をクリアする
    void ClearScanResultsGeneral();
    // General Scan のスキャン結果を追加する。対応するスキャンフィルタがなければfalse。
    bool AddScanResultsGeneral(const user::ScanResult result);
    // General Scan のスキャン結果のうち、BLE_SCAN_RESULT_TIMEOUTしたものを削除する
    void RefreshScanResultGeneral();
    // General Scan のスキャン結果を取得する
    const ScanResultList* GetScanResultsGeneral();

    // Smart Device Scan のスキャン結果をクリアする
    void ClearScanResultsSd();
    // Smart Device Scan のスキャン結果を追加する
    bool AddScanResultsSd(const user::ScanResult result);
    // Smart Device Scan のスキャン結果のうち、BLE_SCAN_RESULT_TIMEOUTしたものを削除する
    void RefreshScanResultSd();
    // Smart Device Scan のスキャン結果を取得する
    const ScanResultList* GetScanResultsSd();

    struct UnconnectableScanResult
    {
        BdAddress                       address;
        nn::bluetooth::BleAdStructure   data[nn::bluetooth::BleAdStructureCountMax];
        nn::os::Tick                    savedTick;
    };

    struct UnconnectableScanResultList
    {
        UnconnectableScanResult             results[nn::bluetooth::BleScanResultCountMax / 2];
        nn::bluetooth::BleAdvertiseFilter   filter;
        bool                                inUse;
    };

    // Unconnectable Scan Pass の結果をクリアする
    void ClearUnconnectableScanResultGeneral(const nn::bluetooth::BleAdvertiseFilter& condition);
    // Unconnectable Scan の結果を追加する
    void AddUnconnectableScanResultGeneral(const nn::bluetooth::BleAdvertiseFilter& condition, const user::ScanResult result);
    // Unconnectable Scan の特定のフィルタでスキャンされた結果を取得する
    const UnconnectableScanResult* GetUnconnectableScanResultGeneral(const nn::bluetooth::BleAdvertiseFilter& condition) const;

    // Smart Device Unconnectable Scan の結果をクリアする
    void ClearUnconnectableScanResultSd(const nn::bluetooth::BleAdvertiseFilter& condition);
    // Smart Device Unconnectable Scan の結果を追加する
    void AddUnconnectableResultSd(const nn::bluetooth::BleAdvertiseFilter& condition, const user::ScanResult result);
    // Smart Device Unconnectable Scan の特定のフィルタでスキャンされた結果を取得する
    const UnconnectableScanResult* GetUnconnectableScanResultSd(const nn::bluetooth::BleAdvertiseFilter& condition) const;

    // Scan Filter を取得する
    bool GetScanFilter(nn::btm::user::BleAdvFilterForGeneral* pFilter, uint16_t parameterId) const;
    bool GetScanFilter(nn::btm::user::BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId) const;

    bool GetIfMatchScanFilter(nn::bluetooth::BleAdvertiseFilter* pCondition, nn::applet::AppletResourceUserId* pOwner,
                              uint8_t filterIndex, user::ScanResult result);

private:
    bool m_IsScanning;
    bool m_IsScanFilterEnabled;

    ScanFilter m_ScanFilterForGeneral;
    ScanFilter m_ScanFilterForPaired;
    ScanFilter m_ScanFilterForUnconnectableScan;
    ScanFilter m_ScanFilterForSdGeneral;
    ScanFilter m_ScanFilterForSdUnconnectableScan;
    ScanFilter m_ScanFilterForWakeOn;

    ScanFilter* GetScanFilter(uint8_t filterIndex);

    ScanResultList m_ScanResultListGeneral;
    ScanResultList m_ScanResultListSd;

    UnconnectableScanResultList m_UnconnectableScanResultListGeneral[BLE_SCAN_CONDITION_MAX_FOR_UNCONNECTABLE_SCAN];
    UnconnectableScanResultList m_UnconnectableScanResultListSd[BLE_SCAN_CONDITION_MAX_FOR_SD_UNCONNECTABLE_SCAN];
};

class BlePairingInfoContainer
{
public:
    static const nn::bluetooth::GattAttributeUuid NN_GATT_PAIRING_SERVICE_UUID;
    static const nn::bluetooth::GattAttributeUuid NN_GATT_PAIRING_COMMAND_CHARACTERISTIC_UUID;
    static const nn::bluetooth::GattAttributeUuid NN_GATT_PAIRING_ADDRESS_WRITER_CHARACTERISTIC_UUID;
    static const nn::bluetooth::GattAttributeUuid NN_GATT_PAIRING_ADDRESS_READER_CHARACTERISTIC_UUID;
    static const nn::bluetooth::GattAttributeUuid NN_GATT_PAIRING_ADV_DATA_READER_CHARACTERISTIC_UUID;
    static const nn::bluetooth::GattAttributeUuid NN_GATT_PAIRING_ERROR_HANDLER_CHARACTERISTIC_UUID;

    static const uint8_t abortPairingRetryCountMax = 5;

    enum BlePairingStage
    {
        BlePairingStage_Init,
        BlePairingStage_StartPairing,
        BlePairingStage_StartUnpairing,
        BlePairingStage_EnableErrorHandler,
        BlePairingStage_ReadPublicAddress,
        BlePairingStage_ReadAdvData,
        BlePairingStage_WritePublicAddress,
        BlePairingStage_WriteComplete,
        BlePairingStage_WriteAbort,
    };

    enum BlePairingCommand : uint8_t
    {
        BlePairingCommand_CompleteOperation = 0x00,
        BlePairingCommand_AbortOperation    = 0x01,
    };

    enum BlePairingErrorCode : uint8_t
    {
        BlePairingErrorCode_Success                 = 0x00,
        BlePairingErrorCode_InvalidValue            = 0x01,
        BlePairingErrorCode_ServerInternalError     = 0x02,
    };

    BlePairingInfoContainer();

    // 進行中のBLE ペアリングをリセット
    void InitializeBlePairingState();

    struct BlePairingInfo
    {
        BdAddress                       address         = { { 0x00, 0x00, 0x00 , 0x00 , 0x00 , 0x00 } };
        BlePairingAdvDataType           type;

        union
        {
            user::BleAdvFilterForGeneral    advInfo;
        };

        bool operator == (const BlePairingInfo& info) const
        {
            // type はチェックしない

            if (address != info.address)
            {
                return false;
            }

            switch (info.type)
            {
            case BlePairingAdvDataType_ManuCliSerId:
                if (memcmp(advInfo.manufacturerId, info.advInfo.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize) != 0)
                {
                    return false;
                }
                if (memcmp(advInfo.clientId, info.advInfo.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize) != 0)
                {
                    return false;
                }
                if (memcmp(advInfo.serverId, info.advInfo.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize) != 0)
                {
                    return false;
                }
                break;
            case BlePairingAdvDataType_Invalid:
            default:
                return false;
            }

            return true;
        }

        bool operator != (const BlePairingInfo& info) const
        {
            return !(*this == info);
        }
    };

    // ペアリング開始前であればセットして、True。進行中であれば、セットせずFalse。
    bool SetPairingDevice(const BlePairingInfo& info);

    // ペアリング開始後であれば、ペアリング対象のデバイス情報を返し、ペアリング開始前であれば、nullptr を返す。
    const BlePairingInfo* GetPairingDevice() const;

    BlePairingStage GetStage() const { return m_Stage; }

    void SetStage(BlePairingStage stage) { m_Stage = stage; }

    // ペアリング処理（True）か、ペアリング削除処理（False）かをセットする
    void SetPairing(bool pairing) { m_Pairing = pairing; }

    // ペアリング処理（True）か、ペアリング削除処理（False）かをゲットする
    bool GetPairing() const { return m_Pairing; }

    const BlePairingInfo* GetInfoList() const { return m_BlePairingInfoList; }

    bool AddPairingInfo(const BlePairingInfo& info);

    bool RemovePairingInfo(const BlePairingInfo& info);

    bool IsPaired(const BlePairingInfo& info) const;

    // NAND からの読み込み
    void Load();

    // NAND への保存
    void Save() const;
private:
    BlePairingInfo m_BlePairingInfoList[nn::bluetooth::BlePairingCountMax];

    BlePairingInfo m_PairingDeviceInfo;

    BlePairingStage m_Stage;

    // ペアリング処理（True）か、ペアリング削除処理（False）か
    bool m_Pairing;
};

}}  // namespace nn::btm
