﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/i2c/i2c_CommandListFormatter.h>
#include <nn/result/result_HandlingUtility.h>

#include "boot_I2cHelper.h"

namespace nn { namespace boot {

namespace {

// RetrySpan 間隔で約 TimeOutSpan の間リトライし続ける NN_RESULT_DO。
#define NN_BOOT_RESULT_DO_RETRY(function) \
    do \
    { \
        const auto TimeOutSpan = ::nn::TimeSpan::FromMilliSeconds(10000); \
        const auto RetrySpan = ::nn::TimeSpan::FromMilliSeconds(20); \
        \
        auto totalSpan = ::nn::TimeSpan::FromMilliSeconds(0); \
        while  ( NN_STATIC_CONDITION(true) ) \
        { \
            auto result = (function); \
            if ( result.IsSuccess() ) \
            { \
                break; \
            } \
            totalSpan += RetrySpan; \
            if ( totalSpan >= TimeOutSpan) \
            { \
                NN_RESULT_DO(result); \
            } \
            ::nn::os::SleepThread(RetrySpan); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

const nn::i2c::TransactionOption g_I2cTransStart = nn::i2c::TransactionOption_StartCondition;
const nn::i2c::TransactionOption g_I2cTransStartStop =
    static_cast<nn::i2c::TransactionOption>(nn::i2c::TransactionOption_StartCondition
                                            | nn::i2c::TransactionOption_StopCondition);

} // namespace

nn::Result ReadI2cRegister(
    uint8_t* pOutData,
    nn::i2c::driver::I2cSession& session,
    const uint8_t* pCmd, size_t cmdSize,
    size_t dataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(pCmd != nullptr);
    NN_ABORT_UNLESS(pOutData != nullptr);
    NN_ABORT_UNLESS(cmdSize > 0);
    NN_ABORT_UNLESS(dataSize > 0);

    uint8_t commandList[nn::i2c::CommandListLengthCountMax];

    nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));

    NN_ABORT_UNLESS_RESULT_SUCCESS(commandListFormatter.EnqueueSendCommand(g_I2cTransStart, pCmd, cmdSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(commandListFormatter.EnqueueReceiveCommand(g_I2cTransStartStop, dataSize));

    NN_BOOT_RESULT_DO_RETRY(nn::i2c::driver::ExecuteCommandList(pOutData, dataSize, session, commandList,
                                                                commandListFormatter.GetCurrentLength()));
    NN_RESULT_SUCCESS;
}

nn::Result WriteI2cRegister(
    nn::i2c::driver::I2cSession& session,
    const uint8_t* pCmd, size_t cmdSize,
    const uint8_t* pData,
    size_t dataSize) NN_NOEXCEPT
{
    const size_t temporaryBufferSize = 32;

    NN_ABORT_UNLESS(pCmd != nullptr);
    NN_ABORT_UNLESS(pData != nullptr);
    NN_ABORT_UNLESS(cmdSize > 0);
    NN_ABORT_UNLESS(dataSize > 0);
    NN_ABORT_UNLESS(cmdSize + dataSize <= temporaryBufferSize);

    uint8_t temporaryBuffer[temporaryBufferSize];

    memcpy(&(temporaryBuffer[0]), pCmd, cmdSize);
    memcpy(&(temporaryBuffer[cmdSize]), pData, dataSize);

    NN_BOOT_RESULT_DO_RETRY(nn::i2c::driver::Send(session, &temporaryBuffer, cmdSize + dataSize, g_I2cTransStartStop));
    NN_RESULT_SUCCESS;
}

}} // namespace nn::boot
