﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/i2c/driver/i2c.h>
#include <nne/max17050/max17050.h>

namespace nn { namespace boot {

class FuelgaugeDriver
{
public:
    FuelgaugeDriver() NN_NOEXCEPT
    {
        nn::i2c::driver::Initialize();
        nn::i2c::driver::OpenSession(&m_I2cSession, nn::i2c::I2cDevice_Max17050);
    }

    ~FuelgaugeDriver() NN_NOEXCEPT
    {
        nn::i2c::driver::CloseSession(m_I2cSession);
        nn::i2c::driver::Finalize();
    }

    nn::Result InitializeParameter() NN_NOEXCEPT;
    nn::Result GetTemperature(double *temperature);
    nn::Result GetSocRep(double *socRep) NN_NOEXCEPT;
    nn::Result GetAverageVCell(uint32_t *vCell) NN_NOEXCEPT;
    nn::Result SetShutdownTimer() NN_NOEXCEPT;
    nn::Result IsI2cShutdownEnabled(bool* pOut) NN_NOEXCEPT;
    nn::Result SetI2cShutdownEnabled(bool isEnabled) NN_NOEXCEPT;
    nn::Result IsBatteryRemoved(bool* pOut) NN_NOEXCEPT;
    nn::Result IssueSoftPowerOnReset() NN_NOEXCEPT;

private:
    nn::Result StartUp(const ::nne::max17050::CustomParameters& customParameters) NN_NOEXCEPT;
    bool IsPowerOnReset() NN_NOEXCEPT;
    bool IsModelAccessLocked() NN_NOEXCEPT;
    nn::Result LockModelAccess() NN_NOEXCEPT;
    nn::Result UnlockModelAccess() NN_NOEXCEPT;
    nn::Result LockVfSoc0AndQh0WriteAccess() NN_NOEXCEPT;
    nn::Result UnlockVfSoc0AndQh0WriteAccess() NN_NOEXCEPT;
    nn::Result UpdateModel(const uint16_t modelTable[]) NN_NOEXCEPT;
    bool IsModelUpdated(const uint16_t modelTable[]) NN_NOEXCEPT;

    nn::Result Read(const uint8_t addr, uint16_t *data) NN_NOEXCEPT;
    nn::Result Write(const uint8_t addr, uint16_t data) NN_NOEXCEPT;
    nn::Result Update(const uint8_t addr, uint16_t mask, uint16_t data) NN_NOEXCEPT;
    bool WriteAndVerify(const uint8_t addr, uint16_t data) NN_NOEXCEPT;
    void DelayMilliSeconds(uint32_t milliSeconds) NN_NOEXCEPT;

private:
    nn::i2c::driver::I2cSession m_I2cSession;
};

}} // namespace nn::boot
