﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/../../Sources/Libraries/psm/driver/detail/psm_FuelGaugeCustomParameters-hardware.nx.h>
#include <nne/../../Sources/Libraries/max17050/detail/max17050_Util.h>
#include <nne/../../Sources/Libraries/max17050/detail/max17050_Spec.h>
#include <nne/../../Sources/Libraries/max17050/detail/max17050_TargetSpec.h>

#include "boot_FuelgaugeDriver.h"
#include "boot_I2cHelper.h"

namespace nn { namespace boot {

namespace {

const ::nne::max17050::CustomParameters& GetCustomParameters() NN_NOEXCEPT
{
    ::std::unique_ptr<::nn::fs::IStorage> storage;

    if ((::nn::fs::OpenBisPartition(&storage, ::nn::fs::BisPartitionId::CalibrationBinary)).IsFailure())
    {
        // エラーの場合パラメータは ATL を使用する。
        return ::nn::psm::driver::detail::CustomParametersForAtl;
    }

    const size_t batteryLotOffset = 0x002CE0;
    const size_t batteryLotSize = 0x20;
    const int batteryLotVendorIndex = 7;
    char batteryLot[batteryLotSize];
    memset(batteryLot, '\0', batteryLotSize);

    if ((storage.get()->Read(batteryLotOffset, batteryLot, batteryLotSize)).IsFailure())
    {
        // エラーの場合パラメータは ATL を使用する。
        return ::nn::psm::driver::detail::CustomParametersForAtl;
    }

    // SIGLO-41597: データの破損による影響が限定的であり、破損があった場合でも起動するべきなので CRC チェックは省略する。

    switch (batteryLot[batteryLotVendorIndex])
    {
    case 'A':
        return ::nn::psm::driver::detail::CustomParametersForAtl;
    case 'M':
        return ::nn::psm::driver::detail::CustomParametersForMaxell;
    default:
        // 未定義の文字の場合パラメータは ATL を使用する。
        return ::nn::psm::driver::detail::CustomParametersForAtl;
    }
}

} // namespace

nn::Result FuelgaugeDriver::InitializeParameter() NN_NOEXCEPT
{
    NN_RESULT_DO(StartUp(GetCustomParameters()));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::StartUp(const ::nne::max17050::CustomParameters& customParameters) NN_NOEXCEPT
{
    const auto param = &customParameters;

    /* Check for POR */
    if (IsPowerOnReset())
    {
        // SIGLO-66059: POR が発生したことを保存する
        NN_RESULT_DO(Update(nne::max17050::detail::DeviceMemoryMap::MiscCfg,
            nne::max17050::detail::DeviceMemoryMap::MiscCfgDetail::Mask::Mask_NeedToRestoreParameters,
            nne::max17050::detail::DeviceMemoryMap::MiscCfgDetail::Mask::Mask_NeedToRestoreParameters));

        /* Delay at least 500ms */
        DelayMilliSeconds(500);

        /* Initialize Configuration */
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::Config, 0x7210));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::FilterCfg, 0x8784));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::RelaxCfg, param->relaxcfg));

        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::LearnCfg, 0x2603));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::FullSocThr, 0x5F00));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::IAvgEmpty, 0x1D2A));

        /* Unlock Model Access */
        do
        {
            NN_RESULT_DO(UnlockModelAccess());

            /* Write / Read / Verify the Custom Model */
            NN_RESULT_DO(UpdateModel(param->modelTable));
        } while (!IsModelUpdated(param->modelTable));

        /* Lock Model Access */
        const auto TrialCountMax = 10;
        auto trialCount = 0;
        while(true)
        {
            NN_RESULT_DO(LockModelAccess());

            /* Verify that Model Access is locked */
            if (IsModelAccessLocked())
            {
                break;
            }

            ++trialCount;
            if (trialCount >= TrialCountMax)
            {
                // Regarded as a success if the device seems SDEV.
                NN_RESULT_SUCCESS;
            }
        }

        /* Write Custom Parameters */
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::Rcomp0, param->rcomp0));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::TempCo, param->tempco));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::IChgTerm, param->ichgterm));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::TGain, param->tgain));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::TOff, param->toff));

        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::VEmpty, param->vempty));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::QResidual00, param->qrtable00));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::QResidual01, param->qrtable01));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::QResidual02, param->qrtable02));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::QResidual03, param->qrtable03));

        /* Update Full Capacity Parameters */
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::FullCap, param->capacity));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::DesignCap, param->vffullcap));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::FullCapNom, param->vffullcap));

        /* Delay at least 350ms */
        DelayMilliSeconds(350);

        /* Write VFSOC and QH values to VFSOC0 and QH0 */
        uint16_t vfsoc, qh;
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::SocVf, &vfsoc));
        NN_RESULT_DO(UnlockVfSoc0AndQh0WriteAccess());
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::VfSoc0, vfsoc));
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::Qh, &qh));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::Qh0, qh));
        NN_RESULT_DO(LockVfSoc0AndQh0WriteAccess());

        /* Advance to Coulomb-Counter Mode */
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::Cycles, 0x0060));

        /* Load New Capacity Parameters */
        uint16_t remcap = (vfsoc * param->vffullcap) / 25600;
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::RemCapMix, remcap));
        uint16_t repcap = remcap * (param->capacity / param->vffullcap) / nne::max17050::detail::ModelScaling;
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::RemCapRep, repcap));

        uint16_t dqacc = param->vffullcap / 16;
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::DPAcc, 0x0C80));

        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::DQAcc, dqacc));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::FullCap, param->capacity));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::DesignCap, param->vffullcap));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::FullCapNom, param->vffullcap));

        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::SocRep, vfsoc));

        /* Initialization Complete */
        uint16_t status;
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::Status, &status));
        while (!WriteAndVerify(nne::max17050::detail::DeviceMemoryMap::Status, status & 0xFFFD));

        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::CGain, 0x7FFF));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::GetSocRep(double *socRep) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(socRep);

    uint16_t data;
    NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::SocRep, &data));

    *socRep = (double)data * nne::max17050::detail::SocRepMultiplier;

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::GetAverageVCell(uint32_t *vCell) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(vCell);

    uint16_t data;
    NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::AverageVCell, &data));

    *vCell = (data >> nne::max17050::detail::VCellOffset) * nne::max17050::detail::VCellUnit_uV
        / nne::max17050::detail::mV2uV;

    NN_RESULT_SUCCESS;
}

bool FuelgaugeDriver::IsPowerOnReset() NN_NOEXCEPT
{
    uint16_t data;
    Read(nne::max17050::detail::DeviceMemoryMap::Status, &data);

    data &= nne::max17050::detail::DeviceMemoryMap::StatusDetail::Mask_POR;

    return (data == nne::max17050::detail::DeviceMemoryMap::StatusDetail::POR_EventOccurred);
}

nn::Result FuelgaugeDriver::GetTemperature(double *temperature)
{
    NN_ABORT_UNLESS_NOT_NULL(temperature);

    int16_t data;

    NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::Temperature, (uint16_t *)&data));

    *temperature = static_cast<double>(data);
    *temperature /= 256;

    NN_RESULT_SUCCESS;
}

bool FuelgaugeDriver::IsModelAccessLocked() NN_NOEXCEPT
{
    uint16_t data;
    bool retval = true;

    for (uint8_t addr = nne::max17050::detail::ModelStartAddress; addr < nne::max17050::detail::ModelStartAddress + nne::max17050::detail::ModelSize; addr++)
    {
        Read(addr, &data);
        retval = retval && (data == 0);
    }

    return retval;
}

nn::Result FuelgaugeDriver::LockModelAccess() NN_NOEXCEPT
{
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::ModelAccess1, nne::max17050::detail::DeviceMemoryMap::ModelAccess1Detail::Lock));
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::ModelAccess2, nne::max17050::detail::DeviceMemoryMap::ModelAccess2Detail::Lock));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::UnlockModelAccess() NN_NOEXCEPT
{
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::ModelAccess1, nne::max17050::detail::DeviceMemoryMap::ModelAccess1Detail::Unlock));
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::ModelAccess2, nne::max17050::detail::DeviceMemoryMap::ModelAccess2Detail::Unlock));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::LockVfSoc0AndQh0WriteAccess() NN_NOEXCEPT
{
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::VfSoc0AndQh0WriteAccess, nne::max17050::detail::DeviceMemoryMap::VfSoc0AndQh0WriteAccessDetail::Lock));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::UnlockVfSoc0AndQh0WriteAccess() NN_NOEXCEPT
{
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::VfSoc0AndQh0WriteAccess, nne::max17050::detail::DeviceMemoryMap::VfSoc0AndQh0WriteAccessDetail::Unlock));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::UpdateModel(const uint16_t modelTable[]) NN_NOEXCEPT
{
    for (uint8_t i = 0; i < nne::max17050::detail::ModelSize; i++)
        NN_RESULT_DO(Write(nne::max17050::detail::ModelStartAddress + i, modelTable[i]));

    NN_RESULT_SUCCESS;
}

bool FuelgaugeDriver::IsModelUpdated(const uint16_t modelTable[]) NN_NOEXCEPT
{
    uint16_t data;
    bool isWrittenCorrectly = true;

    for (uint8_t i = 0; i < nne::max17050::detail::ModelSize; i++)
    {
        Read(nne::max17050::detail::ModelStartAddress + i, &data);
        isWrittenCorrectly = isWrittenCorrectly && (data == modelTable[i]);
    }

    return isWrittenCorrectly;
}

nn::Result FuelgaugeDriver::Read(const uint8_t addr, uint16_t *data) NN_NOEXCEPT
{
    return ReadI2cRegister(reinterpret_cast<uint8_t*>(data), m_I2cSession, &addr, sizeof(addr), sizeof(*data));
}

nn::Result FuelgaugeDriver::Write(const uint8_t addr, uint16_t data) NN_NOEXCEPT
{
    return WriteI2cRegister(m_I2cSession, &addr, sizeof(addr), reinterpret_cast<uint8_t*>(&data), sizeof(data));
}

nn::Result FuelgaugeDriver::Update(const uint8_t addr, uint16_t mask, uint16_t data) NN_NOEXCEPT
{
    uint16_t temp;

    NN_RESULT_DO(Read(addr, &temp));

    temp &= ~mask;
    temp |= (mask & data);

    NN_RESULT_DO(Write(addr, temp));

    NN_RESULT_SUCCESS;
}

bool FuelgaugeDriver::WriteAndVerify(const uint8_t addr, uint16_t data) NN_NOEXCEPT
{
    uint16_t temp;

    Write(addr, data);
    DelayMilliSeconds(3);
    Read(addr, &temp);

    return (data == temp);
}

void FuelgaugeDriver::DelayMilliSeconds(uint32_t milliSeconds) NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(milliSeconds));
}

nn::Result FuelgaugeDriver::SetShutdownTimer() NN_NOEXCEPT
{
    const uint8_t ShutdownTimerAddress = 0x3f;

    // SIGLO-65162: 1.6 時間（デフォルト値）
    const uint16_t ShutdownTimerValue = 0xE000;

    NN_RESULT_DO(Write(ShutdownTimerAddress, ShutdownTimerValue));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::IsI2cShutdownEnabled(bool* pOut) NN_NOEXCEPT
{
    uint16_t data;
    NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::Config, &data));

    auto shutdownConfig = data & nne::max17050::detail::DeviceMemoryMap::ConfigDetail::Mask_I2CSh;
    *pOut = (shutdownConfig == nne::max17050::detail::DeviceMemoryMap::ConfigDetail::I2CSh_I2CTimeoutToShutdown);

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::SetI2cShutdownEnabled(bool isEnabled) NN_NOEXCEPT
{
    auto shutdownConfig = nne::max17050::detail::DeviceMemoryMap::ConfigDetail::I2CSh_I2CTimeoutToNoShutdown;
    if (isEnabled)
    {
        shutdownConfig = nne::max17050::detail::DeviceMemoryMap::ConfigDetail::I2CSh_I2CTimeoutToShutdown;
    }

    NN_RESULT_DO(Update(nne::max17050::detail::DeviceMemoryMap::Config,
                nne::max17050::detail::DeviceMemoryMap::ConfigDetail::Mask_I2CSh,
                shutdownConfig));

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::IsBatteryRemoved(bool* pOut) NN_NOEXCEPT
{
    uint16_t data;
    Read(nne::max17050::detail::DeviceMemoryMap::Status, &data);

    data &= nne::max17050::detail::DeviceMemoryMap::StatusDetail::Mask_Bst;
    *pOut = (data == nne::max17050::detail::DeviceMemoryMap::StatusDetail::Bst_BatteryRemoved);

    NN_RESULT_SUCCESS;
}

nn::Result FuelgaugeDriver::IssueSoftPowerOnReset() NN_NOEXCEPT
{
    while (true)
    {
        // Lock the model and clear the POR bit
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::ModelAccess1, nne::max17050::detail::DeviceMemoryMap::ModelAccess1Detail::Lock));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::ModelAccess2, nne::max17050::detail::DeviceMemoryMap::ModelAccess2Detail::Lock));
        NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::Status, 0x0000));

        // Verify model lock
        uint16_t modelAccess1;
        uint16_t modelAccess2;
        uint16_t status;
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::ModelAccess1, &modelAccess1));
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::ModelAccess2, &modelAccess2));
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::Status, &status));
        if (modelAccess1 == nne::max17050::detail::DeviceMemoryMap::ModelAccess1Detail::Lock
            && modelAccess2 == nne::max17050::detail::DeviceMemoryMap::ModelAccess2Detail::Lock
            && status == 0x0000)
        {
            break;
        }
    }

    // Send SoftPOR
    NN_RESULT_DO(Write(nne::max17050::detail::DeviceMemoryMap::VfSoc0AndQh0WriteAccess, 0x000F));

    // Wait for at least 2ms
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2));

    const auto TrialCountMax = 10;
    auto trialCount = 0;
    while (true)
    {
        // Verify POR bit is set
        uint16_t status;
        NN_RESULT_DO(Read(nne::max17050::detail::DeviceMemoryMap::Status, &status));
        if ((status & nne::max17050::detail::DeviceMemoryMap::StatusDetail::Mask_POR) != 0)
        {
            break;
        }

        ++trialCount;
        if (trialCount >= TrialCountMax)
        {
            // Regarded as a success if the device seems SDEV.
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_SUCCESS;
}

}} // namespace nn::boot
