﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/gpio/driver/gpio.h>
#include <nn/gpio/driver/gpio_PadAccessor.h>
#include <nn/i2c/driver/i2c.h>
#include <nne/bq2419x/bq2419x.h>

namespace nn { namespace boot {

class ChargerDriver
{
public:
    ChargerDriver() NN_NOEXCEPT
    {
        nn::i2c::driver::Initialize();
        nn::i2c::driver::OpenSession(&m_I2cSession, nn::i2c::I2cDevice_Bq24193);

        // TORIAEZU : Initialize() を呼ぶと割り込み用のスレッドを回してしまうので呼ばない
        // ドライバの動作には影響はない
        //nn::gpio::driver::Initialize();
        nn::gpio::driver::OpenSession(&m_ChargeEnableSession, ::nn::gpio::GpioPadName_BattChgEnableN);
        nn::gpio::driver::SetDirection(&m_ChargeEnableSession, nn::gpio::Direction_Output);
    }

    ~ChargerDriver() NN_NOEXCEPT
    {
        nn::gpio::driver::CloseSession(&m_ChargeEnableSession);
        //nn::gpio::driver::Finalize();

        nn::i2c::driver::CloseSession(m_I2cSession);
        nn::i2c::driver::Finalize();
    }

    nn::Result Initialize() NN_NOEXCEPT;
    nn::Result Initialize(bool initializeInputCurrentLimit) NN_NOEXCEPT;
    nn::Result SetChargeEnabled(bool isEnabled) NN_NOEXCEPT;
    nn::Result SetFastChargeCurrentLimit(uint32_t current) NN_NOEXCEPT;
    nn::Result SetChargeVoltageLimit(uint32_t voltage) NN_NOEXCEPT;
    nn::Result GetChargeStatus(nne::bq2419x::ChargeStatusType *chargeStatus);

private:
    nn::Result SwitchHiZ(bool enable) NN_NOEXCEPT;
    nn::Result SetInputCurrentLimit(uint32_t current) NN_NOEXCEPT;
    nn::Result GetInputCurrentLimit(uint32_t *current) NN_NOEXCEPT;
    nn::Result ResetI2CWatchdogTimer() NN_NOEXCEPT;
    nn::Result ConfigureCharger(nne::bq2419x::ChargerType chargerType) NN_NOEXCEPT;
    nn::Result SetMinimumSystemVoltageLimit(uint32_t voltage) NN_NOEXCEPT;
    nn::Result GetMinimumSystemVoltageLimit(uint32_t *voltage) NN_NOEXCEPT;
    nn::Result SetBoostModeCurrentLimit(uint32_t current) NN_NOEXCEPT;
    nn::Result GetBoostModeCurrentLimit(uint32_t *current) NN_NOEXCEPT;
    nn::Result GetFastChargeCurrentLimit(uint32_t *current) NN_NOEXCEPT;
    nn::Result Force20PercentChargeCurrent(bool enable) NN_NOEXCEPT;
    nn::Result SetPreChargeCurrentLimit(uint32_t current) NN_NOEXCEPT;
    nn::Result GetPreChargeCurrentLimit(uint32_t *current) NN_NOEXCEPT;
    nn::Result SetTerminationCurrentLimit(uint32_t current) NN_NOEXCEPT;
    nn::Result GetTerminationCurrentLimit(uint32_t *current) NN_NOEXCEPT;
    nn::Result GetChargeVoltageLimit(uint32_t *voltage) NN_NOEXCEPT;
    nn::Result SetI2CWatchdogTimerSetting(uint32_t seconds) NN_NOEXCEPT;
    nn::Result GetI2CWatchdogTimerSetting(uint32_t *seconds) NN_NOEXCEPT;
    nn::Result SwitchChargingSafetyTimer(bool enable) NN_NOEXCEPT;

    nn::Result Read(const uint8_t addr, uint8_t *data) NN_NOEXCEPT;
    nn::Result Write(const uint8_t addr, uint8_t data) NN_NOEXCEPT;
    nn::Result Update(const uint8_t addr, uint8_t mask, uint16_t data) NN_NOEXCEPT;

private:
    nn::i2c::driver::I2cSession m_I2cSession;
    nn::gpio::driver::GpioPadSession m_ChargeEnableSession;
};

}} // namespace nn::boot
