﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include "circ_buffer.h"

/*---------------------------------------------------------------------------*
 *
 *    Globals/Externs
 *    -- Variables/Functions --
 *
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 *
 *    Data types defined for this file
 *    -- Struct's, Typedef's, Enum's --
 *
 *---------------------------------------------------------------------------*/


/*******************************************************************************
 **
 ** Class       CircBufferC
 **
 ** Description class to handle buffering data efficiently between (possibly)
 ** many senders and one receiver
 **
 *******************************************************************************/

// for signaling events between writer and reader
nn::os::EventType CircBufferC::m_event;


//------------------------------------------------------------------------------
// initializer
//------------------------------------------------------------------------------
void CircBufferC::qInitialize(uint8_t* buffer, size_t size, const char* name)
{
    //NN_SDK_LOG("%s(%s)\n", __FUNCTION__, name);
    m_bufferSize = size;
    m_bufferPtr  = buffer;
    m_readPtr    = m_bufferPtr;
    m_writePtr   = m_bufferPtr;
    m_name       = name;
    m_minSpace   = size;

    nn::os::InitializeMutex(&m_mutex, false, 0);
    if (m_event._state == nn::os::EventType::State_NotInitialized) {
        NN_SDK_LOG("%s(%s) initializing event\n", __FUNCTION__, name);
        nn::os::InitializeEvent(&m_event, false, nn::os::EventClearMode_AutoClear);
    }
}

//------------------------------------------------------------------------------
// finalize the object
//------------------------------------------------------------------------------
void CircBufferC::qFinalize()
{
    //NN_SDK_LOG("%s %s\n", __FUNCTION__, m_name);
    m_bufferSize = 0;
    nn::os::FinalizeMutex(&m_mutex);
}

//------------------------------------------------------------------------------
// get if the queue is initialized
//------------------------------------------------------------------------------
bool CircBufferC::qIsInitialized() NN_NOEXCEPT
{
    return m_bufferSize != 0;
}

//------------------------------------------------------------------------------
// get the amount of writeable space
//------------------------------------------------------------------------------
size_t CircBufferC::qWriteableSize() const
{
    uint8_t *w = m_writePtr;
    uint8_t *r = m_readPtr;
    size_t s = 0;

    if (m_bufferSize == 0) {
        return 0;
    }

    if (r > w) {
        s = r - w - 1;
    } else {
        s = m_bufferSize - 1 - (w - r);
    }

    if (s >= m_bufferSize) {
        NN_SDK_LOG("[bluetooth] %s(%s): calc error, s=%d (w=%p r=%p, b=%p)\n",
                __FUNCTION__, m_name, s, w, r, m_bufferPtr);
        return 0;
    }

    return s;
}

//------------------------------------------------------------------------------
// helper function for qWrite
//------------------------------------------------------------------------------
bool CircBufferC::_write(uint8_t type, uint16_t size, void* dataPtr)
{
    HAL_PACKET_CONTEXT *hpc = (HAL_PACKET_CONTEXT*)m_writePtr;
    hpc->cbType     = type;
    hpc->dataLen    = size;
    hpc->createTime = nn::os::GetSystemTick();

    if (type == DUMMY_PACKET_TYPE) {
        //memset(hpc->buf, 0x55, size);
    } else {
        if (size && dataPtr) {
            memcpy(hpc->buf, dataPtr, size);
        }
        else if (size && !dataPtr) {
            NN_SDK_LOG("[bluetooth] %s(%s): NULL pointer, size=%d, type=0x%x\n", __FUNCTION__, m_name, size, type);
            return -1;
        }
    }

    uint8_t *newWrPtr = m_writePtr + size + sizeof(HAL_PACKET_CONTEXT);
    if (newWrPtr > m_bufferPtr + m_bufferSize) {
        NN_SDK_LOG("[bluetooth] %s(%s): pointer error\n", __FUNCTION__, m_name);
        return -1;
    } else if (newWrPtr == m_bufferPtr + m_bufferSize) {
        m_writePtr = m_bufferPtr;
    } else {
        m_writePtr = newWrPtr;
    }


    return 0;
}

//------------------------------------------------------------------------------
// add to the queue, with timestamp
// return 0 if successful, non-zero on failure (e.g. full)
//------------------------------------------------------------------------------
bool CircBufferC::qWrite(uint8_t cbType, uint16_t size, void* dataPtr)
{
    //NN_SDK_LOG("%s(%s)\n", __FUNCTION__, m_name);
    bool rval = -1;

    nn::os::LockMutex(&m_mutex);

    do {
        if (qWriteableSize() < (size_t)size + sizeof(HAL_PACKET_CONTEXT)) {
            NN_SDK_LOG("[bluetooth] %s(%s): buffer full (1)\n", __FUNCTION__, m_name);
            break;
        }

        // write dummy packet to fill buffer near the end and avoid wraps.
        // make sure there is enough room for this packet and one more header
        // (in case a dummy needs to be added).
        uint16_t remaining = m_bufferSize - (uint16_t)(m_writePtr - m_bufferPtr);
        if (size + 2 * sizeof(HAL_PACKET_CONTEXT) > remaining)
        {
            if (m_readPtr == m_bufferPtr) {
                NN_SDK_LOG("[bluetooth] %s(%s): write+pad would overflow\n", __FUNCTION__, m_name);
                break;
            }
            else {
                _write(DUMMY_PACKET_TYPE, remaining - sizeof(HAL_PACKET_CONTEXT), NULL);
            }
        }

        if (qWriteableSize() < (size_t)size + sizeof(HAL_PACKET_CONTEXT)) {
            NN_SDK_LOG("[bluetooth] %s(%s): buffer full (2)\n", __FUNCTION__, m_name);
            break;
        }

        _write(cbType, size, dataPtr);
        rval = 0;

        size_t s = qWriteableSize();
        if (s + 1000 < m_minSpace) {
            m_minSpace = s;
            NN_SDK_LOG("[bluetooth] %s(%s): new minimum reserve = %d\n", __FUNCTION__, m_name, m_minSpace);
        }

    } while(0);

    nn::os::UnlockMutex(&m_mutex);

    nn::os::SignalEvent(&m_event);

    return rval;
}

//------------------------------------------------------------------------------
// help function for qRead
//------------------------------------------------------------------------------
void* CircBufferC::_read()
{
    //NN_SDK_LOG("%s(%s): %p %p\n", __FUNCTION__, m_name, m_readPtr, m_writePtr);
    if (m_bufferSize == 0)
    {
        NN_SDK_LOG("[bluetooth] %s(%s): not initialized\n", __FUNCTION__, m_name);
        return NULL;
    }

    if (m_readPtr == m_writePtr) {
        //NN_SDK_LOG("%s: buffer empty\n", __FUNCTION__);
        return NULL;
    }

    HAL_PACKET_CONTEXT *hpc = reinterpret_cast<HAL_PACKET_CONTEXT*>(m_readPtr);

    // if this is a dummy filler packet at the end of the array, free it and return the next
    // qRead packet.  maximum one level of recursion.
    if (hpc->cbType == DUMMY_PACKET_TYPE) {
        //NN_SDK_LOG("%s(%s): Free dummy packet\n", __FUNCTION__,m_name);
        qFree();
        return _read();
    } else {
        //NN_SDK_LOG("%s(%s): %p %p\n", __FUNCTION__, m_name, m_readPtr, m_writePtr);
        return m_readPtr;
    }
}

//------------------------------------------------------------------------------
// read from the queue (get pointer to the data), without freeing.
// this function blocks if the queue is empty
// returns NULL on failure, non-NULL otherwise
//------------------------------------------------------------------------------
void* CircBufferC::qRead()
{
    //NN_SDK_LOG("%s(%s): %p %p\n", __FUNCTION__, m_name, m_readPtr, m_writePtr);
    if (m_bufferSize ==0)
    {
        NN_SDK_LOG("[bluetooth] %s(%s): not initialized\n", __FUNCTION__, m_name);
        return NULL;
    }

    return _read();
}

//------------------------------------------------------------------------------
// free a message from the queue
// return 0 if successful, non-zero on failure (e.g. empty)
//------------------------------------------------------------------------------
bool CircBufferC::qFree()
{
    //NN_SDK_LOG("%s(%s)\n", __FUNCTION__, m_name);
    if (m_readPtr == m_writePtr) {
        return -1;
    }

    HAL_PACKET_CONTEXT *hpc = (HAL_PACKET_CONTEXT*)m_readPtr;
    //NN_SDK_LOG("%s: %5d %2x %5d\n", __FUNCTION__, m_readPtr-m_bufferPtr, hpc->cbType, hpc->dataLen);

    if (m_readPtr + sizeof(HAL_PACKET_CONTEXT) + hpc->dataLen < m_bufferPtr + m_bufferSize) {
        m_readPtr += sizeof(HAL_PACKET_CONTEXT) + hpc->dataLen;
    } else {
        m_readPtr = m_bufferPtr;
    }

    return 0;
}


