﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "bluetooth_le.h"

#define BLE_WRITE_MAX BSA_BLE_MAX_ATTR_LEN

/*******************************************************************************
**
** Function:        BtHalLeClientRegister
**
** Description:     Register BLE client application
**
** Parameters:      [in] tBT_UUID uuid: an identifier of client application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientRegister(tBT_UUID uuid);

/*******************************************************************************
**
** Function:        BtHalLeClientDeregister
**
** Description:     Deregister BLE client application
**
** Parameters:      [in] tBSA_BLE_IF clientIf : identifier of client application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregister(tBSA_BLE_IF clientIf);

/*******************************************************************************
**
** Function:        BtHalLeClientDeregisterAll
**
** Description:     Deregister all client applications.
**                  Internally call BtHalLeClientDeregister for all client applications.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregisterAll();

/*******************************************************************************
**
** Function:        BtHalLeClientDiscoverService
**
** Description:     Discover services from profile cache.
**                  Currently, does not included Gatt characteristics neither descriptors.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       Currently, none
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDiscoverService(uint32_t connId, tBT_UUID filterUuid);

/*******************************************************************************
**
** Function:        BtHalLeClientReadCharacteristic
**
** Description:     Read a characteristic of a connected BLE server
**
** Parameters:      [in] uint32_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target characeteristic belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientReadCharacteristic(uint32_t connId,
                                                  tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                  tBT_UUID charUuid, uint8_t charInstId,
                                                  uint8_t authenticationType);

/*******************************************************************************
**
** Function         BtHalLeClientReadDescriptor
**
** Description:     Read a descriptor of a connected BLE server
**
** Parameters:      [in] uint32_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target descriptor belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the characteristic which the target descriptor belongs to
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] tBT_UUID descrUuid             : attribute UUID of the target descriptor
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientReadDescriptor(uint32_t connId,
                                              tBT_UUID srvcUuid, bool isPrimarySrvc,
                                              tBT_UUID charUuid,
                                              tBT_UUID descrUuid,
                                              uint8_t authenticationType);

/*******************************************************************************
**
** Function         BtHalLeClientWriteCharacteristic
**
** Description      Write to a characteristic of a connected BLE server
**
** Parameters:      [in] uint32_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target characeteristic belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] uint8_t* pData                 : data to write
** Parameters:      [in] uint16_t len                   : data length in byte
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
** Parameters:      [in] bool withResponse              : flag indicating if the operation waits a confirmation response from the server
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientWriteCharacteristic(uint32_t connId,
                                                   tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                   tBT_UUID charUuid, uint8_t charInstId,
                                                   const uint8_t* pData, uint16_t len,
                                                   uint8_t authenticationType,
                                                   bool withResponse);

/*******************************************************************************
**
** Function         BtHalLeClientWriteDescriptor
**
** Description:     Write to a descriptor of a connected BLE server
**
** Parameters:      [in] uint32_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target descriptor belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the characteristic which the target descriptor belongs to
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] tBT_UUID descrUuid             : attribute UUID of the target descriptor
** Parameters:      [in] uint8_t* pData                 : data to write
** Parameters:      [in] uint16_t len                   : data length in byte
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientWriteDescriptor(uint32_t connId,
                                               tBT_UUID srvcUuid, bool isPrimarySrvc,
                                               tBT_UUID charUuid,
                                               tBT_UUID descrUuid,
                                               const uint8_t* pData, uint16_t len,
                                               uint8_t authenticationType);

/*******************************************************************************
**
** Function:        BtHalLeClientRegisterNotification
**
** Description:     Register notification to be received to BLE stack
**
** Parameters:      [in] tBSA_BLE_IF clientIf   : identifier of client application
** Parameters:      [in] tBT_UUID srvcUuid      : attribute UUID of the service which the target characteristic belongs to
** Parameters:      [in] uint8_t srvcInstId     : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc     : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId     : instance ID of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientRegisterNotification(uint32_t connId,
                                                    tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                    tBT_UUID charUuid, uint8_t charInstId);

/*******************************************************************************
**
** Function:        BtHalLeClientDeregisterNotification
**
** Description:     Deregister notification not to be received from BLE stack
**
** Parameters:      [in] tBSA_BLE_IF clientIf   : identifier of client application
** Parameters:      [in] tBT_UUID srvcUuid      : attribute UUID of the service which the target characteristic belongs to
** Parameters:      [in] uint8_t srvcInstId     : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc     : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId     : instance ID of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregisterNotification(uint32_t connId,
                                                      tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                      tBT_UUID charUuid, uint8_t charInstId);

/*******************************************************************************
**
** Function:        BtHalLeClientDeregisterNotificationAll
**
** Description:     Deregister all notifications not to be received from BLE stack
**
** Parameters:      [in] tBSA_BLE_IF clientIf   : identifier of client application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregisterNotificationAll(tBSA_BLE_IF clientIf, const BluetoothAddress& serverBdAddr);

/*******************************************************************************
**
** Function:        BtHalLeClientSendIndicationRsp
**
** Description:     Response to an indication from a BLE server
**
** Parameters:      [in] uint16_t connId    : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid  : attribute UUID of the service which the target characteristic belongs to
** Parameters:      [in] uint8_t srvcInstId : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid  : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId : instance ID of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientSendIndicationRsp(uint32_t connId,
                                                 tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                 tBT_UUID charUuid, uint8_t charInstId);

/*******************************************************************************
**
** Function:        BtHalLeClientOpenConnection
**
** Description:     Connect to a BLE server
**
** Parameters:      [in] tBSA_BLE_IF clientIf       : identifier of client application
** Parameters:      [in] Btbdaddr* pServerBdAddr    : BD Address of the target server
** Parameters:      [in] bool isDirectConnection    : flag indicating if direct or background connection
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientOpenConnection(tBSA_BLE_IF clientIf,
                                              const Btbdaddr* pServerBdAddr,
                                              bool isDirectConnection);

/*******************************************************************************
**
** Function:        BtHalLeClientCancelOpenConnection
**
** Description:     Cancel connection to a BLE server
**
** Parameters:      [in] tBSA_BLE_IF clientIf       : identifier of client application
** Parameters:      [in] Btbdaddr* pServerBdAddr    : BD Address of the target server
** Parameters:      [in] bool isDirectConnection    : flag indicating if direct or background connection
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientCancelOpenConnection(tBSA_BLE_IF clientIf,
                                                    const Btbdaddr* pServerBdAddr,
                                                    bool isDirectConnection);

/*******************************************************************************
**
** Function:        BtHalLeClientCloseConnection
**
** Description:     Disconnect from a BLE server
**
** Parameters:      [in] uint16_t connId    : connection ID of the target server
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientCloseConnection(uint32_t connId);

/*******************************************************************************
**
** Function:        BtHalLeClientLoadAttributeCache
**
** Description:     Get GATT profile from a BLE server
**
** Parameters:      [in] uint16_t connId            : connection ID of the target server
** Parameters:      [in] Btbdaddr* pServerBdAddr    : BD Address of the target server
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeClientGattServiceDiscoveryCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientLoadAttributeCache(uint32_t connId, const Btbdaddr* pServerBdAddr);

/*******************************************************************************
**
** Function:        BtHalLeClientConfigureMtu
**
** Description:     Start updating MTU
**
** Parameters:      [in] uint16_t connId    : connection ID of the target server
** Parameters:      [in] size_t mtu         : Prefferred MTU in Byte
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeClientMtuConfigurationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientConfigureMtu(uint32_t connId, size_t mtu);

/*******************************************************************************
**
** Function:        BtHalLeClientAddScanFilterCondition
**
** Description:     Add a scan filter condition to a scan filter
**
** Parameters:      [in] uint8_t filterIndex    : target filter index
** Parameters:      [in] BleAdType adType       : AD Type of filter condition
** Parameters:      [in] uint8_t* data          : an array of filter value
** Parameters:      [in] uint8_t dataLength     : length of filter value
** Parameters:      [in] uint8_t* mask          : mask for filter value
** Parameters:      [in] uint8_t maskLength     : length of mask
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientAddScanFilterCondition(uint8_t filterIndex, BleAdType adType, const uint8_t* data, uint8_t dataLength, const uint8_t* mask, uint8_t maskLength);

/*******************************************************************************
**
** Function:        BtHalLeClientDeleteScanFilterCondition
**
** Description:     Delete a scan filter condition from a scan filter
**
** Parameters:      [in] uint8_t filterIndex    : target filter index
** Parameters:      [in] BleAdType adType       : AD Type of filter condition
** Parameters:      [in] uint8_t* data          : an array of filter value
** Parameters:      [in] uint8_t dataLength     : length of filter value
** Parameters:      [in] uint8_t* mask          : mask for filter value
** Parameters:      [in] uint8_t maskLength     : length of mask
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeleteScanFilterCondition(uint8_t filterIndex, BleAdType adType, const uint8_t* data, uint8_t dataLength, const uint8_t* mask, uint8_t maskLength);

/*******************************************************************************
**
** Function:        BtHalLeClientDeleteScanFilter
**
** Description:     Delete a scan filter. Until disabling scan filter, any advertising packets will be filtered out.
**
** Parameters:      [in] uint8_t filterIndex    : target filter index
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeleteScanFilter(uint8_t filterIndex);

/*******************************************************************************
**
** Function:        BtHalLeClientClearScanFilters
**
** Description:     Clear all the scan filters. Until disabling scan filter, any advertising packets will be filtered out.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientClearScanFilters();

/*******************************************************************************
**
** Function:        BtHalLeClientEnableScanFilter
**
** Description:     Enable/Disable scan filters.
**                  Enabling scan filter with no conditions will filter outs any advertising packets
**                  Disabling scan filter let any advertising be passed.
**
** Parameters:      [in] bool enable    : flag indicating if enable/disable scan filter
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientEnableScanFilter(bool enable);

/*******************************************************************************
**
** Function:        BtHalLeGetFirstCharacteristic
**
** Description:     Get the first characteristic from a service
**
** Parameters:      [in] tBSA_BLE_CL_GET_FIRST_CHAR bsaFirstChar
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetFirstCharacteristic(tBT_UUID* pOutFirstCharUuid, uint8_t* pOutCharInstId, uint8_t* pOutProperty,
                                                uint32_t connId,
                                                tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                                tBT_UUID filterCharUuid);

/*******************************************************************************
**
** Function:        BtHalLeGetNextCharacteristic
**
** Description:     Get the next characteristic from a service
**
** Parameters:      [in] tBSA_BLE_CL_GET_NEXT_CHAR bsaNextChar
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetNextCharacteristic(tBT_UUID* pOutCharUuid, uint8_t* pOutCharInstId, uint8_t* pOutProperty,
                                               uint32_t connId,
                                               tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                               tBT_UUID previousCharUuid, uint8_t previousCharInstId, tBT_UUID filterCharUuid);

/*******************************************************************************
**
** Function:        BtHalLeGetFirstDescriptor
**
** Description:     Find the first descriptor of the
 **                 characteristic on the given server from cache.
**
** Parameters:      [in] tBSA_BLE_CL_GET_FIRST_CHAR_DESCR bsaFirstDescr
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetFirstDescriptor(tBT_UUID* pOutDescUuid,
                                            uint32_t connId,
                                            tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                            tBT_UUID charUuid, uint8_t charInstId,
                                            tBT_UUID filterDescUuid);

/*******************************************************************************
**
** Function:        BtHalLeGetNextDescriptor
**
** Description:     Find the next descriptor of the
 **                 characteristic on the given server from cache.
**
** Parameters:      [in] tBSA_BLE_CL_GET_NEXT_CHAR_DESCR bsaNextDescr
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetNextDescriptor(tBT_UUID* pOutDescUuid,
                                           uint32_t connId,
                                           tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                           tBT_UUID charUuid, uint8_t charInstId,
                                           tBT_UUID previousDescUuid, tBT_UUID filterDescUuid);

