﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/* For exit */
#include <cstdlib>

#include "bt_target.h"
#include "uipc.h"

extern "C"{
#include "bsa_api.h"
}

#include "bluetooth_hal.h"
#include "bluetooth_hal_hh.h"
#include "bluetooth_hal_utils.h"

#include <nn/bluetooth/bluetooth_TypesHal.h>
#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>
#include "bluetooth_queue.h"
#include "fs_wrapper.h"
#include "bluetooth.h"
#include "bluetooth_hid.h"

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>

/*---------------------------------------------------------------------------*
 *
 *    Globals/Externs
 *    -- Variables/Functions --
 *
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/
#ifndef APP_DEFAULT_UIPC_PATH
#define APP_DEFAULT_UIPC_PATH "./"
#endif

/*---------------------------------------------------------------------------*
 *
 *    Data types defined for this file
 *    -- Struct's, Typedef's, Enum's --
 *
 *---------------------------------------------------------------------------*/
static nn::os::ThreadType ptr_BSA_client_thread;
static enum { INIT_NOTDONE, INIT_DONE, INIT_FAIL, INIT_RESTART } initState = INIT_NOTDONE;

enum {BSA_CLIENT_STACK_SIZE = 16 * 1024};

static __attribute__((aligned(nn::os::StackRegionAlignment))) char ApplRobsonStack[BSA_CLIENT_STACK_SIZE];

/* Management Custom Callback */
typedef BOOLEAN (tAPP_MGT_CUSTOM_CBACK)(tBSA_MGT_EVT event, tBSA_MGT_MSG *p_data);
struct
{
    tAPP_MGT_CUSTOM_CBACK* mgt_custom_cback;
} app_mgt_cb;

/*---------------------------------------------------------------------------*
 *
 *    File scope functions/symbols defined for this file
 *    -- Function Prototypes, File Scope Data --
 *
 *---------------------------------------------------------------------------*/
static void _BtHalIntBsaCoreClientInit(void *p_param);

/*******************************************************************************
 **
 ** Function         _BtHalIntInitBsaClientCore
 **
 ** Description      ROBSON initialization function
 **
 ** Returns          int
 **
 *******************************************************************************/
static int _BtHalIntInitBsaClientCore()
{
    memset(&app_robson_cb, 0, sizeof(app_robson_cb));
    app_robson_init_xml();
    memset(&app_mgt_cb, 0, sizeof(app_mgt_cb));
    return 0;
}

/*******************************************************************************
 **
 ** Function         _BtHalIntStartBsaClientCore
 **
 ** Description      ROBSON start function
 **
 ** Returns          int
 **
 *******************************************************************************/
static int _BtHalIntStartBsaClientCore()
{
    /* Init specific robson */
    if (app_robson_set_config() < 0)
    {
        NN_SDK_LOG("[bluetooth] Unable to apply ROBSON configuration\n");
        return -1;
    }

    /* BT configuration from XML file */
    if (app_robson_apply_xml() < 0)
    {
        NN_SDK_LOG("[bluetooth] Unable to apply XML configuration\n");
        return -1;
    }

    return 0;
}

/*******************************************************************************
 **
 ** Function         _BtHalIntSetBluetoothSecurity
 **
 ** Description      Security callback
 **
 ** Parameters
 **
 ** Returns          0 if success/-1 otherwise
 **
 *******************************************************************************/
int BtHalSetBluetoothSecurity()
{
    tBSA_STATUS             status;
    int                     ret_val;
    tBSA_SEC_SET_SECURITY   set_security;
    tAPP_XML_CONFIG xml_local_config;

    BTHAL_BSA_DEBUG("Set Bluetooth Security");

    /* Read IO cap configuration */
    ret_val = app_read_xml_config(&xml_local_config);
    if (ret_val < 0)
    {
        BTHAL_BSA_DEBUG("app_read_xml_config failed: %d", ret_val);
        return -1;
    }

    /* Update security control block */
    strncpy((char*) app_sec_cb.pin_code,(char*) xml_local_config.pin_code,sizeof(app_sec_cb.pin_code));
    app_sec_cb.pin_code[sizeof(app_sec_cb.pin_code) - 1] = 0;
    app_sec_cb.pin_len = xml_local_config.pin_len;

    BSA_SecSetSecurityInit(&set_security);
    set_security.simple_pairing_io_cap = xml_local_config.io_cap;

    set_security.sec_cback = (tBSA_SEC_CBACK *)BtHalCallbackSecurity;
    set_security.ssp_debug = app_sec_cb.ssp_debug;

    status = BSA_SecSetSecurity(&set_security);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] BSA_SecSetSecurity failed: Unable to Set Security status:%d\n", status);
        return -1;
    }
    return 0;
}


/*******************************************************************************
 **
 ** Function         _BtHalIntTmReadVersion
 **
 ** Description      This function is used to read version
 **
 ** Parameters
 **
 ** Returns          int
 **
 *******************************************************************************/
static int _BtHalIntTmReadVersion()
{
    tBSA_TM_READ_VERSION bsa_read_version;
    tBSA_STATUS bsa_status;

    bsa_status = BSA_TmReadVersionInit(&bsa_read_version);
    bsa_status = BSA_TmReadVersion(&bsa_read_version);
    if (bsa_status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] BSA_TmReadVersion failed status:%d\n", bsa_status);
        return(-1);
    }

    BTHAL_BSA_DEBUG("Server status:%d", bsa_read_version.status);
    BTHAL_BSA_DEBUG("FW Version:%d.%d.%d.%d",
            bsa_read_version.fw_version.major,
            bsa_read_version.fw_version.minor,
            bsa_read_version.fw_version.build,
            bsa_read_version.fw_version.config);
    BTHAL_BSA_DEBUG("BSA Server Version:%s", bsa_read_version.bsa_server_version);
    return 0;
}

/*******************************************************************************
 **
 ** Function        _BtHalIntOpenBsaServerCallback
 **
 ** Description     This callback function is called in case of server
 **                 disconnection (e.g. server crashes)
 **
 ** Parameters      event: Management event
 **                 p_data: parameters
 **
 ** Returns         BOOLEAN: True if the generic callback must return (do not handle the event)
 **                          False if the generic callback must handle the event
 **
 *******************************************************************************/
static BOOLEAN _BtHalIntOpenBsaServerCallback(tBSA_MGT_EVT event, tBSA_MGT_MSG *p_data)
{
    switch(event)
    {
    case BSA_MGT_STATUS_EVT:
        BTHAL_BSA_DEBUG("BSA_MGT_STATUS_EVT");
        if (p_data->status.enable == FALSE)
        {
            BTHAL_BSA_DEBUG("\tBluetooth Stopped");

            /* Signal that we can complete the shutdown process */
            nn::bluetooth::serverShutdownDone = true;
        }
        else
        {
            BTHAL_BSA_DEBUG("\tBluetooth restarted => re-start the application");
            _BtHalIntStartBsaClientCore();
        }

        break;
    case BSA_MGT_DISCONNECT_EVT:
        /* Connection with the Server lost => Application will have to reconnect */
        BTHAL_BSA_DEBUG("BSA_MGT_DISCONNECT_EVT reason:%d", p_data->disconnect.reason);

        switch(p_data->disconnect.reason)
        {
            case BSA_ERROR_ROBSON_DEV_STATUS_DOWN:
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::ResultInternalDeviceDown());
                break;
            case BSA_ERROR_ROBSON_DEV_STATUS_CMD_TOUT:
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::ResultInternalTimeout());
                break;
            case BSA_ERROR_ROBSON_DEV_STATUS_HW_ERR_EVT:
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::ResultInternalHwError());
                break;
            default:
                NN_ABORT("[bluetooth] Fatal error occurred\n");
                break;
        }

        // Only signal the upper layer the need to restart due to fatal error once
        if (initState != INIT_RESTART)
        {
            BluetoothHalInsertToQueue(&p_data->disconnect,
                                      sizeof(tBSA_MSG_DISCONNECT_MSG),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_FATAL_ERROR);
            initState = INIT_RESTART;
        }
        break;
    default:
        break;
    }

    /* Do not execute generic Management callback */
    return TRUE;
}


/*******************************************************************************
 **
 ** Function        _BtHalIntOpenBsaServerCallbackMain
 **
 ** Description     This is an example of Generic Management callback function.
 **                 The Management Callback function is called in case of server
 **                 disconnection (e.g. server crashes) or when the Bluetooth
 **                 status changes (enable/disable)
 **
 ** Parameters      event: the event received (Status or Disconnect event)
 **                 p_data:associated data
 **
 ** Returns         void
 **
 *******************************************************************************/
static void _BtHalIntOpenBsaServerCallbackMain(tBSA_MGT_EVT event, tBSA_MGT_MSG *p_data)
{
    BOOLEAN exit_generic_cback= FALSE;

    /*  If Application provided its own custom callback */
    if(app_mgt_cb.mgt_custom_cback != NULL)
    {
        /* Call it */
        exit_generic_cback = app_mgt_cb.mgt_custom_cback(event, p_data);
    }

    /* If custom callback indicates that does not need the generic callback to execute */
    if(exit_generic_cback != FALSE)
    {
        return;
    }

    switch(event)
    {
    case BSA_MGT_STATUS_EVT:
        BTHAL_BSA_DEBUG("BSA_MGT_STATUS_EVT");
        if (p_data->status.enable == FALSE)
        {
            BTHAL_BSA_DEBUG("\tBluetooth Stopped");
        }
        else
        {
            BTHAL_BSA_DEBUG("\tBluetooth restarted");
        }
        break;

    case BSA_MGT_DISCONNECT_EVT:
        /* Connection with the Server lost => Application will have to reconnect */
        BTHAL_BSA_DEBUG("BSA_MGT_DISCONNECT_EVT reason:%d", p_data->disconnect.reason);
        break;
    default:
        break;
    }
}

/*******************************************************************************
 **
 ** Function        _BtHalIntOpenBsaServer
 **
 ** Description     Open communication to BSA Server
 **
 ** Parameters      uipc_path: path to UIPC channels (if NULL, points to CWD)
 **                 p_mgt_callback: Application's custom callback
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
static int _BtHalIntOpenBsaServer(const char *p_uipc_path, tAPP_MGT_CUSTOM_CBACK *p_mgt_callback)
{
    tBSA_MGT_OPEN bsa_open_param;
    tBSA_STATUS bsa_status;
    int i;

    BSA_MgtOpenInit(&bsa_open_param);

    /* If the application passed a NULL UIPC path, use the default one */
    if (p_uipc_path == NULL)
    {
        strncpy(bsa_open_param.uipc_path, APP_DEFAULT_UIPC_PATH, sizeof(bsa_open_param.uipc_path) - 1);
        bsa_open_param.uipc_path[sizeof(bsa_open_param.uipc_path) - 1] = '\0';
    }
    /* Else, use the given path */
    else
    {
        strncpy(bsa_open_param.uipc_path, p_uipc_path, sizeof(bsa_open_param.uipc_path) - 1);
        bsa_open_param.uipc_path[sizeof(bsa_open_param.uipc_path) - 1] = '\0';
    }

    /* Use the Generic Callback */
    bsa_open_param.callback = _BtHalIntOpenBsaServerCallbackMain;

    /* Save the Application's custom callback */
    app_mgt_cb.mgt_custom_cback = p_mgt_callback;

    /* Let's try to connect several time */
    for (i = 0 ; i < 100 ; i++)
    {
        /* Connect to BSA Server */
        bsa_status = BSA_MgtOpen(&bsa_open_param);
        if (bsa_status == BSA_SUCCESS)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }
    if (bsa_status != BSA_SUCCESS)
    {
        return -1;
    }

    return 0;
}


/*******************************************************************************
 **
 ** Function        CloseBsaServer
 **
 ** Description     This function is used to closes the BSA connection
 **
 ** Returns         The status of the operation
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int BtHalBsaCloseServer()
{
    tBSA_MGT_CLOSE  bsa_close_param;
    tBSA_STATUS bsa_status;

    BSA_MgtCloseInit(&bsa_close_param);
    bsa_status = BSA_MgtClose(&bsa_close_param);
    if(bsa_status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] BSA_MgtClose failed status:%d\n", bsa_status);
        return -1;
    }
    return 0;
}

/*******************************************************************************
 **
 ** Function        BsaClientCoreInit
 **
 ** Description
 **
 **
 ** Parameters
 **
 **
 ** Returns
 **
 **
 *******************************************************************************/
int BtHalBsaCoreClientInit()
{
    initState = INIT_NOTDONE;

    nn::os::CreateThread(
        &ptr_BSA_client_thread, _BtHalIntBsaCoreClientInit, NULL, ApplRobsonStack,
        BSA_CLIENT_STACK_SIZE, NN_SYSTEM_THREAD_PRIORITY(bluetooth, BsaClient));
    nn::os::SetThreadNamePointer(&ptr_BSA_client_thread, NN_SYSTEM_THREAD_NAME(bluetooth, BsaClient));
    nn::os::StartThread(&ptr_BSA_client_thread);

    while (initState==INIT_NOTDONE)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }

    if (initState == INIT_FAIL)
    {
        NN_SDK_LOG("[bluetooth] %s: init failed\n", __func__);
        return -1;
    }

    BSA_RobsonRegisterCallback(BtHalCallbackExtension);

    return 0;
}

/*******************************************************************************
 **
 ** Function        BtHalBsaCloseClient
 **
 ** Description     Destroy the client thread
 **
 **
 ** parameters      None
 **
 **
 ** Returns         Void
 **
 **
 *******************************************************************************/
void BtHalBsaCloseClient()
{
    NN_SDK_LOG("[bluetooth] --Closing BSA client core--\n");

    /* Exit HH */
    BtHalBsaExitHh();

    nn::os::DestroyThread(&ptr_BSA_client_thread);
}

/*******************************************************************************
 **
 ** Function        BsaClientHidProfileInit
 **
 ** Description
 **
 **
 ** Parameters
 **
 **
 ** Returns
 **
 **
 *******************************************************************************/
void BtHalBsaHidClientInit(uint16_t interfaceVersion)
{
    NN_SDK_LOG("[bluetooth] --Launching BSA client HID profile---\n");

    BtHalBsaInitHh();

    if(interfaceVersion == 0)
    {
        NN_SDK_LOG("[bluetooth] %s: InterfaceVersion = 0. Pairing won't be saved\n", __func__);
    }
}

/*******************************************************************************
 **
 ** Function         _BtHalIntBsaCoreClientInit
 **
 ** Description      This is the main function
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
static void _BtHalIntBsaCoreClientInit(void *p_param)
{

    if (_BtHalIntInitBsaClientCore() < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: InitBsaClientCore failed\n", __func__);
        exit(-1);
    }

    /* Open connection to BSA Server */
    if (_BtHalIntOpenBsaServer(NULL, _BtHalIntOpenBsaServerCallback) < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to connect to server\n", __func__);
        initState = INIT_FAIL;
        return;
    }

    if (_BtHalIntStartBsaClientCore() < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: StartBsaClientCore failed\n", __func__);
        initState = INIT_FAIL;
        return;
    }

    /* Display versions */
    _BtHalIntTmReadVersion();

    initState = INIT_DONE;
}
