﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <cstring> // for memcpy

#include "bluetooth.h"
#include "bluetooth_hal.h"
#include "bluetooth_hid.h"
#include "bluetooth_ext.h"
#include "bluetooth_le_interface.h"
#include "bluetooth_hal_hh.h"
#include "bluetooth_InternalTypes.h"
#include "bluetooth_queue.h"
#include "bluetooth_le.h"
#include "bta_api.h"

namespace nn {
namespace bluetooth {


/*******************************************************************************
 Globals
 *******************************************************************************/
bool serverShutdownDone;

/*******************************************************************************
 **
 ** Function        InitializeBluetoothInterface
 **
 ** Description     Initialize the bluetooth core interface
 **
 ** Parameters      None
 **
 ** Returns         None
 **
 *******************************************************************************/
void BluetoothInterfaceC::InitializeBluetoothInterface() NN_NOEXCEPT
{
    BTHAL_IF_DEBUG("called");

    // Initialize our BT interface
    m_BluetoothInterface.size = sizeof(m_BluetoothInterface);

    m_BluetoothInterface.InitializeBluetooth            = InitializeBluetooth;
    m_BluetoothInterface.EnableBluetooth                = EnableBluetooth;
    m_BluetoothInterface.DisableBluetooth               = DisableBluetooth;
    m_BluetoothInterface.CleanupBluetooth               = CleanupBluetooth;
    m_BluetoothInterface.GetAdapterProperties           = GetAdapterProperties;
    m_BluetoothInterface.GetAdapterProperty             = GetAdapterProperty;
    m_BluetoothInterface.SetAdapterProperty             = SetAdapterProperty;
    m_BluetoothInterface.GetRemoteDeviceProperties      = GetRemoteDeviceProperties;
    m_BluetoothInterface.GetRemoteDeviceProperty        = GetRemoteDeviceProperty;
    m_BluetoothInterface.SetRemoteDeviceProperty        = SetRemoteDeviceProperty;
    m_BluetoothInterface.GetRemoteServiceRecord         = GetRemoteServiceRecord;
    m_BluetoothInterface.GetRemoteServices              = GetRemoteServices;
    m_BluetoothInterface.StartDiscovery                 = StartDiscovery;
    m_BluetoothInterface.CancelDiscovery                = CancelDiscovery;
    m_BluetoothInterface.CreateBond                     = CreateBond;
    m_BluetoothInterface.RemoveBond                     = RemoveBond;
    m_BluetoothInterface.CancelBond                     = CancelBond;
    m_BluetoothInterface.PinReply                       = PinReply;
    m_BluetoothInterface.SspReply                       = SspReply;
    m_BluetoothInterface.GetProfileInterface            = GetProfileInterface;
    m_BluetoothInterface.DutModeConfigure               = DutModeConfigure;
    m_BluetoothInterface.DutModeSend                    = DutModeSend;
    m_BluetoothInterface.LEtestMode                     = LEtestMode;
    m_BluetoothInterface.ConfigHciSnoopLog              = ConfigHciSnoopLog;
    m_BluetoothInterface.ReadEnergyInfo                 = ReadEnergyInfo;
    m_BluetoothInterface.EnableBluetoothBoostSetting   = EnableBluetoothBoostSetting;
    m_BluetoothInterface.IsBluetoothBoostEnabled        = IsBluetoothBoostEnabled;
    m_BluetoothInterface.EnableBluetoothAfhSetting     = EnableBluetoothAfhSetting;
    m_BluetoothInterface.IsBluetoothAfhEnabled          = IsBluetoothAfhEnabled;

}


/*******************************************************************************
 **
 ** Function        GetProfileInterface
 **
 ** Description     Get the interface for the requested profile id
 **
 ** Parameters      pProfileId - profile type
 **
 ** Returns         pointer to the profile interface
 **
 *******************************************************************************/
const void* BluetoothInterfaceC::GetProfileInterface(const char *pProfileId)
{
    if (*pProfileId != NULL && (!memcmp(BT_PROFILE_HIDHOST_ID,pProfileId,sizeof(BT_PROFILE_HIDHOST_ID))))
    {
        static nn::bluetooth::BluetoothHidInterfaceC bluetoothHidIf;
        auto& bluetoothHidInterface = bluetoothHidIf.GetBluetoothHidInterface();

        BTHAL_IF_DEBUG("called");

        // Initialize the HID interface
        bluetoothHidInterface.size = sizeof(bluetoothHidInterface);

        bluetoothHidInterface.InitializeHid     = BluetoothHidInterfaceC::InitializeHid;
        bluetoothHidInterface.connect           = BluetoothHidInterfaceC::connect;
        bluetoothHidInterface.disconnect        = BluetoothHidInterfaceC::disconnect;
        bluetoothHidInterface.VirtualUnplug     = BluetoothHidInterfaceC::VirtualUnplug;
        bluetoothHidInterface.SetInfo           = BluetoothHidInterfaceC::SetInfo;
        bluetoothHidInterface.GetProtocol       = BluetoothHidInterfaceC::GetProtocol;
        bluetoothHidInterface.SetProtocol       = BluetoothHidInterfaceC::SetProtocol;
        bluetoothHidInterface.GetReport         = BluetoothHidInterfaceC::GetReport;
        bluetoothHidInterface.SetReport         = BluetoothHidInterfaceC::SetReport;
        bluetoothHidInterface.SendData          = BluetoothHidInterfaceC::SendData;
        bluetoothHidInterface.WakeController    = BluetoothHidInterfaceC::WakeController;
        bluetoothHidInterface.AddPairedDevice   = BluetoothHidInterfaceC::AddPairedDevice;
        bluetoothHidInterface.GetPairedDevice   = BluetoothHidInterfaceC::GetPairedDevice;
        bluetoothHidInterface.CleanupHid        = BluetoothHidInterfaceC::CleanupHid;
        bluetoothHidInterface.GetLatestPlr      = BluetoothHidInterfaceC::GetLatestPlr;
        bluetoothHidInterface.GetChannelMap     = BluetoothHidInterfaceC::GetChannelMap;

        return &bluetoothHidInterface;
    }

    if (*pProfileId != NULL && (!memcmp(BT_PROFILE_EXTENSION,pProfileId,sizeof(BT_PROFILE_EXTENSION))))
    {
        static nn::bluetooth::BluetoothExtC BluetoothExtC;
        auto& bluetoothExtension = BluetoothExtC.GetBluetoothExtension();

        BTHAL_IF_DEBUG("EXTENSIONS called");

        // Initialize the Bluetooth Extension
        bluetoothExtension.size = sizeof(bluetoothExtension);

        bluetoothExtension.InitializeExtension         = BluetoothExtC::InitializeExtension;
        bluetoothExtension.HalIfSetTSI                 = BluetoothExtC::HalIfSetTSI;
        bluetoothExtension.HalIfExitTSI                = BluetoothExtC::HalIfExitTSI;
        bluetoothExtension.HalIfSetBurstMode           = BluetoothExtC::HalIfSetBurstMode;
        bluetoothExtension.HalIfExitBurstMode          = BluetoothExtC::HalIfExitBurstMode;
        bluetoothExtension.HalIfSetSfh                 = BluetoothExtC::HalIfSetSfh;
        bluetoothExtension.HalIfSetZeroRetran          = BluetoothExtC::HalIfSetZeroRetran;
        bluetoothExtension.HalIfSetMcMode              = BluetoothExtC::HalIfSetMcMode;
        bluetoothExtension.HalIfStartLlrMode           = BluetoothExtC::HalIfStartLlrMode;
        bluetoothExtension.HalIfExitLlrMode            = BluetoothExtC::HalIfExitLlrMode;
        bluetoothExtension.HalIfSetRadio               = BluetoothExtC::HalIfSetRadio;
        bluetoothExtension.HalIfSetVisibility          = BluetoothExtC::HalIfSetVisibility;
        bluetoothExtension.HalIfSetTbfcScan            = BluetoothExtC::HalIfSetTbfcScan;
        bluetoothExtension.HalIfGetPendingConnections  = BluetoothExtC::HalIfGetPendingConnections;

        return &bluetoothExtension;
    }

    if (*pProfileId != NULL && (!memcmp(BT_PROFILE_LOW_ENERGY,pProfileId,sizeof(BT_PROFILE_LOW_ENERGY))))
    {
        static nn::bluetooth::BluetoothLeInterfaceC BluetoothLeInterfaceC;
        auto& bluetoothLowEnergy = BluetoothLeInterfaceC.GetBluetoothLeInterface();

        BTHAL_IF_DEBUG("LOW ENERGY called");

        // Initialize Bluetooth Low Engergy
        bluetoothLowEnergy.size = sizeof(bluetoothLowEnergy);

        bluetoothLowEnergy.InitializeBle                     = BluetoothLeInterfaceC::InitializeBle;
        bluetoothLowEnergy.EnableBle                         = BluetoothLeInterfaceC::EnableBle;
        bluetoothLowEnergy.DisableBle                        = BluetoothLeInterfaceC::DisableBle;
        bluetoothLowEnergy.CleanupBle                        = BluetoothLeInterfaceC::CleanupBle;
        bluetoothLowEnergy.SetLeVisibility                   = BluetoothLeInterfaceC::SetLeVisibility;
        bluetoothLowEnergy.SetLeConnectionParameter          = BluetoothLeInterfaceC::SetLeConnectionParameter;
        bluetoothLowEnergy.SetLeDefaultConnectionParameter   = BluetoothLeInterfaceC::SetLeDefaultConnectionParameter;
        bluetoothLowEnergy.SetLeAdvertiseData                = BluetoothLeInterfaceC::SetLeAdvertiseData;
        bluetoothLowEnergy.SetLeAdvertiseParameter           = BluetoothLeInterfaceC::SetLeAdvertiseParameter;
        bluetoothLowEnergy.StartLeScan                       = BluetoothLeInterfaceC::StartLeScan;
        bluetoothLowEnergy.StopLeScan                        = BluetoothLeInterfaceC::StopLeScan;
        bluetoothLowEnergy.SetLeScanParameter                = BluetoothLeInterfaceC::SetLeScanParameter;
        bluetoothLowEnergy.AddLeScanFilterCondition          = BluetoothLeInterfaceC::AddLeScanFilterCondition;
        bluetoothLowEnergy.DeleteLeScanFilterCondition       = BluetoothLeInterfaceC::DeleteLeScanFilterCondition;
        bluetoothLowEnergy.DeleteLeScanFilter                = BluetoothLeInterfaceC::DeleteLeScanFilter;
        bluetoothLowEnergy.ClearLeScanFilters                = BluetoothLeInterfaceC::ClearLeScanFilters;
        bluetoothLowEnergy.EnableLeScanFilter                = BluetoothLeInterfaceC::EnableLeScanFilter;
        bluetoothLowEnergy.RegisterLeClient                  = BluetoothLeInterfaceC::RegisterLeClient;
        bluetoothLowEnergy.UnregisterLeClient                = BluetoothLeInterfaceC::UnregisterLeClient;
        bluetoothLowEnergy.UnregisterLeClientAll             = BluetoothLeInterfaceC::UnregisterLeClientAll;
        bluetoothLowEnergy.LeClientConnect                   = BluetoothLeInterfaceC::LeClientConnect;
        bluetoothLowEnergy.LeClientCancelConnection          = BluetoothLeInterfaceC::LeClientCancelConnection;
        bluetoothLowEnergy.LeClientDisconnect                = BluetoothLeInterfaceC::LeClientDisconnect;
        bluetoothLowEnergy.LeClientGetAttributes             = BluetoothLeInterfaceC::LeClientGetAttributes;
        bluetoothLowEnergy.LeClientDiscoverService           = BluetoothLeInterfaceC::LeClientDiscoverService;
        bluetoothLowEnergy.LeClientReadCharacteristic        = BluetoothLeInterfaceC::LeClientReadCharacteristic;
        bluetoothLowEnergy.LeClientReadDescriptor            = BluetoothLeInterfaceC::LeClientReadDescriptor;
        bluetoothLowEnergy.LeClientWriteCharacteristic       = BluetoothLeInterfaceC::LeClientWriteCharacteristic;
        bluetoothLowEnergy.LeClientWriteDescriptor           = BluetoothLeInterfaceC::LeClientWriteDescriptor;
        bluetoothLowEnergy.LeClientRegisterNotification      = BluetoothLeInterfaceC::LeClientRegisterNotification;
        bluetoothLowEnergy.LeClientDeregisterNotification    = BluetoothLeInterfaceC::LeClientDeregisterNotification;
        bluetoothLowEnergy.LeClientConfigureMtu              = BluetoothLeInterfaceC::LeClientConfigureMtu;
        bluetoothLowEnergy.RegisterLeServer                  = BluetoothLeInterfaceC::RegisterLeServer;
        bluetoothLowEnergy.UnregisterLeServer                = BluetoothLeInterfaceC::UnregisterLeServer;
        bluetoothLowEnergy.LeServerConnect                   = BluetoothLeInterfaceC::LeServerConnect;
        bluetoothLowEnergy.LeServerDisconnect                = BluetoothLeInterfaceC::LeServerDisconnect;
        bluetoothLowEnergy.CreateLeService                   = BluetoothLeInterfaceC::CreateLeService;
        bluetoothLowEnergy.StartLeService                    = BluetoothLeInterfaceC::StartLeService;
        bluetoothLowEnergy.AddLeCharacteristic               = BluetoothLeInterfaceC::AddLeCharacteristic;
        bluetoothLowEnergy.AddLeDescriptor                   = BluetoothLeInterfaceC::AddLeDescriptor;
        bluetoothLowEnergy.SetLeResponse                     = BluetoothLeInterfaceC::SetLeResponse;
        bluetoothLowEnergy.LeSendIndication                  = BluetoothLeInterfaceC::LeSendIndication;
        bluetoothLowEnergy.LeGetFirstCharacteristic          = BluetoothLeInterfaceC::LeGetFirstCharacteristic;
        bluetoothLowEnergy.LeGetNextCharacteristic           = BluetoothLeInterfaceC::LeGetNextCharacteristic;
        bluetoothLowEnergy.LeGetFirstDescriptor              = BluetoothLeInterfaceC::LeGetFirstDescriptor;
        bluetoothLowEnergy.LeGetNextDescriptor               = BluetoothLeInterfaceC::LeGetNextDescriptor;
        bluetoothLowEnergy.RegisterLeCoreDataPath            = BluetoothLeInterfaceC::RegisterLeCoreDataPath;
        bluetoothLowEnergy.RegisterLeHidDataPath             = BluetoothLeInterfaceC::RegisterLeHidDataPath;
        bluetoothLowEnergy.RegisterLeDataPath                = BluetoothLeInterfaceC::RegisterLeDataPath;
        bluetoothLowEnergy.UnregisterLeCoreDataPath          = BluetoothLeInterfaceC::UnregisterLeCoreDataPath;
        bluetoothLowEnergy.UnregisterLeHidDataPath           = BluetoothLeInterfaceC::UnregisterLeHidDataPath;
        bluetoothLowEnergy.UnregisterLeDataPath              = BluetoothLeInterfaceC::UnregisterLeDataPath;

        return &bluetoothLowEnergy;
    }

    return NULL;
} // NOLINT(impl/function_size)


/*******************************************************************************
 **
 ** Function        InitializeBluetooth
 **
 ** Description     Initialize the core Bluetooth HAL module. Here we enable
 **                 the Bluetooth hardware or any hardware related features.
 **
 ** Parameters      callbacks - user registered callbacks
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::InitializeBluetooth(BluetoothCallbacks* callbacks)
{
    BTHAL_IF_DEBUG("Bluetooth chip being initialized");

    /* provide the callback routines to the implemenation of this interface. */
    BtHalCallbackRegisterCore(callbacks);

    return 0;
}

/*******************************************************************************
 **
 ** Function        EnableBluetooth
 **
 ** Description     Enable the Bluetooth stack
 **
 ** Parameters      None
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::EnableBluetooth()
{
    BTHAL_IF_DEBUG("called");

    // proper shutdown doesn't happen yet, so before starting force reset and short delay
    bluetoothGpioPowerEnable(false);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    bluetoothGpioPowerEnable(true);

    /* create the Bluetooth HAL CB and HID queues */
    BluetoothHalCreateQueue(HAL_QUEUE_HID);
    BluetoothHalCreateQueue(HAL_QUEUE_CB);

    app_xml_init();

    /* initialize the BSA server */
    bcmServerInit();

    BtHalInitialize();

    /* initialize the BSA client core */
    int result = BtHalBsaCoreClientInit();

    // Shutdown bsa/bta server
    BtHalBsaStartHh();

    // Set the Bluetooth Security callback
    if (BtHalSetBluetoothSecurity() < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set security\n", __func__);
    };

    if (settings::system::IsBluetoothBoostEnabled())
    {
        if (BtHalSet10DbBoost() == 0)
        {
            NN_SDK_LOG("[bluetooth] %s: 10 dB boost enabled\n", __func__);
        }
        else
        {
            NN_SDK_LOG("[bluetooth] %s: Unable to set 10 dB boost\n", __func__);
        }
    }

    if (!settings::system::IsBluetoothAfhEnabled())
    {
        if(BtHalDisableAfh() == 0)
        {
            NN_SDK_LOG("[bluetooth] %s: Bluetooth AFH disabled\n", __func__);
        }
        else
        {
            NN_SDK_LOG("[bluetooth] %s: Unable to disable bluetooth AFH\n", __func__);
        }
    }

    return result;
}

/*******************************************************************************
 **
 ** Function        DisableBluetooth
 **
 ** Description     Disable the Bluetooth stack
 **
 ** Parameters      None
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::DisableBluetooth()
{
    BTHAL_IF_DEBUG("called");

    int count = 0;

    BluetoothHalDestroyQueue(HAL_QUEUE_HID);
    BluetoothHalDestroyQueue(HAL_QUEUE_CB);

    // Close HAL client
    BtHalBsaCloseClient();
    BtHalFinalize();

    serverShutdownDone = false;
    BTA_DisableBluetooth();
    // Wait for shutdown bsa/bta server to finish, exit after 5 seconds
    while(!serverShutdownDone && count++ < 500) {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    if(!serverShutdownDone) {
        NN_SDK_LOG("[bluetooth] %s: timed-out on BTA_DisableBluetooth\n", __func__);
    }

    // Close HAL server
    BtHalBsaCloseServer();

    // - close UIPC, serial, and GKI threads
    bcmServerShutdown();

    /* Power down bluetooth chip */
    bluetoothGpioPowerEnable(false);
    return 0;
}

/*******************************************************************************
 **
 ** Function        CleanupBluetooth
 **
 ** Description     Use this function to reset the hardware
 **
 ** Parameters      None
 **
 ** Returns         None
 **
 *******************************************************************************/
void BluetoothInterfaceC::CleanupBluetooth()
{
    BTHAL_IF_DEBUG("called");
    BluetoothCallbacks invalidateCallbacks;

    memset(&invalidateCallbacks, 0, sizeof(BluetoothCallbacks));
    invalidateCallbacks.size = sizeof(BluetoothCallbacks);
    BtHalCallbackRegisterCore(&invalidateCallbacks);

    NN_SDK_LOG("[bluetooth] CleanupBluetooth complete\n");
}

/*******************************************************************************
 **
 ** Function        GetAdapterProperties
 **
 ** Description     Get all the adapter (host) properties
 **
 ** Parameters      None
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::GetAdapterProperties(AdapterProperty *pAdapterProperty)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("called");

    if (pAdapterProperty == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: error: NULL pointer passed\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    status = BtHalGetAdapterProperties(pAdapterProperty);

    return status;
}

/*******************************************************************************
 **
 ** Function        GetAdapterProperty
 **
 ** Description     Get a single Adapter (host) property
 **
 ** Parameters      type - (e.g. BT_PROPERTY_ADAPTER_DISCOVERY_TIMEOUT)
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::GetAdapterProperty(BluetoothProperty *property)
{
    BluetoothHhStatus status = BTHH_OK;
    uint8_t receivedLen = 0;
    BTHAL_IF_DEBUG("called");

    if (property       != NULL &&
        property->pVal != NULL &&
        property->len  != 0)
    {
        switch(property->type)
        {
            case BT_PROPERTY_BDNAME:
                Btbdname hostName;
                status = BtHalGetHostBdName(&hostName, &receivedLen);

                if(receivedLen > property->len)
                {
                    /* only copy what we have room for */
                    receivedLen = property->len;
                    NN_SDK_LOG("[bluetooth] %s: warning: length of recieved name %d is greater than parameter length %d\n", __func__, receivedLen, property->len);
                }

                memcpy(property->pVal, hostName.name, receivedLen);
                break;

            case BT_PROPERTY_BDADDR:
                Btbdaddr hostAddr;
                status = BtHalGetHostBdAddr(&hostAddr);
                memcpy(property->pVal, hostAddr.address, property->len);
                break;

            case BT_PROPERTY_CLASS_OF_DEVICE:
                ClassOfDevice deviceClass;
                status = BtHalGetHostClassOfDevice(&deviceClass);
                memcpy(property->pVal, deviceClass.cod, property->len);
                break;

            case BT_PROPERTY_FEATURE_SET:
                uint8_t featureSet;
                status = BtHalGetHostFeatureSet(&featureSet);
                memcpy(property->pVal, &featureSet, property->len);
                break;

            default:
                status = BTHH_ERR_BAD_PARAM;
                NN_SDK_LOG("[bluetooth] %s: Warning: unsupported property. type=%d pVal=%p len= %d\n", __func__,
                      property->type, property->pVal, property->len);
                break;
        }
    }
    else
    {
        status = BTHH_ERR_BAD_PARAM;
    }
    return status;
}

/*******************************************************************************
 **
 ** Function        SetAdapterProperty
 **
 ** Description     Set a single Adapter(host) property. Usually this is only
 **                 used for GET BD_ADDR however for our system we need to set
 **                 the host bd_addr
 **
 ** Parameters      property - (e.g. BT_PROPERTY_ADAPTER_DISCOVERY_TIMEOUT)
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::SetAdapterProperty(const BluetoothProperty *property)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("called");

    if (property       != NULL &&
        property->pVal != NULL &&
        property->len  != 0)
    {
        switch(property->type)
        {
            case BT_PROPERTY_BDNAME:
                Btbdname hostName;
                memcpy(hostName.name, property->pVal, property->len);
                status = BtHalSetHostBdName(&hostName, property->len);
                break;

            case BT_PROPERTY_BDADDR:
                Btbdaddr hostAddr;
                memcpy(hostAddr.address, property->pVal, property->len);
                status = BtHalSetHostBdAddr(&hostAddr);
                break;

            case BT_PROPERTY_SUPER_BDADDR:
                Btbdaddr hostSuperAddr;
                memcpy(hostSuperAddr.address, property->pVal, property->len);
                status = BtHalSetHostSuperBdAddr(&hostSuperAddr);
                break;

            case BT_PROPERTY_CLASS_OF_DEVICE:
                ClassOfDevice deviceClass;
                memcpy(deviceClass.cod, property->pVal, property->len);
                status = BtHalSetHostClassOfDevice(&deviceClass);
                break;

            case BT_PROPERTY_FEATURE_SET:
                uint8_t featureSet;
                memcpy(&featureSet, property->pVal, property->len);
                status = BtHalSetHostFeatureSet(&featureSet);
                break;

            default:
                status = BTHH_ERR_BAD_PARAM;
                NN_SDK_LOG("[bluetooth] %s: Warning: unsupported property. type=%d pVal=%p len= %d\n", __func__,
                      property->type, property->pVal, property->len);
                break;
        }
    }
    else
    {
        status = BTHH_ERR_BAD_PARAM;
    }
    return status;
}

/*******************************************************************************
 **
 ** Function        GetRemoteDeviceProperties
 **
 ** Description     Get all the remote device properties
 **                 (e.g. BT_PROPERTY_ADAPTER_DISCOVERY_TIMEOUT |
 **                       BT_PROPERTY_ADAPTER_BONDED_DEVICES)
 **
 ** Parameters      pRemoteAddr - Bluetooth device address of the host device
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::GetRemoteDeviceProperties(Btbdaddr *pRemoteAddr)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        GetRemoteDeviceProperty
 **
 ** Description     Get a specific remote property
 **                 (e.g. BT_PROPERTY_REMOTE_FRIENDLY_NAME)
 **
 ** Parameters      pRemoteAddr - Bluetooth device address of remote device
 **                 type        - remote device property
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::GetRemoteDeviceProperty(Btbdaddr *pRemoteAddr, BluetoothPropertyType type)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        SetRemoteDeviceProperty
 **
 ** Description     Set a specific remote device property
 **                 (e.g. BT_PROPERTY_REMOTE_FRIENDLY_NAME)
 **
 ** Parameters      pRemoteAddr - Bluetooth device address of remote device
 **                 property    - Bluetooth device property
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::SetRemoteDeviceProperty(Btbdaddr *pRemoteAddr,const BluetoothProperty *property)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        GetRemoteServiceRecord
 **
 ** Description
 **
 ** Parameters      pRemoteAddr - Bluetooth device address of remote device
 **                 uuid        - Universally Unique Identifier
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::GetRemoteServiceRecord(Btbdaddr *pRemoteAddr, BluetoothUUID *uuid)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        GetRemoteServices
 **
 ** Description     Start SDP to get remote services
 **                 The Bluetooth SDP provides a means by which service applications
 **                 running on different Bluetooth enabled devices may discover each
 **                 other's existence, and exchange information to determine their
 **                 characteristics.
 **
 ** Parameters      pRemoteAddr   - Bluetooth device address of remote device
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::GetRemoteServices(Btbdaddr *pRemoteAddr)
{
    BTHAL_IF_DEBUG("called");
    BtHalGetRemoteServices(pRemoteAddr);
    return 0;
}

/*******************************************************************************
 **
 ** Function        StartDiscovery
 **
 ** Description     Start the discovery process
 **
 ** Parameters      None
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::StartDiscovery()
{
    BTHAL_IF_DEBUG("now starting discovery process (inquiry)");
    BtHalStartDiscovery();
    return 0;
}

/*******************************************************************************
 **
 ** Function        CancelDiscovery
 **
 ** Description     Cancel the discovery process
 **
 ** Parameters      None
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::CancelDiscovery()
{
    BTHAL_IF_DEBUG("now canceling the discovery process");
    BtHalCancelDiscovery();
    return 0;
}

/*******************************************************************************
 **
 ** Function        CreateBond
 **
 ** Description     Create a bond with the remote device
 **
 ** Parameters      pBdAddr   - Bluetooth device address of remote device
 **                 transport - UNUSED
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::CreateBond(const Btbdaddr *pBdAddr, int transport)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("now creating a bond");
    status = BtHalCreateBond(pBdAddr);
    return status;
}

/*******************************************************************************
 **
 ** Function        RemoveBond
 **
 ** Description     Remove a bond with the remote device
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::RemoveBond(const Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("now removing a bond");

    status = BtHalRemoveBond(pBdAddr);

    return status;
}

/*******************************************************************************
 **
 ** Function        CancelBond
 **
 ** Description     Cancel bond with the remote device
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::CancelBond(const Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;

    BTHAL_IF_DEBUG("called");
    status = BtHalCancelBond(pBdAddr);
    return status;
}

/*******************************************************************************
 **
 ** Function        PinReply
 **
 ** Description     Function to reply to pin request
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **                 accept  - 1 to accept, 0 to reject
 **                 pin_len - length of the pin code
 **                 pPinCode- pin code
 **
 **
 ** Returns         0 if success/-1 otherwise
 **
 *******************************************************************************/
int BluetoothInterfaceC::PinReply(const Btbdaddr *pBdAddr, uint8_t accept,uint8_t pin_len, BluetoothPinCode *pPinCode)
{
    BluetoothHhStatus status = BTHH_OK;

    BTHAL_IF_DEBUG("called");
    status = BtHalPinReply(pBdAddr,accept,pin_len,pPinCode);
    return status;
}

/*******************************************************************************
 **
 ** Function        SspReply
 **
 ** Description     Function used to accept/refuse Simple Pairing
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **                 variant - Just works,Numeric comparison or passkey entry
 **                 accept  - 1 to accept, 0 to reject
 **                 passkey - pass key provided
 **
 ** Returns         0 if success/-1 otherwise
 **
 *******************************************************************************/
int BluetoothInterfaceC::SspReply(const Btbdaddr *pBdAddr, BluetoothSspVariant variant,uint8_t accept, uint32_t passkey)
{
    BluetoothHhStatus status = BTHH_OK;

    BTHAL_IF_DEBUG("called");
    status = BtHalSspReply(pBdAddr,variant,accept,passkey);
    return status;
}

/*******************************************************************************
 **
 ** Function        DutModeConfigure
 **
 ** Description     Puts the device in test mode (DUT = Device under test)
 **
 ** Parameters      enable - 0 to disable test mode 1 to enable test mode
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::DutModeConfigure(uint8_t enable)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        DutModeSend
 **
 ** Description     Sends device test mode
 **
 ** Parameters      opcode - test mode
 **                 buffer - buffer of the test data
 **                 len    - length of the buffer
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::DutModeSend(uint16_t opcode, uint8_t *buffer, uint8_t len)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        LEtestMode
 **
 ** Description     Enable Low Energy test mode
 **
 ** Parameters      opcode - test mode
 **                 buffer - buffer of the test data
 **                 len    - length of the buffer
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::LEtestMode(uint16_t opcode, uint8_t *buffer, uint8_t len)
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

/*******************************************************************************
 **
 ** Function        ConfigHciSnoopLog
 **
 ** Description     Enable HCI snoop log
 **
 ** Parameters      enable - 1 to enable 0 to disable
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::ConfigHciSnoopLog(uint8_t enable)
{
    BTHAL_IF_DEBUG("called");
//    StartBtSnoop();
    return 0;
}

/*******************************************************************************
 **
 ** Function        ReadEnergyInfo
 **
 ** Description
 **
 ** Parameters      None
 **
 ** Returns         int
 **
 *******************************************************************************/
int BluetoothInterfaceC::ReadEnergyInfo()
{
    NN_SDK_LOG("[bluetooth] %s: CURRENTLY NOT SUPPORTED\n", __FUNCTION__);
    return 0;
}

void BluetoothInterfaceC::EnableBluetoothBoostSetting(bool enable)
{
    BTHAL_IF_DEBUG("called");

    nn::settings::system::SetBluetoothBoostEnabled(enable);

    return;
}

bool BluetoothInterfaceC::IsBluetoothBoostEnabled()
{
    BTHAL_IF_DEBUG("called");

    bool enabled = nn::settings::system::IsBluetoothBoostEnabled();

    return enabled;
}

void BluetoothInterfaceC::EnableBluetoothAfhSetting(bool enable)
{
    BTHAL_IF_DEBUG("called");

    nn::settings::system::SetBluetoothAfhEnabled(enable);

    return;
}

bool BluetoothInterfaceC::IsBluetoothAfhEnabled()
{
    BTHAL_IF_DEBUG("called");

    bool enabled = nn::settings::system::IsBluetoothAfhEnabled();

    return enabled;
}


} // bluetooth
} // nn

