﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <cstdio>
#include <cstring>
#include "fcntl.h"
#include <cstdlib>
#include "fs_wrapper.h"

enum {MAX_FILE_HANDLE=20};
enum {WRITE_BUF_SIZE=1024};

#if 0
#   define DEBUG_LOG(format,...) NN_SDK_LOG(format,##__VA_ARGS__)
#else
#   define DEBUG_LOG(format,...) (void*)0
#endif


#undef printf
extern "C" int printf(const char *FORMAT, ...)
{
    NN_SDK_LOG("Called the wrong printf! --------------------------------\n");
    return 0;
}

extern "C" int scanf(const char *FORMAT, ...)
{
    NN_SDK_LOG("Called the wrong scanf! ---------------------------------\n");
    return 0;
}

extern "C" FILE *fopen(const char* file, const char *mode)
{
    NN_SDK_LOG("Called the wrong fopen! ---------------------------------\n");
    return 0;
}

extern "C" int fclose(FILE *stream)
{
    NN_SDK_LOG("Called the wrong fclose! ---------------------------------\n");
    return 0;
}

extern "C" int unlink(const char *filename)
{
    NN_SDK_LOG("Called the wrong unlink! ---------------------------------\n");
    return 0;
}



typedef struct
{
    unsigned char used;
    nn::fs::FileHandle fileHandle;
    int64_t offset;
    int8_t *buffer;
    int64_t boffset;
} tFileHandle_Entry;

static tFileHandle_Entry filehandle_cb[MAX_FILE_HANDLE];

static enum { I_NONE, I_ALLOC, I_ALLOC_AND_MOUNT } initialized = I_NONE;

static bool inline is_handle_valid(int fd)
{
    if((fd < 0) || (fd >= MAX_FILE_HANDLE) || (!filehandle_cb[fd].used)) {
        NN_SDK_LOG("Invalid file handle: %d\n", fd);
        return false;
    }
    return true;
}


static void init_filehandle()
{
    int i;

    for (i = 0; i < MAX_FILE_HANDLE; i++)
    {
        filehandle_cb[i].used = false;
        filehandle_cb[i].offset = 0;
        filehandle_cb[i].buffer = NULL;
        filehandle_cb[i].boffset = 0;
    }
}

static int allocate_file_handle_index()
{
    int i;

    for (i = 0; i < MAX_FILE_HANDLE; i++)
    {
        if (filehandle_cb[i].used == false)
        {
            filehandle_cb[i].used = true;
            return i;
        }
    }
    return -1;
}

extern "C" int FileOpen(const char * file_Name, int mode)
{
    int m_Mode=0, fd;
    nn::Result result;

    if (initialized!=I_ALLOC_AND_MOUNT)
    {
        NN_SDK_LOG("%s of %s failed, FS not mounted\n", __FUNCTION__, file_Name);
        return -1;
    }

    fd = allocate_file_handle_index();
    if (fd == -1)
    {
        NN_SDK_LOG("Failed to open file %s, no handles\n", file_Name);
        return -1;
    }

    tFileHandle_Entry *fe = &filehandle_cb[fd];

    if (mode & O_CREAT)
    {
        m_Mode = nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend;

        // create folders up to the file - may not be necessary if the folders
        // already exist, but doesn't cause any harm
        char path[256];
        for (int i=0; file_Name[i] && i<sizeof(path); i++)
        {
            if ((file_Name[i] == '/') && (i > 3))
            {
                path[i] = 0;
                int r = CreateDirectory(path);
                DEBUG_LOG("%s fd=%d file=%s createDir=%s %s\n",
                        __FUNCTION__, fd, file_Name, path,
                        r >= 0 ? "passed" : "failed");
                NN_UNUSED(r);
            }
            path[i] = file_Name[i];
        }

        result = nn::fs::CreateFile(file_Name, 0);
        DEBUG_LOG("%s fd=%d create for file %s %s (%x)\n",
                __FUNCTION__, fd, file_Name,
                result.IsSuccess() ? "passed" : "failed",
                result.GetInnerValueForDebug() );
    }

    switch (mode & (O_RDONLY|O_WRONLY|O_RDWR))
    {
        case O_RDONLY:
            m_Mode |= nn::fs::OpenMode_Read;
            break;
        case O_WRONLY:
            m_Mode |= nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend;
            break;
        case O_RDWR:
            m_Mode |= nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend;
            break;
        default:
            m_Mode |= nn::fs::OpenMode_Read;
            break;
    }

#if 1
    // THIS CODE IS A WORK-AROUND FOR A PROBLEM IN THE FS AND SHOULD BE REMOVED
    //  - "Flush" is called for a read-only file which asserts.
    //  - Setting the mode to Read/Write prevents the assert.
    m_Mode |= nn::fs::OpenMode_Write;
    result = nn::fs::OpenFile(&fe->fileHandle, file_Name, m_Mode);
#endif

    if (result.IsFailure())
    {
        if (result.GetInnerValueForDebug() != 0x202)
        {
            DEBUG_LOG("Failed to open file %s (%x)\n", file_Name, result.GetInnerValueForDebug() );
        }
        fe->used = false;
        return -1;
    }

    fe->offset = 0;
    fe->buffer = (m_Mode && nn::fs::OpenMode_Write) ? (int8_t*)std::malloc(WRITE_BUF_SIZE) : NULL;
    fe->boffset = 0;

    DEBUG_LOG("%s succeeded - %s fd=%d, mode=%d %x\n", __FUNCTION__, file_Name, fd, m_Mode, fe->fileHandle);
    return fd;
}

extern "C" int FileClose(int fd)
{
    if (!is_handle_valid(fd))
    {
        NN_SDK_LOG("Failed to close file -- invalid fd (%d)\n", fd);
        return -1;
    }

    tFileHandle_Entry *fe = &filehandle_cb[fd];

    // write out anything in the output buffer
    if (fe->buffer)
    {
        if (fe->boffset)
        {
            nn::fs::WriteFile( fe->fileHandle, fe->offset, fe->buffer, fe->boffset, nn::fs::WriteOption());
        }
        std::free(fe->buffer);
        fe->buffer = NULL;
        nn::fs::FlushFile(fe->fileHandle);
    }

    nn::fs::CloseFile(fe->fileHandle);
    fe->used = false;
    DEBUG_LOG("%s succeeded fd=%d %x\n", __FUNCTION__, fd, fe->fileHandle);
    return 0;
}

extern "C" int FileWrite(int fd, const void *buffer, unsigned int size)
{
    if (!is_handle_valid(fd))
    {
        NN_SDK_LOG("%s failed invalid fd=%d\n", __FUNCTION__, fd);
        return -1;
    }

    unsigned int input_size = size;

    tFileHandle_Entry *fe = &filehandle_cb[fd];

    if (fe->buffer == NULL)
    {
        nn::fs::WriteFile(fe->fileHandle, fe->offset, buffer, size, nn::fs::WriteOption());
        fe->offset += size;
    }
    else
    {
        if (fe->boffset + size >= WRITE_BUF_SIZE)
        {
            // write the internal buffer
            //NN_SDK_LOG("writing boffset %lld\n", fe->boffset);
            nn::fs::WriteFile(fe->fileHandle, fe->offset, fe->buffer, fe->boffset, nn::fs::WriteOption());
            fe->offset += fe->boffset;
            fe->boffset = 0;

            // write the new data or copy to internal buffer
            if (size > WRITE_BUF_SIZE)
            {
                //NN_SDK_LOG("writing size %d\n", size);
                nn::fs::WriteFile(fe->fileHandle, fe->offset, buffer, size, nn::fs::WriteOption());
                fe->offset += size;
                size = 0;
            }
        }

        if (size)
        {
            memcpy(fe->buffer + fe->boffset, buffer, size);
            fe->boffset += size;
        }
    }

    return input_size;
}

extern "C" int FileRead(int fd, void* buffer, unsigned int size)
{
    nn::Result result;
    size_t readBytes;
    int64_t offset;

    if (!is_handle_valid(fd))
    {
        NN_SDK_LOG("%s failed invalid fd=%d\n", __FUNCTION__, fd);
        return -1;
    }

    offset = filehandle_cb[fd].offset;
    result = nn::fs::ReadFile(&readBytes, filehandle_cb[fd].fileHandle, offset, buffer, size);

    if (result.IsFailure())
    {
        NN_SDK_LOG("Failed to read file fd : %d (%x)\n", fd, result.GetInnerValueForDebug() );
        return -1;
    }
    filehandle_cb[fd].offset += readBytes;

    DEBUG_LOG("%s succeeded fd=%d size=%d, readbytes=%d\n", __FUNCTION__, fd, size, readBytes);
    return readBytes;
}

extern "C" int FileSize(int fd)
{
    nn::Result result;
    int64_t size;

    if (!is_handle_valid(fd))
    {
        NN_SDK_LOG("%s failed invalid fd=%d\n", __FUNCTION__, fd);
        return -1;
    }

    result = nn::fs::GetFileSize(&size, filehandle_cb[fd].fileHandle);
    if (result.IsFailure())
    {
        NN_SDK_LOG("Failed to read file size : %d\n", fd);
        return -1;
    }

    DEBUG_LOG("%s succeeded fd=%d - size=%lld\n", __FUNCTION__, fd, size);
    return size;
}

namespace {
    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }
    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
}

extern "C" int InitHostFileSystem()
{
    nn::Result result;
    if (initialized==I_ALLOC_AND_MOUNT)
    {
        return 1;
    }

    if (initialized==I_NONE)
    {
        nn::fs::SetAllocator(Allocate, Deallocate);
        init_filehandle();
        initialized = I_ALLOC;
    }

    result = nn::fs::MountHostRoot();
    if (result.IsFailure())
    {
        NN_SDK_LOG("%s Failed to mount the FS (%x)\n", __FUNCTION__, result.GetInnerValueForDebug() );
        return -1;
    }

    initialized = I_ALLOC_AND_MOUNT;
    DEBUG_LOG("%s succeeded\n", __FUNCTION__);
    return 1;
}

extern "C" void EndHostFileSystem()
{
    if (initialized==I_ALLOC_AND_MOUNT)
    {
        nn::fs::UnmountHostRoot();
        initialized = I_ALLOC;
        DEBUG_LOG("%s succeeded\n", __FUNCTION__);
    }
}

extern "C" int CreateDirectory(const char *name)
{
    if (initialized!=I_ALLOC_AND_MOUNT)
    {
        NN_SDK_LOG("%s failed, FS not mounted\n", __FUNCTION__);
        return -1;
    }

    nn::Result result;
    result = nn::fs::CreateDirectory(name);
    return result.IsSuccess() ? 0 : -1;
}

extern "C" int DeleteDirectory(const char *name)
{
    if (initialized!=I_ALLOC_AND_MOUNT)
    {
        NN_SDK_LOG("%s failed, FS not mounted\n", __FUNCTION__);
        return -1;
    }

    nn::Result result;
    result = nn::fs::DeleteDirectory(name);
    return result.IsSuccess() ? 0 : -1;
}

extern "C" int FileRemove(const char *file_Name)
{
    if (initialized!=I_ALLOC_AND_MOUNT)
    {
        NN_SDK_LOG("%s failed, FS not mounted\n", __FUNCTION__);
        return -1;
    }

    nn::Result result;
    nn::fs::DeleteFile(file_Name);
    return result.IsSuccess() ? 0 : -1;
}

extern "C" void FilePrintf(int fd, char *format, ...)
{
    va_list ap;
    char buffer[255];

    va_start(ap, format);
    vsnprintf(buffer, sizeof(buffer), format ,ap);
    va_end(ap);

    FileWrite(fd, buffer, strlen(buffer));
}

extern "C" char* FileGets(char *data, unsigned short *len, int fd)
{
    nn::Result result;
    char *p = data, buffer[512], *p_buffer;
    size_t readBytes;
    unsigned short length, total_length = 0;
    int64_t offset;

    if (!is_handle_valid(fd))
    {
        NN_SDK_LOG("%s failed invalid fd=%d\n", __FUNCTION__, fd);
        return NULL;
    }

    while (true)
    {
        offset = filehandle_cb[fd].offset;
        result = nn::fs::ReadFile(&readBytes, filehandle_cb[fd].fileHandle, offset, buffer, sizeof(buffer));
        if (result.IsFailure())
        {
            NN_SDK_LOG("Failed to read file fd : %d\n", fd);
            return NULL;
        }

        p_buffer = buffer;
        length = 0;
        while (length < readBytes)
        {
            *p++ = *p_buffer;
            length++;
            if (*p_buffer == '\n')
            {
               filehandle_cb[fd].offset += length;
               *p = '\0';
               total_length += length;
               *len = total_length;
               return data;
            }
            p_buffer++;
         }
        filehandle_cb[fd].offset += length;
     }
}

extern "C" void FileBegine(int fd)
{
    filehandle_cb[fd].offset = 0;
}

