﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Thread.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/spsm/spsm_ServiceName.h>
#include <nn/spsm/server/spsm_Server.h>
#include <nn/spsm/observer/spsm_Observer.h>
#include "am_PowerStateManager.h"

namespace nn { namespace am {

    namespace
    {
        const int SpsmSessionCountMax = 4;
        class SpsmServerManager
            : public nn::sf::HipcSimpleAllInOneServerManager<SpsmSessionCountMax, 1>
        {};
        SpsmServerManager g_SpsmServerManager;

        // Power State Handler Thread
        nn::os::ThreadType              g_IpcServerThread;
        NN_OS_ALIGNAS_THREAD_STACK char g_IpcServerThreadStack[0x2000];
        nn::os::ThreadType              g_EventHandlerThread;
        NN_OS_ALIGNAS_THREAD_STACK char g_EventHandlerThreadStack[0x4000];
        nn::os::ThreadType              g_PowerStateHandlerThread;
        NN_OS_ALIGNAS_THREAD_STACK char g_PowerStateHandlerThreadStack[0x2000];
    }   // anonymous namespace

    void StartSpsmHipcServer(nn::sf::SharedPointer<nn::spsm::detail::IPowerStateInterface> manager) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SpsmServerManager.RegisterObjectForPort(manager, SpsmSessionCountMax, nn::spsm::ServiceName));

        // IPC サーバの開始
        auto f = [](void* p_)
        {
            auto* p = reinterpret_cast<SpsmServerManager*>(p_);
            p->LoopAuto();
        };
        auto result = nn::os::CreateThread( &g_IpcServerThread,
                                            f,
                                            &g_SpsmServerManager,
                                            g_IpcServerThreadStack,
                                            sizeof(g_IpcServerThreadStack),
                                            NN_SYSTEM_THREAD_PRIORITY(spsm, IpcServer));
        nn::os::SetThreadNamePointer(&g_IpcServerThread, NN_SYSTEM_THREAD_NAME(spsm, IpcServer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::StartThread(&g_IpcServerThread);
    }

    void StartSpsmPowerStateMachine() NN_NOEXCEPT
    {
        bool skipPscWait = true;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::settings::fwdbg::GetSettingsItemValue(
                &skipPscWait, sizeof(skipPscWait), "systemstartup", "skip_psc_wait");
#endif // !NN_BUILD_CONFIG_OS_HORIZON

        if ( skipPscWait )
        {
            // 全プロセスが入っていない環境では, spsm が Psm のモジュール登録待ちをしないように設定する
            nn::spsm::server::InitializePowerStateMachine(nn::spsm::server::InitializeMode_NoWait);
        }
        else if (nn::pm::GetBootMode() == pm::BootMode_Maintenance)
        {
            nn::spsm::server::InitializePowerStateMachine(nn::spsm::server::InitializeMode_Maintenance);
        }
        else
        {
            nn::spsm::server::InitializePowerStateMachine(nn::spsm::server::InitializeMode_Normal);
        }
        // システム電源ステートマシンの開始
        auto f = [](void* p)
        {
            NN_UNUSED(p);
            nn::spsm::server::LoopPowerStateMachine();
        };
        auto result = nn::os::CreateThread( &g_PowerStateHandlerThread,
                                            f,
                                            nullptr,
                                            g_PowerStateHandlerThreadStack,
                                            sizeof(g_PowerStateHandlerThreadStack),
                                            NN_SYSTEM_THREAD_PRIORITY(spsm, PowerStateMachine));
        nn::os::SetThreadNamePointer(&g_PowerStateHandlerThread, NN_SYSTEM_THREAD_NAME(spsm, PowerStateMachine));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::StartThread(&g_PowerStateHandlerThread);
    }

    void StartSpsmEventHandler(nn::os::SystemEventType* pCecEventType) NN_NOEXCEPT
    {
        // イベントハンドラの開始
        auto f = [](void* p)
        {
            auto* pCecEventType = reinterpret_cast<nn::os::SystemEventType*>(p);
            nn::spsm::observer::LoopEventHandler(pCecEventType);
        };
        auto result = nn::os::CreateThread( &g_EventHandlerThread,
                                            f,
                                            pCecEventType,
                                            g_EventHandlerThreadStack,
                                            sizeof(g_EventHandlerThreadStack),
                                            NN_SYSTEM_THREAD_PRIORITY(spsm, EventHandler));
        nn::os::SetThreadNamePointer(&g_EventHandlerThread, NN_SYSTEM_THREAD_NAME(spsm, EventHandler));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::StartThread(&g_EventHandlerThread);
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // ユニットテストでプログラムを正常終了するために必要
    void StopSpsmEventHandler() NN_NOEXCEPT
    {
        nn::spsm::observer::StopEventHandler();
        nn::os::WaitThread(&g_EventHandlerThread);
    }
#endif

}}  // namespace nn::am

