﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>

#include <nn/init.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/apm/apm.h>
#include <nn/apm/apm_System.h>
#include <nn/omm/omm_Api.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/nfc/am/nfc_Am.h>
#include <nn/spsm/spsm_ShimInternal.h>
#include <nn/spsm/server/spsm_Server.h>
#include <nn/spsm/observer/spsm_Observer.h>
#include <nn/account/account_ApiForSystemServices.h>

#include <nn/applet/applet_ServiceName.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/btm/btm_Api.h>
#include <nn/lbl/lbl.h>
#include <nn/ae/ae_Types.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/settings/system/settings_Boot.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/cec/cec_Api.h>
#include <nn/grc/grc_Api.h>

#include <nn/am/service/am_ServiceName.h>
#include <nn/am/service/am_AppletService.h>
#include <nn/am/service/am_AppletProxyService.sfdl.h>
#include <nn/am/service/am_SystemAppletProxyService.sfdl.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/service/am_GpuErrorControl.h>
#include <nn/am/service/am_GpuResourceControl.h>
#include <nn/am/service/am_OverlayNotificationControl.h>
#include <nn/am/service/am_PlayDataManager.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/am_SystemReportManager.h>
#include <nn/am/service/am_Monitoring.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/am/service/am_ContentActionTable.h>

#include <nn/diag/diag_AbortObserver.h>

#include "am_AppletManager.h"
#include "am_LaunchApplets.h"
#include "am_IdleStateManager.h"
#include "am_PowerStateManager.h"
#include "am_TcapServer.h"
#include "am_ScreenShotApplicationServer.h"

#include "am_OperationModeManagerMain.h"
#include "../../Libraries/am/service/am_OmmEvent.h"

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_ResultHandler.h>

using namespace nn;

NN_ALIGNAS(4096) union
{
    uint8_t mallocBuffer[512 * 1024];
} g_MallocBufferAndStartupLogo;

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBufferAndStartupLogo.mallocBuffer, sizeof(g_MallocBufferAndStartupLogo.mallocBuffer));
}

extern "C" void nndiagStartup()
{
}

#ifdef NN_PROCESS_AM_USE_LR_REDIRECT
    void SetupLrRedirectForTest() NN_NOEXCEPT;
#endif

namespace
{
    class PowerStateInterfaceServerImpl
        : public spsm::server::ServerInterfaceImpl
        , public spsm::observer::ObserverInterfaceImpl
    {
    };
    sf::UnmanagedServiceObject<spsm::detail::IPowerStateInterface, PowerStateInterfaceServerImpl> g_PowerStateInterfaceServer;

    nn::am::service::AppletService g_AppletService;
    nn::sf::UnmanagedServiceObjectByPointer<nn::am::service::IApplicationProxyService, decltype(g_AppletService)> g_ApplicationProxyService(&g_AppletService);
    nn::sf::UnmanagedServiceObjectByPointer<nn::am::service::IAllSystemAppletProxiesService, decltype(g_AppletService)> g_AllSystemAppletProxiesService(&g_AppletService);

    struct MyHipcServerManagerOption
    {
        static const bool CanDeferInvokeRequest = false;
        static const int ObjectInSubDomainCountMax = 1000;
        static const size_t PointerTransferBufferSize = 0;
        static const int SubDomainCountMax = 16;
    };

    class MyHipcServerManager
        : public sf::HipcSimpleAllInOneServerManager<100, 2, MyHipcServerManagerOption> // セッション数等は、後程調整
    {
    };

    MyHipcServerManager g_MyHipcServerManager;

    void InitializeMonitoring() NN_NOEXCEPT
    {
        static os::ThreadType g_MonitoringThread;
        static NN_OS_ALIGNAS_THREAD_STACK char g_MonitoringThreadStack[16 * 1024];
        os::CreateThread(&g_MonitoringThread, [](void*)
        {
            NN_AM_SERVICE_DIAGNOSTICS_SET_CONTEXT_NAME(WatchDog);
            am::service::RunMonitor();
        }, nullptr, g_MonitoringThreadStack, sizeof(g_MonitoringThreadStack), NN_SYSTEM_THREAD_PRIORITY(am, WatchDog));
        os::StartThread(&g_MonitoringThread);
    }

}   // anonymous namespace

extern "C" void nnMain() NN_NOEXCEPT
{
    InitializeMonitoring();

    nv::SetGraphicsAllocator(
        [] (size_t size, size_t, void*) -> void*
        {
            return std::malloc(size);
        },
        [] (void* p, void*)
        {
            std::free(p);
        },
        [] (void* p, size_t size, void*)
        {
            return std::realloc(p, size);
        },
        nullptr);
    static NN_ALIGNAS(4096) uint8_t g_NvBuffer[1024 * 1024];

    // システムプロセス用のポートを指定して初期化
    nv::SetGraphicsServiceName("nvdrv:s");
    nv::InitializeGraphics(g_NvBuffer, sizeof(g_NvBuffer));

    // fs を複数セッションで初期化
    fs::InitializeWithMultiSessionForSystem();

    // am では fs shim レイヤでアボートしないように
    fs::SetEnabledAutoAbort(false);

    // am, omm, idle の全員が使用するのでここで初期化
    lbl::Initialize();

    // cec 初期化
    // 現在、 cec::Initialize() を一度しか呼べないという制約のため、ここで初期化している。
    // TODO: cec を利用する omm / spsm がそれぞれ独立で cec::Initialize() を呼んでも問題ないように、cec の shim を改善する。
    os::SystemEventType cecEventType;
    NN_ABORT_UNLESS_RESULT_SUCCESS(cec::Initialize(&cecEventType));

    am::service::SetConfigsByFirmwareDebugSettings();

    // ディスプレイ管理を初期化。初期状態はシステムモード。
    am::service::InitializeDisplayLayerControl();

    // NX ロゴ表示の開始
    am::InitializeOperationModeManagerForShowStartupLogo();
    am::ShowStartupLogoOfOperationModeManager();

#if defined(NN_PROCESS_AM_FOR_REPAIR)
    // NX ロゴ表示の終了をここで呼ぶ
    am::HideStartupLogoOfOperationModeManager();
    for (;;)
    {
        NN_SDK_LOG("[am] process is now stalled intentionally. \n");
        nn::os::SleepThread( nn::TimeSpan::FromDays(1) );
    }
#else
    // NX ロゴ表示終了のためのタイマーを発動
    // 起動が遅い場合にこの時間より長くロゴを表示することを保証するだけで、
    // この時間で確実にロゴが消えるわけではない。むしろ、EventHandler の
    // スレッドが動き出すまでに 3.5 秒程度かかるため、それ未満の時間を
    // 指定してもあまり意味がない。
    // ただし、これらの時間に到達する前に SA や OA の初期化が呼ばれた場合には、
    // その時点でロゴ表示は強制的に終了する。
    am::service::GetOmmEvent().StartStartupLogoTimer(TimeSpan::FromSeconds(5));

    account::InitializeForSystemService();
    am::service::InitializeGpuErrorControl();
    am::service::StartGpuResourceControl();
    am::service::InitializeOverlayNotification();

    // 無操作状態マネージャを開始
    am::StartIdleStateManager();

    // 高度な温度制御ポリシーサーバを開始
    // TCAP を DFC で初期化するために StartOperationModeManagerMain と spsm::InitializeWith より前でなければならない
    am::StartTcapServer();

    // spsm のために btm を初期化
    btm::InitializeBtmInterface();

    // spsm サーバを開始
    am::StartSpsmPowerStateMachine();
    am::StartSpsmEventHandler(&cecEventType);
    spsm::InitializeWith(g_PowerStateInterfaceServer.GetShared());
    am::StartSpsmHipcServer(g_PowerStateInterfaceServer.GetShared());

    am::RegisterOperationModeManager();
    am::StartOperationModeManagerMain();

#ifdef NN_PROCESS_AM_USE_LR_REDIRECT
    SetupLrRedirectForTest();
#endif

    hid::system::InitializeHomeButton();
    hid::system::InitializeSleepButton();
    hid::system::InitializeCaptureButton();
    hid::system::InitializeInputDetector();

    ns::Initialize();
    omm::Initialize();
    am::service::InitializePlayDataManager();
    am::service::StartSystemReportManager();

    // capsrv 接続の初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeScreenShotControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    am::service::SetupScreenShotApplicationServer();

    // grc 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(grc::InitializeByHipc());

    // nfc 初期化
    nfc::am::Initialize();

    // am 内 abort ハンドラの設定
    diag::AbortObserverHolder abortObserverHodler;
    diag::InitializeAbortObserverHolder(&abortObserverHodler, [] (const diag::AbortInfo&)
    {
        NN_AM_SERVICE_LOG(error, "abort in am process.\n");
        g_AppletService.GetAppletSystem().GetWindowManager()->NotifyFatal();
    });
    diag::RegisterAbortObserver(&abortObserverHodler);
    NN_UTIL_SCOPE_EXIT
    {
        diag::UnregisterAbortObserver(&abortObserverHodler);
    };

    // IPC ポートの登録
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_MyHipcServerManager.RegisterObjectForPort(g_ApplicationProxyService.GetShared(), 1, nn::am::service::AppletServiceNameForApplication));
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_MyHipcServerManager.RegisterObjectForPort(g_AllSystemAppletProxiesService.GetShared(), 9, nn::am::service::AppletServiceNameForSystem));

    // WindowManager スレッドの起動
    am::service::StartWindowManager(&g_AppletService);

    // 初期起動アプレットの起動（SA, OA）
    am::service::LaunchApplets(&g_AppletService);

    // EventHandler スレッドの起動
    // SA に対する通知があるため、SA が起動した後に動作させる。
    am::service::StartAppletManager(&g_AppletService);

    #ifdef NN_PROCESS_AM_USES_CONTENT_ACTION_TABLE_SYSTEM_DATA
        am::service::contentAction::UseSystemData("CAction");
    #endif

    // メインスレッドに対しては名前を変更しない
    //NN_AM_SERVICE_DIAGNOSTICS_SET_CONTEXT_NAME("IpcHandler(am.main)");
    g_MyHipcServerManager.Start();
    for (;;)
    {
        auto p = g_MyHipcServerManager.Wait();
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_Ipc_Loop, 60);
        g_MyHipcServerManager.ProcessAuto(p);
    }
#endif

}
