﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_AccountServer.h"

#include <nn/account/account_ServiceNames.h>
#include <nn/account/account_RuntimeResource.h>
#include <nn/account/account_RuntimeUtil.h>

#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace
{

/* サービススレッドの資源 ----------------------------------------------------------------------------*/
NN_ALIGNAS(4096) char g_AccountServerThreadStack[AccoutServerThreadStackSize];
nn::os::ThreadType g_AccountServerThread;

AccountServer* g_pAccountServer;
std::aligned_storage<sizeof(AccountServer), NN_ALIGNOF(AccountServer)>::type g_ServerStroage;
void AccountServerThreadFunction(void* p) NN_NOEXCEPT
{
    auto& server = *reinterpret_cast<AccountServer*>(p);
    server.LoopAuto();
}

/* Executorスレッドの資源 ----------------------------------------------------------------------------*/
NN_ALIGNAS(4096) char g_AccountExecThreadStackSys[AccountExecThreadStackSize];
nn::os::ThreadType g_AccountExecThreadSys;
NN_ALIGNAS(4096) char g_AccountExecThreadStackApp[AccountExecThreadStackSize];
nn::os::ThreadType g_AccountExecThreadApp;

template <size_t Size>
class Allocator
    : public nn::account::LargeBufferAllocator
{
private:
    typename std::aligned_storage<Size, std::alignment_of<std::max_align_t>::value>::type m_Buffer;
public:
    Allocator() NN_NOEXCEPT
        : LargeBufferAllocator(&m_Buffer, sizeof(m_Buffer))
    {
    }
};
util::optional<Allocator<512 * 1024>> g_pAllocator;

typedef nn::account::ExecutionManager<nn::account::Executor, nn::account::ExecutionResource> ExecutionManagerType;
struct ExecutionContext
{
    util::optional<ExecutionManagerType> pManager;
    std::aligned_storage<AccountExecThreadBufferSize, std::alignment_of<std::max_align_t>::value>::type buffer;
    nn::account::LargeBufferAllocator *pAllocator;
};
ExecutionContext g_ExecContextSys {};
ExecutionContext g_ExecContextApp {};


/* Executorスレッドの実装 ----------------------------------------------------------------------------*/
void AccountExecThreadFunction(void* p) NN_NOEXCEPT
{
    auto& context = *reinterpret_cast<ExecutionContext*>(p);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(&context.buffer) % std::alignment_of<std::max_align_t>::value == 0);

    /* 確保 -------------------------------------------------- */
    nn::account::ExecutionResource resource = { curl_easy_init(), { &context.buffer, sizeof(context.buffer) }, context.pAllocator };

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // CURL ハンドルのリフレッシュ
    nn::os::TimerEvent curlTimer(nn::os::EventClearMode_ManualClear);
    nn::os::MultiWaitHolderType curlTimerHolder;
    nn::os::InitializeMultiWaitHolder(&curlTimerHolder, curlTimer.GetBase());

    context.pManager->Link(&multiWait);
    nn::os::LinkMultiWaitHolder(&multiWait, &curlTimerHolder);

    /* 解放 -------------------------------------------------- */
    NN_UTIL_SCOPE_EXIT {
        nn::os::UnlinkMultiWaitHolder(&curlTimerHolder);
        context.pManager->Unlink();

        nn::os::FinalizeMultiWaitHolder(&curlTimerHolder);
        curlTimer.Stop();

        nn::os::FinalizeMultiWait(&multiWait);

        curl_easy_cleanup(resource.curlHandle);
    };

    while (NN_STATIC_CONDITION(true))
    {
        auto *pHolder = nn::os::WaitAny(&multiWait);

        if (pHolder == &curlTimerHolder)
        {
            curlTimer.Clear();
            curl_easy_cleanup(resource.curlHandle);
            resource.curlHandle = curl_easy_init();
            NN_ABORT_UNLESS(resource.curlHandle);
        }
        else
        {
            auto userData = nn::os::GetMultiWaitHolderUserData(pHolder);
            NN_SDK_ASSERT(userData == reinterpret_cast<uintptr_t>(&context.pManager.value()));
            NN_SDK_ASSERT(*context.pManager);
            NN_UNUSED(userData);

            if (!context.pManager->Execute(pHolder, resource))
            {
                context.pManager->Unlink();
                break;
            }
            curlTimer.StartOneShot(nn::TimeSpan::FromSeconds(10));
        }
    }
}

/* 各スレッドの初期化 --------------------------------------------------------------------------------*/
nn::Result InitializeServiceThread(AccountResourceType& resource) NN_NOEXCEPT
{
    g_pAccountServer = new(&g_ServerStroage) AccountServer(resource);

    NN_RESULT_DO(g_pAccountServer->InitializePort(
        AccountServerPort_Administrators,
        8,
        nn::account::ServiceNameForAdministrator));

    NN_RESULT_DO(g_pAccountServer->InitializePort(
        AccountServerPort_SystemServices,
        16,
        nn::account::ServiceNameForSystemService));

    NN_RESULT_DO(g_pAccountServer->InitializePort(
        AccountServerPort_Applications,
        4,
        nn::account::ServiceNameForApplication));

    NN_RESULT_DO(g_pAccountServer->InitializePort(
        AccountServerPort_BaasAccessTokenAccessors,
        4,
        nn::account::ServiceNameForBaasAccessTokenAccessor));

    NN_RESULT_DO(nn::os::CreateThread(
        &g_AccountServerThread,
        AccountServerThreadFunction, g_pAccountServer,
        g_AccountServerThreadStack, sizeof(g_AccountServerThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(account, IpcServer)));
    nn::os::SetThreadNamePointer(
        &g_AccountServerThread,
        NN_SYSTEM_THREAD_NAME(account, IpcServer));

    NN_RESULT_SUCCESS;
}

nn::Result InitializeExecutorThread(AccountResourceType& resource) NN_NOEXCEPT
{
    g_ExecContextSys.pManager.emplace(resource.GetExecutorForSystem());
    g_ExecContextApp.pManager.emplace(resource.GetExecutorForApplication());
    g_pAllocator.emplace();
    g_ExecContextSys.pAllocator = g_ExecContextApp.pAllocator = &g_pAllocator.value();

    NN_RESULT_DO(nn::os::CreateThread(
        &g_AccountExecThreadSys,
        AccountExecThreadFunction, &g_ExecContextSys,
        g_AccountExecThreadStackSys, sizeof(g_AccountExecThreadStackSys),
        NN_SYSTEM_THREAD_PRIORITY(account, Executor_Sys)));
    nn::os::SetThreadNamePointer(
        &g_AccountExecThreadSys,
        NN_SYSTEM_THREAD_NAME(account, Executor_Sys));

    NN_RESULT_DO(nn::os::CreateThread(
        &g_AccountExecThreadApp,
        AccountExecThreadFunction, &g_ExecContextApp,
        g_AccountExecThreadStackApp, sizeof(g_AccountExecThreadStackApp),
        NN_SYSTEM_THREAD_PRIORITY(account, Executor_App)));
    nn::os::SetThreadNamePointer(
        &g_AccountExecThreadApp,
        NN_SYSTEM_THREAD_NAME(account, Executor_App));

    NN_RESULT_SUCCESS;
}

} // ~namespace nn::<anonymous>

nn::Result StartAccountServer(AccountResourceType& resource) NN_NOEXCEPT
{
    // サービススレッド, Executor スレッドの初期化
    NN_RESULT_DO(InitializeServiceThread(resource));
    NN_RESULT_DO(InitializeExecutorThread(resource));

    // 開始
    nn::os::StartThread(&g_AccountExecThreadSys);
    nn::os::StartThread(&g_AccountExecThreadApp);
    nn::os::StartThread(&g_AccountServerThread);
    NN_RESULT_SUCCESS;
}

void StopAccountServer() NN_NOEXCEPT
{
    // 全スレッドに終了要求
    g_pAccountServer->RequestStop();
    g_ExecContextSys.pManager->RequestStop();
    g_ExecContextApp.pManager->RequestStop();

    // 終了待ち, 破棄
    nn::os::WaitThread(&g_AccountServerThread);
    nn::os::WaitThread(&g_AccountExecThreadApp);
    nn::os::WaitThread(&g_AccountExecThreadSys);
    nn::os::DestroyThread(&g_AccountServerThread);
    nn::os::DestroyThread(&g_AccountExecThreadApp);
    nn::os::DestroyThread(&g_AccountExecThreadSys);

    // リソース解放
    g_ExecContextApp.pManager = util::nullopt;
    g_ExecContextSys.pManager = util::nullopt;
    g_pAllocator = util::nullopt;
    g_pAccountServer->~AccountServer();
}

} // ~namespace nn
