﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/detail/account_Log.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "account_AccountDaemonSettings.h"
#include "account_Util.h"

namespace nn {

class AccountDaemonTaskScheduler
{
private:
    static const int32_t InitialScheduleDelaySeconds = 1 * 60; // default: 1 min

    static const int32_t DefaultSchedulePeriodicitySeconds = 1 * 60 * 60; // default: 1 hrs
    static const int32_t DefaultNintendoAccountInfoRefreshIntervalSeconds = 13 * 60 * 60; // default: 13 hrs
    static const int32_t DefaultProfileSynchronizationIntervalSeconds = 5 * 60 * 60; // default: 5 hrs

    os::TimerEvent m_Event;
    AccountDaemonSettings::SchedulerSettings m_Settings {
        DefaultSchedulePeriodicitySeconds,
        DefaultProfileSynchronizationIntervalSeconds,
        DefaultNintendoAccountInfoRefreshIntervalSeconds
    };
    mutable os::SdkMutex m_Lock;
    TimeSpan m_Interval {};
    util::optional<TimeSpan> m_pLastExecution;
    TimeSpan m_NextExecution {};

    struct UserState
    {
        account::Uid uid;
        struct LastExecutionState
        {
            std::pair<bool, uint64_t> profileUpdate;
        } lastExecutionState;

        void Reset(const account::Uid& uid) NN_NOEXCEPT
        {
            this->uid = uid;
            this->lastExecutionState.profileUpdate = std::pair<bool, uint64_t>(false, 0);
        }
    } m_UserStates[account::UserCountMax];

    void InitializeImpl() NN_NOEXCEPT
    {
        for (auto& s : m_UserStates)
        {
            s.Reset(account::InvalidUid);
        }
        ResetInterval();
    }
    Result UpdateUsers(int* pOutCount) NN_NOEXCEPT;

    void ScheduleImpl() NN_NOEXCEPT;
    void RescheduleImmediatelyImpl() NN_NOEXCEPT;
    void UpdateLastExecutionTime() NN_NOEXCEPT;

    Result ExecuteScheduledTasksImpl(CancelPoint& cp) NN_NOEXCEPT;
    Result RefreshNintendoAccountStatus(const account::Uid& uid, CancelPoint& cp) NN_NOEXCEPT;
    Result SynchronizeProfileImmediately(const account::Uid& uid, CancelPoint& cp) NN_NOEXCEPT;
    Result SynchronizeProfile(const account::Uid& uid, CancelPoint& cp) NN_NOEXCEPT;

public:
    AccountDaemonTaskScheduler() NN_NOEXCEPT
        : m_Event(os::EventClearMode_ManualClear)
    {
        InitializeImpl();
    }
    explicit AccountDaemonTaskScheduler(const AccountDaemonSettings::SchedulerSettings& settings) NN_NOEXCEPT
        : m_Event(os::EventClearMode_ManualClear)
        , m_Settings(settings)
    {
        InitializeImpl();
    }
    void ExecuteScheduledTasks(CancelPoint& cp) NN_NOEXCEPT
    {
        // 次回実行をスケジュール
        bool isCanceled = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (isCanceled)
            {
                RescheduleImmediately();
                return;
            }
            UpdateLastExecutionTime();
        };

        // キャンセル済みであれば実行しない
        if (cp.Test())
        {
            NN_ACCOUNT_INFO_WITH_TIMESTAMP("Execution already cancelled\n");
            isCanceled = true;
            return;
        }

        // タスクを実行
        auto r = ExecuteScheduledTasksImpl(cp);
        if (!r.IsSuccess())
        {
            if (account::ResultCancelled::Includes(r))
            {
                NN_ACCOUNT_INFO_WITH_TIMESTAMP(
                    "ExecuteScheduledTasksImpl cancelled\n");
                isCanceled = true;
            }
            else if (account::ResultInternetConnectionUnavailable::Includes(r))
            {
                NN_ACCOUNT_INFO_WITH_TIMESTAMP(
                    "ExecuteScheduledTasksImpl couldn't establish internet connection\n");
            }
            else
            {
                NN_ACCOUNT_ERROR_WITH_TIMESTAMP(
                    "ExecuteScheduledTasksImpl failed with: %03d-%04d\n",
                    r.GetModule(), r.GetDescription());
                ReportError(r, false);
            }
        }
    }

    void UpdateInterval(TimeSpan interval) NN_NOEXCEPT;
    void ResetInterval() NN_NOEXCEPT
    {
        UpdateInterval(TimeSpan::FromSeconds(m_Settings.schedulePeriodicity));
    }
    void RescheduleImmediately() NN_NOEXCEPT;
    TimeSpan GetNextExecutionInUptime() const NN_NOEXCEPT;

    os::TimerEventType* GetEventBase() NN_NOEXCEPT
    {
        return m_Event.GetBase();
    }
    bool TryWait() NN_NOEXCEPT
    {
        return m_Event.TryWait();
    }
    void Clear() NN_NOEXCEPT
    {
        m_Event.Clear();
    }
};

} // ~namespace nn
