﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>

#include "dmnt_HtcsReceiveBuffer.h"

namespace nn { namespace dmnt { namespace io {

int HtcsReceiveBuffer::Read(void* dst, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(size, static_cast<size_t>(0), HtcsReceiveBufferSize);

    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if (!IsReadable() || !IsValid())
        {
            return -1;
        }

        size_t readSize = std::min(size, m_DataSize);
        if (readSize <= 0)
        {
            return -1;
        }
        std::memcpy(dst, &(m_Buffer[m_HeadIndex]), readSize);

        m_DataSize -= readSize;
        m_HeadIndex += readSize;
        if (m_DataSize == 0)
        {
            m_DataSize = 0;
            m_HeadIndex = 0;
            m_ReadableEvent.Clear();
            m_WritableEvent.Signal();
        }

        NN_SDK_ASSERT_MINMAX(m_DataSize, static_cast<size_t>(0), HtcsReceiveBufferSize);
        NN_SDK_ASSERT_MINMAX(m_HeadIndex, static_cast<size_t>(0), HtcsReceiveBufferSize);

        return readSize;
    }
}

int HtcsReceiveBuffer::Write(void* dst, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(size, static_cast<size_t>(0), HtcsReceiveBufferSize);

    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if (!IsWritable() || !IsValid())
        {
            return -1;
        }

        std::memcpy(m_Buffer, dst, size);

        m_DataSize = size;
        m_HeadIndex = 0;
        m_ReadableEvent.Signal();
        m_WritableEvent.Clear();

        NN_SDK_ASSERT_MINMAX(m_DataSize, static_cast<size_t>(0), HtcsReceiveBufferSize);
        NN_SDK_ASSERT_MINMAX(m_HeadIndex, static_cast<size_t>(0), HtcsReceiveBufferSize);

        return size;
    }
}

int HtcsReceiveBuffer::WaitToBeReadable() NN_NOEXCEPT
{
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        if (IsReadable())
        {
            return 1;
        }
        else if (!IsValid())
        {
            return -1;
        }
        else
        {
            m_ReadableEvent.Clear();
        }
    }

    m_ReadableEvent.Wait();
    if (!IsValid())
    {
        return -1;
    }
    return 1;
}

int HtcsReceiveBuffer::WaitToBeReadable(nn::TimeSpan timeout) NN_NOEXCEPT
{
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        if (IsReadable())
        {
            return 1;
        }
        else if(!IsValid())
        {
            return -1;
        }
        else
        {
            m_ReadableEvent.Clear();
        }
    }

    int ret = m_ReadableEvent.TimedWait(timeout);
    if (!IsValid())
    {
        return -1;
    }
    return ret;
}

int HtcsReceiveBuffer::WaitToBeWritable() NN_NOEXCEPT
{
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        if (IsWritable())
        {
            return 1;
        }
        else if(!IsValid())
        {
            return -1;
        }
        else
        {
            m_WritableEvent.Clear();
        }
    }

    m_WritableEvent.Wait();
    if (!IsValid())
    {
        return -1;
    }
    return 1;
}

void HtcsReceiveBuffer::Invalidate() NN_NOEXCEPT
{
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_Valid = false;
        m_ReadableEvent.Signal();
        m_WritableEvent.Signal();
    }
}

}}}
