﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/osdbg.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/fs/fs_Directory.h>
#include "dmnt_ModuleDefinition.h"
#include "SoftwareBreakPoints.h"
#include "HardwareBreakPoints.h"
#include "HardwareWatchPoints.h"

namespace nn { namespace dmnt {
    enum ProcessStatus
    {
        BREAK,
        RUN,
        EXIT
    };

    enum ContinueMode
    {
        STOPPED,
        CONTINUE,
        STEP
    };

    class DebugProcess
    {
    public:
        static const int NUM_THREAD = 266;
    public:
        DebugProcess();
        DebugProcess* Get(nn::svc::Handle handle);
        Result Attach(uint64_t pid);
        Result Attach(nn::svc::Handle handle, const char* name);
        Result GetThreadContext(nn::svc::ThreadContext* pContext, uint32_t tid, uint32_t flags);
        Result SetThreadContext(const nn::svc::ThreadContext* pContext, uint32_t tid, uint32_t flags);

        Result WriteMemory(const void* buf, uintptr_t addr, size_t size);
        Result ReadMemory(void* buf, uintptr_t addr, size_t size);
        Result Continue();
        Result Continue(uint32_t tid);
        void ClearStep();
        Result Step();
        Result Step(uint32_t tid);
        Result Break();
        Result GetProcessDebugEvent(nn::svc::DebugEventInfo* pEventInfo);
        void SetBreak()
        {
            m_Status = BREAK;
        }
        ProcessStatus GetStatus()
        {
            return m_Status;
        }
        Result Initialize();
        int ThreadCreate(uint32_t tid);
        void ThreadExit(uint32_t tid);
        Result GetThreadList(int32_t* pNumThreads, uint32_t* pThreadId, int32_t numThreads);
        Result GetThreadInfoList(int32_t* pNumThreads, nn::osdbg::ThreadInfo* pThreadInfo[], int32_t numThreads);
        void SetLastThreadId(uint32_t tid)
        {
            m_LastTid = tid;
            m_TidOverride = m_LastTid;
            m_TidOverrideStepContinue = m_LastTid;
        }
        uint32_t GetLastThreadId();
        uint32_t GetTidOverride();
        uint32_t GetTidOverrideStepContinue();
        void SetTidOverride(uint32_t tid)
        {
            m_TidOverride = tid;
        }
        void SetTidOverrideStepContinue(uint32_t tid)
        {
            m_TidOverrideStepContinue = tid;
        }

        void SetLastSignal(uint32_t sig)
        {
            m_LastSignal = sig;
        }
        uint32_t GetLastSignal()
        {
            return m_LastSignal;
        }
        uint32_t GetContinueThreadId()
        {
            return m_ContinueTid;
        }
        void ClearContinueThreadId()
        {
            m_ContinueTid = 0;
        }

        nn::Result SetBreakPoint(uintptr_t addr, size_t size, bool isStep);
        nn::Result ClearBreakPoint(uintptr_t addr, size_t size);

        nn::Result SetHardwareBreakPoint(uintptr_t addr, size_t size, bool isStep);
        nn::Result ClearHardwareBreakPoint(uintptr_t addr, size_t size);

        nn::Result SetWatchPoint(nn::Bit64 address, nn::Bit64 len, bool readAccess, bool writeAccess) NN_NOEXCEPT;
        nn::Result ClearWatchPoint(nn::Bit64 address, nn::Bit64 len) NN_NOEXCEPT;
        nn::Result GetWatchPointInfo(nn::Bit64 address, bool &readAccess, bool &writeAccess) NN_NOEXCEPT;
        static bool IsValidWatch(nn::Bit64 address, nn::Bit64 len);

        nn::Result Terminate();
        void Detach();
        bool IsValid()
        {
            return m_IsValid;
        }
        bool Is64Bit()
        {
            return m_Is64Bit;
        }
        bool Is64BitAddress()
        {
            return m_Is64BitAddress;
        }
        nn::svc::Handle GetHandle()
        {
            return m_Handle;
        }

        size_t GetNumModule() const { return m_NumModule; }
        size_t GetMainModuleIndex() const { return m_MainModule; }
        const char* GetModuleName(size_t index) const { return m_ModuleDefinitions[index].GetName(); }
        uintptr_t GetBaseAddress(size_t index) const { return m_ModuleDefinitions[index].GetAddress(); }
        void GetBranchTarget(nn::svc::ThreadContext &context, uint32_t tid, uint64_t& CurNext, uint64_t& Target);
    private:
        int m_ThreadCount;
        Result Start();
        bool m_IsValid;
        bool m_Is64Bit;
        bool m_Is64BitAddress;
        ProcessStatus m_Status;
        nn::svc::Handle m_Handle;
        uint64_t m_Pid;
        uint32_t m_LastTid;
        uint32_t m_TidOverrideStepContinue;
        uint32_t m_TidOverride;

        uint32_t m_ContinueTid;
        uint32_t m_LastSignal;
        bool m_Stepping;
        struct
        {
            uint32_t tid;
            bool isValid;
        } m_Thread[NUM_THREAD];
        nn::osdbg::ThreadInfo   m_ThreadInfo[NUM_THREAD];
        nn::svc::DebugInfoCreateProcess  m_DebugInfoCreateProcess;

        SoftwareBreakPoints m_SoftBreakPoints;
        HardwareBreakPoints m_HardBreakPoints;
        HardwareWatchPoints m_WatchPoints;
        BreakPointManager &m_StepBreakPoints;
        ModuleDefinition m_ModuleDefinitions[MAX_NUMBER_OF_MODULES_QUERY];
        enum KnownLibraries
        {
            NNSDK,
            NNRTLD,
            NVN,
            NNDISPLAY,
            NUM_KnownLibraries
        };
        int m_KnownLibraryIndex[NUM_KnownLibraries];
        char m_FileName[nn::fs::EntryNameLengthMax];
        char m_nnSdkLang[3];
        size_t m_NumModule;
        size_t m_MainModule;

        Result CountModules(nn::svc::Handle handle);
        void MakeSdkPaths();
        void MatchModulesToKnownLibs();
        void NullTerminateModuleDefs();
    };
}}
