﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/ns_DevelopApi.h>
#include <nn/ldr/ldr_DebugMonitorApi.h>
#include <nn/util/util_FormatString.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Base.h>
#include <nn/osdbg.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/ldr/ldr_Types.h>
#include <nn/dmnt/dmnt_Result.h>
#include <nn/dbg/dbg_Api.h>
#include <algorithm>
#include <mutex>
#include <cstring>
#include <cinttypes>
#include <alloca.h>
#include "gdbserver_log.h"
#include "dmnt_Rsp.h"
#include "dmnt_ModuleDefinition.h"
#include "dmnt_DebugMonitor.h"
#include "dmnt_ArraySize.h"

#define SOFTSTEP
//==============================================================================

namespace nn { namespace dmnt {

static char const * const KnownLibraryNames[] =
{
    "nnSdk",
    "nnrtld",
    "nvn",
    "nnDisplay"
};

DebugProcess::DebugProcess()
:
    m_IsValid(false),
    m_ThreadCount(0),
    m_SoftBreakPoints(this),
    m_HardBreakPoints(this),
    m_WatchPoints(this),
    #ifdef SOFTSTEP
    m_StepBreakPoints(m_SoftBreakPoints)
    #else
    m_StepBreakPoints(m_HardBreakPoints)
    #endif
{
}

Result DebugProcess::Initialize()
{
    m_Status = BREAK;
    for (int i = 0; i < ARRAYSIZE(m_Thread); i++)
    {
        m_Thread[i].tid = 0;
        m_Thread[i].isValid = false;
    }
    m_LastTid = 0;
    m_TidOverride = m_LastTid;
    m_TidOverrideStepContinue = m_LastTid;
    m_ContinueTid = 0;
    m_LastSignal = 0;
    m_ThreadCount = 0;
    m_NumModule = 0;
    m_MainModule = -1;
    m_Stepping = false;
    m_Pid = 0;
    m_nnSdkLang[0] = 0;

    for (size_t i = 0; i < ARRAYSIZE(m_KnownLibraryIndex); ++i)
    {
        m_KnownLibraryIndex[i] = -1;
    }

    return ResultSuccess();
}
int DebugProcess::ThreadCreate(uint32_t tid)
{
    for (int i = 0; i < ARRAYSIZE(m_Thread); i++)
    {
        if (!m_Thread[i].isValid)
        {
            m_Thread[i].isValid = true;
            m_Thread[i].tid = tid;
            SetLastThreadId(tid);
            SetLastSignal(nn::dmnt::rsp::GDB_SIGNAL_TRAP);
            m_ThreadCount++;
            return i;
        }
    }
    return -1;
}
void DebugProcess::ThreadExit(uint32_t tid)
{
    for (int i = 0; i < ARRAYSIZE(m_Thread); i++)
    {
        if (m_Thread[i].isValid && m_Thread[i].tid == tid)
        {
            m_Thread[i].isValid = false;
            m_Thread[i].tid = 0;
            SetLastThreadId(tid);
            SetLastSignal(nn::dmnt::rsp::GDB_SIGNAL_TRAP);
            m_ThreadCount--;
            break;
        }
    }
}
Result DebugProcess::GetThreadList(int32_t* pNumThreads, uint32_t* pThreadId, int32_t numThreads)
{
    int32_t maxThreads = numThreads;
    int32_t num = 0;
    for (int i = 0; i < ARRAYSIZE(m_Thread); i++)
    {
        if (m_Thread[i].isValid)
        {
            if (num < maxThreads)
            {
                pThreadId[num++] = m_Thread[i].tid;
            }
        }
    }
    *pNumThreads = num;
    return ResultSuccess();
}
Result DebugProcess::GetThreadInfoList(int32_t* pNumThreads, nn::osdbg::ThreadInfo* pThreadInfo[], int32_t numThreads)
{
    int32_t maxThreads = numThreads;
    int32_t num = 0;
    for (int i = 0; i < ARRAYSIZE(m_Thread); i++)
    {
        if (m_Thread[i].isValid)
        {
            if (num < maxThreads)
            {
                pThreadInfo[num++] = &m_ThreadInfo[i];
            }
        }
    }
    *pNumThreads = num;
    return ResultSuccess();
}
uint32_t DebugProcess::GetLastThreadId()
{
    if (m_LastTid == 0)
    {
        for (int i = 0; i < ARRAYSIZE(m_Thread); i++)
        {
            if (m_Thread[i].isValid)
            {
                SetLastThreadId(m_Thread[i].tid);
                break;
            }
        }
    }
    return m_LastTid;
}

uint32_t DebugProcess::GetTidOverride()
{
    GetLastThreadId();
    return m_TidOverride;
}
uint32_t DebugProcess::GetTidOverrideStepContinue()
{
    GetLastThreadId();
    return m_TidOverrideStepContinue;
}

Result DebugProcess::GetProcessDebugEvent(nn::svc::DebugEventInfo* pEventInfo)
{
    Result result = nn::svc::GetDebugEvent(pEventInfo, m_Handle);
    if (result.IsSuccess())
    {
        switch (pEventInfo->event)
        {
        case nn::svc::DebugEvent_CreateProcess:
            {
                m_DebugInfoCreateProcess = pEventInfo->info.createProcess;
                if (pEventInfo->info.createProcess.flag & nn::svc::CreateProcessParameterFlag_64Bit)
                {
                    m_Is64Bit = true;
                }
                if ((pEventInfo->info.createProcess.flag & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
                {
                    m_Is64BitAddress = true;
                }
            }
            break;
        case nn::svc::DebugEvent_CreateThread:
            {
                int index = ThreadCreate(pEventInfo->threadId);
                if (index >= 0)
                {
                    nn::Result result = nn::osdbg::InitializeThreadInfo(&(m_ThreadInfo[index]), m_Handle, &m_DebugInfoCreateProcess, &pEventInfo->info.createThread);

                    if( ! result.IsSuccess() )
                    {
                        NN_SDK_LOG("nn::dmnt::DebugProcess::GetProcessDebugEvent: nn::osdbg::InitializeThreadInfo FAILED for thread %d:  Error = 0x%x\n", pEventInfo->threadId, result.GetInnerValueForDebug() );
                    }
                }
            }
            break;
        case nn::svc::DebugEvent_ExitThread:
            {
                ThreadExit(pEventInfo->threadId);
            }
            break;
        default:
            {
            }
            break;
        }

        if (pEventInfo->flags & nn::svc::DebugEventFlag_Stopped)
        {
            SetBreak();
        }
    }

    return result;
}

Result DebugProcess::Break()
{
    Result result = ResultSuccess();
    if (GetStatus() == RUN)
    {
        GDB_TRACE_T("[DebugProcess::Break()] Line:%d\n", __LINE__);
        result = nn::svc::BreakDebugProcess(m_Handle);
    }
    else
    {
        GDB_TRACE_E("[DebugProcess::Break()] Line:%d -- Process Not Running!\n", __LINE__);
    }
    return result;
}

Result DebugProcess::Continue()
{
    GDB_TRACE_T("%s:%d DebugProcess::Continue() all\n",__func__,__LINE__);
    nn::Bit64 threadIds[] = { 0 };
    Result result = nn::svc::ContinueDebugEvent(
            m_Handle,
            nn::svc::ContinueFlag_ExceptionHandled |
            nn::svc::ContinueFlag_EnableExceptionEvent |
            nn::svc::ContinueFlag_ContinueAll,
            threadIds,
            NN_ARRAY_SIZE(threadIds));
    if (result.IsSuccess())
    {
        m_ContinueTid = 0;
        m_Status = RUN;
        SetLastThreadId(0);
        SetLastSignal(nn::dmnt::rsp::GDB_SIGNAL_0);
    }
    return result;
}
Result DebugProcess::Continue(uint32_t tid)
{
    GDB_TRACE_T("%s:%d DebugProcess::Continue() tid=%d\n",__func__,__LINE__, tid);
    nn::Bit64 threadIds[] = { tid };
    Result result = nn::svc::ContinueDebugEvent(
            m_Handle,
            nn::svc::ContinueFlag_ExceptionHandled |
            nn::svc::ContinueFlag_EnableExceptionEvent,
            threadIds,
            NN_ARRAY_SIZE(threadIds));
    if (result.IsSuccess())
    {
        m_ContinueTid = tid;
        m_Status = RUN;
        SetLastThreadId(0);
        SetLastSignal(nn::dmnt::rsp::GDB_SIGNAL_0);
    }
    return result;
}

void DebugProcess::ClearStep()
{
    if (m_Stepping)
    {
        m_StepBreakPoints.ClearStep();
        m_Stepping = false;
    }
}

Result DebugProcess::Step()
{
    GDB_TRACE_T("%s:%d DebugProcess::Step() all\n",__func__,__LINE__);
    return Step(GetLastThreadId());
}

Result DebugProcess::Step(uint32_t tid)
{
    GDB_TRACE_T("%s:%d DebugProcess::Step() tid=%d\n",__func__,__LINE__, tid);
    nn::svc::ThreadContext context;
    Result result = GetThreadContext(&context, tid, nn::svc::ContextFlag_Control);
    if (!result.IsSuccess())
    {
        GDB_TRACE_E("[DebugProcess::Step] Line:%d, tid:%d FAILED:0x%x\n", __LINE__, tid, result.GetInnerValueForDebug());
        return result;
    }

    m_Stepping = true;
    uint64_t currentPc = context.pc;
    uint64_t stepAddress = 0;
    GetBranchTarget(context, tid, currentPc, stepAddress);
    if (currentPc)
    {
        result = m_StepBreakPoints.SetBreakPoint(currentPc, sizeof(uint32_t), true); // TODO: Detect 32-bit
    }

    if (result.IsSuccess() && stepAddress)
    {
        result = m_StepBreakPoints.SetBreakPoint(stepAddress, sizeof(uint32_t), true); // TODO: Detect 32-bit
    }

    if (!result.IsSuccess())
    {
        GDB_TRACE_E("[DebugProcess::Step] Line:%d, tid:%d FAILED:0x%x\n", __LINE__, tid, result.GetInnerValueForDebug());
        ClearStep();
        return result;
    }

    return ResultSuccess();
}

Result DebugProcess::WriteMemory(const void* buf, uintptr_t addr, size_t size)
{
    return nn::svc::WriteDebugProcessMemory(m_Handle, reinterpret_cast<uintptr_t>(buf), addr, size);
}
Result DebugProcess::ReadMemory(void* buf, uintptr_t addr, size_t size)
{
    return nn::svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(buf), m_Handle, addr, size);
}
Result DebugProcess::GetThreadContext(nn::svc::ThreadContext* pContext, uint32_t tid, uint32_t flags)
{
    return nn::svc::GetDebugThreadContext(pContext, m_Handle, tid, flags);
}
Result DebugProcess::SetThreadContext(const nn::svc::ThreadContext* pContext, uint32_t tid, uint32_t flags)
{
    return nn::svc::SetDebugThreadContext(m_Handle, tid, *pContext, flags);
}

Result DebugProcess::CountModules(nn::svc::Handle handle)
{
    uintptr_t addr = 0;

    Result result;
    GDB_TRACE_T("[DebugProcess::CountModules] Line:%d\n", __LINE__);
    for (;;)
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo pageInfo;

        result = nn::svc::QueryDebugProcessMemory(&blockInfo, &pageInfo, handle, addr);
        if (result.IsSuccess() && blockInfo.permission == nn::svc::MemoryPermission_ReadExecute && blockInfo.state == nn::svc::MemoryState_Code )
        {
            NN_ABORT_UNLESS(m_NumModule < MAX_NUMBER_OF_MODULES_QUERY);
            m_ModuleDefinitions[m_NumModule].SetAddressSize(blockInfo.baseAddress, blockInfo.size);

            uint64_t buffer = 0;
            uint64_t address = blockInfo.baseAddress + blockInfo.size;
            GDB_TRACE_T("[DebugProcess::CountModules] Line:%d     ReadDebugProcessMemory@0x%p\n", __LINE__, address);
            result = nn::svc::ReadDebugProcessMemory( (uintptr_t)&buffer, handle, (uintptr_t)address, sizeof(uint64_t) );
            uint64_t length = buffer >> 32;
            length |= (buffer << 32);

            if( result.IsSuccess() && length > 0 )
            {
                length = std::min(length, (uint64_t)MAX_PATH);
                char *name = m_ModuleDefinitions[m_NumModule].GetNameBuffer();
                name[0] = 0;
                result = nn::svc::ReadDebugProcessMemory( (uintptr_t)name, handle, (uintptr_t)(address + sizeof(uint64_t)), length );
                name[MAX_PATH - 1] = 0;
            }
            ++m_NumModule;
        }
        if (blockInfo.state == nn::svc::MemoryState_Inaccessible)
        {
            break;
        }
        if (static_cast<uintptr_t>(blockInfo.baseAddress + blockInfo.size) <= addr)
        {
            break;
        }
        addr = blockInfo.baseAddress + blockInfo.size;
    }
    return result;
}

Result DebugProcess::Attach(uint64_t pid)
{
    std::strcpy(m_FileName, "Application");
    Result result = nn::svc::DebugActiveProcess(&m_Handle, pid);
    if (result.IsSuccess())
    {
        result = Start();
    }

    if (result.IsSuccess())
    {
        result = CountModules(m_Handle);
    }
    if (result.IsSuccess())
    {
        MakeSdkPaths();
    }

    nn::Bit64 getpid = 0;
    nn::svc::GetProcessId(&getpid, m_Handle);
    GDB_TRACE_S("Attach(x%llx) getpid=x%llx\n", pid, getpid);
    m_Pid = getpid;
    return result;
}
char const *DirOutputs[] =
{
    "Outputs\\"
};

char const *DirPrograms[] =
{
    "Programs\\",
    "Samples\\"
    "Tools\\"
    "Tests\\"
    "Externals\\"
};

char const *DirPlatforms[] =
{
    "NX-NXFP2-a32\\",
    "NX-NXFP2-a64\\"
};

char const *DirBuildTypes[] =
{
    "Develop\\",
    "Debug\\",
    "Release\\"
};

template <size_t N> bool IsOneOf(char const *(&haystack)[N], char const *needle, size_t needleLength)
{
    for(size_t i = 0; i < N; ++i)
    {
        if (strncasecmp(haystack[i], needle, needleLength) == 0)
        {
            return true;
        }
    }
    return false;
}

void DebugProcess::MatchModulesToKnownLibs()
{
    for (size_t moduleIndex = 0; moduleIndex < m_NumModule; ++moduleIndex)
    {
        bool matched = false;
        // Special case nnSdk because its name is localized.
        if (strncmp(m_ModuleDefinitions[moduleIndex].GetName(), "nnSdk", 5) == 0)
        {
            GDB_TRACE_T("Found nnsdk moduleIndex:%d, name:%s\n", moduleIndex, m_ModuleDefinitions[moduleIndex].GetName());
            m_KnownLibraryIndex[NNSDK] = moduleIndex;
            strncpy(m_nnSdkLang, m_ModuleDefinitions[moduleIndex].GetName() + 5, ARRAYSIZE(m_nnSdkLang));
            m_nnSdkLang[ARRAYSIZE(m_nnSdkLang) - 1] = 0;
            matched = true;
        }

        for (size_t knownLibIndex = 0; knownLibIndex < ARRAYSIZE(KnownLibraryNames); ++knownLibIndex)
        {
            if (strcmp(m_ModuleDefinitions[moduleIndex].GetName(), KnownLibraryNames[knownLibIndex]) == 0)
            {
                GDB_TRACE_T("Found moduleIndex:%d, knownLibIndex:%d, name:%s\n", moduleIndex, knownLibIndex, m_ModuleDefinitions[moduleIndex].GetName());
                m_KnownLibraryIndex[knownLibIndex] = moduleIndex;
                matched = true;
                break;
            }
        }
        if (!matched && m_MainModule == -1)
        {
            if (strchr(m_ModuleDefinitions[moduleIndex].GetName(), '/') != 0 || strchr(m_ModuleDefinitions[moduleIndex].GetName(), '\\') != 0)
            {
                GDB_TRACE_T("Found main moduleIndex:%d, name:%s\n", moduleIndex, m_ModuleDefinitions[moduleIndex].GetName());
                m_MainModule = moduleIndex;
            }
        }
    }
    if (m_MainModule == -1 && m_NumModule == 1)
    {
        GDB_TRACE_T("main module not found, assuming it is module 0\n");
        m_MainModule = 0;
    }
}

void DebugProcess::NullTerminateModuleDefs()
{
    // Ensure all these strings are NULL terminated and have the ".nss" extension
    for (size_t knownLibIndex = 0; knownLibIndex < ARRAYSIZE(KnownLibraryNames); ++knownLibIndex)
    {
        static const char nssExtension[] = ".nss";
        static const size_t extLength = sizeof(nssExtension) - 1;
        if (m_KnownLibraryIndex[knownLibIndex] != -1)
        {
            char *name = m_ModuleDefinitions[m_KnownLibraryIndex[knownLibIndex]].GetNameBuffer();
            int length = strlen(name);
            if (strncmp(name + length - extLength, nssExtension, extLength) != 0 && length < MAX_PATH - extLength)
            {
                strcat(name, nssExtension);
            }
            name[MAX_PATH - 1] = 0;
        }
    }
}

void DebugProcess::MakeSdkPaths()
{
    // PREFIX\Programs\Chris\Intermediates\PLATFORM\Libraries\rtld\BUILD\nnrtld.debug.nss
    // PREFIX\Programs\Iris\Intermediates\PLATFORM\Libraries\nnSdk\BUILD\nnSdk.debug.nss
    // PREFIX\Iris\Libraries\PLATFORM\BUILD\nvn.nss
    // PREFIX\Iris\Libraries\PLATFORM\BUILD\nnDisplay.nss
    //
    static char const * const nnsdkFmt =  "%.*sPrograms\\Iris\\Intermediates\\%.*s\\Libraries\\nnSdk%s\\%.*s\\nnSdk%s.debug.nss"; // or nnSdkJp
    static char const * const nnrtldFmt = "%.*sPrograms\\Chris\\Intermediates\\%.*s\\Libraries\\rtld\\%.*s\\nnrtld.debug.nss";
    static char const * const nvnFmt       = "%.*sIris\\Libraries\\%.*s\\%.*s\\nvn.nss";
    static char const * const nnDisplayFmt = "%.*sIris\\Libraries\\%.*s\\%.*s\\nnDisplay.nss";

    char const *p = m_FileName + strlen(m_FileName);
    char const *next = p;
    char const *outputs = 0;
    char const *programs = 0;
    char const *platform = 0;
    char const *buildType = 0;

    int platformLen = 0;
    int buildTypeLen = 0;

    m_nnSdkLang[0] = 0;

    GDB_TRACE_T("MakeSdkPaths: m_FileName:%s\n", m_FileName);
    MatchModulesToKnownLibs();
    while (p-- > m_FileName)
    {
        if (*p == ':')
        {
            break;
        }

        if (*p == '\\' || *p == '/')
        {
            if (IsOneOf(DirOutputs, p + 1, next - p))
            {
                outputs = p + 1;
            }
            if (IsOneOf(DirPrograms, p + 1, next - p))
            {
                programs = p + 1;
            }
            if (IsOneOf(DirPlatforms, p + 1, next - p))
            {
                platform = p + 1;
                platformLen = ((int) (next - p - 1));
            }
            if (IsOneOf(DirBuildTypes, p + 1, next - p))
            {
                buildType = p + 1;
                buildTypeLen = ((int) (next - p - 1));
            }
            next = p;
        }
    }

    if (outputs && programs && platform && buildType)
    {
        if (m_KnownLibraryIndex[NNSDK] != -1)
        {
            char *name = m_ModuleDefinitions[m_KnownLibraryIndex[NNSDK]].GetNameBuffer();
            nn::util::SNPrintf(name, MAX_PATH, nnsdkFmt, (int)(programs - m_FileName), m_FileName, platformLen, platform, m_nnSdkLang, buildTypeLen, buildType, m_nnSdkLang);
        }
        if (m_KnownLibraryIndex[NNRTLD] != -1)
        {
            char *name = m_ModuleDefinitions[m_KnownLibraryIndex[NNRTLD]].GetNameBuffer();
            nn::util::SNPrintf(name, MAX_PATH, nnrtldFmt, (int)(programs - m_FileName), m_FileName, platformLen, platform, buildTypeLen, buildType);
        }
        if (m_KnownLibraryIndex[NVN] != -1)
        {
            char *name = m_ModuleDefinitions[m_KnownLibraryIndex[NVN]].GetNameBuffer();
            nn::util::SNPrintf(name, MAX_PATH, nvnFmt, (int)(programs - m_FileName), m_FileName, platformLen, platform, buildTypeLen, buildType);
        }
        if (m_KnownLibraryIndex[NNDISPLAY] != -1)
        {
            char *name = m_ModuleDefinitions[m_KnownLibraryIndex[NNDISPLAY]].GetNameBuffer();
            nn::util::SNPrintf(name, MAX_PATH, nnDisplayFmt, (int)(programs - m_FileName), m_FileName, platformLen, platform, buildTypeLen, buildType);
        }
    }

    NullTerminateModuleDefs();
}

Result DebugProcess::Attach(nn::svc::Handle handle, const char* name)
{
    m_FileName[0] = 0;

    m_Handle = handle;
    Result result = Start();
    if (result.IsSuccess() && name)
    {
        // name の形式は "@Host:/***.nca"
        // m_FileName の形式は "***.nss"
        const char* const MountName = "@Host:/";

        NN_ABORT_UNLESS(std::strlen(name) > std::strlen(MountName) + 4);
        NN_ABORT_UNLESS(std::strlen(name) < sizeof(m_FileName));
        NN_ABORT_UNLESS(std::memcmp(name, MountName, std::strlen(MountName)) == 0);
        NN_ABORT_UNLESS(std::strcmp(name + std::strlen(name) - 4, ".nca") == 0 ||
                std::strcmp(name + std::strlen(name) - 4, ".NCA") == 0);

        std::strcpy(m_FileName, name + std::strlen(MountName));
        std::strcpy(m_FileName + std::strlen(m_FileName) - 4, ".nss");
    }

    nn::Bit64 getpid = 0;
    nn::svc::GetProcessId(&getpid, m_Handle);

    if (result.IsSuccess())
    {
        result = CountModules(m_Handle);
    }

    if (result.IsSuccess())
    {
        MakeSdkPaths();
    }

    GDB_TRACE_S("Attach(handle) getpid=x%llx\n", getpid);

    m_Pid = getpid;
    return result;
}

void DebugProcess::Detach()
{
    if (m_IsValid)
    {
        m_SoftBreakPoints.ClearAll();
        m_HardBreakPoints.ClearAll();
        m_WatchPoints.ClearAll();

        svc::CloseHandle(m_Handle);
        m_Handle = nn::svc::Handle();
    }
    m_IsValid = false;
    SetLastThreadId(0);
    m_SoftBreakPoints.Reset();
    m_HardBreakPoints.Reset();
    m_WatchPoints.Reset();
}

Result DebugProcess::Terminate()
{
    if (IsValid())
    {
        NN_ABORT_UNLESS(nn::svc::TerminateDebugProcess(m_Handle).IsSuccess());
        Detach();
    }

    return ResultSuccess();
}
Result DebugProcess::Start()
{
    int threadCount = 0;
    bool isAttached = false;
    while (threadCount == 0 || !isAttached)
    {
        int32_t index;
        NN_ABORT_UNLESS(nn::svc::WaitSynchronization(&index, &m_Handle, 1, nn::svc::WAIT_INFINITE).IsSuccess());

        nn::svc::DebugEventInfo eventInfo;
        NN_ABORT_UNLESS(nn::svc::GetDebugEvent(&eventInfo, m_Handle).IsSuccess());

        switch (eventInfo.event)
        {
        case nn::svc::DebugEvent_CreateProcess:
            {
                m_DebugInfoCreateProcess = eventInfo.info.createProcess;
                if (eventInfo.info.createProcess.flag & nn::svc::CreateProcessParameterFlag_64Bit)
                {
                    m_Is64Bit = true;
                }
                if ((eventInfo.info.createProcess.flag & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
                {
                    m_Is64BitAddress = true;
                }
            }
            break;
        case nn::svc::DebugEvent_Exception:
            {
                if (eventInfo.info.exception.exceptionCode == nn::svc::DebugException_AttachBreak)
                {
                    isAttached = true;
                }
            }
            break;
        case nn::svc::DebugEvent_CreateThread:
            {
                threadCount++;
                int index = ThreadCreate(eventInfo.threadId);
                if (index >= 0)
                {
                    nn::Result result = nn::osdbg::InitializeThreadInfo(&(m_ThreadInfo[index]), m_Handle, &m_DebugInfoCreateProcess, &eventInfo.info.createThread);

                    if( ! result.IsSuccess() )
                    {
                        NN_SDK_LOG("nn::dmnt::DebugProcess::Start: nn::osdbg::InitializeThreadInfo FAILED for thread %d:  Error = 0x%x\n", eventInfo.threadId, result.GetInnerValueForDebug() );
                    }
                }
            }
            break;
        case nn::svc::DebugEvent_ExitThread:
            {
                threadCount--;
                ThreadExit(eventInfo.threadId);
            }
            break;
        default:
            {
            }
            break;
        }
    }
    m_IsValid = true;
    m_Status = BREAK;

    return ResultSuccess();
}
DebugProcess* DebugProcess::Get(nn::svc::Handle handle)
{
    if (m_Handle == handle)
    {
        return this;
    }
    return NULL;
}

nn::Result DebugProcess::SetBreakPoint(uintptr_t addr, size_t len, bool isStep)
{
    return m_SoftBreakPoints.SetBreakPoint(addr, len, isStep);
}

nn::Result DebugProcess::ClearBreakPoint(uintptr_t addr, size_t size)
{
    m_SoftBreakPoints.ClearBreakPoint(addr,size);
    return nn::ResultSuccess();
}

nn::Result DebugProcess::ClearWatchPoint(nn::Bit64 address, nn::Bit64 len) NN_NOEXCEPT
{
    return m_WatchPoints.ClearBreakPoint(address, len);
}

bool DebugProcess::IsValidWatch(nn::Bit64 address, nn::Bit64 len)
{
    return HardwareWatchPoints::IsValidWatch(address, len);
}

nn::Result DebugProcess::GetWatchPointInfo(nn::Bit64 address, bool &readAccess, bool &writeAccess) NN_NOEXCEPT
{
    return m_WatchPoints.GetWatchPointInfo(address, readAccess, writeAccess);
}

nn::Result DebugProcess::SetWatchPoint(nn::Bit64 address, nn::Bit64 len, bool readAccess, bool writeAccess) NN_NOEXCEPT
{
    return m_WatchPoints.SetWatchPoint(address, len, readAccess, writeAccess);
}

nn::Result DebugProcess::SetHardwareBreakPoint(uintptr_t addr, size_t len, bool isStep)
{
    return m_HardBreakPoints.SetBreakPoint(addr, len, isStep);
}

nn::Result DebugProcess::ClearHardwareBreakPoint(uintptr_t addr, size_t size)
{
    m_HardBreakPoints.ClearBreakPoint(addr, size);
    return nn::ResultSuccess();
}

uint64_t GetRegisterValue32(nn::svc::ThreadContext &context, uint64_t pc, int tid, int regno)
{
    if (regno == 15) // Special handling for ARM32 PC
    {
        return pc + 8;
    }
    else
    {
        return context.r[regno];
    }
}

int32_t SignExtendValue( uint32_t Value, uint32_t nBits )
{
    return( (int32_t)( Value << ( 32 - nBits ) ) >> ( 32 - nBits ) );
}

void DebugProcess::GetBranchTarget(nn::svc::ThreadContext &context, uint32_t tid, uint64_t& CurNext, uint64_t& Target)
{
    DebugProcess *debugProcess = this;
    uint64_t const pc = CurNext;
    Target      = 0;
    bool isCallInstr = false;

    CurNext += 4;
    uint32_t instruction = 0;  // Original instruction
    debugProcess->ReadMemory((void*)&instruction, pc, sizeof(instruction) );

    // ======== ARM64 Instructions ========

    if( debugProcess->Is64Bit() )
    {
        // Only support debugging 64-bit process on 64-bit architecture
        #if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)

////----------------------------------------------------------------------"x001 01xx xxxx xxxx xxxx xxxx xxxx xxxx" // Unconditional branch (immediate)
//    { 0, 0, "", "b",                "$Im26:00;",                            "0001 01 Im26" },
//    { 0, 0, "", "bl",               "$Im26:00;",                            "1001 01 Im26" },
        if( (instruction & 0x7C000000) == 0x14000000 )
        {
            if( instruction != 0x14000001 ) // Ignore NOP - Branch to Next
            {
                isCallInstr = (instruction & 0x80000000) == 0x80000000;
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
                Target = SignExtendValue( ((instruction & 0x03FFFFFF) << 2), 28 ) + pc;
            }
        }

////----------------------------------------------------------------------"x011 010x xxxx xxxx xxxx xxxx xxxx xxxx" // Compare & branch (immediate)
//    { 0, 0, "", "cbz",              "$?sf?X?W?d;, $Im19:00;",               "sf 011 0100 Im19 Rd" },
//    { 0, 0, "", "cbnz",             "$?sf?X?W?d;, $Im19:00;",               "sf 011 0101 Im19 Rd" },
        else if( (instruction & 0x7E000000) == 0x34000000 )
        {
            Target = SignExtendValue( ((instruction & 0x00FFFFE0) >> 3), 21 ) + pc;
        }

////----------------------------------------------------------------------"x011 011x xxxx xxxx xxxx xxxx xxxx xxxx" // Test & branch (immediate)
//    { 0, 0, "", "tbz",              "$?sf?X?W?d;, #$b31:M40;, $Im14:00;",   "sf 011 0110 M40 Im14 Rd" },
//    { 0, 0, "", "tbnz",             "$?sf?X?W?d;, #$b31:M40;, $Im14:00;",   "sf 011 0111 M40 Im14 Rd" },
        else if( (instruction & 0x7E000000) == 0x36000000 )
        {
            Target = SignExtendValue( ((instruction & 0x0007FFE0) >> 3), 16 ) + pc;
        }

////----------------------------------------------------------------------"0101 010x xxxx xxxx xxxx xxxx xxxx xxxx" // Conditional branch (immediate)
//    { 0, 0, "", "b.$cond;",         "$Im19:00;",                            "0101 0100 Im19 0 cond" },
//    }
        else if( (instruction & 0xFF000010) == 0x54000000 )
        {
            if( (instruction & 0x0F) == 0x0E )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }
            Target = SignExtendValue( ((instruction & 0x00FFFFE0) >> 3), 21 ) + pc;
        }

////----------------------------------------------------------------------"1101 011x xxxx xxxx xxxx xxxx xxxx xxxx" // Unconditional branch (register)
//    { 0, 0, "", "br",               "$Xn;",                                 "1101 0110 0001 1111 0000 00 Rn    00000" },
//    { 0, 0, "", "blr",              "$Xn;",                                 "1101 0110 0011 1111 0000 00 Rn    00000" },
//    { 0, 0, "", "ret",              "?Xn=11110??$Xn;?",                     "1101 0110 0101 1111 0000 00 Rn    00000" },
//    { 0, 0, "", "eret",             "",                                     "1101 0110 1001 1111 0000 00 11111 00000" },
//    { 0, 0, "", "drps",             "",                                     "1101 0110 1011 1111 0000 00 11111 00000" },
        else if( (instruction & 0xFF8FFC1F) == 0xD60F0000 )
        {
            isCallInstr = (instruction & 0x00F00000) == 0x00300000;

            if( !isCallInstr )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            nn::svc::ThreadContext context;
            Result result = debugProcess->GetThreadContext(&context, tid, nn::svc::ContextFlag_Control | nn::svc::ContextFlag_General);
            if (result.IsSuccess())
            {
                int regno = (instruction & 0x03E0) >> 5;
                if (regno < 29)
                {
                    Target = context.r[regno];
                }
                else if (regno == 29)
                {
                    Target = context.fp;
                }
                else if (regno == 30)
                {
                    Target = context.lr;
                }
                else if (regno == 31)
                {
                    Target = context.sp;
                }
            }
        }
        #endif
    }

    // ====================================
    // ======== ARM32 Instructions ========
    // ====================================

    else
    {

        #if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
        Bit64 sp = context.r[13];
        #elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
        Bit32 sp = context.sp;
        #endif
//    { 0, 0, "", "blx",              "$Offset24:00+1000;",                   "1111 1010 Offset24" },                             // Branch to Thumb
//    { 0, 0, "", "blx",              "$Offset24:10+1000;",                   "1111 1011 Offset24" },                             // Branch to Thumb
//    { 0, 0, "", "b$Cond;",          "$Offset24:00+1000;",                   "Cond 1010 Offset24" },                             // Branch
//    { 0, 0, "", "bl$Cond;",         "$Offset24:00+1000;",                   "Cond 1011 Offset24" },                             // Branch with Link
        if( (instruction & 0x0E000000) == 0x0A000000 )          // B<c>/BL<c>/BLX imm
        {
            if( instruction != 0xEAFFFFFF ) // Ignore NOP - Branch to Next
            {
                isCallInstr = ((instruction & 0xF0000000) == 0xF0000000) || ((instruction & 0x01000000) != 0);

                if( (!isCallInstr) && ((instruction & 0xF0000000) == 0xE0000000) )
                {
                    CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
                }

                Target = SignExtendValue( (instruction & 0x00FFFFFF) << 2, 26 ) + 8 + pc;

                if( (instruction & 0xFF000000) == 0xFB000000 )
                {
                    Target += 2; // Adjust for Thumb target
                }
            }
        }

//    { 0, 0, "", "bx$Cond;",         "$Rm;",                                 "Cond 0001 0010 1111 1111 1111 0001 Rm" },          // Branch and Exchange
//    { 0, 0, "", "blx$Cond;",        "$Rm;",                                 "Cond 0001 0010 1111 1111 1111 0011 Rm" },          // Branch with Link and Exchange
        else if( (instruction & 0x0FFFFFD0) == 0x012FFF10 )     // BX<c>/BLX<c> reg
        {
            isCallInstr = ((instruction & 0x00000020) != 0);

            if( (!isCallInstr) && ((instruction & 0xF0000000) == 0xE0000000) )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            int regno = instruction & 0x0F;
            Target = GetRegisterValue32(context, pc, tid, regno);
        }

//    { 0, 0, "", "mov$S;$Cond;",     "$Rd;, $Rm;, $ST; $Rs;",                "Cond 0001 101 S Rn Rd Rs 0 ST 1 Rm" },             // Rd = Op2
        else if( (instruction & 0x0FFFFFF0) == 0x01A0F000 )     // MOV PC, reg
        {
            if( (instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            int regno = instruction & 0x0F;
            Target = GetRegisterValue32(context, pc, tid, regno);
        }

//    { 0, 0, "", "ldr$B;$Cond;",     "$Rd;, [$Rn;]$W;",                      "Cond 0101 U B W 1 Rn Rd 000000000000" },
//    { 0, 0, "", "ldr$B;$Cond;",     "$Rd;, [$Rn;, #$U;$Offset12;]$W;",      "Cond 0101 U B W 1 Rn Rd Offset12" },
        else if( (instruction & 0x0F10F000) == 0x0510F000 )    // LDR PC, [Rx, #Imm]{!}
        {
            if( (instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            uint64_t Addr    = 0;
            int64_t Offset  = instruction & 0x00000FFF;
            int32_t RegN    = ( instruction >> 16 ) & 0x0F;
            uint32_t Value   = 0;

            Addr = GetRegisterValue32(context, pc, tid, RegN);

            if( (instruction & 0x01000000) != 0 ) // Offset or Pre-Indexed
            {
                if( (instruction & 0x00800000) == 0 ) // Subtract offset
                {
                    Offset = 0 - Offset;
                }
                Addr += Offset;
            }
            debugProcess->ReadMemory(&Value, Addr, sizeof(uint32_t)); // Read new PC from memory
            Target      = (uint64_t)Value;
        }

//    { 0, 0, "", "add$S;$Cond;",     "$Rd;, $Rn;, $Rm;$ShiftST;",            "Cond 0000 100 S Rn Rd ShiftST 0 Rm" },             // Rd = Op1  +  Op2
        else if( (instruction & 0x0FF0F060) == 0x0080F000 )    // ADD PC, Rn, Rm, LSL #Shift
        {
            if( (instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            int64_t Shift   = ( instruction >>  7 ) & 0x0000001F;
            int32_t RegN    = ( instruction >> 16 ) & 0x0F;
            int32_t RegM    = instruction & 0x0F;
            uint64_t ValueN  = 0;
            uint64_t ValueM  = 0;

            ValueN = GetRegisterValue32(context, pc, tid, RegN);
            ValueM = GetRegisterValue32(context, pc, tid, RegM);

            Target      = (uint64_t)( (int32_t)ValueN + (int32_t)( ValueM << Shift ) );

            if( Target & 0x03 )
            {
                Target  = ( Target >> 2 ) << 2; // Branch to Thumb or misaligned ARM32
            }
        }

//    { 0, 0, "", "pop$Cond;",        "{$Rd;}",                               "Cond 0100 1001 1101 Rd 0000 0000 0100" },
        else if( (instruction & 0x0FFFFFFF) == 0x049DF004 )     // POP<c> {PC}
        {
            if( (instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            uint32_t  Value;
            debugProcess->ReadMemory((void*)&Value, sp, sizeof(uint32_t) ); // Read new PC from stack
            Target = (uint64_t)Value;
        }

//    { 0, 0, "", "pop$Cond;",        "{$RegList;}",                          "Cond 1000 1011 1101 RegList" },
        else if( (instruction & 0x0FFF8000) == 0x08BD8000 )     // POP<c> {...,PC}
        {
            if( (instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            uint32_t  RegList = instruction & 0x0000FFFF;
            uint64_t  Addr    = sp;

            for( int32_t i = 0; i < 14; i++ ) // Advance address to point to PC by skipping all other regs
            {
                if( RegList & 1 )
                {
                    Addr += 4;
                }
                RegList >>= 1;
            }

            uint32_t Value;
            debugProcess->ReadMemory((void*)&Value, Addr, sizeof(uint32_t) ); // Read new PC from stack
            Target = (uint64_t)Value;
        }
    }
} // NOLINT(readability/fn_size)

}}
