﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <mutex>
#include <nn/os.h>
#include <nn/osdbg.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/dmnt/dmnt_Result.h>
#include "dmnt_Rsp.h"
#include "dmnt_DebugMonitor.h"
#include <alloca.h>
#include <cstring>
#include "gdbserver_log.h"

#include "SoftwareBreakPoints.h"
#include "dmnt_ArraySize.h"

namespace
{
uint8_t const AArch64BRK[4]  = {0xff, 0xff, 0xff, 0xe7};
uint8_t const ArmBRK[4]      = {0xfe, 0xde, 0xff, 0xe7};
uint8_t const ArmThumbBRK[2] = {0x80, 0xb6};
}
namespace nn { namespace dmnt {

nn::Result SoftBreakPoint::Set(DebugProcess *debugProcess, uintptr_t addr, size_t len, bool isStep)
{
    m_IsStep = isStep;
    m_Addr = addr;
    m_Length = len;

    Result result = debugProcess->ReadMemory((uint8_t *)&m_OriginalData, m_Addr, m_Length);
    GDB_TRACE_T("SoftBreakPoint::SetBreakPoint:%d @0x%llx, oldCode=0x%x, SoftBreakPoint=0x%llx\n", __LINE__, m_Addr, m_OriginalData, this);
    if (debugProcess->Is64Bit())
    {
        if (len != 4)
        {
            result = nn::svc::ResultInvalidArgument();
        }
        else
        {
            result = debugProcess->WriteMemory(AArch64BRK, m_Addr, m_Length);
        }
    }
    else
    {
        if (len != 4 && len != 2)
        {
            result = nn::svc::ResultInvalidArgument();
        }
        else
        {
            result = debugProcess->WriteMemory(m_Length == 4 ? ArmBRK : ArmThumbBRK, m_Addr, m_Length);
        }
    }

    if (result.IsSuccess())
    {
        m_InUse = true;
    }
    return result;
}

nn::Result SoftBreakPoint::Clear(DebugProcess *debugProcess)
{
    nn::Result result;
    if (m_InUse)
    {
        if (m_Addr)
        {
            result = debugProcess->WriteMemory((uint8_t *)&m_OriginalData, m_Addr, m_Length);
            GDB_TRACE_T("SoftBreakPoint::ClearBreakPoint @0x%llx, oldCode=0x%x, SoftBreakPoint=0x%llx\n", m_Addr, m_OriginalData, this);
            if (!result.IsSuccess())
            {
                GDB_TRACE_T("SoftBreakPoint::ClearBreakPoint @0x%llx, oldCode=0x%x FAILED error:0x%x\n", m_Addr, m_OriginalData, result.GetInnerValueForDebug());
            }
            Reset();
        }
        else
        {
            GDB_TRACE_E("SoftBreakPoint::ClearBreakPoint @0x%llx, oldCode=0x%x ** OOPS ** Address is null, SoftBreakPoint=0x%llx\n", m_Addr, m_OriginalData, this);
            result = nn::svc::ResultInvalidArgument();
        }
    }
    return result;
}

SoftwareBreakPoints::SoftwareBreakPoints(DebugProcess *debugProcess)
: BreakPointManager(debugProcess)
{
    GDB_TRACE_T("SoftwareBreakPoints %s:%d this=0x%llx, debugProcess=0x%llx", __func__,__LINE__, this, debugProcess);
}

BreakPoint *SoftwareBreakPoints::GetBreakPoint(unsigned index)
{
    if (index < ARRAYSIZE(m_BreakPoints))
    {
        return &m_BreakPoints[index];
    }
    return nullptr;
}

}}

