﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <mutex>
#include <nn/os.h>
#include <nn/osdbg.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/dmnt/dmnt_Result.h>
#include "dmnt_Rsp.h"
#include "dmnt_DebugMonitor.h"
#include <alloca.h>
#include <cstring>
#include "gdbserver_log.h"

#include "HardwareBreakPoints.h"
#include "dmnt_ArraySize.h"

namespace
{
// Hardware breakpoint register information
//
int g_LastBreakPointRegister = -1;
int g_FirstBreakPointContextRegister = -1;
int g_LastBreakPointContextRegister = -1;
int g_LastWatchPointRegister = -1;
}

namespace
{
// MultiCoreThread data objects
bool                        g_MultiCoreThreadInitialized = false;
NN_ALIGNAS(nn::os::StackRegionAlignment)
nn::Bit8                    g_MultiCoreStack[ 16 * 1024 ];
nn::os::ThreadType          g_MultiCoreThread;
nn::os::MessageQueueType    g_MultiCoreRequestMQ;
nn::os::MessageQueueType    g_MultiCoreResultMQ;
uintptr_t                   g_MultiCoreRequestQueue[ 2 ];
uintptr_t                   g_MultiCoreResultQueue[  2 ];
}

namespace nn { namespace dmnt {

char *toBin(uint32_t value, int bits, char *buffer)
{
    buffer[bits] = 0;
    for (int i = 0; i < bits; ++i)
    {
        buffer[bits - i - 1] = '0' + (value & 1);
        value >>= 1;
    }
    return buffer;
}

nn::Result HardBreakPoint::Set(DebugProcess *debugProcess, uintptr_t addr, size_t len, bool isStep)
{
    m_Addr = addr;
    m_Length = len;
    m_IsStep = isStep;

    Result result = HardwareBreakPoints::SetContextBreakpoint(m_RegContext, debugProcess->GetHandle());

    if (result.IsSuccess())
    {
        result = HardwareBreakPoints::SetExecutionBreakpoint(m_RegNumber, m_RegContext, addr);
    }

    if (result.IsSuccess())
    {
        m_InUse = true;
    }
    return result;
}

nn::Result HardBreakPoint::Clear(DebugProcess *debugProcess)
{
    nn::Result result;
    if (m_InUse)
    {
        GDB_TRACE_T("HardBreakPoint::ClearBreakPoint @0x%llx\n", m_Addr);
        result = HardwareBreakPoints::SetExecutionBreakpoint(m_RegNumber, m_RegContext, 0);
        Reset();
    }
    return result;
}

HardwareBreakPoints::HardwareBreakPoints(DebugProcess *debugProcess)
: BreakPointManager(debugProcess)
{
    CountBreakPointRegisters();
    const nn::svc::HardwareBreakPointRegisterName contextReg = (nn::svc::HardwareBreakPointRegisterName)(g_FirstBreakPointContextRegister);
    for (int i = 0; i < ARRAYSIZE(m_BreakPoints); ++i)
    {
        m_BreakPoints[i].Initialize( (nn::svc::HardwareBreakPointRegisterName) (nn::svc::HardwareBreakPointRegisterName_I0 + i), contextReg);
    }
}

BreakPoint *HardwareBreakPoints::GetBreakPoint(unsigned index)
{
    if (index < ARRAYSIZE(m_BreakPoints))
    {
        return &m_BreakPoints[index];
    }
    return nullptr;
}

//==============================================================================
// MultiCore thread
void HardwareBreakPoints::MultiCoreThread( void* pArg )
{
    //GDB_TRACE_T( "MultiCoreThread starting on Core %d, AvailableCoreMask = %llx\n", nn::svc::GetCurrentProcessorNumber(), nn::os::GetThreadAvailableCoreMask() );

    nn::Result  Result = nn::ResultSuccess();
    int32_t     CurCore;
    nn::Bit64   Mask;
    nnHandle    Handle;
    Handle.value = 0xFFFF8000; // PSEUDO_HANDLE_CURRENT_THREAD

    Result = nn::svc::GetThreadCoreMask( &CurCore, &Mask, static_cast<nn::svc::Handle>( Handle ) );

    if( ! Result.IsSuccess() )
    {
        GDB_TRACE_E( "GetThreadCoreMask failed: %08x\n", Result.GetInnerValueForDebug() );
    }

    while( true )
    {
        uintptr_t pRequest;
        nn::os::ReceiveMessageQueue( &pRequest, &g_MultiCoreRequestMQ );
        uint64_t* Request = (uint64_t*)pRequest;

        if( Request[0] == 1 )
        {
            for(int32_t Core = 0; Core < 4; Core++)
            {
                Mask    = 1 << Core;
                Result  = nn::svc::SetThreadCoreMask( static_cast<nn::svc::Handle>( Handle ), Core, Mask );

                if( ! Result.IsSuccess() )
                {
                    GDB_TRACE_E( "SetThreadCoreMask( %d, %d, %llx ) failed: %08x\n", Handle, Core, Mask, Result.GetInnerValueForDebug() );
                    //break;
                }

                Mask        = 0;
                CurCore     = -1;

                Result = nn::svc::GetThreadCoreMask( &CurCore, &Mask, static_cast<nn::svc::Handle>( Handle ) );

                if( ! Result.IsSuccess() )
                {
                    GDB_TRACE_E( "GetThreadCoreMask failed: %08x\n", Result.GetInnerValueForDebug() );
                }

                Result = nn::svc::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)Request[1], Request[2], Request[3] );

                if( ! Result.IsSuccess() )
                {
                    GDB_TRACE_E( "SetHardwareBreakPoint failed: %08x, Reg:%lld, Control:0x%llx, Value:0x%llx\n", Result.GetInnerValueForDebug(), Request[1], Request[2], Request[3] );
                    break;
                }
            }
        }
        nn::os::SendMessageQueue( &g_MultiCoreResultMQ, (uintptr_t)Result.IsSuccess());
    }

    GDB_TRACE_T( "MultiCoreThread ending\n" );
}

bool HardwareBreakPoints::SendMultiCoreRequest( void* pMultiCoreRequest )
{
    if( ! g_MultiCoreThreadInitialized )
    {
        nn::os::InitializeMessageQueue( &g_MultiCoreRequestMQ, g_MultiCoreRequestQueue, sizeof(g_MultiCoreRequestQueue) / sizeof(g_MultiCoreRequestQueue[ 0 ]) );
        nn::os::InitializeMessageQueue( &g_MultiCoreResultMQ , g_MultiCoreResultQueue,  sizeof(g_MultiCoreResultQueue)  / sizeof(g_MultiCoreResultQueue[  0 ]) );

        nn::Result Result = nn::os::CreateThread( &g_MultiCoreThread, &MultiCoreThread, nullptr, g_MultiCoreStack, sizeof(g_MultiCoreStack), 16 );

        if( ! Result.IsSuccess() )
        {
            GDB_TRACE_E( "SetThreadCoreMask failed: %08x\n", Result.GetInnerValueForDebug() );
        }
        nn::os::StartThread( &g_MultiCoreThread );

        g_MultiCoreThreadInitialized = true;
    }

    nn::os::SendMessageQueue( &g_MultiCoreRequestMQ, (uintptr_t)pMultiCoreRequest );

    uintptr_t RetVal;

    nn::os::ReceiveMessageQueue( &RetVal, &g_MultiCoreResultMQ );

    return RetVal != 0;
}

nn::Result HardwareBreakPoints::SetHardwareBreakPoint( /*nn::svc::HardwareBreakPointRegisterName*/ uint32_t regNo,  nn::Bit64 control, nn::Bit64 value ) NN_NOEXCEPT
{
    uint64_t Request[4];

    Request[0] = 1;
    Request[1] = regNo;
    Request[2] = control;
    Request[3] = value;

    bool result = SendMultiCoreRequest( (void*)Request );

    if(!result)
    {
        return nn::dmnt::ResultUnknown();
    }

    return nn::ResultSuccess();
}

void HardwareBreakPoints::CountBreakPointRegisters()
{
    if (g_LastBreakPointRegister == -1)
    {
        // There doesn't appear to be a way for the debugger to query this information directly.
        // So, do it indirectly by trying to use the registers and see which ones work.
        // Count the break point registers
        for (int i = nn::svc::HardwareBreakPointRegisterName_I0; i <= nn::svc::HardwareBreakPointRegisterName_I15; ++i)
        {
            nn::Result result = nn::svc::SetHardwareBreakPoint((nn::svc::HardwareBreakPointRegisterName)i, 0, 0);
            if (!result.IsSuccess())
            {
                break;
            }
            g_LastBreakPointRegister = i;
        }
        GDB_TRACE_T("Last valid break register: %d\n", g_LastBreakPointRegister);

        // Count the context registers
        nn::Bit64 control =
            (0x0 << 16) | // LBN
            (0x3 << 20) | // BT - Context match
            (0xf << 5)  | // BAS - both 32/64 bit instructions
            1; // Enable

        g_LastBreakPointContextRegister = g_LastBreakPointRegister;
        for (int i = g_LastBreakPointRegister; i >= (int)nn::svc::HardwareBreakPointRegisterName_I0; --i)
        {
            Result result = nn::svc::SetHardwareBreakPoint((nn::svc::HardwareBreakPointRegisterName)i, control, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS.value);
            nn::svc::SetHardwareBreakPoint((nn::svc::HardwareBreakPointRegisterName)i, 0, 0);

            if (!result.IsSuccess())
            {
                // We expect ResultInvalidHandle because we need a debug handle to the process (KDebug).
                if (result.GetInnerValueForDebug() != nn::svc::ResultInvalidHandle().GetInnerValueForDebug())
                {
                    break;
                }
            }
            g_FirstBreakPointContextRegister = i;
        }
        GDB_TRACE_T("Context registers:%d - %d\n", g_FirstBreakPointContextRegister, g_LastBreakPointContextRegister);

        for (int i = nn::svc::HardwareBreakPointRegisterName_D0; i <= nn::svc::HardwareBreakPointRegisterName_D15; ++i)
        {
            Result result = nn::svc::SetHardwareBreakPoint((nn::svc::HardwareBreakPointRegisterName)i, 0, 0);
            if (!result.IsSuccess())
            {
                break;
            }
            g_LastWatchPointRegister = i - nn::svc::HardwareBreakPointRegisterName_D0;
        }
        GDB_TRACE_T("Last valid watch register: %d\n", g_LastWatchPointRegister);
    }
}

nn::Result HardwareBreakPoints::SetExecutionBreakpoint(nn::svc::HardwareBreakPointRegisterName regNo, nn::svc::HardwareBreakPointRegisterName regContext, nn::Bit64 address) NN_NOEXCEPT
{
    // DBGBCR
    const uint8_t fieldBT  = 1;     // 8 bits
    const uint8_t fieldBAS = 0x0f;  // 4 bits (byte address select)

    nn::Bit64 control =
        (regContext << 16) |    // LBN - 4 bits
        (fieldBT << 20) |       // BT - matched address, linked (NOTE: Arm64 doesn't support mismatched address break)
        (fieldBAS << 5)  |      // BAS - both 32/64 bit instructions
        ((address != 0) ? 1 : 0); // Enable

    GDB_TRACE_T("[SetExecutionBreakpoint] Line:%d, reg:%d, control:0x%llx, address=0x%llx\n", __LINE__, regNo, control, address);

    //char bContext[32];
    //char bBT[32];
    //char bBAS[32];
    //char bControl[34];
    //GDB_TRACE_T("[DebugProcess::SetDataBreakpoint] Line:%d, reg:%d, address=0x%llx, Ctx:%s, BT:%s, BAS:%s, Control:%s\n", __LINE__, regNo, address, toBin(regContext, 4, bContext), toBin(fieldBT, 4, bBT), toBin(fieldBAS, 4, bBAS), toBin((uint32_t)control, 32, bControl));

    Result result = HardwareBreakPoints::SetHardwareBreakPoint(regNo, control, address);

    if (!result.IsSuccess())
    {
        GDB_TRACE_E("[SetExecutionBreakpoint] Line:%d, FAILED:0x%x, reg:%d, address=0x%llx\n", __LINE__, result.GetInnerValueForDebug(), regNo, address);
        return result;
    }
    return result;
}

nn::svc::HardwareBreakPointRegisterName HardwareBreakPoints::GetWatchPointContextRegister()
{
    HardwareBreakPoints::CountBreakPointRegisters();
    return (nn::svc::HardwareBreakPointRegisterName)(g_FirstBreakPointContextRegister + 1);
}

nn::Result HardwareBreakPoints::SetContextBreakpoint(nn::svc::HardwareBreakPointRegisterName regContext, nn::svc::Handle processDebuggerHandle) NN_NOEXCEPT
{
    uint32_t handleValue = ((nnHandle) processDebuggerHandle).value;
    // DBGBCR
    nn::Bit64 control =
        (0x0 << 16) | // LBN
        (0x3 << 20) | // BT - Context match
        (0xf << 5)  | // BAS - both 32/64 bit instructions
        1; // Enable
    GDB_TRACE_T("[SetContextBreakpoint] Line:%d, reg:%d, value(handle)=0x%x\n", __LINE__, regContext, handleValue);
    Result result = HardwareBreakPoints::SetHardwareBreakPoint(regContext, control, handleValue);

    if (!result.IsSuccess())
    {
        GDB_TRACE_E("[SetContextBreakpoint] Line:%d, FAILED:0x%x, reg:%d\n", __LINE__, result.GetInnerValueForDebug(), regContext);
        return result;
    }
    return result;
}
}}


