﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_DeviceState.h>
#include <nn/xcd/xcd_Vibration.h>
#include "xcd_Peripheral.h"
#include "xcd_VibratorAgent.h"
#include "xcd_VibrationOnConnect.h"

namespace nn { namespace xcd {

//!< Vibratorの実装のためのベースとなるクラス
class VibratorBase final : public Peripheral, public ICommandListener
{
public:
    VibratorBase() NN_NOEXCEPT :
        m_VibrationOnConnectLeft(this),
        m_VibrationOnConnectRight(this),
        m_Mutex(true) {}
    virtual ~VibratorBase() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual void ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT NN_OVERRIDE;
    virtual size_t GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;

public:
    //!< 指定された position の振動子が存在しているか判定します
    Result CheckExistence(VibratorPosition position) NN_NOEXCEPT;

    //!< 振動子に接続されているアンプの電源状態を制御します
    Result SetEnabled(bool isEnabled, VibratorPosition position) NN_NOEXCEPT;

    //!< 振動子に接続されているアンプの電源状態を取得します
    Result IsEnabled(bool* pOutValue, VibratorPosition position) NN_NOEXCEPT;

    //!< 個々の振動子を制御するための Agent を取得します
    Result GetVibratorAgent(VibratorAgent** pOutValue, VibratorPosition position) NN_NOEXCEPT;

    //!< 本体との通信方式を設定します
    void SetInterfaceType(InterfaceType interfaceType) NN_NOEXCEPT;

    //!< 接続時振動を開始します
    void StartVibrationOnConnect() NN_NOEXCEPT;

    //!< 接続時振動が終了したことを通知します
    void NotifyVibrationOnConnectFinished() NN_NOEXCEPT;

private:
    static const size_t InputReportSize;
    static const size_t OutputReportSize;
    static const int InputReportBitShiftLeft;
    static const int InputReportBitShiftRight;
    static const int OutputReportOffsetLeft;
    static const int OutputReportOffsetRight;

private:
    //!< 振動子に接続されているアンプの電源状態
    enum class PowerStatus : uint8_t
    {
        Disabled,           //!< 電源 OFF 状態
        DisableRequested,   //!< Disable コマンドを送って Ack 待ちの状態
        Enabled,            //!< 電源 ON 状態
        EnableRequested,    //!< Enable コマンドを送って Ack 待ちの状態
    };

    PowerStatus m_PowerStatusLeft;
    PowerStatus m_PowerStatusRight;
    VibratorAgent m_VibratorAgentLeft;
    VibratorAgent m_VibratorAgentRight;
    VibrationOnConnect m_VibrationOnConnectLeft;
    VibrationOnConnect m_VibrationOnConnectRight;
    nn::os::Mutex m_Mutex;
    InterfaceType m_InterfaceType;
    bool m_IsPowerDisableScheduled;

private:
    PowerStatus& RefPowerStatus(VibratorPosition position) NN_NOEXCEPT
    {
        return (position == VibratorPosition_Left) ? m_PowerStatusLeft : m_PowerStatusRight;
    }
    VibratorAgent& RefVibratorAgent(VibratorPosition position) NN_NOEXCEPT
    {
        return (position == VibratorPosition_Left) ? m_VibratorAgentLeft : m_VibratorAgentRight;
    }
};

}} // namespace nn::xcd
