﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   AMFM 符号をパックする際のフォーマットに関する定義
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_BitPack.h>

namespace nn { namespace xcd {

struct VibrationAmFmPackTag;

/**
 * @brief   AMFM 符号を 32 ビットにパックした型です。
 * @details
 *  nn::xcd::VibrationAmFmPackFormat::(フォーマット名)::Make() を利用することで
 *  特定のフォーマットにしたがってインスタンスを生成することができます。
 */
typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag> VibrationAmFmPack;

/**
 * @brief   AMFM 符号をパックする際のフォーマットをまとめた構造体です。
 */
struct VibrationAmFmPackFormat
{
    /**
     * @brief   3 個の振動値をすべて 5bit AMFM 符号で表すフォーマットです。
     */
    struct Three
    {
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<30, 2, int> Count;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<25, 5, int> AmFm5bitLow0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<20, 5, int> AmFm5bitHigh0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<15, 5, int> AmFm5bitLow1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<10, 5, int> AmFm5bitHigh1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 5, 5, int> AmFm5bitLow2;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 0, 5, int> AmFm5bitHigh2;

        /**
         * @brief       フォーマット Three にしたがって AMFM 符号をパックします。
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 3 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 3 以上)
         */
        static VibrationAmFmPack Make(int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            VibrationAmFmPack pack;
            pack.Set<Count>(3);
            pack.Set<AmFm5bitLow0>(codesLow[0]);
            pack.Set<AmFm5bitHigh0>(codesHigh[0]);
            pack.Set<AmFm5bitLow1>(codesLow[1]);
            pack.Set<AmFm5bitHigh1>(codesHigh[1]);
            pack.Set<AmFm5bitLow2>(codesLow[2]);
            pack.Set<AmFm5bitHigh2>(codesHigh[2]);
            return pack;
        }
    };

    /**
     * @brief   2 個の振動値をすべて 5bit AMFM 符号で表すフォーマットです。
     */
    struct Two
    {
        typedef Three::Count            Count;
        typedef Three::AmFm5bitLow0     AmFm5bitLow0;
        typedef Three::AmFm5bitHigh0    AmFm5bitHigh0;
        typedef Three::AmFm5bitLow1     AmFm5bitLow1;
        typedef Three::AmFm5bitHigh1    AmFm5bitHigh1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<0, 10, int> LowerBits;

        /**
         * @brief       フォーマット Two にしたがって AMFM 符号をパックします。
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 2 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 2 以上)
         */
        static VibrationAmFmPack Make(int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            VibrationAmFmPack pack;
            pack.Set<Count>(2);
            pack.Set<AmFm5bitLow0>(codesLow[0]);
            pack.Set<AmFm5bitHigh0>(codesHigh[0]);
            pack.Set<AmFm5bitLow1>(codesLow[1]);
            pack.Set<AmFm5bitHigh1>(codesHigh[1]);
            pack.Set<LowerBits>(0);
            return pack;
        }
    };

    /**
     * @brief   1 個の振動値をすべて 5bit AMFM 符号で表すフォーマットです。
     */
    struct One
    {
        typedef Three::Count            Count;
        typedef Three::AmFm5bitLow0     AmFm5bitLow0;
        typedef Three::AmFm5bitHigh0    AmFm5bitHigh0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<0, 20, int> LowerBits;

        /**
         * @brief       フォーマット One にしたがって AMFM 符号をパックします。
         * @param[in]   codeLow     低周波帯の AMFM 符号
         * @param[in]   codeHigh    高周波帯の AMFM 符号
         */
        static VibrationAmFmPack Make(int codeLow, int codeHigh) NN_NOEXCEPT
        {
            VibrationAmFmPack pack;
            pack.Set<Count>(1);
            pack.Set<AmFm5bitLow0>(codeLow);
            pack.Set<AmFm5bitHigh0>(codeHigh);
            pack.Set<LowerBits>(0);
            return pack;
        }
    };

    /**
     * @brief   AMFM 符号をひとつもパックしないフォーマットです。
     */
    struct Zero
    {
        typedef Three::Count            Count;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<0, 30, int> LowerBits;

        /**
         * @brief       フォーマット Zero にしたがってパックします。
         */
        static VibrationAmFmPack Make() NN_NOEXCEPT
        {
            VibrationAmFmPack pack;
            pack.Set<Count>(0);
            pack.Set<LowerBits>(0);
            return pack;
        }
    };

    /**
     * @brief   1 個の振動値をすべて 7bit AM 符号と 7bit FM 符号で表すフォーマットです。
     */
    struct One28bit
    {
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<30, 2, int> Count;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<23, 7, int> Am7bitLow0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<16, 7, int> Fm7bitLow0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 9, 7, int> Am7bitHigh0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 2, 7, int> Fm7bitHigh0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 0, 2, int> LowerBits;
        static const int CountValue = 1;

        /**
         * @brief       フォーマット One28bit にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitLow   低周波帯の 7bit AM 符号
         * @param[in]   fm7bitLow   低周波帯の 7bit FM 符号
         * @param[in]   am7bitHigh  高周波帯の 7bit AM 符号
         * @param[in]   fm7bitHigh  高周波帯の 7bit FM 符号
         */
        static VibrationAmFmPack Make(int am7bitLow, int fm7bitLow, int am7bitHigh, int fm7bitHigh) NN_NOEXCEPT
        {
            VibrationAmFmPack pack;
            pack.Set<Count>(CountValue);
            pack.Set<Am7bitLow0>(am7bitLow);
            pack.Set<Fm7bitLow0>(fm7bitLow);
            pack.Set<Am7bitHigh0>(am7bitHigh);
            pack.Set<Fm7bitHigh0>(fm7bitHigh);
            pack.Set<LowerBits>(0);
            return pack;
        }
    };

    //!< {Two,Three}14bit{Low,High} に共通する定義の抜粋
    struct Two14bit
    {
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<30, 2, int> Count;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<23, 7, int> Am7bit0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<18, 5, int> AmFm5bit0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<13, 5, int> AmFm5bitLow1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 8, 5, int> AmFm5bitHigh1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 1, 7, int> Fm7bit0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 0, 1, bool> IsHigh;
        static const int CountValueForTwo = 2;
        static const int CountValueForThree = 0;
    };

    /**
     * @brief   2 個の振動値のうち、ひとつめの振動値の amplitudeLow と frequencyLow を
     *          7bit AM 符号と 7bit FM 符号で表すフォーマットです。
     */
    struct Two14bitLow
    {
        typedef Two14bit::Count         Count;
        typedef Two14bit::Am7bit0       Am7bitLow0;
        typedef Two14bit::Fm7bit0       Fm7bitLow0;
        typedef Two14bit::AmFm5bit0     AmFm5bitHigh0;
        typedef Two14bit::AmFm5bitLow1  AmFm5bitLow1;
        typedef Two14bit::AmFm5bitHigh1 AmFm5bitHigh1;
        typedef Two14bit::IsHigh        IsHigh;

        /**
         * @brief       フォーマット Two14bitLow にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitLow0  ひとつめの振動値の低周波帯の 7bit AM 符号
         * @param[in]   fm7bitLow0  ひとつめの振動値の低周波帯の 7bit FM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 2 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 2 以上)
         */
        static VibrationAmFmPack Make(int am7bitLow0, int fm7bitLow0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            // codesLow[0] は利用しません
            VibrationAmFmPack pack;
            pack.Set<Count>(Two14bit::CountValueForTwo);
            pack.Set<Am7bitLow0>(am7bitLow0);
            pack.Set<Fm7bitLow0>(fm7bitLow0);
            pack.Set<AmFm5bitHigh0>(codesHigh[0]);
            pack.Set<AmFm5bitLow1>(codesLow[1]);
            pack.Set<AmFm5bitHigh1>(codesHigh[1]);
            pack.Set<IsHigh>(false);
            return pack;
        }
    };

    /**
     * @brief   2 個の振動値のうち、ひとつめの振動値の amplitudeHigh と frequencyHigh を
     *          7bit AM 符号と 7bit FM 符号で表すフォーマットです。
     */
    struct Two14bitHigh
    {
        typedef Two14bit::Count         Count;
        typedef Two14bit::AmFm5bit0     AmFm5bitLow0;
        typedef Two14bit::Am7bit0       Am7bitHigh0;
        typedef Two14bit::Fm7bit0       Fm7bitHigh0;
        typedef Two14bit::AmFm5bitLow1  AmFm5bitLow1;
        typedef Two14bit::AmFm5bitHigh1 AmFm5bitHigh1;
        typedef Two14bit::IsHigh        IsHigh;

        /**
         * @brief       フォーマット Two14bitHigh にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitHigh0 ひとつめの振動値の高周波帯の 7bit AM 符号
         * @param[in]   fm7bitHigh0 ひとつめの振動値の高周波帯の 7bit FM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 2 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 2 以上)
         */
        static VibrationAmFmPack Make(int am7bitHigh0, int fm7bitHigh0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            // codesHigh[0] は利用しません
            VibrationAmFmPack pack;
            pack.Set<Count>(Two14bit::CountValueForTwo);
            pack.Set<AmFm5bitLow0>(codesLow[0]);
            pack.Set<Am7bitHigh0>(am7bitHigh0);
            pack.Set<Fm7bitHigh0>(fm7bitHigh0);
            pack.Set<AmFm5bitLow1>(codesLow[1]);
            pack.Set<AmFm5bitHigh1>(codesHigh[1]);
            pack.Set<IsHigh>(true);
            return pack;
        }
    };

    /**
     * @brief   3 個の振動値のうち最初の 2 個を Two14bitLow と同様の方法で表すフォーマットです。
     */
    struct Three14bitLow
    {
        typedef Two14bitLow::Count          Count;
        typedef Two14bitLow::Am7bitLow0     Am7bitLow0;
        typedef Two14bitLow::Fm7bitLow0     Fm7bitLow0;
        typedef Two14bitLow::AmFm5bitHigh0  AmFm5bitHigh0;
        typedef Two14bitLow::AmFm5bitLow1   AmFm5bitLow1;
        typedef Two14bitLow::AmFm5bitHigh1  AmFm5bitHigh1;
        typedef Two14bitLow::IsHigh         IsHigh;

        /**
         * @brief       フォーマット Three14bitLow にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitLow0  ひとつめの振動値の低周波帯の 7bit AM 符号
         * @param[in]   fm7bitLow0  ひとつめの振動値の低周波帯の 7bit FM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 2 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 2 以上)
         */
        static VibrationAmFmPack Make(int am7bitLow0, int fm7bitLow0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            VibrationAmFmPack pack = Two14bitLow::Make(am7bitLow0, fm7bitLow0, codesLow, codesHigh);
            pack.Set<Count>(Two14bit::CountValueForThree);
            return pack;
        }
    };

    /**
     * @brief   3 個の振動値のうち最初の 2 個を Two14bitHigh と同様の方法で表すフォーマットです。
     */
    struct Three14bitHigh
    {
        typedef Two14bitHigh::Count         Count;
        typedef Two14bitHigh::AmFm5bitLow0  AmFm5bitLow0;
        typedef Two14bitHigh::Am7bitHigh0   Am7bitHigh0;
        typedef Two14bitHigh::Fm7bitHigh0   Fm7bitHigh0;
        typedef Two14bitHigh::AmFm5bitLow1  AmFm5bitLow1;
        typedef Two14bitHigh::AmFm5bitHigh1 AmFm5bitHigh1;
        typedef Two14bitHigh::IsHigh        IsHigh;

        /**
         * @brief       フォーマット Three14bitHigh にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitHigh0 ひとつめの振動値の高周波帯の 7bit AM 符号
         * @param[in]   fm7bitHigh0 ひとつめの振動値の高周波帯の 7bit FM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 2 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 2 以上)
         */
        static VibrationAmFmPack Make(int am7bitHigh0, int fm7bitHigh0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            VibrationAmFmPack pack = Two14bitHigh::Make(am7bitHigh0, fm7bitHigh0, codesLow, codesHigh);
            pack.Set<Count>(Two14bit::CountValueForThree);
            return pack;
        }
    };

    //!< Three7bit{Amplitude,Frequency}{Low,High} に共通する定義の抜粋
    struct Three7bit
    {
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<30, 2, int> Count;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<23, 7, int> AmFm7bit0;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<18, 5, int> AmFm5bitLow1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field<13, 5, int> AmFm5bitHigh1;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 8, 5, int> AmFm5bitLow2;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 3, 5, int> AmFm5bitHigh2;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 2, 1, bool> IsFrequency;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 1, 1, bool> IsThree;
        typedef nn::util::BitPack<uint32_t, VibrationAmFmPackTag>::Field< 0, 1, bool> IsHigh;
        static const int CountValue = 1;
        static VibrationAmFmPack Make(int amfm7bit0, int codesLow[], int codesHigh[], bool isFrequency, bool isHigh) NN_NOEXCEPT
        {
            // codesLow[0] と codesHigh[0] は利用しません
            VibrationAmFmPack pack;
            pack.Set<Count>(CountValue);
            pack.Set<AmFm7bit0>(amfm7bit0);
            pack.Set<AmFm5bitLow1>(codesLow[1]);
            pack.Set<AmFm5bitHigh1>(codesHigh[1]);
            pack.Set<AmFm5bitLow2>(codesLow[2]);
            pack.Set<AmFm5bitHigh2>(codesHigh[2]);
            pack.Set<IsFrequency>(isFrequency);
            pack.Set<IsThree>(true);
            pack.Set<IsHigh>(isHigh);
            return pack;
        }
    };

    /**
     * @brief   3 個の振動値のうち、ひとつめの振動値の amplitudeLow を 7bit AM 符号で表すフォーマットです。
     */
    struct Three7bitAmplitudeLow
    {
        typedef Three7bit::Count            Count;
        typedef Three7bit::AmFm7bit0        Am7bitLow0;
        typedef Three7bit::AmFm5bitLow1     AmFm5bitLow1;
        typedef Three7bit::AmFm5bitHigh1    AmFm5bitHigh1;
        typedef Three7bit::AmFm5bitLow2     AmFm5bitLow2;
        typedef Three7bit::AmFm5bitHigh2    AmFm5bitHigh2;
        typedef Three7bit::IsFrequency      IsFrequency;
        typedef Three7bit::IsThree          IsThree;
        typedef Three7bit::IsHigh           IsHigh;

        /**
         * @brief       フォーマット Three7bitAmplitudeLow にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitLow0  ひとつめの振動値の低周波帯の 7bit AM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 3 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 3 以上)
         */
        static VibrationAmFmPack Make(int am7bitLow0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            return Three7bit::Make(am7bitLow0, codesLow, codesHigh, false, false);
        }
    };

    /**
     * @brief   3 個の振動値のうち、ひとつめの振動値の frequencyLow を 7bit FM 符号で表すフォーマットです。
     */
    struct Three7bitFrequencyLow
    {
        typedef Three7bit::Count            Count;
        typedef Three7bit::AmFm7bit0        Fm7bitLow0;
        typedef Three7bit::AmFm5bitLow1     AmFm5bitLow1;
        typedef Three7bit::AmFm5bitHigh1    AmFm5bitHigh1;
        typedef Three7bit::AmFm5bitLow2     AmFm5bitLow2;
        typedef Three7bit::AmFm5bitHigh2    AmFm5bitHigh2;
        typedef Three7bit::IsFrequency      IsFrequency;
        typedef Three7bit::IsThree          IsThree;
        typedef Three7bit::IsHigh           IsHigh;

        /**
         * @brief       フォーマット Three7bitFrequencyLow にしたがって AMFM 符号をパックします。
         * @param[in]   fm7bitLow0  ひとつめの振動値の低周波帯の 7bit FM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 3 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 3 以上)
         */
        static VibrationAmFmPack Make(int fm7bitLow0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            return Three7bit::Make(fm7bitLow0, codesLow, codesHigh, true, false);
        }
    };

    /**
     * @brief   3 個の振動値のうち、ひとつめの振動値の amplitudeHigh を 7bit AM 符号で表すフォーマットです。
     */
    struct Three7bitAmplitudeHigh
    {
        typedef Three7bit::Count            Count;
        typedef Three7bit::AmFm7bit0        Am7bitHigh0;
        typedef Three7bit::AmFm5bitLow1     AmFm5bitLow1;
        typedef Three7bit::AmFm5bitHigh1    AmFm5bitHigh1;
        typedef Three7bit::AmFm5bitLow2     AmFm5bitLow2;
        typedef Three7bit::AmFm5bitHigh2    AmFm5bitHigh2;
        typedef Three7bit::IsFrequency      IsFrequency;
        typedef Three7bit::IsThree          IsThree;
        typedef Three7bit::IsHigh           IsHigh;

        /**
         * @brief       フォーマット Three7bitAmplitudeHigh にしたがって AMFM 符号をパックします。
         * @param[in]   am7bitHigh0 ひとつめの振動値の高周波帯の 7bit AM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 3 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 3 以上)
         */
        static VibrationAmFmPack Make(int am7bitHigh0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            return Three7bit::Make(am7bitHigh0, codesLow, codesHigh, false, true);
        }
    };

    /**
     * @brief   3 個の振動値のうち、ひとつめの振動値の frequencyHigh を 7bit FM 符号で表すフォーマットです。
     */
    struct Three7bitFrequencyHigh
    {
        typedef Three7bit::Count            Count;
        typedef Three7bit::AmFm7bit0        Fm7bitHigh0;
        typedef Three7bit::AmFm5bitLow1     AmFm5bitLow1;
        typedef Three7bit::AmFm5bitHigh1    AmFm5bitHigh1;
        typedef Three7bit::AmFm5bitLow2     AmFm5bitLow2;
        typedef Three7bit::AmFm5bitHigh2    AmFm5bitHigh2;
        typedef Three7bit::IsFrequency      IsFrequency;
        typedef Three7bit::IsThree          IsThree;
        typedef Three7bit::IsHigh           IsHigh;

        /**
         * @brief       フォーマット Three7bitAmplitudeHigh にしたがって AMFM 符号をパックします。
         * @param[in]   fm7bitHigh0 ひとつめの振動値の高周波帯の 7bit FM 符号
         * @param[in]   codesLow    低周波帯の AMFM 符号の配列 (長さは 3 以上)
         * @param[in]   codesHigh   高周波帯の AMFM 符号の配列 (長さは 3 以上)
         */
        static VibrationAmFmPack Make(int fm7bitHigh0, int codesLow[], int codesHigh[]) NN_NOEXCEPT
        {
            return Three7bit::Make(fm7bitHigh0, codesLow, codesHigh, true, true);
        }
    };
};

}} // namespace nn::xcd
