﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   AMFM 符号のデコーダ (内部実装)
 */

#pragma once

#include "xcd_VibrationAmFmCommand.h"

namespace nn { namespace xcd {

/**
 * @brief       AMFM デコーダを表すクラスです。
 * @details
 *  公開ヘッダのデコード関連 API が振幅や周波数を float で扱うのに対し、
 *  このクラスでは高速化のため固定小数点型で扱います。
 *
 *  静的メンバ関数はデコーダコンテクストを引数にとり、
 *  デコーダのインスタンスを生成しなくてもデコード処理を実行できる機能を提供します。@n
 *  静的ではないメンバ関数を実行すると、デコーダのインスタンスが
 *  内部で保持しているデコーダコンテクストに対して処理が実行されます。
 */
class VibrationAmFmDecoder
{
    NN_DISALLOW_COPY(VibrationAmFmDecoder);
    NN_DISALLOW_MOVE(VibrationAmFmDecoder);

public:
    VibrationAmFmDecoder() NN_NOEXCEPT {}
    ~VibrationAmFmDecoder() NN_NOEXCEPT {}

    //!< このデコーダインスタンスをリセットします。
    void Reset(VibrationAmFmFormat format, VibrationAmpFreqT::Type baseFrequency) NN_NOEXCEPT
    {
        Reset(&m_Context, format, baseFrequency);
    }

    //!< AMFM 符号をデコードしてデコーダの内部状態を更新します。
    void Decode(int amFmCode) NN_NOEXCEPT
    {
        Decode(amFmCode, &m_Context);
    }

    //!< 7bit AM 符号をデコードしてデコーダの内部状態を更新します。
    void DecodeAm7bitCode(int amCode) NN_NOEXCEPT
    {
        DecodeAm7bitCode(amCode, &m_Context);
    }

    //!< 7bit FM 符号をデコードしてデコーダの内部状態を更新します。
    void DecodeFm7bitCode(int fmCode) NN_NOEXCEPT
    {
        DecodeFm7bitCode(fmCode, &m_Context);
    }

    //!< デコーダが現在保持している振幅値を取得します。
    VibrationAmpFreqT::Type GetAmplitude() const NN_NOEXCEPT
    {
        return GetAmplitude(&m_Context);
    }

    //!< デコーダが現在保持している周波数値を取得します。
    VibrationAmpFreqT::Type GetFrequency() const NN_NOEXCEPT
    {
        return GetFrequency(&m_Context);
    }

public:
    //!< コンテクストをリセットします。
    static inline void Reset(
        VibrationAmFmDecoderContext* pOutContext,
        VibrationAmFmFormat format,
        VibrationAmpFreqT::Type baseFrequency) NN_NOEXCEPT
    {
        const VibrationAmFmCommand::Config* pConfig = VibrationAmFmCommand::GetConfig(format);
        pOutContext->_pConfig = pConfig;
        pOutContext->_logAmp = pConfig->logResetAmp;
        pOutContext->_logFreq = pConfig->logResetFreq;
        pOutContext->_baseFreq = baseFrequency;
    }

    //!< AMFM 符号をデコードしてコンテクストを更新します。
    static inline void Decode(int amFmCode, VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        const VibrationAmFmCommand::Config* pConfig = reinterpret_cast<const VibrationAmFmCommand::Config*>(pContext->_pConfig);
        const VibrationAmFmCommand::AmFmCommand& cmd = pConfig->pTable[amFmCode];
        pContext->_logAmp = ApplyCommandToAmp(cmd, pContext->_logAmp, pConfig);
        pContext->_logFreq = ApplyCommandToFreq(cmd, pContext->_logFreq, pConfig);
    }

    //!< 7bit AM 符号をデコードしてコンテクストを更新します。
    static inline void DecodeAm7bitCode(int amCode, VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        const float LogAmpCoeffUnder16  =   8.0f / 32.0f;
        const float LogAmpOffsetUnder16 = 248.0f / 32.0f;
        const float LogAmpCoeffUnder32  =   2.0f / 32.0f;
        const float LogAmpOffsetUnder32 = 158.0f / 32.0f;
        const float LogAmpCoeffDefault  =   1.0f / 32.0f;
        const float LogAmpOffsetDefault = 127.0f / 32.0f;
        const VibrationAmFmCommand::Config* pConfig = reinterpret_cast<const VibrationAmFmCommand::Config*>(pContext->_pConfig);

        if(amCode == 0)
        {
            pContext->_logAmp = pConfig->logMinAmp;
        }
        else if(amCode < 16)
        {
            pContext->_logAmp = VibrationLogT::FromFloat(LogAmpCoeffUnder16 * amCode - LogAmpOffsetUnder16);
        }
        else if(amCode < 32)
        {
            pContext->_logAmp = VibrationLogT::FromFloat(LogAmpCoeffUnder32 * amCode - LogAmpOffsetUnder32);
        }
        else if(amCode < 128)
        {
            pContext->_logAmp = VibrationLogT::FromFloat(LogAmpCoeffDefault * amCode - LogAmpOffsetDefault);
        }
    }

    //!< 7bit FM 符号をデコードしてコンテクストを更新します。
    static inline void DecodeFm7bitCode(int fmCode, VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        const float LogFreqCoeff    =  1.0f / 32.0f;
        const float LogFreqOffset   = 64.0f / 32.0f;

        if(fmCode != 0)
        {
            pContext->_logFreq = VibrationLogT::FromFloat(LogFreqCoeff * fmCode - LogFreqOffset);
        }
    }

    //!< コンテクストから振幅値を取得します。
    static inline VibrationAmpFreqT::Type GetAmplitude(const VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        const VibrationAmFmCommand::Config* pConfig = reinterpret_cast<const VibrationAmFmCommand::Config*>(pContext->_pConfig);
        if(pContext->_logAmp < pConfig->logZeroAmp)
        {
            // コンテクストが内部で保持する振幅値がゼロ閾値を下回る場合にはゼロを返す
            return VibrationAmpFreqT::FromFloat(0.0f);
        }
        else
        {
            return VibrationPowTable::CalcPow(pContext->_logAmp);
        }
    }

    //!< コンテクストから周波数値を取得します。
    static inline VibrationAmpFreqT::Type GetFrequency(const VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        // 内部値はベース周波数との比率として保持しているので、ベース周波数と乗算した値を返す
        VibrationAmpFreqT::Type freq = VibrationPowTable::CalcPow(pContext->_logFreq);
        freq = (freq >> BaseFrequencyBitShift) * (pContext->_baseFreq >> (VibrationAmpFreqT::ScaleBit - BaseFrequencyBitShift));
        return freq;
    }

private:
    static const int BaseFrequencyBitShift = 4;     //!< 内部の周波数値とベース周波数の乗算時に用いるビットシフト量

private:
    nn::xcd::VibrationAmFmDecoderContext m_Context; //!< インスタンスが内部に保持するコンテクスト

private:
    //!< 指定された更新処理を現在の値に対して実行した結果を返します。
    static inline VibrationLogT::Type ApplyCommand(
        const VibrationAmFmCommand::AmFmAction& action, //!< 更新処理の種類
        const VibrationLogT::Type& arg,                 //!< 更新処理実行時に用いる引数
        const VibrationLogT::Type& current,             //!< 現在の値の対数
        const VibrationLogT::Type& reset,               //!< リセット値の対数
        const VibrationLogT::Type& min,                 //!< 最小値の対数
        const VibrationLogT::Type& max                  //!< 最大値の対数
        ) NN_NOEXCEPT
    {
        switch(action)
        {
        case VibrationAmFmCommand::AmFmAction::None:
            return current;
        case VibrationAmFmCommand::AmFmAction::Reset:
            return reset;
        case VibrationAmFmCommand::AmFmAction::Set:
            return arg;
        case VibrationAmFmCommand::AmFmAction::Multiply:
            // 処理対象が対数なので、乗算処理が指定された場合は加算処理を実行する
            return VibrationLogT::GetClamped(VibrationLogT::GetSum(current, arg), min, max);
        default:
            return reset;
        }
    }

    //!< AMFM コマンドにしたがって現在の振幅値を更新した結果を返します。
    static inline VibrationLogT::Type ApplyCommandToAmp(
        const VibrationAmFmCommand::AmFmCommand& cmd,   //!< AMFM コマンド
        const VibrationLogT::Type& currentLogAmp,       //!< 現在の振幅値の対数
        const VibrationAmFmCommand::Config* pConfig) NN_NOEXCEPT
    {
        return ApplyCommand(
            cmd.amAction, cmd.amArg, currentLogAmp,
            pConfig->logResetAmp, pConfig->logMinAmp, pConfig->logMaxAmp);
    }

    //!< AMFM コマンドにしたがって現在の周波数値を更新した結果を返します。
    static inline VibrationLogT::Type ApplyCommandToFreq(
        const VibrationAmFmCommand::AmFmCommand& cmd,   //!< AMFM コマンド
        const VibrationLogT::Type& currentLogFreq,      //!< 現在の周波数値の対数
        const VibrationAmFmCommand::Config* pConfig) NN_NOEXCEPT
    {
        return ApplyCommand(
            cmd.fmAction, cmd.fmArg, currentLogFreq,
            pConfig->logResetFreq, pConfig->logMinFreq, pConfig->logMaxFreq);
    }
};

}} // namespace nn::xcd
