﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <new>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include "xcd_TeraBase.h"
#include "xcd_TeraStateMachine.h"
#include "xcd_ReportTypes.h"
#include "detail/xcd_TeraBaseParser.h"
#include "detail/xcd_TeraCommon.h"

namespace nn { namespace xcd {

namespace
{

// ステート遷移のタイムアウト時間
// この時間が経過してもステートが変化しない場合はリトライする。値は暫定。
const auto StateSetTimeout = nn::TimeSpan::FromMilliSeconds(500);

// 無効な InputReport/McuRead を受信し続けた際に IAP 破損と見なすタイムアウト。値は暫定。
const auto InvalidInputTimeoutForIapBroken = nn::TimeSpan::FromSeconds(5);

// バージョン読み込みを諦めて IAP 破損と見なすまでのタイムアウト
// * InputReport/McuRead すら取れない状況下でハマらないための安全機構。
const auto ForceTimeoutForVersionReading = nn::TimeSpan::FromSeconds(10);

}  // anonymous

// デバッグ設定の初期値は All OFF
TeraBase::DebugOptionSet TeraBase::g_DebugOptions = TeraBase::DebugOptionSet();

void TeraBase::SetMcuBrokenEmulationEnabled(bool isEnabled) NN_NOEXCEPT
{
    g_DebugOptions.Set<DebugOption::McuBrokenEmulation>(isEnabled);
}

void TeraBase::SetFirmwareUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT
{
    ITeraUpdater::SetUpdateFailureEmulationEnabled(isEnabled);
}

void TeraBase::Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    Peripheral::Activate(type, firmwareVersion);

    // TODO: Tera が載っていないデバイスでは各種操作ができないようにする

    m_McuFirmwareVersion.SetDeviceType(type);

    // 状態管理クラスの初期化
    // コールバックの引数に自身を渡してもらうようにする
    m_StateMachine.Activate(type, m_pCommand, StateTransitionCallback, this);

    // コマンドハンドラの登録
    m_IrsensorBase.SetCommandHandler(m_pCommand);

    // ファームウェア更新モジュールの起動
    ActivateFirmwareUpdater();

    // 内部状態をクリアする
    ClearInternalState();
    m_McuFirmwareVersion.Invalidate();
    m_StateMachine.ClearInternalState();

    m_pUpdateFinishEvent     = nullptr;
    m_pSetDataFormatFunction = nullptr;
    m_pSetDataFormatArgument = nullptr;
}

void TeraBase::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // 切断時の処理
    // TeraMCU は切断時に BTMCU によって Standby 状態に入れられるため、次回起動時には
    // 再起動が必要になる。 そのため内部状態を初期状態と同じになるようにクリアしておく。
    // 既に取得していたイベントがある場合は、Signal してから破棄する。
    ClearInternalState();
    m_McuFirmwareVersion.Invalidate();
    m_StateMachine.ClearInternalState();
    NotifyGotMcuFirmwareVersion();

    // アップデート終了時に Standby 遷移するため、ClearInternalState ではなく Deactivate でクリアする
    m_pUpdateFinishEvent     = nullptr;
    m_pSetDataFormatFunction = nullptr;
    m_pSetDataFormatArgument = nullptr;

    // NFC の終了処理 (念のため)
    m_NfcProcessor.Deactivate();

    // 状態管理クラスの終了
    m_StateMachine.Deactivate();

    // ファームウェア更新モジュールの破棄
    if (m_pFirmwareUpdater != nullptr)
    {
        DeactivateFirmwareUpdater();
        m_pFirmwareUpdater->~ITeraUpdater();
        m_pFirmwareUpdater = nullptr;
    }

    m_Type      = DeviceType_Unknown;
    m_Activated = false;

    Peripheral::Deactivate();
}

void TeraBase::SetHidAccessor(detail::HidAccessor* pHidAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHidAccessor);

    m_pHidAccessor = pHidAccessor;
}

void TeraBase::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    if (result.IsFailure())
    {
        return;
    }

    if (id == Command_McuResume.Id)
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        if (!m_StateMachine.IsMcuReady() && !m_McuFirmwareVersion.IsValid())
        {
            SendNopCommandByMcuWrite();
        }
    }
}

void TeraBase::NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    ParseInputByMcuRead(pBuffer, size);
}

nn::Result TeraBase::ActivatePeripheral(detail::InternalMcuState nextMcuState) NN_NOEXCEPT
{
    switch (nextMcuState)
    {
    case detail::InternalMcuState_Ir:
        {
            // IR センサーの初期化
            m_IrsensorBase.Activate(m_Type, m_FirmwareVersion);
        }
        break;
    case detail::InternalMcuState_Nfc:
        {
            // NFC の初期化
            m_NfcProcessor.Activate(m_Type, m_FirmwareVersion);
        }
        break;
    case detail::InternalMcuState_Standby:
        {
            // Tera MCU は終了しているため、これ以降の処理も含めて何もしない
            NN_RESULT_THROW(nn::xcd::ResultInvalidMcuState());
        }
        break;
    case detail::InternalMcuState_Background:
        // 今のところ何もしない
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

void TeraBase::DeactivatePeripheral() NN_NOEXCEPT
{
    m_CommonResponseStatus.Clear();

    // IR センサーの終了処理 (初期化されていない場合は何もしない)
    m_IrsensorBase.Deactivate();

    // NFC の終了処理
    m_NfcProcessor.Deactivate();
}

void TeraBase::ActivateFirmwareUpdater() NN_NOEXCEPT
{
    // 既存のインスタンスを破棄
    DeactivateFirmwareUpdater();
    if (m_pFirmwareUpdater != nullptr)
    {
        m_pFirmwareUpdater->~ITeraUpdater();
    }

    if (IsAvailableFastFirmwareUpdate())
    {
        // 高速版
        m_pFirmwareUpdater = new(m_FirmwareUpdaterStorage) detail::TeraUpdater();
    }
    else
    {
        // 低速版 (互換用)
        m_pFirmwareUpdater = new(m_FirmwareUpdaterStorage) detail::TeraUpdaterLegacy();
    }

    m_pFirmwareUpdater->Activate(m_Type, FirmwareUpdateCallback, this);
}

void TeraBase::DeactivateFirmwareUpdater() NN_NOEXCEPT
{
    if (m_pFirmwareUpdater != nullptr)
    {
        m_pFirmwareUpdater->Deactivate();
    }
}

nn::Result TeraBase::SetMcuStateImpl(McuState mcuState, bool isHidCommandOnly) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        mcuState == McuState_Standby ||
        mcuState == McuState_Background ||
        mcuState == McuState_Nfc ||
        mcuState == McuState_Ir);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // バージョンチェックが動作中の場合はエラーを返す
    NN_RESULT_THROW_UNLESS(
        !m_IsVersionReading,
        nn::xcd::ResultMcuBusy());

    // 遷移先ステートの設定
    // Tera の起動も SetDestinationStateXxx() 内で行う
    NN_RESULT_DO(
        isHidCommandOnly ?
        m_StateMachine.SetDestinationStateWithoutMcuInput(mcuState) :
        m_StateMachine.SetDestinationState(mcuState)
    );

    // 以前 FW 破損判定されていた場合は、再起動時にバージョンを取得し直す (誤判定対策)
    if (m_McuFirmwareVersion.IsCorrupted() &&
        mcuState != McuState_Standby)
    {
        m_McuFirmwareVersion.Invalidate();
    }

    // ステート切り替え時は MCU がリセットされるので、IAP 破損判定を止める
    m_IapBrokenCheckCounter.Stop();
    m_GetVersionTimeoutCounter.Stop();

    m_pOutputReporter = nullptr;

    NN_RESULT_SUCCESS;
}

void TeraBase::SendNopCommandByMcuWrite() NN_NOEXCEPT
{
    // MCU Write で送るコマンドのペイロード長
    // (BluetoothTransferSizeList の定義は配列のサイズに使えない)
    const size_t payloadLengthForMcuWrite = 38;

    uint8_t buf[payloadLengthForMcuWrite];
    auto commandSize = detail::CreateTeraControlNopCommand(buf, sizeof(buf), detail::BluetoothMode_McuWrite);
    m_pCommand->McuWrite(buf, commandSize, this);
}

nn::Result TeraBase::ExtControl(bool isOn, ICommandListener* pListener) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_RESULT_DO(m_StateMachine.CheckMcuState(McuState_Background));
    // MCU Write で送るコマンドのペイロード長
    // (BluetoothTransferSizeList の定義は配列のサイズに使えない)
    const size_t payloadLengthForMcuWrite = 38;

    uint8_t buf[payloadLengthForMcuWrite];
    auto commandSize = detail::CreateTeraExtControlCommand(buf, sizeof(buf), detail::BluetoothMode_McuWrite, isOn);
    m_pCommand->McuWrite(buf, commandSize, pListener);

    NN_RESULT_SUCCESS;
}

void TeraBase::ParseInputByMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // Tera が通常起動しており、ステート遷移判定も不要な場合は何もしない
    if (m_StateMachine.IsMcuReady() &&
        !m_StateMachine.IsTransitionWithoutMcuInput())
    {
        return;
    }

    InputReportBasicInfo info;
    ParseInputReportBasicInfo(&info, detail::BluetoothMode_McuWrite, pBuffer, size, 0);
    bool isCommonProtocol =
        info.isValid &&
        info.needsParse &&
        (info.protocol.GetProtocolType() == detail::CommandProtocol_Common);
    if (isCommonProtocol)
    {
        ParseCommonInputReport(pBuffer, size, 0);
    }
    else if (!info.isValid)
    {
        UpdateIapBrokenCheck(true);
    }

    if (m_StateMachine.IsTransitionWithoutMcuInput())
    {
        // HidCommand によるステート遷移処理中の場合は StateMachine へ通知
        if (m_StateMachine.IsProcessing())
        {
            m_StateMachine.UpdateStateByMcuRead(
                pBuffer,
                size,
                static_cast<detail::InternalMcuState>(m_CommonResponseStatus.state));

            // MCU Write が送られなかった場合は、状態更新のために NOP を送る
            if (!m_StateMachine.IsMcuWriteSent())
            {
                SendNopCommandByMcuWrite();
            }
        }
    }
    else if (m_McuFirmwareVersion.IsValid())
    {
        // バージョン情報が確定したら Tera を寝かせる
        // FullKey の場合は起動状態を維持する必要があるのでそのまま
        if (m_Type != DeviceType_FullKey)
        {
            m_pCommand->McuResume(McuResumeValueType_Suspend, this);
        }
    }
    else
    {
        // バージョン情報が確定していなければ再度 NOP を送る
        SendNopCommandByMcuWrite();
    }
}

void TeraBase::ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_EQUAL(size, McuInReportSize_Payload);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // MCU 停止中は解析しない
    if (!m_StateMachine.IsMcuReady())
    {
        if (!m_IsVersionReading)
        {
            StopIapBrokenCheck();
        }
        return;
    }

    if (!ParseInputReportImpl(pBuffer, size, sampleNumber))
    {
        // 無効なパケットを受信したら IAP 破損判定を行う
        UpdateIapBrokenCheck(true);
    }
}

void TeraBase::UpdateIapBrokenCheck(bool isInParsing) NN_NOEXCEPT
{
    // FW バージョンが取得できたら破損判定を中断
    if (m_McuFirmwareVersion.IsValid())
    {
        StopIapBrokenCheck();
        return;
    }

    // パケットの解析中なら IAP 応答のチェックを開始する
    if (isInParsing && !m_IapBrokenCheckCounter.IsStarted())
    {
        m_IapBrokenCheckCounter.Start(InvalidInputTimeoutForIapBroken);
    }

    if (!m_GetVersionTimeoutCounter.IsStarted())
    {
        m_GetVersionTimeoutCounter.Start(ForceTimeoutForVersionReading);
    }

    // 正常な応答が規定時間得られない、またはバージョン取得に時間がかかりすぎている場合は IAP 破損と見なす
    if (m_IapBrokenCheckCounter.IsExpired() || m_GetVersionTimeoutCounter.IsExpired())
    {
        m_McuFirmwareVersion.SetIapCorruption();

        NotifyGotMcuFirmwareVersion();
        NotifyFirmwareUpdateFinished();
        m_StateMachine.AbortStateTransition();
    }
}

bool TeraBase::ParseInputReportImpl(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    if (ProcessReservedAction())
    {
        // 予約済みアクションが発動した場合は他の処理はしない
        return true;
    }

    InputReportBasicInfo info;
    ParseInputReportBasicInfo(
        &info,
        detail::BluetoothMode_Common,
        pBuffer,
        size,
        sampleNumber);

    // プロトコル毎の解析が不要か、不正パケットなら解析しない
    if (!(info.needsParse && info.isValid))
    {
        return info.isValid;
    }

    switch (info.protocol.GetProtocolType())
    {
    case detail::CommandProtocol_Common:
        {
            ParseCommonInputReport(pBuffer, size, sampleNumber);
            m_StateMachine.UpdateState(static_cast<detail::InternalMcuState>(m_CommonResponseStatus.state));
        }
        break;
    case detail::CommandProtocol_Nfc:
        {
            m_NfcProcessor.ParseInputReport(pBuffer, size, sampleNumber);
        }
        break;
    case detail::CommandProtocol_Ir:
        {
            m_IrsensorBase.ParseInputReport(pBuffer, size, sampleNumber);
        }
        break;
    case detail::CommandProtocol_Idle:
        // 何もしない
        break;
    default:
        {
            // 不正なプロトコルは解析しない
            // (例えば、全データが 0x00 のパケットの場合は CRC チェックを通過してここに入る)
            NN_XCD_TERA_LOG(
                "Invalid protocol (%d). Ignored.\n",
                info.protocol.GetProtocolType());
            return false;
        }
    }

    return true;
}

void TeraBase::ParseInputReportBasicInfo(
    InputReportBasicInfo* pOutInfo,
    detail::BluetoothMode bluetoothMode,
    const uint8_t* pBuffer,
    size_t size,
    uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(sampleNumber);

    if (m_StateMachine.IsSystemBoot())
    {
        // システムブート中は何もしない
        pOutInfo->isValid    = true;
        pOutInfo->needsParse = false;
        return;
    }

    if (g_DebugOptions.Test<DebugOption::McuBrokenEmulation>())
    {
        // MCU 破損エミュレーション時は全てエラー扱いする
        pOutInfo->isValid    = false;
        pOutInfo->needsParse = false;
        return;
    }

    pOutInfo->protocol.SetValue(pBuffer[detail::McuCommonInputOffsetByte_MisoProtocol]);

    // 特殊なプロトコルの判定
    switch (pOutInfo->protocol.GetValue())
    {
    case detail::IapProtocolValue:
        {
            // IAP モードで動作中
            ParseIapInputReport(pBuffer, size, sampleNumber);
            pOutInfo->isValid    = true;
            pOutInfo->needsParse = false;
        }
        return;
    case detail::NoDataProtocolValue:
        {
            // MCU から正常にデータを受け取れていない場合はエラー扱いして無視する
            NN_XCD_TERA_LOG("No data. Ignored.\n");
            pOutInfo->isValid    = false;
            pOutInfo->needsParse = false;
        }
        return;
    default:
        // 通常のプロトコル
        break;
    }

    // CRC 不一致の場合はエラー扱いして無視する
    if (!detail::IsCrc8Match(pBuffer, size, bluetoothMode))
    {
        pOutInfo->isValid    = false;
        pOutInfo->needsParse = false;
        return;
    }

    // 有効なデータ
    pOutInfo->isValid    = true;
    pOutInfo->needsParse = true;
}

void TeraBase::ParseCommonInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!detail::TeraBaseParser::ParseCommonInputReport(
        &m_CommonResponseStatus,
        pBuffer,
        size))
    {
        return;
    }

    if (!m_McuFirmwareVersion.IsValid())
    {
        m_McuFirmwareVersion.SetVersion(
            m_CommonResponseStatus.version.major,
            m_CommonResponseStatus.version.minor);
        NotifyGotMcuFirmwareVersion();
    }
}

void TeraBase::ParseIapInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);

    uint8_t resultCode;
    if (!detail::TeraBaseParser::ParseIapInputReport(&resultCode, pBuffer, size))
    {
        return;
    }

    DispatchIapResult(resultCode);
}

void TeraBase::DispatchIapResult(uint8_t resultCode) NN_NOEXCEPT
{
    switch (resultCode)
    {
    case detail::McuIapResultCode_Corrupted:
        {
            // FW が壊れている
            ProcessIapCorruption();
        }
        break;
    case detail::McuIapResultCode_Validating:
        {
            // CRC 検証中は何もしない
            NN_XCD_TERA_LOG("[IAP] Validating...\n");
            m_McuFirmwareVersion.ClearCustomerCodeCorruptionCount();
        }
        break;
    case detail::McuIapResultCode_ResetRequired:
        {
            // CRC 検証完了後は MCU リセットが必要
            NN_XCD_TERA_LOG("[IAP] Need reset\n");
            m_McuFirmwareVersion.Invalidate();
            m_StateMachine.Reset([](ICommandListener* pListener)
            {
                // 既に Customer code が正しいことは検証されているため、
                // タイムアウト時もアップデートは完了扱いにする
                NN_XCD_TERA_LOG("[IAP] Reset timed out\n");
                auto* pSelf = static_cast<TeraBase*>(pListener);
                pSelf->NotifyFirmwareUpdateFinished();
            }, this);
        }
        break;
    default:
        {
            // 未知のリザルトコードは無視する
            m_McuFirmwareVersion.ClearCustomerCodeCorruptionCount();
        }
        break;
    }
}

void TeraBase::ProcessIapCorruption() NN_NOEXCEPT
{
    if (m_McuFirmwareVersion.IsValid())
    {
        return;
    }

    // アップデート進行中は破損チェックをしない
    if (m_IsMcuFirmwareUpdating)
    {
        McuUpdateStateInfo info;
        m_pFirmwareUpdater->GetState(&info);
        if (info.state != McuUpdateState_Reboot &&
            info.state != McuUpdateState_End)
        {
            return;
        }
    }

    // 規定回数 FW 破損応答を受けたら破損確定
    m_McuFirmwareVersion.ProceedCustomerCodeCorruptionCount();
    if (m_McuFirmwareVersion.IsCorrupted())
    {
        NN_XCD_TERA_LOG("[IAP] Corrupted\n");

        NotifyGotMcuFirmwareVersion();
        NotifyFirmwareUpdateFinished();
        m_StateMachine.AbortStateTransition();
    }
}

size_t TeraBase::GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (g_DebugOptions.Test<DebugOption::McuBrokenEmulation>())
    {
        // MCU 破損エミュレーション時は何も送らない
        return 0;
    }

    if (!m_StateMachine.IsMcuReady())
    {
        if (m_Type == DeviceType_FullKey)
        {
            // フルキーコンの場合は、振動を維持するために NOP を送る
            return GetCommonOutput(pOutValue, size);
        }
        else
        {
            return 0;
        }
    }

    // Initializing ステート中は常に TERA_CONTROL を発行
    if (m_StateMachine.IsMcuInitializing())
    {
        return GetCommonOutput(pOutValue, size);
    }

    if (m_pOutputReporter == nullptr)
    {
        return GetCommonOutput(pOutValue, size);
    }
    else
    {
        return m_pOutputReporter->GetOutputReport(pOutValue, size);
    }
}

size_t TeraBase::GetCommonOutput(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    return detail::CreateTeraControlNopCommand(pOutValue, size, detail::BluetoothMode_Common);
}

McuState TeraBase::GetMcuState() NN_NOEXCEPT
{
    return m_StateMachine.GetCurrentState();
}

nn::Result TeraBase::SetMcuState(nn::xcd::McuState mcuState) NN_NOEXCEPT
{
    return SetMcuStateImpl(mcuState, false);
}

nn::Result TeraBase::SetMcuStateImmediate(nn::xcd::McuState mcuState) NN_NOEXCEPT
{
    return SetMcuStateImpl(mcuState, true);
}

bool TeraBase::IsAvailableFastFirmwareUpdate() NN_NOEXCEPT
{
    // JoyRight の BT FW が 03.82 以降の場合のみ高速更新可能
    if (m_Type == nn::xcd::DeviceType_Right &&
        m_FirmwareVersion.bluetooth >= FirmwareVersionBt_0382)
    {
        return true;
    }

    return false;
}

nn::Result TeraBase::RequestMcuFirmwareVersion() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // バージョン取得中なら即時 NN_RESULT_SUCCESS で返す
    if (m_IsVersionReading == true)
    {
        NN_RESULT_SUCCESS;
    }

    m_IsVersionReading = true;

    // 既にバージョン取得済みなら即通知
    if (m_McuFirmwareVersion.IsValid())
    {
        NotifyGotMcuFirmwareVersion();
    }
    else
    {
        if (m_Type == DeviceType_FullKey)
        {
            // FullKey は常に Tera が起動しているので、即バージョン取得を開始
            SendNopCommandByMcuWrite();
        }
        else if (!m_StateMachine.IsMcuReady())
        {
            // Tera が起動していない場合はバージョン取得のために起動
            // (公には起動していない状態として扱うため、StateMachine には通知しない)
            m_pCommand->McuResume(McuResumeValueType_Resume, this);
        }

        // バージョン取得中断判定を開始
        m_GetVersionTimeoutCounter.Start(ForceTimeoutForVersionReading);
    }

    NN_RESULT_SUCCESS;
}

void TeraBase::ClearInternalState() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // フィールドのリセット
    m_CommonResponseStatus.Clear();
    m_pOutputReporter       = nullptr;
    m_IsMcuFirmwareUpdating = false;
    m_IsVersionReading      = false;
    m_ReservedAction.Clear();
    StopIapBrokenCheck();
}

void TeraBase::NotifyGotMcuFirmwareVersion() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // バージョン取得が完了したので中断判定を止める
    StopIapBrokenCheck();

    m_IsVersionReading = false;
}

nn::Result TeraBase::CheckFirmwareVersionValidity() NN_NOEXCEPT
{
    // バージョン取得の中断判定
    if (m_GetVersionTimeoutCounter.IsStarted())
    {
        UpdateIapBrokenCheck(false);
    }

    // FW バージョンの取得が完了していない
    NN_RESULT_THROW_UNLESS(
        m_McuFirmwareVersion.IsValid(),
        nn::xcd::ResultMcuVersionNotAvailable());

    NN_RESULT_SUCCESS;
}

bool TeraBase::ProcessReservedAction() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!m_ReservedAction.isEnabled)
    {
        return false;
    }
    else if (!m_ReservedAction.IsTimerExpired())
    {
        // 待機時間を満了していない
        return false;
    }

    switch (m_ReservedAction.action)
    {
    case ReservedActionType::StartFwUpdate:
        if (m_StateMachine.IsSystemBoot())
        {
            NN_XCD_TERA_LOG("Starting update sequence\n");

            // Tera のシステムブート待ちが終了したので、ブートローダとの同期を開始
            m_pFirmwareUpdater->StartSyncWithBootLoader();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_ReservedAction.Clear();
    return true;
}

nn::Result TeraBase::GetMcuFirmwareVersion(McuVersionData* pOutVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_RESULT_DO(CheckFirmwareVersionValidity());

    m_McuFirmwareVersion.GetVersion(pOutVersion);

    NN_RESULT_SUCCESS;
}

nn::Result TeraBase::GetMcuFirmwareVersionForNfc(McuVersionDataForNfc* pOutVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(CheckFirmwareVersionValidity());

    // NFC 向けには、IAP 破損は HW 異常として扱う
    NN_RESULT_THROW_UNLESS(
        !m_McuFirmwareVersion.IsIapCorrupted(),
        nn::xcd::ResultMcuHardwareError());

    NN_RESULT_THROW_UNLESS(
        !m_McuFirmwareVersion.IsCorrupted(),
        nn::xcd::ResultMcuFirmwareCorrupted());

    m_McuFirmwareVersion.GetVersion(pOutVersion);

    NN_RESULT_SUCCESS;
}

nn::Result TeraBase::StartMcuFirmwareUpdate(
    const FirmwareImage& image,
    nn::os::SystemEventType* pFinishEvent,
    ITeraUpdater::UpdateMode updateMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinishEvent);
    NN_SDK_REQUIRES_NOT_NULL(m_pSetDataFormatFunction);
    NN_SDK_REQUIRES_NOT_NULL(m_pSetDataFormatArgument);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    ActivateFirmwareUpdater();

    NN_RESULT_TRY(m_pFirmwareUpdater->SetFirmwareImage(image, updateMode))
        NN_RESULT_CATCH(nn::xcd::ResultMcuInvalidFirmwareImage)
        {
            // Updater を止めて戻る
            DeactivateFirmwareUpdater();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    m_IsMcuFirmwareUpdating = true;
    m_pUpdateFinishEvent    = pFinishEvent;

    // アップデート用の DataFormat に設定
    m_pSetDataFormatFunction(
        nn::xcd::PeriodicDataFormat_McuUpdate,
        m_pSetDataFormatArgument
    );

    // TODO: DataFormat 変更を待てるなら待った方が良い
    m_StateMachine.SetDestinationStateForFirmwareUpdate(McuState_FirmwareUpdate);

    // 高速通信モードを有効化
    m_pHidAccessor->SetFastModeEnabled(true);

    NN_RESULT_SUCCESS;
}

void TeraBase::SetChangeDataFormatFunction(DataFormatFunctionType function, void* pArgument) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(function);
    NN_SDK_REQUIRES_NOT_NULL(pArgument);

    m_pSetDataFormatFunction = function;
    m_pSetDataFormatArgument = pArgument;
}

void TeraBase::SetNfcEvent(nn::os::NativeHandle* pOutCommonHandle, nn::os::NativeHandle* pOutDetectHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_NfcProcessor.SetEvent(pOutCommonHandle, pOutDetectHandle);
}

void TeraBase::ProcessStateTransition(detail::InternalMcuState state) NN_NOEXCEPT
{
    switch (state)
    {
    case detail::InternalMcuState_Idle:
        if (m_IsMcuFirmwareUpdating)
        {
            NN_XCD_TERA_LOG("Transit to IDLE\n");

            NotifyFirmwareUpdateFinished();
        }
        break;
    case detail::InternalMcuState_Standby:
        {
            DeactivatePeripheral();
            ClearInternalState();
        }
        break;
    case detail::InternalMcuState_SystemBoot:
        {
            NN_XCD_TERA_LOG("Transit to system boot\n");

            // システムブート状態で操作可能になるまで待ってから次の処理を行う
            m_ReservedAction.Set(
                ReservedActionType::StartFwUpdate,
                nn::TimeSpan::FromMilliSeconds(200));
        }
        break;
    default:
        {
            ActivatePeripheral(state);
        }
        break;
    }
}

void TeraBase::ProcessFirmwareUpdateForward(ITeraUpdater::UpdatePhase phase) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    switch (phase)
    {
    case ITeraUpdater::UpdatePhase::BootFinish:
        {
            // 起動処理が完了したら ROM を消去
            m_McuFirmwareVersion.Invalidate();
            m_pFirmwareUpdater->StartEraseRom();
        }
        break;
    case ITeraUpdater::UpdatePhase::EraseFinish:
        {
            // 消去が完了したら書き込みを開始
            m_pFirmwareUpdater->StartWriteRom();
        }
        break;
    case ITeraUpdater::UpdatePhase::FinalizeFinish:
    case ITeraUpdater::UpdatePhase::Error:
        {
            if (phase == ITeraUpdater::UpdatePhase::FinalizeFinish)
            {
                NN_XCD_TERA_LOG("Finished! Reboot for get version\n");
            }
            else
            {
                NN_XCD_TERA_LOG("An error has occurred\n");
            }

            m_McuFirmwareVersion.Invalidate();

            // バージョンチェックのため、DataFormat を MCU に戻してシステムブートを抜ける
            m_pSetDataFormatFunction(
                nn::xcd::PeriodicDataFormat_MCU,
                m_pSetDataFormatArgument);
            m_StateMachine.SetDestinationStateForFirmwareUpdate(
                nn::xcd::McuState_Idle);
        }
        break;
    default:
        // 何もしない
        break;
    }
}

void TeraBase::NotifyFirmwareUpdateFinished() NN_NOEXCEPT
{
    if (!m_IsMcuFirmwareUpdating)
    {
        return;
    }

    // 高速通信モードを終了
    m_pHidAccessor->SetFastModeEnabled(false);

    // アップデート完了処理
    m_pFirmwareUpdater->Finish();

    // Standby に戻す
    m_StateMachine.SetDestinationState(nn::xcd::McuState_Standby);

    // DataFormat を Basic に戻しておく
    m_pSetDataFormatFunction(
        nn::xcd::PeriodicDataFormat_Basic,
        m_pSetDataFormatArgument);

    m_IsMcuFirmwareUpdating = false;
    DeactivateFirmwareUpdater();

    // 完了イベントの通知
    if (m_pUpdateFinishEvent != nullptr)
    {
        nn::os::SignalSystemEvent(m_pUpdateFinishEvent);
    }
}

void TeraBase::StateTransitionCallback(detail::InternalMcuState state, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    // 引数は TeraBase のインスタンスを指している
    auto* pTeraBase = static_cast<TeraBase*>(pListener);
    pTeraBase->ProcessStateTransition(state);
}

void TeraBase::FirmwareUpdateCallback(ITeraUpdater::UpdatePhase phase, void* pArgument) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pArgument);

    // 引数は TeraBase のインスタンスを指している
    auto* pTeraBase = reinterpret_cast<TeraBase*>(pArgument);
    pTeraBase->ProcessFirmwareUpdateForward(phase);
}

void TeraBase::ParseMcuUpdateInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsActivated())
    {
        return;
    }

    // MCU Update In パケット受信中は IAP 破損判定を止める
    StopIapBrokenCheck();

    if (ProcessReservedAction())
    {
        // 予約済みアクションが発動した場合は他の処理はしない
        return;
    }

    if (g_DebugOptions.Test<DebugOption::McuBrokenEmulation>())
    {
        // MCU 破損エミュレーション時はデータを空にする
        uint8_t buffer[McuUpdateInReportSize_Payload] = {};
        m_pFirmwareUpdater->ParseMcuUpdateInputReport(buffer, sizeof(buffer), sampleNumber);
    }
    else
    {
        m_pFirmwareUpdater->ParseMcuUpdateInputReport(pBuffer, size, sampleNumber);
    }

#if 0
    // データをパース
    NN_SDK_LOG("[%d] Mcu Update Data : ", sampleNumber);
    for (int i = 0; i < size; i++)
    {
        NN_SDK_LOG("%02x ", pBuffer[i]);
    }
    NN_SDK_LOG("\n");
#endif
}

size_t TeraBase::GetMcuUpdateOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsActivated())
    {
        return 0;
    }

    if (g_DebugOptions.Test<DebugOption::McuBrokenEmulation>())
    {
        // MCU 破損エミュレーション時は何も送らない
        return 0;
    }

    return m_pFirmwareUpdater->GetMcuUpdateOutputReport(pOutValue, size);
}

}} // namespace nn::xcd
