﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/xcd/xcd.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/TargetConfigs/build_Base.h>
#include "detail/xcd_CreateSystemServer.h"

namespace nn { namespace xcd {

namespace {

// MCU アクセスのためのサービスオブジェクト参照を取得
nn::sf::SharedPointer<detail::ISystemServer> GetSystemServer() NN_NOEXCEPT
{
    static nn::sf::SharedPointer<detail::ISystemServer> s_pServer = nullptr;

    if (s_pServer == nullptr)
    {
        s_pServer =
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            detail::CreateSystemServerByHipc();
#else
            // HOS 以外は DFC を使用
            detail::CreateSystemServerByDfc();
#endif
    }

    return s_pServer;
}

}  // anonymous namespace

Result GetDataFormat(PeriodicDataFormat* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    std::uint8_t dataFormat;
    auto result = GetSystemServer()->GetDataFormat(
        &dataFormat,
        handle);

    *pOutValue = static_cast<PeriodicDataFormat>(dataFormat);

    return result;
}

nn::Result SetDataFormat(nn::xcd::PeriodicDataFormat dataFormat, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->SetDataFormat(
        dataFormat,
        handle);
}

nn::Result GetMcuState(McuState* pOutMcuState, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuState);

    std::uint8_t mcuState;
    auto result = GetSystemServer()->GetMcuState(
        &mcuState,
        handle);

    *pOutMcuState = static_cast<McuState>(mcuState);

    return result;
}

nn::Result SetMcuState(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->SetMcuState(
        static_cast<std::uint8_t>(mcuState),
        handle);
}

nn::Result SetMcuStateImmediate(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->SetMcuStateImmediate(
        static_cast<std::uint8_t>(mcuState),
        handle);
}

nn::Result GetMcuVersionForNfc(McuVersionDataForNfc* pOutMcuVersionData, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuVersionData);

    return GetSystemServer()->GetMcuVersionForNfc(
        pOutMcuVersionData,
        handle);
}

nn::Result CheckNfcDevicePower(DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->CheckNfcDevicePower(
        handle);
}

nn::Result SetNfcEvent(nn::os::SystemEventType* pNfcEvent, nn::os::SystemEventType* pNfcDetectEvent, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNfcEvent);
    NN_SDK_REQUIRES_NOT_NULL(pNfcDetectEvent);

    nn::sf::NativeHandle sfHandle;
    nn::sf::NativeHandle sfHandleForDetect;

    NN_RESULT_DO(GetSystemServer()->SetNfcEvent(
        &sfHandle,
        &sfHandleForDetect,
        handle));

    // 取得したハンドルをイベントに関連付ける
    nn::os::AttachReadableHandleToSystemEvent(
        pNfcEvent,
        sfHandle.GetOsHandle(),
        sfHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear);
    nn::os::AttachReadableHandleToSystemEvent(
        pNfcDetectEvent,
        sfHandleForDetect.GetOsHandle(),
        sfHandleForDetect.IsManaged(),
        nn::os::EventClearMode_ManualClear);

    // 渡されたイベントにハンドルの管理を移管したので、管理権を放棄
    sfHandle.Detach();
    sfHandleForDetect.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result GetNfcInfo(NfcInfo* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->GetNfcInfo(
        pOutValue,
        handle);
}

nn::Result StartNfcDiscovery(const NfcDiscoveryParameter& nfcDiscoveryParam, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->StartNfcDiscovery(
        nfcDiscoveryParam,
        handle);
}

nn::Result StopNfcDiscovery(DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->StopNfcDiscovery(
        handle);
}

nn::Result StartNtagRead(const NtagReadParameter& ntagReadParam, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->StartNtagRead(
        ntagReadParam,
        handle);
}

nn::Result StartNtagWrite(const NtagWriteParameter& ntagWriteParam, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->StartNtagWrite(
        ntagWriteParam,
        handle);
}

nn::Result SendNfcRawData(const NfcPassThruParameter& passThruParam, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->SendNfcRawData(
        passThruParam,
        handle);
}

nn::Result RegisterMifareKey(const MifareKeyWriteParameter& keyWriteParameter, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->RegisterMifareKey(
        keyWriteParameter,
        handle);
}

nn::Result ClearMifareKey(const MifareKeyClearParameter& keyClearParameter, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->ClearMifareKey(
        keyClearParameter,
        handle);
}

nn::Result StartMifareRead(const MifareReadParameter& readParameter, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->StartMifareRead(
        readParameter,
        handle);
}

nn::Result StartMifareWrite(const MifareWriteParameter& writeParameter, DeviceHandle handle) NN_NOEXCEPT
{
    return GetSystemServer()->StartMifareWrite(
        writeParameter,
        handle);
}

AwakeTriggerReason GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT
{
    uint64_t outValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemServer()->GetAwakeTriggerReasonForLeftRail(&outValue));

    switch (outValue)
    {
    case AwakeTriggerReason_None:
        return AwakeTriggerReason_None;
    case AwakeTriggerReason_HomePressed:
        return AwakeTriggerReason_HomePressed;
    case AwakeTriggerReason_BatteryLevelChanged:
        return AwakeTriggerReason_BatteryLevelChanged;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

AwakeTriggerReason GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT
{
    uint64_t outValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemServer()->GetAwakeTriggerReasonForRightRail(&outValue));

    switch (outValue)
    {
    case AwakeTriggerReason_None:
        return AwakeTriggerReason_None;
    case AwakeTriggerReason_HomePressed:
        return AwakeTriggerReason_HomePressed;
    case AwakeTriggerReason_BatteryLevelChanged:
        return AwakeTriggerReason_BatteryLevelChanged;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}


}} // namespace nn::xcd
