﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/xcd/xcd_Input.h>

namespace nn { namespace xcd {

inline int16_t LittleEndianToInt16(const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    return (buffer[0] | ((buffer[1] << 8) & 0xff00));
}

inline void Int16ToLittleEndian(uint8_t* buffer, const int16_t value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    buffer[0] = static_cast<uint8_t>(value & 0xff);
    buffer[1] = static_cast<uint8_t>((value >> 8) & 0xff);
}

inline uint32_t LittleEndianToUint32(const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    return static_cast<uint32_t>(
        buffer[0] |
        (buffer[1] << 8) |
        (buffer[2] << 16) |
        (buffer[3] << 24)
        );
}

inline void Uint32ToLittleEndian(uint8_t* buffer, const uint32_t value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    buffer[0] = static_cast<uint8_t>(value & 0xff);
    buffer[1] = static_cast<uint8_t>((value >> 8) & 0xff);
    buffer[2] = static_cast<uint8_t>((value >> 16) & 0xff);
    buffer[3] = static_cast<uint8_t>((value >> 24) & 0xff);
}

inline void ParseAnalogStick(AnalogStickState* pOutValue, const uint8_t data[])
{
    pOutValue->x = data[0] | ((data[1] << 8) & 0x0f00);
    pOutValue->y = ((data[1] >> 4) &0x0f) | ((data[2] << 4) & 0x0ff0);
}

inline void ParseSensorData(SensorState* pOutState, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    pOutState->x = static_cast<int32_t>(LittleEndianToInt16(&(buffer[0])));
    pOutState->y = static_cast<int32_t>(LittleEndianToInt16(&(buffer[2])));
    pOutState->z = static_cast<int32_t>(LittleEndianToInt16(&(buffer[4])));
}

inline void ParseSensorData(SensorState* pOutState, uint8_t coef, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    pOutState->x = static_cast<int32_t>(LittleEndianToInt16(&(buffer[0]))) * coef;
    pOutState->y = static_cast<int32_t>(LittleEndianToInt16(&(buffer[2]))) * coef;
    pOutState->z = static_cast<int32_t>(LittleEndianToInt16(&(buffer[4]))) * coef;
}

inline void ParseSensorData(nn::util::Float3* pOutState, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    pOutState->x = static_cast<float>(LittleEndianToInt16(&(buffer[0])));
    pOutState->y = static_cast<float>(LittleEndianToInt16(&(buffer[2])));
    pOutState->z = static_cast<float>(LittleEndianToInt16(&(buffer[4])));
}

inline void ParseSensorData(nn::util::Float3* pOutState, uint8_t coef, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    pOutState->x = static_cast<float>(LittleEndianToInt16(&(buffer[0]))) * coef;
    pOutState->y = static_cast<float>(LittleEndianToInt16(&(buffer[2]))) * coef;
    pOutState->z = static_cast<float>(LittleEndianToInt16(&(buffer[4]))) * coef;
}

inline void GenerateFromAnalogStick(uint8_t* buffer, const AnalogStickState& value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    buffer[0] = static_cast<uint8_t>(value.x & 0xff);
    buffer[1] = static_cast<uint8_t>(((value.x >> 8) & 0x0f) | ((value.y << 4) & 0xf0));
    buffer[2] = static_cast<uint8_t>((value.y >> 4) & 0xff);
}

inline void GenerateFromSensorState(uint8_t* buffer, const SensorState& value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    Int16ToLittleEndian(&buffer[0], static_cast<const int16_t>(value.x));
    Int16ToLittleEndian(&buffer[2], static_cast<const int16_t>(value.y));
    Int16ToLittleEndian(&buffer[4], static_cast<const int16_t>(value.z));
}

}} // namespace nn::xcd
