﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_LightEventApi.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Hid.h>

#include "xcd_Bluetooth.h"
#include "xcd_ColorUtil.h"
#include "xcd_PairingManager.h"

// #define ENABLE_PRE_INSTALL_PAIRING

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR

namespace nn { namespace xcd {

namespace {

const uint8_t NxControllerSettingsInterfaceAvailableMask = 0x80;   //!< seettings 用にペアリング情報の有効性をラフに担保するためのマスク

::nn::settings::system::NxControllerSettings GetNxControllerSettings(RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    ::nn::settings::system::NxControllerSettings output = {};
    output.bd_addr          = deviceInfo.bluetooth.address;
    output.device_type      = deviceInfo.deviceType;
    std::memcpy(output.identification_code, deviceInfo.identificationCode._code, IdentificationCodeLength);
    output.mainColor        = deviceInfo.color.color[0];
    output.subColor         = deviceInfo.color.color[1];
    output.thirdColor       = deviceInfo.color.color[2];
    output.forthColor       = deviceInfo.color.color[3];
    output.design_Variation = deviceInfo.designVariation;
    output.interface_type   = deviceInfo.interfaceType | NxControllerSettingsInterfaceAvailableMask;
    return output;
}

RegisteredDeviceInfo GetRegisteredDevice(::nn::settings::system::NxControllerSettings& settings) NN_NOEXCEPT
{
    // registrationCount は呼び出し元でセットする
    RegisteredDeviceInfo output = {};
    output.bluetooth.address = settings.bd_addr;
    std::memcpy(output.identificationCode._code, settings.identification_code, IdentificationCodeLength);
    output.color.color[0]    = settings.mainColor;
    output.color.color[1]    = settings.subColor;
    output.color.color[2]    = settings.thirdColor;
    output.color.color[3]    = settings.forthColor;
    output.deviceType        = static_cast<DeviceType>(settings.device_type);
    output.subType           = 0;
    output.isIncomplete      = false;
    output.isUpdated         = false;
    output.designVariation   = settings.design_Variation;
    switch (settings.interface_type)
    {
    case InterfaceType_Bluetooth | NxControllerSettingsInterfaceAvailableMask:
        output.interfaceType = InterfaceType_Bluetooth;
        break;
    case InterfaceType_Uart | NxControllerSettingsInterfaceAvailableMask:
        output.interfaceType = InterfaceType_Uart;
        break;
    case InterfaceType_Usb | NxControllerSettingsInterfaceAvailableMask:
        output.interfaceType = InterfaceType_Usb;
        break;
    default:
        output.interfaceType = InterfaceType_Unknown;
    }
    return output;
}
}


PairingManager::PairingManager() NN_NOEXCEPT
    : m_pEvent(nullptr)
    , m_RegisteredDeviceList()
    , m_RegisteredDeviceCount(0)
    , m_RegistrationCount(0)
    , m_Mutex(false)
    , m_RegistrationPendingQueue()
    , m_PendingDeviceCount(0)
    , m_PairingUpdateEvent()
    , m_pDatabaseClearedEvent(nullptr)
{
    // 何もしない
}

PairingManager::~PairingManager() NN_NOEXCEPT
{
    // 何もしない
}

void PairingManager::Activate() NN_NOEXCEPT
{
    SetPairingUpdateEvent(&m_PairingUpdateEvent);
    GetTaskManager().RegisterPeriodicTask(this);

    m_RegistrationCount = std::numeric_limits<int64_t>::min();
    m_PendingDeviceCount = 0;

    SyncWithPairingDatabaseOnBoot();

    // ペアリング情報の更新を上位に通知
    if (m_pEvent != nullptr)
    {
        ::nn::os::SignalSystemEvent(m_pEvent);
    }
}

void PairingManager::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void PairingManager::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitSystemEvent(&m_PairingUpdateEvent))
    {
        nn::os::ClearSystemEvent(&m_PairingUpdateEvent);

        // ペアリングデータベースとの同期を行う
        // ペアリングデータベースにあってコントローラーデータベースにないデータの保管はしない
        SyncWithPairingDatabase();

        // 登録待ちになっているデバイスの追加を行う
        if (TryAddPendingDeviceToNxControllerInfo() == true)
        {
            // 本体設定を更新
            UpdateNxControllerDatabaseSettings();
        }

        // ペアリング情報の更新を上位に通知
        if (m_pEvent != nullptr)
        {
            ::nn::os::SignalSystemEvent(m_pEvent);
        }
    }
}

void PairingManager::BindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    m_pEvent = pSystemEvent;
    // 最初はシグナル状態とする
    ::nn::os::SignalSystemEvent(m_pEvent);
}

void PairingManager::UnbindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    if (m_pEvent == pSystemEvent)
    {
        m_pEvent = nullptr;
    }
}

size_t PairingManager::GetRegisteredDevices(RegisteredDeviceInfo* pOutValues, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    size_t returnCount = 0;
    for (returnCount = 0; returnCount < length && returnCount < static_cast<size_t>(m_RegisteredDeviceCount); returnCount++)
    {
        pOutValues[returnCount] = m_RegisteredDeviceList[returnCount];
        pOutValues[returnCount].color = ConvertColor(pOutValues[returnCount].color);
    }

    return returnCount;
}

void PairingManager::RegisterDevice(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    // Bluetooth のペアリング情報を登録
    if(deviceInfo.interfaceType != InterfaceType_Bluetooth)
    {
        RegisteredWiredPairingInfo(deviceInfo);
    }

    // NxController の情報を登録
    if (TryAppendNxControllerInfo(deviceInfo) == true)
    {
        // 本体設定を更新
        UpdateNxControllerDatabaseSettings();

        // ペアリング情報の更新を上位に通知
        if (m_pEvent != nullptr)
        {
            ::nn::os::SignalSystemEvent(m_pEvent);
        }
    }
    else
    {
        // リストの中にない場合はまだ Bluetooth ペアリングされていない
        // ペアリングデータベースに登録されていない場合は、ペアリングデータベースが更新されるまで、登録待ちキューに足す
        AddToPendingQueue(deviceInfo);
    }
}

bool PairingManager::GetRegisteredDeviceInfo(RegisteredDeviceInfo* pOutControllerInfo, ::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    for (auto& device : m_RegisteredDeviceList)
    {
        if (device.bluetooth.address == address &&
            device.isIncomplete == false)
        {
            *pOutControllerInfo = device;
            return true;
        }
    }

    // デバイスがみつからなかった
    return false;
}

bool PairingManager::IsDevicePaired(::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    return IsRegisteredBluetoothDevice(address);
}

void PairingManager::SyncWithPairingDatabaseOnBoot() NN_NOEXCEPT
{
    // コントローラー設定の呼び出し
    ::nn::settings::system::NxControllerSettings controllerDatabase[::nn::settings::system::NxControllerSettingsCountMax];
    auto nxDeviceCount = ::nn::settings::system::GetNxControllerSettings(controllerDatabase, NN_ARRAY_SIZE(controllerDatabase));

    // ペアリング情報の読み出し
    BluetoothDeviceInfo pairingDatabase[PairingDeviceCountMax];
    auto pairingDeviceCount = GetRegisteredBluetoothDevices(pairingDatabase, NN_ARRAY_SIZE(pairingDatabase));

    // 登録数を初期化
    m_RegisteredDeviceCount = 0;

    // ペアリングデバイス情報をマスターとしてデバイスリストを同期
    for (int pairingIndex = 0; pairingIndex < pairingDeviceCount; pairingIndex++)
    {
        bool deviceRegistered = false;

        // コントローラーの登録リストに保存されているか探索
        for (int countrollerIndex = 0; countrollerIndex < nxDeviceCount; countrollerIndex++)
        {
            if (controllerDatabase[countrollerIndex].bd_addr == pairingDatabase[pairingIndex].address)
            {
                // 登録済み
                deviceRegistered = true;
                m_RegisteredDeviceList[m_RegisteredDeviceCount] = GetRegisteredDevice(controllerDatabase[countrollerIndex]);
                break;
            }
        }

        // コントローラー(hid)データベースには存在しないがペアリング情報にだけ存在するデバイスを追加
        // 1.0.0 のシステムで登録されて最新に更新された場合にこの状態になる
        if (deviceRegistered == false)
        {
            // 空のデータをリストに追加する
            m_RegisteredDeviceList[m_RegisteredDeviceCount] = RegisteredDeviceInfo();
            m_RegisteredDeviceList[m_RegisteredDeviceCount].isIncomplete = true;
        }

        m_RegisteredDeviceList[m_RegisteredDeviceCount].bluetooth = pairingDatabase[pairingIndex];
        m_RegisteredDeviceList[m_RegisteredDeviceCount].registerationCount = m_RegistrationCount;

        m_RegisteredDeviceCount++;
    }
    // 本体設定を更新
    UpdateNxControllerDatabaseSettings();
}

void PairingManager::SyncWithPairingDatabase() NN_NOEXCEPT
{
    // ペアリング情報の読み出し
    BluetoothDeviceInfo pairingDatabase[PairingDeviceCountMax];
    auto pairingDeviceCount = GetRegisteredBluetoothDevices(pairingDatabase, NN_ARRAY_SIZE(pairingDatabase));
    auto newCount = 0;
    RegisteredDeviceInfo newList[RegisteredDeviceCountMax] = {};

    // ペアリングデバイス情報をマスターとしてデバイスリストを同期
    for (int pairingIndex = 0; pairingIndex < pairingDeviceCount; pairingIndex++)
    {
        bool deviceRegistered = false;

        // コントローラーの登録リストに保存されているか探索
        for (int countrollerIndex = 0; countrollerIndex < m_RegisteredDeviceCount; countrollerIndex++)
        {
            if (m_RegisteredDeviceList[countrollerIndex].bluetooth.address == pairingDatabase[pairingIndex].address)
            {
                // 登録済み
                deviceRegistered = true;
                newList[newCount] = m_RegisteredDeviceList[countrollerIndex];

                if (std::memcmp(m_RegisteredDeviceList[countrollerIndex].bluetooth.linkKey.key, pairingDatabase[pairingIndex].linkKey.key, LinkkeyLength) != 0)
                {
                    // Link Key だけ更新されているときは、Bluetooth 情報だけ更新しつつ新規ペアリングデバイスとして ReisrationCount をインクリメント
                    ++m_RegistrationCount;
                    newList[newCount].registerationCount = m_RegistrationCount;
                    newList[newCount].isUpdated = true;
                    newList[newCount].bluetooth = pairingDatabase[pairingIndex];
                    newList[newCount].interfaceType = InterfaceType_Bluetooth;
                }

                newCount++;
                break;
            }
        }

        // 未登録のデバイスを追加
        if (deviceRegistered == false)
        {
            // 空のデータをリストに追加する
            newList[newCount] = RegisteredDeviceInfo();
            newList[newCount].bluetooth = pairingDatabase[pairingIndex];

            // 新規デバイスとして追加

            // Bluetooth として新規に登録されたデバイスは RegistrtionCount を追加
            // 上位で新たに登録されたデバイスかどうかの判別に用いられる
            ++m_RegistrationCount;
            newList[newCount].registerationCount = m_RegistrationCount;
            newList[newCount].isUpdated = false;
            newList[newCount].isIncomplete = true;
            newList[newCount].interfaceType = InterfaceType_Bluetooth;

            newCount++;
        }
    }

    std::memcpy(m_RegisteredDeviceList, newList, sizeof(m_RegisteredDeviceList));

    // ペアリング情報が削除された場合は通知を発行する
    if (newCount == 0 &&
        m_RegisteredDeviceCount != 0 &&
        m_pDatabaseClearedEvent != nullptr)
    {
        nn::os::SignalLightEvent(m_pDatabaseClearedEvent);
    }

    m_RegisteredDeviceCount = newCount;

    // 本体設定を更新
    UpdateNxControllerDatabaseSettings();
}

void PairingManager::RegisteredWiredPairingInfo(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    // 既にコントローラーデータベースに存在するかどうかを確認
    for (int i = 0; i < m_RegisteredDeviceCount; i++)
    {
        // データがあらかじめ登録されている
        if (m_RegisteredDeviceList[i].bluetooth.address == deviceInfo.bluetooth.address)
        {
            if (std::memcmp(m_RegisteredDeviceList[i].bluetooth.linkKey.key, deviceInfo.bluetooth.linkKey.key, LinkkeyLength) == 0)
            {
                // LinkKey が一致している場合は追加登録なし
            }
            else
            {
                // LinkKey を更新して再登録
                m_RegisteredDeviceList[i].bluetooth = deviceInfo.bluetooth;
                // 再登録(更新) された
                m_RegisteredDeviceList[i].isUpdated = true;
                // RegistrationCount はインクリメントする
                ++m_RegistrationCount;
                m_RegisteredDeviceList[i].registerationCount = m_RegistrationCount;
                // Bluetooth のペアリング情報の登録処理
                RegisterBluetoothDevice(deviceInfo.bluetooth);
            }
            return;
        }
    }
    // デバイスが登録されていない場合は、Bluetooth のペアリング情報の登録処理
    RegisterBluetoothDevice(deviceInfo.bluetooth);
}

bool PairingManager::TryAppendNxControllerInfo(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    // コントローラーデータベースに登録されている Bluetooth の情報を探索
    for (int i = 0; i < m_RegisteredDeviceCount; i++)
    {
        // データがあらかじめ登録されている
        if (m_RegisteredDeviceList[i].bluetooth.address == deviceInfo.bluetooth.address)
        {
            m_RegisteredDeviceList[i].deviceType = deviceInfo.deviceType;
            m_RegisteredDeviceList[i].identificationCode = deviceInfo.identificationCode;
            m_RegisteredDeviceList[i].interfaceType = deviceInfo.interfaceType;
            m_RegisteredDeviceList[i].color = deviceInfo.color;
            if (m_RegisteredDeviceList[i].isIncomplete == true)
            {
                // もともと不完全なデータであった場合は、RegistrationCount を更新しつつフラグを落とす
                m_RegisteredDeviceList[i].isIncomplete = false;
            }
            return true;
        }
    }

    return false;
}

bool PairingManager::TryAddPendingDeviceToNxControllerInfo() NN_NOEXCEPT
{
    int queueIndex = 0;
    bool isRegistered = false;
    for (queueIndex = 0; queueIndex < m_PendingDeviceCount; )
    {
        // 登録待ちキューのデバイスをデバイスリストに追加
        if (TryAppendNxControllerInfo(m_RegistrationPendingQueue[queueIndex]) == true)
        {
            // 登録待ちキューからデバイスを削除
            RemoveFromPendingQueue(queueIndex);
            isRegistered = true;
        }
        // 登録待ちキューに登録されているデバイスがペアリングされていない場合はスキップ
        else
        {
            ++queueIndex;
        }
    }
    return isRegistered;
}

void PairingManager::UpdateNxControllerDatabaseSettings() NN_NOEXCEPT
{
    ::nn::settings::system::NxControllerSettings newSettings[::nn::settings::system::NxControllerSettingsCountMax];
    int settingsCount = 0;
    // settings を更新
    for (int i = 0; i < m_RegisteredDeviceCount; i++)
    {
        // isIncomplete == true の場合は、NxControllerSettings が存在しないのでスキップ
        if (m_RegisteredDeviceList[i].isIncomplete == false)
        {
            newSettings[settingsCount] = GetNxControllerSettings(m_RegisteredDeviceList[i]);
            settingsCount++;
        }
    }
#ifndef ENABLE_PRE_INSTALL_PAIRING
    ::nn::settings::system::SetNxControllerSettings(newSettings, settingsCount);
#endif
}

bool PairingManager::IsWiredPairingRequired(InterfaceType interfaceForPairing, DeviceType deviceType) NN_NOEXCEPT
{
    // TODO: InterfaceType 以外も判定に使えるようにするべき
    switch (deviceType)
    {
        case DeviceType_MiyabiLeft:
        case DeviceType_MiyabiRight:
            return false;
        default:
            break;
    }

    switch (interfaceForPairing)
    {
        case InterfaceType_Uart:
        case InterfaceType_Usb:
            return true;
        default:
            return false;
    }
}

void PairingManager::SetDatabaseClearedEvent(::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pDatabaseClearedEvent = pEvent;
}

void PairingManager::AddToPendingQueue(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    // 既にキューに登録されていないかをチェック
    for (int i = 0; i < m_PendingDeviceCount; i++)
    {
        if (m_RegistrationPendingQueue[i].bluetooth.address == deviceInfo.bluetooth.address)
        {
            RemoveFromPendingQueue(i);
            break;
        }
    }

    // キューがいっぱいの場合は頭のデバイスを削除する
    if (m_PendingDeviceCount == NN_ARRAY_SIZE(m_RegistrationPendingQueue))
    {
        // 先頭のデバイスを削除
        RemoveFromPendingQueue(0);
    }

    // リストの末尾に追加
    if (m_PendingDeviceCount < NN_ARRAY_SIZE(m_RegistrationPendingQueue))
    {
        m_RegistrationPendingQueue[m_PendingDeviceCount] = deviceInfo;
        ++m_PendingDeviceCount;
    }
}

void PairingManager::RemoveFromPendingQueue(int index) NN_NOEXCEPT
{
    if (m_PendingDeviceCount < index)
    {
        return;
    }

    // 前づめする
    for (int i = index; i < m_PendingDeviceCount - 1; i++)
    {
        m_RegistrationPendingQueue[i] = m_RegistrationPendingQueue[i + 1];
    }
    m_RegistrationPendingQueue[m_PendingDeviceCount - 1] = RegisteredDeviceInfo();
    --m_PendingDeviceCount;
}

}} // namespace nn::xcd
